/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	The program page.
 *
 *	by Tony Sideris	(11:27PM Apr 28, 2002)
 *================================================*/
#include "arson.h"

#include <qbuttongroup.h>
#include <qlineedit.h>
#include <qcombobox.h>
#include <qlistbox.h>
#include <qdom.h>
#include <qlayout.h>

#include <kurllabel.h>
#include <klocale.h>
#include <krun.h>

#include "programpage.h"
#include "programdlg.h"
#include "listwnd.h"

/*========================================================*/

/*	This XML content defines the content
 *	of the 'Program Preference' controls.
 *	the descriptions are filled in with
 *	sprintf() at runtime so they can
 *	be i18n'd easier.
 *
 *	The contents and order must match
 *	the PROGGRP_ and *SEQ_ enums.
 */
const char *program_sequence =
"<?xml version='1.0'?><programs>
	<group desc='%1'>
		<program name='cdrdao'/>
		<program name='cdrecord'/>
	</group>

	<group desc='%2'>
		<program name='cdrecord'/>
		<program name='cdrdao'/>
	</group>

	<group desc='%3'>
		<program name='cdrdao'/>
		<program name='readcd -> cdrecord'/>
	</group>

	<group desc='%4'>
		<program name='bladeenc'/>
		<program name='LAME'/>
	</group>

	<group desc='%5'>
		<program name='cdrecord'/>
		<program name='cdrdao'/>
	</group>

	<group desc='%6'>
		<program name='cdda2wav'/>
		<program name='cdparanoia'/>
	</group>

	<group desc='%7'>
		<program name='cdrecord'/>
		<program name='cdrdao'/>
	</group>
</programs>";

ArsonProgramPage::ArsonProgramPage (ArsonConfig &config, ArsonConfigDlg *pd, QWidget *parent)
	: ArsonProgramPageBase(parent), ArsonConfigPage(config),
	m_current_group(-1),
	m_seq(config.seq())
{
	KURLLabel *pl;
	static ArsonListHeader hdrs[] = {
		ArsonListHeader(i18n("Program Name"), 25),
		ArsonListHeader(i18n("Program Path"), 45),
		ArsonListHeader(i18n("Extra Parameters"), 30),
	};

	//	Initialize and fill in the program list
	program_list->setListHeaders(hdrs, ARRSIZE(hdrs));
	program_list->setSorting(-1);
	program_list->setAllColumnsShowFocus(true);
	program_list->setSelectionMode(QListView::Single);
	paths->setText(cfg().extraProgramPaths());
	fillProgramList();

	//	cdrdao Drivers
	fillDriverList(writer_drv);
	fillDriverList(source_drv);
	writer_drv->setEditText(cfg().driver());
	source_drv->setEditText(cfg().srcDriver());

	//	Add a helpful hyperlink for setting driver
	pl = new KURLLabel(QString("http://cdrdao.sourceforge.net/drives.html"),
		i18n("Need Help? Heres a driver list..."), driver_group);
	driver_groupLayout->addWidget(pl);
	pl->setGlow(true);

	setupProgramPrefs();

	QObject::connect(program_list, SIGNAL(doubleClicked(QListViewItem*)),
		this, SLOT(program_dblclk(QListViewItem*)));
	QObject::connect(pl, SIGNAL(leftClickedURL(const QString &)),
		this, SLOT(url_clicked(const QString &)));
}

/*==========================================================*/

void ArsonProgramPage::setupProgramPrefs (void)
{
	QString content(program_sequence);
	QDomDocument doc;
	QDomElement root;
	int index = 0;

	doc.setContent(content
		.arg(i18n("Audio CD Writer"))		//	Program group descriptions
		.arg(i18n("ISO Writer"))
		.arg(i18n("CD Copy"))
		.arg(i18n("MP3 Encoder"))
		.arg(i18n("Device Scanner"))
		.arg(i18n("CD Ripper"))
		.arg(i18n("CDRW Blanking"))
		);

	root = doc.documentElement();

	//	Iterate through the program groups defined in the XML content
	for (QDomNode node = root.firstChild(); !node.isNull(); node = node.nextSibling(), ++index)
	{
		QStringList pl;
		QDomElement el = node.toElement();

		Assert(!el.isNull() && el.tagName() == "group");

		//	Iterate through the programs defined in the current group
		for (QDomNode in = el.firstChild(); !in.isNull(); in = in.nextSibling())
		{
			QDomElement il = in.toElement();

			Assert(!il.isNull() && il.tagName() == "program");

			pl.append(il.attribute("name"));
		}

		//	Create the new program group
		if (!pl.isEmpty())
		{
			prog_group->insertItem(el.attribute("desc"));
			m_groups.append(pl);
		}

		m_seq.setSeq(index, cfg().seq().seq(index));
	}

	Assert(m_groups.count() > 0);
	
	prog_group->setCurrentItem(0);
	program_group_changed(0);
}

/*========================================================*/

void ArsonProgramPage::fillProgramList (void)
{
	QListViewItem *pLast = NULL;
	const ArsonConfig::Program *papp;

	program_list->clear();

	for (int index = 0; (papp = cfg().program(index)); index++)
		pLast = new QListViewItem(program_list, pLast,
			ArsonConfig::programName(index),
			papp->m_program,
			papp->m_params);
}

/*==========================================================*/

void ArsonProgramPage::program_group_changed (int cur)
{
	int index;
	const int count = m_groups[cur].count();

	//	Save the current selection (but not first time)
	saveCurrentGroup();

	//	Reset the listbox contents
	prog_list->clear();

	//	Fill the listbox
	for (index = 0; index < count; ++index)
		prog_list->insertItem(m_groups[cur][index]);

	//	Select the last selected item
	prog_list->setCurrentItem(
		m_seq.seq((m_current_group = cur)));
}

void ArsonProgramPage::selected_program_changed (int sel)
{
	saveCurrentGroup();
}

void ArsonProgramPage::saveCurrentGroup (void)
{
	if (m_current_group != -1)
		m_seq.setSeq(m_current_group, prog_list->currentItem());
}

/*========================================================*/

void ArsonProgramPage::url_clicked (const QString &url)
{
	KRun::runURL(url, "text/html");
}

/*==========================================================*/

void ArsonProgramPage::auto_detect_clicked()
{
	//	Rescan and refill the program list.
	cfg().autoDetectPrograms(paths->text());
	fillProgramList();
}

/*==========================================================*/

void ArsonProgramPage::program_dblclk (QListViewItem *pItem)
{
	const int index = program_list->itemIndex(pItem);
	ArsonConfig::Program *papp = cfg().program(index);

	if (ArsonProgramDlg(this, index, *papp).exec() == QDialog::Accepted)
		fillProgramList();
}

void ArsonProgramPage::modify_clicked (void)
{
	Trace("modify clicked\n");

	if (QListViewItem *pi = program_list->currentItem())
		program_dblclk(pi);
}

/*==========================================================*/

void ArsonProgramPage::fillDriverList (QComboBox *ptr)
{
	const char *drivers[] = {
		"plextor",
		"plextor-scan",
		"cdd2600",
		"generic-mmc",
		"generic-mmc-raw",
		"ricoh-mp6200",
		"sony-cdu920",
		"sony-cdu948",
		"taiyo-yuden",
		"teac-cdr55",
		"yamaha-cdr10x",
	};

	for (int index = 0; index < ARRSIZE(drivers); ++index)
		ptr->insertItem(drivers[index]);

	ptr->listBox()->sort();
}

/*==========================================================*/

void ArsonProgramPage::on_accept (void)
{
	saveCurrentGroup();
	cfg().seq() = m_seq;

	cfg().setSrcDriver(source_drv->currentText().stripWhiteSpace());
	cfg().setDriver(writer_drv->currentText().stripWhiteSpace());
	cfg().setExtraProgramPaths(paths->text());
}

/*========================================================*/
