/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	The ripper page.
 *
 *	by Tony Sideris	(11:24PM Apr 28, 2002)
 *================================================*/
#include "arson.h"

#include <qradiobutton.h>
#include <qpushbutton.h>
#include <qtoolbutton.h>
#include <qpopupmenu.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qspinbox.h>
#include <qslider.h>

#include <kfiledialog.h>
#include <klocale.h>

#include "ripperpage.h"
#include "listwnd.h"
#include "lookup.h"
#include "konfig.h"
#include "cdripper.h"
#include "lookup.h"

/*========================================================*/
/*	A listview item which contains a lookup item
 *========================================================*/

class arsonLookupListItem : public QListViewItem
{
public:
	arsonLookupListItem (QListView *pl, QListViewItem *after, ArsonLookup *ptr)
		: QListViewItem(pl, after), m_ptr(ptr)
	{
		setText(0, ptr->typeDesc());
		setText(1, ptr->display());
	}

	virtual ~arsonLookupListItem (void) { delete m_ptr; }

	ArsonLookup *lookup (void) { return m_ptr; }
	void setLookup (ArsonLookup *ptr) { m_ptr = ptr; }

private:
	ArsonLookup *m_ptr;
};

/*========================================================*/

namespace arson
{
	struct popupBtn
	{
		const char *code;
		const QString desc;
	};

	int popup (popupBtn *items, QWidget *btn)
	{
		int index;
		QPopupMenu menu;
		const QPoint bl (0, btn->geometry().height() + 1);	//	Bottom left of button

		for (index = 0; items[index].code; ++index)
			menu.insertItem(i18n("(%1) %2")
				.arg(items[index].code)
				.arg(items[index].desc),
				index);

		return menu.exec(btn->mapToGlobal(bl));
	}
};

/*========================================================*/

ArsonRipperPage::ArsonRipperPage (ArsonConfig &config, ArsonConfigDlg *pd, QWidget *parent)
	: ArsonRipperPageBase(parent), ArsonConfigPage(config)
{
	arsonLookupListItem *last = NULL;
	const ArsonLookupOrder &lp = cfg().ripper().lookups();
	static ArsonListHeader hdrs[] = {
		ArsonListHeader(i18n("Type"), 35),
		ArsonListHeader(i18n("Source"), 65),
	};

	updateDevs(cfg().ripper().srcdev(), cfg().devices());

	setupQuality();
	
	lookup->toggle();
	if (!cfg().ripper().is(ArsonConfig::RipperCfg::ripperCdLookup))
		lookup->toggle();

	handleCheckboxes(false);

	default_outdir->setText(cfg().ripper().outdir());
	RIPPERFORMAT::fillComboBox(default_format);

	cdi_format->insertStringList(cfg().ripper().recentCdiFormats());
	cdi_format->setEditText(cfg().ripper().cdiFormat());

	comment->setText(cfg().ripper().defaultComment());
	
	lookup_sources->setSelectionMode(QListView::Single);
	lookup_sources->setListHeaders(hdrs, ARRSIZE(hdrs));
	lookup_sources->setItemsMovable(true);
	
	for (int index = 0; index < lp.count(); ++index)
		last = new arsonLookupListItem(lookup_sources,
			last, lp.lookup(index)->clone());
	
	QObject::connect(confdev, SIGNAL(clicked()),
		pd, SLOT(configure_devices()));
	QObject::connect(lookup_sources,
		SIGNAL(selectionChanged(QListViewItem*)),
		this, SLOT(lookup_selected(QListViewItem*)));
	QObject::connect(up_src, SIGNAL(clicked()),
		lookup_sources, SLOT(moveItemUp()));
	QObject::connect(down_src, SIGNAL(clicked()),
		lookup_sources, SLOT(moveItemDown()));
	QObject::connect(up_src, SIGNAL(clicked()),
		this, SLOT(lookup_selected()));
	QObject::connect(down_src, SIGNAL(clicked()),
		this, SLOT(lookup_selected()));

	lookup_selected();
}

/*========================================================*/

void ArsonRipperPage::setupQuality (void)
{
	int index;
	const char *brs[] = {
		"32", "40", "48", "56", "64", "80", "96", "112",
		"128", "160", "192", "224", "256", "320"
	};
	const QString chans[] = {
		i18n("Stereo"),
		i18n("Mono"),
		i18n("Left Mono (bladeenc)"),
		i18n("Right Mono (bladeenc)"),
		i18n("Joint Stereo (LAME)"),
	};

	for (index = 0; index < ARRSIZE(brs); ++index)
		bitrate->insertItem(brs[index]);

	for (index = 0; index < ARRSIZE(chans); ++index)
		channels->insertItem(chans[index]);

	for (index = 0; index < ArsonEncoderOpts::_encoder_max; ++index)
		encoder->insertItem(i18n(ArsonEncoderOpts::encoder_names[index]));
	
	encoder->setCurrentItem(0);

	preset->insertStringList(m_presets.names());
	default_vbr();
	
	preset_changed(cfg().ripper().quality());
}

/*==========================================================*/

void ArsonRipperPage::browse_clicked (void)
{
	const QString dir = KFileDialog::getExistingDirectory();

	if (dir != QString::null)
		default_outdir->setText(dir);
}

/*========================================================*/

void ArsonRipperPage::add_lookup (void)
{
	ArsonLookupDlg dlg (this);

	if (dlg.exec() == QDialog::Accepted)
		new arsonLookupListItem(lookup_sources,
			lookup_sources->lastChild(), dlg.result());
}

void ArsonRipperPage::del_lookup (void)
{
	delete (arsonLookupListItem *) lookup_sources->selectedItem();
}

/*==========================================================*/

void ArsonRipperPage::updateDevs (const char *dev, const ArsonDeviceList &dl)
{
	QCString sel (dev);

	if (source_device->count() > 0 && !dev)
		sel = m_devs.fromList(source_device);

	m_devs = dl.uiList(false);

	m_devs.fillList(source_device,
		sel.isEmpty() ? NULL : sel.data());
}

void ArsonRipperPage::on_dev_changed (const ArsonDeviceList &dl)
{
	updateDevs(NULL, dl);
}

/*========================================================*/

void ArsonRipperPage::handleCheckboxes (bool save)
{
	struct {
		QCheckBox *pb;
		uint mask;
	}	checks[] = {
		{ email, ArsonConfig::RipperCfg::ripperEmailPrompt },
		{ cdtext, ArsonConfig::RipperCfg::ripperOverCdtext },
		{ lookup, ArsonConfig::RipperCfg::ripperCdLookup },
	};

	for (int index = 0; index < ARRSIZE(checks); ++index)
	{
		if (save)
			cfg().ripper().fromBool(checks[index].mask, checks[index].pb->isChecked());
		else
			checks[index].pb->setChecked(cfg().ripper().is(checks[index].mask));
	}
}

/*========================================================*/

void ArsonRipperPage::lookup_selected (QListViewItem *pi)
{
	const uint mask = lookup_sources->itemListStates();
#define TOBOOL(f)		((mask & ArsonListWndBase::f) != 0)

	del_src->setEnabled(TOBOOL(itemDelete));
	up_src->setEnabled(TOBOOL(itemUp));
	down_src->setEnabled(TOBOOL(itemDown));
}

void ArsonRipperPage::lookup_selected (void)
{
	lookup_selected(NULL);
}

/*========================================================*/

void ArsonRipperPage::default_vbr (void)
{
	vbr_quality->setValue(5);
}

void ArsonRipperPage::preset_changed (const QString &name)
{
	if (const ArsonEncoderOpts *po = m_presets.opts(name))
	{
		if (preset->currentText() != name)
			if (QListBoxItem *pi = preset->listBox()->findItem(name))
				preset->setCurrentItem(preset->listBox()->index(pi));

		setOpts(*po);
	}
}

void ArsonRipperPage::add_preset (void)
{
	const QString name = preset->currentText();
	
	if (!name.isEmpty() && !m_presets.opts(name))
	{
		ArsonEncoderOpts opts;
		getOpts(opts);

		m_presets.addPreset(name, opts);
		preset->insertItem(name);
		preset->listBox()->sort();
	}
}

void ArsonRipperPage::del_preset (void)
{
	m_presets.delPreset(preset->currentText());
	preset->removeItem(preset->currentItem());

	const QString ct = preset->currentText();

	if (ct.isEmpty())
		preset->setCurrentItem(0);
	
	preset_changed(preset->currentText());
}

void ArsonRipperPage::upd_preset (void)
{
	const QString name = preset->currentText();

	if (m_presets.opts(name))
	{
		ArsonEncoderOpts opts;
		getOpts(opts);

		m_presets.addPreset(name, opts);
	}
}

/*========================================================*/

void ArsonRipperPage::getOpts (ArsonEncoderOpts &opts)
{
	if (manual->isChecked())
		opts.setManual(encoder->currentItem(), switches->text());
	else
	{
		uint flags = 0;

		opts.setBitrate(bitrate->currentText().toInt());
		opts.setChannels(channels->currentItem());
		opts.setVbr(vbr->isChecked() ? (9 - vbr_quality->value()) : -1);

#define FROMCHX(ctrl,flag)		if ((ctrl)->isChecked()) flags |= ArsonEncoderOpts::flag

		FROMCHX(crc, flagCRC);
		FROMCHX(copyright, flagCopyright);
		FROMCHX(copy, flagCopy);
		opts.setFlags(flags);
	}
}

void ArsonRipperPage::setOpts (const ArsonEncoderOpts &opts)
{
	const bool man = (opts.manualEncoder() != ArsonEncoderOpts::encoderNone);

	manual->setChecked(man);

	if (man)
	{
		encoder->setCurrentItem(opts.manualEncoder());
		switches->setText(opts.manualSwitches());
	}
	else
	{
		switches->setText(QString::null);

		if (QListBoxItem *pi = bitrate->listBox()->findItem(QString::number(opts.bitrate())))
			bitrate->setCurrentItem(bitrate->listBox()->index(pi));

		channels->setCurrentItem(opts.channels());

		cbr->setChecked(opts.vbr() == -1);
		vbr->setChecked(opts.vbr() != -1);

		if (opts.vbr() != -1)
			vbr_quality->setValue(9 - opts.vbr());

#define SETCHX(ctrl,flag)		(ctrl)->setChecked(opts.is(ArsonEncoderOpts::flag))

		SETCHX(crc, flagCRC);
		SETCHX(copyright, flagCopyright);
		SETCHX(copy, flagCopy);
	}
}

/*========================================================*/

void ArsonRipperPage::encoder_changed (int enc)
{
	encoder_help_btn->setEnabled(
		enc > 0 && enc < ArsonEncoderOpts::_encoder_max
		&& manual->isChecked());
}

void ArsonRipperPage::encoder_help (void)
{
	const int sel = encoder->currentItem();
	const int progs[ArsonEncoderOpts::_encoder_max] = {
		-1,
		ArsonConfig::PROGRAM_LAME,
		ArsonConfig::PROGRAM_BLADEENC,
		ArsonConfig::PROGRAM_OGGENC,
#ifdef FLAC
		ArsonConfig::PROGRAM_FLAC,
#endif
	};

	if (progs[sel] != -1)
	{
		ArsonProgramHelp prog (progs[sel]);

		if (prog.valid())
			prog.show(this);
	}
}

/*========================================================*/

void ArsonRipperPage::format_code_popup (void)
{
	int index;
	arson::popupBtn codes[] = {
		{ "%a", i18n("&Artist") },
		{ "%c", i18n("&CD Title") },
		{ "%t", i18n("&Track Title") },
		{ "%n", i18n("Track &Number (1, 2, etc.)") },
		{ "%N", i18n("T&rack Number (01, 02, etc.)") },
		{ "%z", i18n("Trac&k Number (0, 1, etc.)") },
		{ "%Z", i18n("Track Nu&mber (00, 01, etc.)") },
		{ NULL, QString() }
	};

	if ((index = arson::popup(codes, popup_button)) != -1)
		cdi_format->lineEdit()->insert(codes[index].code);
}

/*========================================================*/

void ArsonRipperPage::comment_popup (void)
{
	int index;
	arson::popupBtn codes[] = {
		{ "%d", i18n("&Date") },
		{ "%t", i18n("&Time") },
		{ "%v", i18n("&Version") },
		{ NULL, QString() },
	};

	if ((index = arson::popup(codes, comment_button)) != -1)
		comment->insert(codes[index].code);
}

/*==========================================================*/

void ArsonRipperPage::on_accept (void)
{
	ArsonLookupOrder order;

	for (arsonLookupListItem *ptr = (arsonLookupListItem *) lookup_sources->firstChild();
		 ptr; ptr = (arsonLookupListItem *) ptr->nextSibling())
		order.addLookup(ptr->lookup()->clone());

	cfg().ripper().setLookups(order);

	cfg().ripper().setSrcdev(m_devs.fromList(source_device));
	cfg().ripper().setFormat(default_format->currentItem());
	cfg().ripper().setOutdir(default_outdir->text());
	cfg().ripper().setCdiFormat(cdi_format->currentText());
	cfg().ripper().setDefaultComment(comment->text());

	upd_preset();	//	Save current changes
	cfg().ripper().setQuality(preset->currentText());	
	m_presets.save();

	handleCheckboxes(true);
}

/*========================================================*/
/*	Lookup Dialog
 *========================================================*/

#define FREEDB_ADDR		"freedb.freedb.org"
#define FREEDB_PORT		80

#define CDINDEX_ADDR	"www.freeamp.org"
#define CDINDEX_PORT	80

ArsonLookupDlg::ArsonLookupDlg (QWidget *parent)
	: ArsonLookupDlgBase(parent, NULL, true)
{
	for (int index = 0; index < ArsonLookup::_lookup_max; ++index)
		types->insertItem(ArsonLookup::typeDesc(index));

	fdb_addr->setText(FREEDB_ADDR);
	fdb_port->setValue(FREEDB_PORT);

	cdi_addr->setText(CDINDEX_ADDR);
	cdi_port->setValue(CDINDEX_PORT);
	
	type_changed(0);
	adjustSize();
}

/*========================================================*/

void ArsonLookupDlg::type_changed (int type)
{
	QGroupBox *grps[] = {
		fdb_http,
		fdb_local,
		cdindex,
	};

	for (int index = 0; index < ARRSIZE(grps); ++index)
	{
		if (index != type)
			grps[index]->hide();
		else
			grps[index]->show();
	}

	types->setCurrentItem(type);
}

/*========================================================*/

void ArsonLookupDlg::browse_clicked (void)
{
	const QString path = KFileDialog::getExistingDirectory();

	if (path != QString::null)
		fdb_path->setText(path);
}

/*========================================================*/

ArsonLookup *ArsonLookupDlg::result (void) const
{
	switch (types->currentItem())
	{
	case ArsonLookup::freedbHttp:
		return new ArsonFreedbHttpLookup(
			fdb_addr->text(),
			fdb_port->cleanText().toShort());

	case ArsonLookup::freedbLocal:
		return new ArsonFreedbLocalLookup(
			fdb_path->text(), autosave->isChecked());

	case ArsonLookup::cdindex:
		return new ArsonCdindexLookup(
			cdi_addr->text(),
			cdi_port->cleanText().toShort());
	}

	Assert(false);
	return NULL;
}

/*========================================================*/
