/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Base socket classes and related utilities.
 *
 *	by Tony Sideris	(03:55PM Feb 10, 2002)
 *================================================*/
#include "arson.h"

#include <qfile.h>

#include <kstddirs.h>
#include <klocale.h>
#include <kglobal.h>

#include "socket.h"

#define CHECK_WRITE(buf,n)	if(writeBlock((buf),(n)) == -1) Trace("writeBlock failed (%s,%d)\n", __FILE__, __LINE__);

/*========================================================*/

ArsonSocket::ArsonSocket (const QString &host, short port)
	: QSocket(),
	m_errCode(0)
{
	QObject::connect(this, SIGNAL(connectionClosed()),
		this, SLOT(slotClosed()));

	QObject::connect(this, SIGNAL(connected()),
		this, SLOT(slotConnected()));

	QObject::connect(this, SIGNAL(readyRead()),
		this, SLOT(slotRead()));

	QObject::connect(this, SIGNAL(error(int)),
		this, SLOT(slotError(int)));

	if (host != QString::null)
	{
		Trace("connecting to %s:%d...\n", host.ascii(), port);
		connectToHost(host, port);
	}
}

/*========================================================*/

void ArsonSocket::slotClosed (void)
{
	done();
}

/*========================================================*/

void ArsonSocket::slotRead (void)
{
	const int navail = bytesAvailable();

	if (navail > 0)
	{
		char *ptr = new char[navail];
		int nread;

		if ((nread = readBlock(ptr, navail)) != -1)
			dispatch(ptr, nread);

		delete[] ptr;
	}
}

/*========================================================*/

void ArsonSocket::slotError (int errCode)
{
	m_errCode = errCode ? errCode : -1;
	done();
}

/*========================================================*/

void ArsonSocket::slotConnected (void)
{
	Trace("connection complete\n");
}

void ArsonSocket::dispatch (const char *data, int nbytes)
{
	//	Nothing...
}

void ArsonSocket::done (void)
{
	emit complete(this);
}

/*========================================================*/
/*	This socket class retreives data to a temp file
 *========================================================*/

ArsonRetrSocket::ArsonRetrSocket (const QString &host, short port)
	: ArsonSocket(host, port),
	m_tempFile("socket", "tmp")
{
	if (m_tempFile.status())
		throw ArsonError(
			i18n("Failed to create temp file for socket"));

	m_tempFile.setAutoDelete(true);
}

/*========================================================*/

void ArsonRetrSocket::dispatch (const char *data, int nbytes)
{
	m_tempFile.file()->writeBlock(data, nbytes);
}

void ArsonRetrSocket::done (void)
{
	m_tempFile.close();
	ArsonSocket::done();
}

/*========================================================*/
/*	A socket class which sends a file
 *========================================================*/

ArsonSendSocket::ArsonSendSocket (const QString &host, short port)
	: ArsonSocket(QString::null, 0),
	m_blockSize(0),
	m_host(host),
	m_port(port)
{
	QObject::connect(this, SIGNAL(bytesWritten(int)),
		this, SLOT(slotBytesWritten(int)));
}

/*========================================================*/

bool ArsonSendSocket::sendFile (const char *filename, uint blockSize)
{
	m_file.setName(filename);
	m_buf.resize((m_blockSize = blockSize));

	if (m_file.open(IO_ReadOnly | IO_Raw))
	{
		connectToHost(m_host, m_port);
		return true;
	}

	Trace("ArsonSendSocket: failed to open file %s\n", filename);
	return false;
}

/*========================================================*/

void ArsonSendSocket::slotConnected (void)
{
	refillBuffer();
}

/*========================================================*/

void ArsonSendSocket::slotBytesWritten (int bytes)
{
	if (!bytesToWrite())
		refillBuffer();
}

/*========================================================*/

void ArsonSendSocket::refillBuffer (void)
{
	Assert(m_blockSize > 0);
	int nread;

	if ((nread = m_file.readBlock(m_buf.data(), m_blockSize)) > 0)
	{
		fwrite(m_buf.data(), 1, nread, stdout);
		CHECK_WRITE(m_buf.data(), nread);
	}
	else
		sendComplete();
}

/*========================================================*/

void ArsonSendSocket::sendComplete (void)
{
	done();
}

/*========================================================*/
/*	A socket to perform an HTTP GET
 *========================================================*/

ArsonHttpGetSocket::ArsonHttpGetSocket (const QString &host, short port)
	: ArsonRetrSocket(host, port)
{
	//	Nothing...
}

/*========================================================*/

void ArsonHttpGetSocket::slotConnected (void)
{
	QString req ("GET "), tmp = request();

	req.append(path());

	if (tmp != QString::null)
		req.append("?").append(tmp);

	req.append(" HTTP/1.0");

	CHECK_WRITE(req.ascii(), req.length());
	CHECK_WRITE("\r\n\r\n", 4);

	Trace("Sent request: %s\n", req.ascii());	
	ArsonRetrSocket::slotConnected();
}

/*========================================================*/
/*	Socket to perform an HTTP POST
 *========================================================*/

ArsonHttpPostSocket::ArsonHttpPostSocket (const QString &host, short port)
	: ArsonSendSocket(host, port)
{
	//	Nothing...
}

/*========================================================*/

void ArsonHttpPostSocket::slotConnected (void)
{
	QString req ("POST ");
	req.append(path()).append(" HTTP/1.0\r\n\r\n");

	Trace("Sending POST: %s", req.ascii());
	CHECK_WRITE(req.ascii(), req.length());
	ArsonSendSocket::slotConnected();
}

/*========================================================*/

void ArsonHttpPostSocket::slotRead (void)
{
	if (canReadLine())
	{
		QString line = readLine();
		Trace("Http POST Response: %s\n", line.ascii());

		if (line.left(4) == "HTTP")
			line = line.data() + line.find(' ') + 1;

		response(line.left(3).toInt(),
			line.mid(3, line.length() - 5));

		done();
	}
}

/*========================================================*/

void ArsonHttpPostSocket::sendComplete (void)
{
	//	Wait for response...
}

/*========================================================*/
/*	Text stream to skip HTTP headers and auto-close FILE*
 *========================================================*/

ArsonHttpStream::ArsonHttpStream (FILE *file)
	: QTextStream(file, IO_ReadOnly), m_file(file)
{
	QString line;

	//	Skip the HTTP headers
	do line = readLine().stripWhiteSpace();
	while (!line.isEmpty());
}

/*========================================================*/

ArsonHttpStream::~ArsonHttpStream (void)
{
	fclose(m_file);
}

/*========================================================*/
/*	An HTTP command
 *========================================================*/

ArsonHttpCommand::ArsonHttpCommand (const QString &cmd)
{
	if (cmd != QString::null)
		setCommand(cmd);
}

/*========================================================*/

ArsonHttpCommand &ArsonHttpCommand::operator<< (const ArsonHttpCommand &rhs)
{
	m_command.append("&").append(rhs.command());
	return *this;
}

ArsonHttpCommand &ArsonHttpCommand::operator<< (const QString &text)
{
	const int length = m_command.length();

	if (length > 0 && m_command[length - 1] != '=')
		m_command.append("+");

	m_command.append(text);
	
	return *this;
}

ArsonHttpCommand &ArsonHttpCommand::operator<< (uint n)
{
	return ((*this) << QString::number(n));
}

ArsonHttpCommand &ArsonHttpCommand::operator<< (int n)
{
	return ((*this) << QString::number(n));
}

/*========================================================*/

void ArsonHttpCommand::setCommand (const QString &cmd)
{
	m_command = cmd + "=";
}

/*========================================================*/
/*	A wait dialog containing a socket connection
 *========================================================*/

ArsonSocketWaitDlg::ArsonSocketWaitDlg (void)
	: ArsonWaitDlg(kapp->mainWidget()),
	m_pSocket(NULL)
{
	//	Nothing...
}

ArsonSocketWaitDlg::~ArsonSocketWaitDlg (void)
{
	delete m_pSocket;
}

/*========================================================*/

void ArsonSocketWaitDlg::beginConnection (const char *slot)
{
	delete m_pSocket;
	m_pSocket = NULL;

	if ((m_pSocket = createSocket()))
	{
		if (!slot)
			slot = SLOT(slotCompleted(ArsonSocket*));

		QObject::connect(m_pSocket,
			SIGNAL(complete(ArsonSocket*)),
			this, slot);
	}
}

/*========================================================*/

int ArsonSocketWaitDlg::exec (void)
{
	beginConnection();

	if (!m_pSocket)
		return Rejected;
	
	return ArsonWaitDlg::exec();
}

/*========================================================*/

void ArsonSocketWaitDlg::slotCompleted (ArsonSocket *pThis)
{
	accept();
}

/*========================================================*/
