/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Read CD class implemenatations
 *
 *	by Tony Sideris	(08:21PM Sep 08, 2001)
 *================================================*/
#include <qcheckbox.h>
#include <qlabel.h>

#include <kfiledialog.h>
#include <klocale.h>

#include "progressdlg.h"
#include "infprogbar.h"
#include "tools.h"
#include "process.h"
#include "konfig.h"
#include "cdcopy.h"
#include "isofs.h"

/*========================================================*/

ArsonReadCdMgr::ArsonReadCdMgr (ArsonProcessUI *pUI, const char *outfile)
	: ArsonProcessMgr(pUI),
	m_strOutFile(outfile)
{
}

/*========================================================*/

void ArsonReadCdMgr::begin (const ArsonProcessOpts &opts)
{
	ArsonProcessMgr::begin(opts);

	ui()->setText(
		i18n("Reading image from device %1 into file %2 ...")
		.arg(ACONFIG.device())
		.arg(m_strOutFile));
	
	try { setProcess(new ArsonReadCdProcess(this, m_strOutFile)); }
	catch (ArsonError &err) {
		err.report();
	}
}

/*========================================================*/

class readcdDlg : public ArsonSimpleProgressDlg
{
public:
	readcdDlg (const char *outfile)
		: ArsonSimpleProgressDlg(
			i18n("Please insert the CD you wish to read and press Start..."),
			i18n("Readcd Progress"), NULL, "readcd"), m_outfile(outfile) {}

private:
	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		return new ArsonReadCdMgr(ui(), m_outfile);
	}

	QCString m_outfile;
};

bool ArsonReadCdMgr::readCd (const char *outfile)
{
	readcdDlg(outfile).exec();
	return true;
}

/*========================================================*/
/*	Unlock the drive
 *========================================================*/

class unlockMgr : public ArsonProcessMgr
{
public:
	unlockMgr (ArsonProcessUI *pUI)
		: ArsonProcessMgr(pUI)
	{
		//	Nothing...
	}

	virtual void begin (const ArsonProcessOpts &opts)
	{
		ArsonProcessMgr::begin(opts);

		try {
			setProcess(
				new ArsonCdrdaoUnlockProcess(this));
		}
		catch (ArsonError &err) {
			err.report();
		}
	}
};

class unlockDlg : public ArsonProgressDlg
{
public:
	unlockDlg (void)
		: ArsonProgressDlg(NULL, "unlockdlg")
	{
		setText(
			i18n("Press Start to unlock drive..."));
	}

private:
	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		setProgressMode(arsonProgressInfinite);

		return new unlockMgr(ui());
	}

	virtual void endWrite (void)
	{
		setProgressMode(arsonProgressBlank);
		ArsonProgressDlg::endWrite();
	}
};

void arsonUnlockDrive (void)
{
	unlockDlg().exec();
}

/*========================================================*/
/*	CDRW blanking
 *========================================================*/

class blankMgr : public ArsonProcessMgr
{
public:
	blankMgr (ArsonProcessUI *pUI)
		: ArsonProcessMgr(pUI) { }

	virtual void begin (const ArsonProcessOpts &opts)
	{
		ArsonProcessMgr::begin(opts);
		
		try {
			ArsonProcess *ptr;
			const long how = opts.getLong("how");

			if (ACONFIG.seq().seq(PROGSEQ_BLANK) == BLANKSEQ_CDRDAO)
				ptr = new ArsonCdrdaoBlankProcess(this, how);
			else
				ptr = new ArsonCdrecordBlankProcess(this, how);

			setProcess(ptr);
		}
		catch (ArsonError &err) {
			err.report();
		}
	}
};

class blankDlg : public ArsonProgressDlg
{
public:
	blankDlg (void)
		: ArsonProgressDlg(NULL, "blankdlg")
	{
		const static QString modes[] = {
			i18n("Full"),
			i18n("Fast"),
		};
		QLabel *pl = new QLabel(i18n("Blank Mode: "), ctrlParent());
		m_pMode = new ArsonProgressCombobox(ctrlParent(), "blankmode");
		pl->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Fixed));

		setText(
			i18n("Insert a CDRW, and press Start to blank the disk..."));

		for (int index = 0; index < ARRSIZE(modes); ++index)
			m_pMode->insertItem(modes[index]);

		m_pMode->setCurrentItem(0);


		layoutRow() << pl << m_pMode;
	}

private:
	virtual void processOpts (ArsonProcessOpts &opts)
	{
		opts.addLong("how", m_pMode->currentItem());
		ArsonProgressDlg::processOpts(opts);
	}

	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		return new blankMgr(ui());
	}

	QComboBox *m_pMode;
};


void arsonBlankCdrw (void)
{
	blankDlg().exec();
}

/*========================================================*/
/*	Image writing
 *========================================================*/

class cueDlg : public ArsonCdWriterProgressDlg
{
public:
	cueDlg (const char *cuefile)
		: ArsonCdWriterProgressDlg(ArsonFrame::getFrame(), "cuedlg"),
		m_cuefile(cuefile) {}

private:
	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		return new ArsonCueWriter(ui(), m_cuefile);
	}

	QCString m_cuefile;
};

class isoDlg : public ArsonCdWriterProgressDlg
{
public:
	isoDlg (const char *isofile)
		: ArsonCdWriterProgressDlg(ArsonFrame::getFrame(), "isodlg"),
		m_isofile(isofile) {}

private:
	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		return new ArsonIsoWriter(ui(), m_isofile);
	}

	QCString m_isofile;
};

/*========================================================*/
/*	Worker image writer functions
 *========================================================*/

void arsonWriteCueFile (const char *cuefile)
{
	cueDlg(cuefile).exec();
}

void arsonWriteIsoFile (const char *isofile)
{
	isoDlg(isofile).exec();
}

/*========================================================*/
/*	Write a single existing directory tree
 *========================================================*/

void arsonWriteDir (const QString &indir)
{
	try
	{
		QStringList sl;
		ArsonIsoWriterProgressDlg dlg (ArsonFrame::getFrame(), sl);
		QString dir (indir);

		if (indir == QString::null)
		{
			dir = KFileDialog::getExistingDirectory();

			if (dir == QString::null)
				return;
		}

		sl.append(dir);
		dlg.exec();
	}
	catch(ArsonError &err) {
		err.report();
	}
}

/*========================================================*/
/*	Class for handling image files
 *========================================================*/

const QString ArsonCdImageFile::exts[ArsonCdImageFile::Max] = {
	I18N_NOOP("*.iso|ISO Files"),
	I18N_NOOP("*.cif|CIF Files"),
	I18N_NOOP("*.cue|CUE Files"),
	I18N_NOOP("*.toc|TOC Files"),
};

ArsonCdImageFile::ArsonCdImageFile (const QString &filename)
	: m_file(filename), m_type(Unknown) { }

int ArsonCdImageFile::imgFormat (void)
{
	if (m_type == Unknown)
	{
		QFileInfo fi (QFile::encodeName(filename()));
		const QString ext = fi.extension(false).lower();

		if (fi.isDir())
			return (m_type = Dir);

		for (int index = 0; index < Max; ++index)
		{
			const int pipe = exts[index].find('|');

			if (exts[index].mid(2, pipe - 2) == ext)
				return (m_type = index);
		}
	}

	return m_type;
}

bool ArsonCdImageFile::write (void)
{
	switch (imgFormat())
	{
	case Iso:
	case Cif:
		arsonWriteIsoFile(filename());
		break;

	case Toc:
	case Cue:
		arsonWriteCueFile(filename());
		break;

	case Dir:
		arsonWriteDir(filename());
		break;

	default:
		return false;
	}

	return true;
}

/*========================================================*/
/*	Write any known image file format
 *========================================================*/

void arsonWriteImg (const char *fn)
{
	ArsonCdImageFile img (fn);

	if (img.imgFormat() == ArsonCdImageFile::Unknown)
		arsonErrorMsg(
			i18n("Unknown image file format."));
	else
		img.write();
}

/*========================================================*/
/*	CD Copy function
 *========================================================*/

void arsonCdCopy (void)
{
	ArsonCdCopyDlg().exec();
}

/*========================================================*/
