------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--       A S I S _ U L . G L O B A L _ S T A T E . U T I L I T I E S        --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                       Copyright (C) 2007, AdaCore                        --
--                                                                          --
-- Asis Utility Library (ASIS UL) is free software; you can redistribute it --
-- and/or  modify  it  under  terms  of  the  GNU General Public License as --
-- published by the Free Software Foundation; either version 2, or (at your --
-- option)  any later version.  ASIS UL  is distributed in the hope that it --
-- will  be  useful,  but  WITHOUT  ANY  WARRANTY; without even the implied --
-- warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the --
-- GNU  General Public License for more details. You should have received a --
-- copy of the  GNU General Public License  distributed with GNAT; see file --
-- COPYING. If not,  write  to the  Free Software Foundation,  51 Franklin  --
-- Street, Fifth Floor, Boston, MA 02110-1301, USA.                         --
--                                                                          --
-- ASIS UL is maintained by AdaCore (http://www.adacore.com).               --
--                                                                          --
------------------------------------------------------------------------------

--  This package defines various utility programs needed for the implementation
--  of the data strucrure representing the progtam global state

package ASIS_UL.Global_State.Utilities is

   function Can_Create_Tasks (El : Asis.Element) return Boolean;
   --  Checks if the execution or elaboration of its argument can cause one or
   --  more tasks to be created. At the moment two cases are considered as
   --  (potential) task creations - object declarations and allocators (in both
   --  cases the type of object/value can be a task type or it can contain
   --  task components).

   function Corresponding_Element (El : Asis.Element) return Asis.Element;
   --  For the argument Element, computes the element that is used as the basis
   --  for representation of the given entity in the data structures
   --  representing the program global state. The rules for defining the
   --  corresponding element are:
   --
   --  * for a task       - the corresponding single task declaration or task
   --                       type declaration
   --
   --  * for a subprogram - if a separate declaration is given, the
   --                       corresponding element is the declaration, otherwise
   --                       the subprogram body, except it is a proper body for
   --                       a subunit, in this case the corresponding element
   --                       is either the corresponding declaration, if
   --                       present, or the body stub
   --
   --  * for a subprogram instantiation - the instantiation itself
   --
   --  * for an expanded subprogram spec or body corresponding to a subprogram
   --    instantiation - the corresponding instantiation
   --
   --  * for a subprogram renaming:
   --     - the Corresponding_Element of the renamed subprogram (or entry), if
   --       the renaming can be resolved statically to some subprogram;
   --
   --     - An_Attribute_Reference Element if the renaming renames an
   --       attribite subprogram
   --
   --     - An_Enumeration_Literal Element if the renaming renames an
   --       enumeration literal
   --
   --     - Nil_Element for all the other cases
   --
   --  * for entry body       - the corresponding entry declaration;
   --
   --  * for accept statement - the corresponding entry declaration;
   --
   --  * for ???  - to be extended...
   --
   --  If the argument Element does not correspond to any of the cases
   --  described above, it is returned as the result.

   function Enclosing_Scope (El : Asis.Element) return Asis.Element;
   --  Returns the closest Is_Scope Element that encloses the argument Element

   function Get_Called_Element (Call : Asis.Element) return Asis.Element;
   --  Supposing that the argument Is_Call, tries to compute the called
   --  element, depending on the kind of the call. If the called element
   --  can be detected, returns Corresponding_Element for it, otherwise
   --  returns Nil_Element.
   --
   --  What about calls to attribute subprograms in case when the corresponding
   --  attribute is predefined???

   function Get_Renamed_Subprogram (El : Asis.Element) return Asis.Element;
   --  Assuming that El is a subprogram renaming element, tries to get the
   --  declaration of renamed callable entity by unwinding renamings. Returns
   --  Nil_Element if the renamed subprogram cannot be determined statically.

   function Is_Call (El : Asis.Element) return Boolean;
   --  Checks if El is a call that is of interest for the call graph.

   function Is_Call_To_Predefined_Operation
     (Call : Asis.Element)
      return Boolean;
   --  Checks if the argument is a call to a predefined operation

   function Is_Predefined_Operation_Renaming
     (Ren  : Asis.Element)
      return Boolean;
   --  Checks if the argument is a renaming of a predefined operation (returns
   --  true for undirect predefined operators renamings as well)

   function Is_Declaration_Of_Callable_Entity
     (El   : Asis.Element)
      return Boolean;
   --  Checks if El is a declaration of a callable entity that is of interest
   --  for call graph analysis. This function does not consider subprogram
   --  renamings as being declarations of callable entities - renamings are
   --  processed separately. Bodies (even if they act as specs) are also not
   --  considered as declarations of callable entities.
   --
   --  This function assumes, that it is called only in the source context that
   --  is of interest for our analyzis. That is, if it see a function
   --  declaration, it does not care if it is declared inside a generic - such
   --  possibilities are supposed to be filtered out by high-level traversal
   --  control.

   function Is_Scope (El : Asis.Element) return Boolean;
   --  Checks if El is a scope in the Call Graph sense - that is, an executable
   --  body of an entity that can be called by another entities and that itself
   --  can call other entities. We also consider as scopes library-level
   --  packages and library-level package instantiations.
   --  What about task entries and their accept statements?????????

end ASIS_UL.Global_State.Utilities;
