------------------------------------------------------------------------------
--                                                                          --
--                 ASIS-for-GNAT IMPLEMENTATION COMPONENTS                  --
--                                                                          --
--                            A 4 G . V C H E C K                           --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--            Copyright (c) 1995-1999, Free Software Foundation, Inc.       --
--                                                                          --
-- ASIS-for-GNAT is free software; you can redistribute it and/or modify it --
-- under terms of the  GNU General Public License  as published by the Free --
-- Software Foundation;  either version 2,  or  (at your option)  any later --
-- version. ASIS-for-GNAT is distributed  in the hope  that it will be use- --
-- ful, but WITHOUT ANY WARRANTY; without even the implied warranty of MER- --
-- CHANTABILITY or  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General  --
-- Public License for more details. You should have received a copy of the  --
-- GNU General Public License  distributed with ASIS-for-GNAT; see file     --
-- COPYING. If not, write to the Free Software Foundation,  59 Temple Place --
-- - Suite 330,  Boston, MA 02111-1307, USA.                                --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- ASIS-for-GNAT was originally developed  by the ASIS-for-GNAT team at the --
-- Software  Engineering  Laboratory  of  the Swiss  Federal  Institute  of --
-- Technology (LGL-EPFL) in Lausanne,  Switzerland, in cooperation with the --
-- Scientific  Research  Computer  Center of  Moscow State University (SRCC --
-- MSU), Russia,  with funding partially provided  by grants from the Swiss --
-- National  Science  Foundation  and  the  Swiss  Academy  of  Engineering --
-- Sciences.  ASIS-for-GNAT is now maintained by  Ada Core Technologies Inc --
-- (http://www.gnat.com).                                                   --
--                                                                          --
------------------------------------------------------------------------------

with Asis;        use Asis;
with Asis.Text;
with Asis.Errors; use Asis.Errors;

with A4G.A_Types; use A4G.A_Types;

package A4G.Vcheck is

--  GNAT-style reformatting required!! Only formal compilability with -gnatg
--  is achieved by now.

--  This package contains procedures for checking the validity of the
--  Context, Compilation_Unit and Element values, raising the
--  Asis-defined exceptions (with previous setting of the status
--  value and forming the Diagnosis string) in some standard
--  situations and utility subprograms to be used in the keeping to
--  the "catch-all" policy defined in the subsection 3.6 of the
--  ASIS: Detailed Semantics and Implementation Document (v. 1.1.1)
--
--  The first prototype implementation uses rather roof approach
--  to keep to the "catch-all" policy in respect to the
--  non-Asis-defined exceptions and to handle the status and Diagnosis
--  information. The main principles of this approach are:
--
--  (a) all non-Asis exceptions are catched by
--
--        when others =>
--          Raise_ASIS_Failed (Diagnosis => "Name of the routine in which"
--                             &" this handler is" );
--
--       exception handler, and as a rule no attempt is undertaken
--       to recover from the error situation (Status is seted as
--       "Value_Error" and ASIS_Failed is raised);
--
--  (b) Diagnosis string contains only general description of the failure
--      and the idication of the Asis query or low-level implementation
--      utility subprogram in which the Asis-defined exception was initially
--      raised;
--
--  (c) if Asis-defined exception propagates through the Asis implementation,
--      then the information related to the dynamic context of the exception
--      propagation is added to the Diagnosis (and separated by the
--      Asis_Types_And_Limits.ASIS_Line_Terminator constant).
--
--  (d) most of the routines in the Asis implementation should contain the
--      exception handler with the "others" choise (as described in the
--      paragraph (a) above. If it is known that the Asis_defined exception
--      could be raised in the procedure or function body, then the body
--      should contain two following exception handlers in the order given:
--
--      when Asis_ASIS_Inappropriate_Context    |
--          ASIS_Inappropriate_Compilation_Unit |
--          ASIS_Inappropriate_Element            =>
--
--          raise; -- nothing should be done additionally;
--                 -- exceprion has beed raised in the argument(s)
--                 -- validity/appropriation checks in the same
--                 -- frame
--
--      when ASIS_Failed -- | other possible Asis-defined exceptions
--                           =>
--        Add_Call_Information (Outer_Call => "Name of the routine in which"
--                           &" this handler is" );
--        raise; ---????????
--
--      when others =>
--        Raise_ASIS_Failed (Diagnosis => "Name of the routine in which"
--                           &" this handler is" );
--


   --------------------------------------------------------------------
   -- Data Objects for Handling the Diagnosis string and Error Staus --
   --------------------------------------------------------------------

   Status_Indicator : Error_Kinds := Not_An_Error;

   Diagnosis_Buffer : String (1 .. Max_Diagnosis_Length);

   Diagnosis_Len    : Natural range 0 .. Max_Diagnosis_Length := 0;

   procedure Add (Phrase : String);
   --  Adds Phrase to Diagnosis_Buffer and resets Diagnosis_Len. Exits with no
   --  (further) change in Diagnosis_Buffer as soon as Diagnosis_Len attains
   --  Max_Diagnosis_Length.

   ------------------------------------------------------------
   --  Checking the validity of Context, Compilation_Unit and
   --  Element
   ------------------------------------------------------------

------------------------------------------------------------
   procedure Check_Validity
     (Compilation_Unit : Asis.Compilation_Unit;
      Query            : String);
------------------------------------------------------------
--   Compilation_Unit - Specifies the unit to check
--   Query            - Specifies the name of query in which the check
--                      is performed. The parameter value is used to
--                      form the Diagnosis string
--
--  Performs the check if the unit does not belong to the closed library
--  Sets the corresponding Status and forms the corresponding Diagnosis
--  if the chesk fails
--
--  BETTER DOCS NEEDED!!
---------------------------------------------------------------
   procedure Check_Validity
     (Element : Asis.Element;
      Query   : String);
------------------------------------------------------------
--   Element - Specifies the element to check
--   Query   - Specifies the name of query in which the check is performed.
--             The parameter value is used to form the Diagnosis string
--
--  Performs the check if the element does not belong to the invalid
--  element
--  Sets the corresponding Status and forms the corresponding Diagnosis
--  if the chesk fails
--
--  BETTER DOCS NEEDED!!
---------------------------------------------------------------
   procedure Check_Validity (Line  : Asis.Text.Line;
                             Query : String);
------------------------------------------------------------
--   Line    - Specifies the Line to check
--   Query   - Specifies the name of query in which the check is performed.
--             The parameter value is used to form the Diagnosis string
--
--  Performs the check if the line does not belong to the invalid
--   Context
--  Sets the corresponding Status and forms the corresponding Diagnosis
--  if the chesk fails
--
--  BETTER DOCS NEEDED!!
------------------------------------------------------------
   procedure Check_Validity (Context : Asis.Context;
                             Query   : String);
------------------------------------------------------------
--   Context - Specifies the ASIS Context to check
--   Query   - Specifies the name of query in which the check is performed.
--             The parameter value is used to form the Diagnosis string
--
--  Performs the check if the Context is nit in inassosiated or inopened
--  state
--  Sets the corresponding Status and forms the corresponding Diagnosis
--  if the chesk fails
--
--  BETTER DOCS NEEDED!!



   -------------------------------------
   -- Raising Asis-defined exceptions --
   -------------------------------------

---------------------------------------------------------------
   procedure Raise_ASIS_Failed (Diagnosis : String);
---------------------------------------------------------------
--   Diagnosis - Specifies the situation, when and where the internal
--               error have taken place.
--
--  Raises ASIS_Failed with Internal_Error status, is to be used
--  to ensure the "catch all" policy for non-ASIS exceptions, see
--  ASIS:  Detailed Semantics and Implementation document, section 3.6
--
--   SHOULD BE SYSTEMATICALLY REPLACED BY THE FOLLOWING FUNCTION
--
--  What is the difference between Internal_Error and
--  Unhandled_Exception_Error error status values?????

   procedure Raise_ASIS_Failed
     (Argument : Asis.Element;
      Diagnosis : String;
      Stat      : Asis.Errors.Error_Kinds := Internal_Error;
      Bool_Par  : Boolean := False);
   --  Raises ASIS_Failed with Stat as the value of ASIS Error Status.
   --  Expects the query name as Diagnosis. Bool_Par is set True, if
   --  a query has an optional boolean parameter and if this parameter
   --  is set on for the given call to this query. Uses Argument, Diagnosis
   --  and Bool_Par parameter to form the ASIS Diagnosis string.
   --  It is a bug to call this procedure with Nil_Element
   --  or unvalid Argument.

--------------------------------------------------------------------
   procedure Raise_ASIS_Inappropriate_Compilation_Unit
                         (Diagnosis : String);
--------------------------------------------------------------------
--  Diagnosis - Specifies the query to which the Compilation Unit
--              with inappropriate kind was passed
--
--  Raises ASIS_Inappropriate_Compilation_Unit with Value_Error status
--
--  BETTER DOCS NEEDED!!
--------------------------------------------------------------------
   procedure Raise_ASIS_Inappropriate_Element
     (Diagnosis : String;
      Status    : Error_Kinds      := Value_Error);
--------------------------------------------------------------------
--  Diagnosis - Specifies the query to which the Element with
--              inappropriate kind was passed
--
--  Raises ASIS_Inappropriate_Element with Status error status
--
--  BETTER DOCS NEEDED!!
--
--------------------------------------------------------------------
   procedure Raise_ASIS_Inappropriate_Line_Number
     (Diagnosis : String;
      Status    : Error_Kinds      := Value_Error);
--------------------------------------------------------------------
   procedure Not_Implemented_Yet (Diagnosis : String);
--------------------------------------------------------------------
--  Diagnosis - Specifies the query which has not been implemented
--              properly yet and has a placeholder as its body
--
--  Raises ASIS_Failed with Not_Implemented_Error status
--
--  This procedure is used in the placeholder bodies of the non-implemented
--  queries
--
--  BETTER DOCS NEEDED!!
--------------------------------------------------------------------

   ----------------------------------------------
   --  Setting the Status and Diagnosis values --
   ----------------------------------------------

--------------------------------------------------------------------
   procedure Set_Error_Status
     (Status    : Error_Kinds      := Not_An_Error;
      Diagnosis : String := Nil_Asis_String);
--------------------------------------------------------------------
--
--  This procedure is the full analog of the A4G.Set_Status
--  procedure, it is intended to be used as the implementation of the
--  A4G.Set_Status (by means of the renaming_as_a_body
--  in the body of Asis_Environment)
--------------------------------------------------------------------


   -----------------------------
   -- Adding Call Information --
   -----------------------------

   procedure Add_Call_Information
     (Outer_Call : String;
      Argument   : Asis.Element := Nil_Element;
      Bool_Par   : Boolean := False);
   --  Adds in the ASIS diagnosis the information about dynamically enclosing
   --  calls when an ASIS exception is propagated from some dynamically
   --  enclosed routine. If Argument is not Nil, adds its debug image in the
   --  Diagnosis, if needed (that is, if Argument_Postponed flag is set on,
   --  and resets this flag OFF after that)

   pragma No_Return (Raise_ASIS_Failed);
   pragma No_Return (Raise_ASIS_Inappropriate_Compilation_Unit);
   pragma No_Return (Raise_ASIS_Inappropriate_Element);
   pragma No_Return (Not_Implemented_Yet);
end A4G.Vcheck;