// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include <ctype.h>

#include "Puma/CRecognizer.h"
#include "Puma/Source.h"

// Include token type definitions
#include "Puma/PreParser.h"
#include "Puma/CTokens.h"
#include "Puma/CWildcardTokens.h"
#include "Puma/PreMacroOpTokens.h"
#include "Puma/CCommentTokens.h"

namespace Puma {


// Include generated scanner tables
#include "PreDirectiveTab.ot"
#include "PreTokenTab.ot"
#include "CDirectiveTab.ot"
#include "CWildcardTab.ot"
#include "CCoreTab.ot"
#include "CIdentifierTab.ot"


// Declaration of orange token recognizer objects

MappedOrangeRecognizer<CScanBuffer> CRecognizer::pre_dir_recognizer (
  PreDirectiveTabMap, PreDirectiveTabExprMap,
  PreDirectiveTabStart, PreDirectiveTabStates, 
  PreDirectiveTabNext, PreDirectiveTabControl, 
  sizeof (PreDirectiveTabNext) / sizeof (int));

MappedOrangeRecognizer<CScanBuffer> CRecognizer::pre_token_recognizer (
  PreTokenTabMap, PreTokenTabExprMap,
  PreTokenTabStart, PreTokenTabStates, 
  PreTokenTabNext, PreTokenTabControl, 
  sizeof (PreTokenTabNext) / sizeof (int));

OrangeRecognizer<CScanBuffer> CRecognizer::dir_recognizer (
  CDirectiveTabMap, CDirectiveTabStart, 
  CDirectiveTabStates, CDirectiveTabNext, 
  CDirectiveTabControl, 
  sizeof (CDirectiveTabNext) / sizeof (int));

MappedOrangeRecognizer<CScanBuffer> CRecognizer::wildcard_recognizer (
  CWildcardTabMap, CWildcardTabExprMap, CWildcardTabStart, 
  CWildcardTabStates, CWildcardTabNext, CWildcardTabControl, 
  sizeof (CWildcardTabNext) / sizeof (int));

MappedOrangeRecognizer<CScanBuffer> CRecognizer::core_recognizer (
  CCoreTabMap, CCoreTabExprMap, CCoreTabStart, CCoreTabStates, 
  CCoreTabNext, CCoreTabControl, sizeof (CCoreTabNext) / sizeof (int));

MappedOrangeRecognizer<CScanBuffer> CRecognizer::identifier_recognizer (
  CIdentifierTabMap, CIdentifierTabExprMap, CIdentifierTabStart,
  CIdentifierTabStates, CIdentifierTabNext, CIdentifierTabControl, 
  sizeof (CIdentifierTabNext) / sizeof (int));


// functions


void CRecognizer::setup () {
  comment_recognizer.mode (CCommentRecognizer::NO_COMMENT);
  directives = true;
  scan_mode = NORMAL;
}


int CRecognizer::recognize (Lang &lang, int &expr, int &len) {
  int result;

  if ((result = comment_recognizer.recognize (&scan_buffer, expr, len)) != 0) { 
    lang = (expr == TOK_WSPACE ? WHITE : COMMENT);
    return result;
  }
  scan_buffer.retry ();

  if (directives && ! macro_ops) {
    directives = false;
    if ((result = pre_dir_recognizer.recognize (&scan_buffer, expr, len)) != 0) {
      if (result > 0) 
        scan_mode = IN_PRE_DIR;
      lang = PRE_DIR;
      return result;
    }
    scan_buffer.retry ();

    if ((result = dir_recognizer.recognize (&scan_buffer, expr, len)) != 0) {
      if (result > 0) 
        scan_mode = IN_COMP_DIR;
      lang = COMP_DIR;
      return result;
    }
    scan_buffer.retry ();
  }

  if (macro_ops || scan_mode == IN_PRE_DIR) {
    if ((result = pre_token_recognizer.recognize (&scan_buffer, expr, len)) != 0) {
      lang = PRE;
      return result;
    }
    scan_buffer.retry ();
  }

  if (wildcards) {
    if ((result = wildcard_recognizer.recognize (&scan_buffer, expr, len)) != 0) {
      lang = WILDCARD;
      return result;
    }
    scan_buffer.retry ();
  }

  if ((result = core_recognizer.recognize (&scan_buffer, expr, len)) != 0) {
    if (expr >= TOK_FIRST_CORE && expr <= TOK_LAST_CORE) {
      lang = CORE;
      return result;
    }
    // if it looks like a keyword we have to check if some character follows
    // that makes it an identifier instead of a keyword
    scan_buffer.more (len);
    if (scan_buffer.state () == ScanBuffer::STATE_OK) {
      char next = scan_buffer.lookahead ();
      if (isalnum ((int)next) || next == '_')
        result = 0;
    }

    if (result) {
      bool is_keyword = false;
      if (expr >= TOK_FIRST_C && expr <= TOK_LAST_C)
        is_keyword = std_c;
      else if (expr >= TOK_FIRST_EXT_C && expr <= TOK_LAST_EXT_C)
        is_keyword = gnu_c || visual_c || visual_cplusplus;
      else if (expr >= TOK_FIRST_GNUC && expr <= TOK_LAST_GNUC)
        is_keyword = gnu_c;
      else if (expr >= TOK_FIRST_VC && expr <= TOK_LAST_VC)
        is_keyword = visual_c || visual_cplusplus;
      else if (expr >= TOK_FIRST_CC && expr <= TOK_LAST_CC)
        is_keyword = std_cplusplus;
      else if (expr >= TOK_FIRST_EXT_CC && expr <= TOK_LAST_EXT_CC)
        is_keyword = gnu_cplusplus || visual_cplusplus;
      else if (expr >= TOK_FIRST_GNUCC && expr <= TOK_LAST_GNUCC)
        is_keyword = gnu_cplusplus;
      else if (expr >= TOK_FIRST_TYPETRAIT && expr <= TOK_LAST_TYPETRAIT)
        is_keyword = type_traits;
      else if (expr >= TOK_FIRST_CC1X && expr <= TOK_LAST_CC1X)
        is_keyword = cc1x;
      else if (expr >= TOK_FIRST_AC && expr <= TOK_LAST_AC)
        is_keyword = aspectc;

      if (is_keyword) {
        lang = KEYWORD;
        return result;
      }
    }
  }
  scan_buffer.retry ();

  long num = additional_keyword_recognizers.length ();
  for (long i = 0; i < num; i++) {
    if ((result = additional_keyword_recognizers[i]->recognize (&scan_buffer, expr, len)) != 0) {
      lang = KEYWORD;
      return result;
    }
    scan_buffer.retry ();
  }

  if ((result = identifier_recognizer.recognize (&scan_buffer, expr, len)) != 0) {
    lang = ID;
    return result;
  }
  scan_buffer.retry ();

  lang = UNKNOWN;
  expr = 0;
  len  = 1;
  return 0;
}


} // namespace Puma
