// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

// AspectC++ includes
#include "PointCutExpr.h"
#include "PointCutContext.h"
#include "Binding.h"
#include "MatchExpr.h"

// PUMA includes
#include "Puma/ErrorStream.h"
#include "Puma/CFunctionInfo.h"
#include "Puma/CArgumentInfo.h"
#include "Puma/CClassInfo.h"
#include "Puma/CClassDatabase.h"
#include "Puma/CTree.h"
#include "Puma/Filter.h"

#include <assert.h>
#include <sstream>
using std::stringstream;
using std::endl;

void PointCutExpr::sem_args (ErrorStream &err, PointCutContext &context) {
  // recursively do a semantic analysis of child nodes
  for (int i = 0; i < args (); i++)
    if (arg (i))
      arg (i)->semantics (err, context);
}

void PointCutExpr::check_arg_types (ErrorStream &err, const char *func, 
				    PCE_Type expected) {
  // it is an error if something is provided, which was not expected
  for (int i = 0; i < args (); i++) {
    if (arg (i) && arg (i)->type () != expected) {
      err << sev_error << node ()->token ()->location ()
	  << "'" << func << "' expects a "
	  << (expected == PCE_NAME ? "name" : "code") << " pointcut"
	  << " as argument";
      if (args  () > 1)
	err << " " << (i + 1);
      err << endMessage;
    }
  }
}

void PointCutExpr::check_arg_types_equal (ErrorStream &err, const char *func) {
  if (!arg (0)) return;
  // it is an error if the types of all arguments are not equal
  PCE_Type first_type = arg (0)->type ();
  for (int i = 1; i < args (); i++) {
    if (arg (i) && arg (i)->type () != first_type) {
      err << sev_error << node ()->token ()->location ()
	  << "'" << func << "' expects that all argument types are equal"
	  << endMessage;
    }
  }
}

const CArgumentInfo *PointCutExpr::get_binding (const char *name,
						ErrorStream &err, 
						PointCutContext &context) {
  if (!context.func ()) {
    err << sev_error << node ()->token ()->location () 
	<< "context variable '" << name
	<< "' is not support in this kind of pointcut expression"
	<< endMessage;
    return 0;
  }
  ArgSet &curr_arg_bindings = *context.arg_bindings ().top ();
  CArgumentInfo *arginfo = (CArgumentInfo*)0;
  unsigned int a;
  for (a = 0; a < context.func ()->Arguments (); a++) {
    if (strcmp (context.func ()->Argument (a)->Name (), name) == 0) {
      arginfo = context.func ()->Argument (a);
      break;
    }
  }
  if (!arginfo) {
    err << sev_error << node ()->token ()->location () 
	<< "'" << name
	<< "' is not in argument list" << endMessage;
    return 0;
  }
  return curr_arg_bindings.lookup (a);
}

bool PCE_SingleArg::check_derived_class (JPL_Class &cls,
  PointCutContext &context, Binding &binding, Condition &cond) {
    
  Condition unused;
  bool do_subclasses = true;
  const set<int> &derived_classes = cls.derived_class_ids ();
  for (set<int>::const_iterator i = derived_classes.begin ();
    i != derived_classes.end (); ++i) {
    JPL_Class *derived = (JPL_Class*)cls.map (*i);
    if (arg (0)->evaluate (*derived, context, binding, unused)) {
      if (context.in_that () || context.in_target ()) {
        cond.matches (*derived);
        do_subclasses = false; // don't check subclasses, but
                               // siblings
      }
      else
        return true;
    }
    if (do_subclasses && check_derived_class (*derived, context,
        binding, cond))
      return true;
  }
  return !do_subclasses;
}

bool PCE_SingleArg::check_derived_func (JPL_Class &cls, JPL_Function &func,
  PointCutContext &context, Binding &binding, Condition &cond) {
  Condition unused;
  bool do_subclasses = true;
  const set<int> &derived_classes = cls.derived_class_ids ();
  for (set<int>::const_iterator i = derived_classes.begin ();
    i != derived_classes.end (); ++i) {
    JPL_Class *derived = (JPL_Class*)cls.map (*i);

    const JPL_Name::CList &children = derived->children ();
    for (JPL_Name::CList::const_iterator i = children.begin ();
         i != children.end (); ++i) {
      if ((*i)->type () != JoinPointLoc::Function)
        continue;
      JPL_Function &curr = *(JPL_Function*)*i;
      if (!func.has_same_name_and_args (curr))
         continue;
      if (arg (0)->evaluate (curr, context, binding, cond))
        return true;
    }

    if (do_subclasses && check_derived_func (*derived, func, context, 
      binding, cond))
      return true;
  }
  return !do_subclasses;
}


bool PCE_SingleArg::check_base_class (JPL_Class &cls,
        PointCutContext &context, Binding &binding, Condition &cond) {
  if (arg (0)->evaluate (cls, context, binding, cond)) {
    return true;
  }
  const set<int> &bases = cls.base_class_ids ();
  for (set<int>::const_iterator i = bases.begin (); i != bases.end (); ++i) {
    JPL_Class *base = (JPL_Class*)cls.map (*i);
    if (check_base_class (*base, context, binding, cond))
      return true;
  }
  return false;
}

bool PCE_SingleArg::check_base_func (JPL_Class &cls, JPL_Function &func,
        PointCutContext &context, Binding &binding, Condition &cond) {
  const JPL_Name::CList &children = cls.children ();
  for (JPL_Name::CList::const_iterator i = children.begin ();
       i != children.end (); ++i) {
    if ((*i)->type () != JoinPointLoc::Function)
      continue;
    JPL_Function &curr = *(JPL_Function*)*i;
    if (!func.has_same_name_and_args (curr))
       continue;
    if (arg (0)->evaluate (curr, context, binding, cond))
      return true;
  }

  const set<int> &bases = cls.base_class_ids ();
  for (set<int>::const_iterator i = bases.begin (); i != bases.end (); ++i) {
    JPL_Class *base = (JPL_Class*)cls.map (*i);
    if (check_base_func (*base, func, context, binding, cond))
      return true;
  }
  return false;
}

bool PCE_SingleArg::check_scopes (JPL_Name *scope,
        PointCutContext &context, Binding &binding, Condition &cond) {
  do {
    // TODO: at the moment the evaluation only matches classes.
    //       later namespaces should also be accepted, e.g. within("Puma")
    if (arg (0)->evaluate (*scope, context, binding, cond))
      return true;
    scope = (JPL_Name*)scope->parent ();
  } while (scope);

  return false;
}

PCE_Type PCE_Classes::type () const {
  return PCE_NAME;
}

void PCE_Classes::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types (err, "classes", PCE_NAME);
  sem_args (err, context);
  _possible_types = JoinPointLoc::Class;
}

bool PCE_Classes::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                            Binding &binding, Condition &cond) {
  return jpl.type () == JoinPointLoc::Class &&
    arg (0)->evaluate (jpl, context, binding, cond);
}

PCE_Type PCE_Base::type () const {
  return PCE_NAME;
}

void PCE_Base::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types (err, "base", PCE_NAME);
  sem_args (err, context);
  _possible_types = (JoinPointLoc::join_point_type)arg (0)->possible_types ();
}

bool PCE_Base::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                         Binding &binding, Condition &cond) {
  if (!(jpl.type () & (int)_possible_types))
    return false;
    
  if (jpl.type () & (JoinPointLoc::Class | JoinPointLoc::Aspect)) {
    // the class is a base of the argument class(es) if one of its derived
    // classes is described by the argument
    return check_derived_class ((JPL_Class&)jpl, context, binding, cond);
  }
  else if ((jpl.type () == JoinPointLoc::Function)) {
    JPL_Function &func = (JPL_Function&)jpl;
    if (func.parent ()->type () & (JoinPointLoc::Class | JoinPointLoc::Aspect)) {
      JPL_Class &cls = *(JPL_Class*)func.parent ();
      return check_derived_func (cls, func, context, binding, cond);
    }
  }
  return false;
}

PCE_Type PCE_Derived::type () const {
  return PCE_NAME;
}

void PCE_Derived::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types (err, "derived", PCE_NAME);
  sem_args (err, context);
  _possible_types = (JoinPointLoc::join_point_type)arg (0)->possible_types ();
}

bool PCE_Derived::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                            Binding &binding, Condition &cond) {
  if (!(jpl.type () & (int)_possible_types))
    return false;
    
  if (jpl.type () & (JoinPointLoc::Class | JoinPointLoc::Aspect)) {
    // the class is derived of the argument class(es) if the class itself or
    // one of its base classes is described by the argument
    return check_base_class ((JPL_Class&)jpl, context, binding, cond);
  }
  else if ((jpl.type () == JoinPointLoc::Function)) {
    JPL_Function &func = (JPL_Function&)jpl;
    if (func.parent ()->type () & (JoinPointLoc::Class | JoinPointLoc::Aspect)) {
      JPL_Class *cls = (JPL_Class*)func.parent ();
      return check_base_func (*cls, func, context, binding, cond);
    }
  }
  return false;
}

PCE_Type PCE_Within::type () const {
  return PCE_CODE;
}

void PCE_Within::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types (err, "within", PCE_NAME);
  sem_args (err, context);
  _possible_types = JoinPointLoc::Code;
}

bool PCE_Within::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                           Binding &binding, Condition &cond) {

  // 'within' matches only code join points
  if (!(jpl.type () & (int)JoinPointLoc::Code))
    return false;
  JPL_Code &jpl_code = (JPL_Code&)jpl;
    
  // accept pseudo method calls
  if (jpl_code.type () == JoinPointLoc::MethodCall &&
      ((JPL_MethodCall&)jpl_code).is_pseudo ()) {
    context.pseudo_true (true);
    return true;
  }

  JPL_Name *scope = jpl_code.lexical_scope ();
  if (scope->type () == JoinPointLoc::Function) {
    JPL_Function *func = (JPL_Function*)scope;
    if (func->is_constructor () || func->is_destructor ())
      scope = func->parent ();
  }
  return check_scopes (scope, context, binding, cond);
}

PCE_Type PCE_Execution::type () const {
  return PCE_CODE;
}

void PCE_Execution::semantics (ErrorStream &err,
			       PointCutContext &context) {
  check_arg_types (err, "execution", PCE_NAME);
  sem_args (err, context);
  _possible_types = JoinPointLoc::Method;
}

bool PCE_Execution::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                              Binding &binding, Condition &cond) {
  // consider execution join point only
  if (jpl.type () != JoinPointLoc::Method)
    return false;

  // check if the executed function's name matches the argument
  JPL_Function *function = ((JPL_Method&)jpl).function ();
  if (arg (0)->evaluate (*function, context, binding, cond))
    return true;

  // otherwise check the scope of the function and all its parent scopes
  JPL_Name *scope = ((JPL_Code&)jpl).lexical_scope ();
  return check_scopes (scope, context, binding, cond);
}

PCE_Type PCE_Call::type () const {
  return PCE_CODE;
}

void PCE_Call::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types (err, "call", PCE_NAME);
  sem_args (err, context);
  _possible_types = JoinPointLoc::MethodCall;
}

bool PCE_Call::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                         Binding &binding, Condition &cond) {
  if (jpl.type () != JoinPointLoc::MethodCall)
    return false;

  // check if the called function's name matches the argument
  JPL_Function *target_function = ((JPL_MethodCall&)jpl).target_function ();
  if (arg (0)->evaluate (*target_function, context, binding, cond))
    return true;

  // otherwise check the scope of the function and all its parent scopes
  JPL_Name *scope = target_function->parent ();
  return scope && check_scopes (scope, context, binding, cond);
}


PCE_Type PCE_Construction::type () const {
  return PCE_CODE;
}

void PCE_Construction::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types (err, "construction", PCE_NAME);
  sem_args (err, context);
  _possible_types = JoinPointLoc::Construction;
}

bool PCE_Construction::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                                 Binding &binding, Condition &cond) {
  if (jpl.type () != JoinPointLoc::Construction)
    return false;
  JPL_Name *scope = ((JPL_Construction&)jpl).lexical_scope ();
  return arg (0)->evaluate (*scope, context, binding, cond);
}

PCE_Type PCE_Destruction::type () const {
  return PCE_CODE;
}

void PCE_Destruction::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types (err, "destruction", PCE_NAME);
  sem_args (err, context);
  _possible_types = JoinPointLoc::Destruction;
}

bool PCE_Destruction::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                                Binding &binding, Condition &cond) {
  if (jpl.type () != JoinPointLoc::Destruction)
    return false;
  JPL_Name *scope = ((JPL_Destruction&)jpl).lexical_scope ();
  return arg (0)->evaluate (*scope, context, binding, cond);
}


PCE_Type PCE_That::type () const {
  return PCE_CODE;
}

void PCE_That::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types (err, "that", PCE_NAME);
  context.enter_that ();
  sem_args (err, context);
  context.leave_that ();
  _possible_types = JoinPointLoc::Code;
}

bool PCE_That::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                         Binding &binding, Condition &cond) {
  // 'that' matches only code join points
  if (!(jpl.type () & (int)JoinPointLoc::Code))
    return false;
  JPL_Code &jpl_code = (JPL_Code&)jpl;

  if (jpl.type () == JoinPointLoc::MethodCall &&
      ((JPL_MethodCall&)jpl).is_pseudo ()) {
    context.pseudo_true (true);
    return true;
  }

  // get the function that is executed when the join point is reached
  JPL_Name *parent = jpl_code.parent ();
  if (!parent->type () == JoinPointLoc::Function)
    return false;
  JPL_Function *jpl_function = (JPL_Function*)parent;

  // static member functions and non-member functions don't have a 'this' pointer
  if (jpl_function->is_non_member () || jpl_function->is_static_member ())
    return false;

  // get the class of which the function is a member
  JPL_Class *class_loc = (JPL_Class*)jpl_function->parent ();
  
  context.enter_that ();
  binding._this = (CArgumentInfo *)0;

  // if any of the class' base classes matches the argument the current class
  // is definitely an object of the required type
  if (!check_base_class (*class_loc, context, binding, cond)) {
    // in a construction or destruction the object has exactly the type
    // of the constructor/destructor scope => no run-time check
    if (jpl_function->is_constructor () || jpl_function->is_destructor ()) {
      context.leave_that ();
      return false;
    }
      
    // create a 'that' condition with a mangled name
    stringstream mangled_check;
    arg (0)->mangle_type_check (mangled_check);
    cond.that (class_loc, mangled_check.str ());

    // find the derived classes that match the argument
    check_derived_class (*class_loc, context, binding, cond);
  }
  context.leave_that ();
  return true;
}

PCE_Type PCE_Target::type () const {
  return PCE_CODE;
}

void PCE_Target::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types (err, "target", PCE_NAME);
  context.enter_target ();
  sem_args (err, context);
  context.leave_target ();
  _possible_types = JoinPointLoc::MethodCall;
}

bool PCE_Target::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                           Binding &binding, Condition &cond) {
  // only calls have a target at the moment
  if (jpl.type () != JoinPointLoc::MethodCall)
    return false;
  JPL_MethodCall &jpl_call = (JPL_MethodCall&)jpl;

  JPL_Function *target_function = jpl_call.target_function ();

  // only non-static member functions are relevant here
  // TODO: how to handle explicit destructor calls?
  if (!target_function->is_non_static_member ())
    return false;

  // get the class of which the function is a member
  JPL_Class *class_loc = (JPL_Class*)target_function->parent ();
  
  context.enter_target ();
  binding._target = (CArgumentInfo *)0;
  // if any of the class' base classes matches the argument the current class
  // is definitely an object of the required type
  if (!check_base_class (*class_loc, context, binding, cond)) {
    // create a 'that' condition with a mangled name
    stringstream mangled_check;
    arg (0)->mangle_type_check (mangled_check);
    cond.target (class_loc, mangled_check.str ());

    // find the derived classes that match the argument
    check_derived_class (*class_loc, context, binding, cond);
  }
  context.leave_target ();
  return true;
}

PCE_Type PCE_CFlow::type () const {
  return PCE_CODE;
}

void PCE_CFlow::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types (err, "cflow", PCE_CODE);
  sem_args (err, context);
  // evaluate the argument pointcut
  JoinPointModel::Selection all;
  context.jpm ().select (JoinPointLoc::Code, all);
  for (JoinPointModel::Selection::iterator iter = all.begin ();
       iter != all.end (); ++iter) {
    JoinPointLoc &jpl = **iter;
    Binding cflow_binding;
    Condition cflow_condition;
    context.pseudo_true (false); // todo sichern!
    if (arg (0)->evaluate (jpl, context, cflow_binding, cflow_condition)) {
      if (cflow_binding._used) {
        err << sev_error 
            << node ()->token ()->location ()
            << "context variables not supported in cflows" << endMessage;
        break;
      }
      if (cflow_condition) {
        err << sev_error 
            << node ()->token ()->location ()
            << "runtime conditions not supported in cflows" << endMessage;
        break;
      }
      _arg_pointcut.append (*new JoinPoint (&jpl, cflow_condition));
    }
  }
  _index = context.cflow (this); // remember this node in the context
  _possible_types = JoinPointLoc::Code;
}

bool PCE_CFlow::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                          Binding &binding, Condition &cond) {
  // check if this join point is one of the cflow starting points
  if (_arg_pointcut.find (&jpl) != _arg_pointcut.end ())
    return true; // no runtime condition check!
    
  // every other joinpoint might be in the cflow (space for improvement)
  cond.cflow (_index);
  return true;
}

PCE_Type PCE_Args::type () const {
  return PCE_CODE;
}

void PCE_Args::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types (err, "args", PCE_NAME);

  // recursively do a semantic analysis of child nodes
  int saved_arg = context.get_arg ();
  for (int i = 0; i < args (); i++) {
    context.set_arg (i);
    if (arg (i))
      arg (i)->semantics (err, context);
  }
  context.set_arg (saved_arg);
  _possible_types = JoinPointLoc::Code;
}

bool PCE_Args::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                         Binding &binding, Condition &cond) {

  // ignore all non-code join points
  if (!(jpl.type () & (int)JoinPointLoc::Code))
    return false;
  JPL_Code *jpl_code = (JPL_Code*)&jpl;
    
  // check if the number of arguments is ok -> not compatible ("...")
  if (args () != jpl_code->arg_count ())
    return false;

  // check if all argument types match
  int saved_arg = context.get_arg ();
  for (unsigned a = 0; a < (unsigned)jpl_code->arg_count (); a++) {
    context.set_arg (a);
    binding._args[a] = (CArgumentInfo*)0;
    JPL_Type &type_loc = (JPL_Type&)jpl_code->arg_type ((int)a);
    if (!arg (a)->evaluate (type_loc, context, binding, cond)) {
      context.set_arg (saved_arg);
      return false;
    }
  }
  context.set_arg (saved_arg);

  return true;
}

PCE_Type PCE_Result::type () const {
  return PCE_CODE;
}

void PCE_Result::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types (err, "result", PCE_NAME);
  context.enter_result ();
  sem_args (err, context);
  context.leave_result ();
  _possible_types = (JoinPointLoc::join_point_type)
    (JoinPointLoc::Method|JoinPointLoc::MethodCall);
}

bool PCE_Result::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                           Binding &binding, Condition &cond) {

  // only execution and call join points have a result
  if (jpl.type () != JoinPointLoc::Method &&
      jpl.type () != JoinPointLoc::MethodCall)
    return false;
  
  JPL_Code *jpl_code = (JPL_Code*)&jpl;
  JPL_Type &type_loc = (JPL_Type&)jpl_code->result_type ();

  context.enter_result ();
  binding._result = (CArgumentInfo *)0;
  bool result = arg (0)->evaluate (type_loc, context, binding, cond);
  context.leave_result ();
  return result;
}


PCE_Type PCE_Or::type () const {
  return arg (0)->type ();
}

void PCE_Or::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types_equal (err, "||");
  sem_args (err, context);
  _possible_types = (JoinPointLoc::join_point_type)
    (arg(0)->possible_types () | arg(1)->possible_types ());
}

bool PCE_Or::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                       Binding &binding, Condition &cond) {
  Condition subcond0, subcond1;
  bool subresult0, subresult1;
  subresult0 = arg (0)->evaluate (jpl, context, binding, subcond0);
  subresult1 = arg (1)->evaluate (jpl, context, binding, subcond1);
  // if both subresults are false the disjunction is false, too
  if (!(subresult0 || subresult1))
    return false;
    
  // if any of the subresults was an unconditional true the result is true
  if ((subresult0 && !subcond0) || (subresult1 && !subcond1))
    return true;
    
  // at least one subresult was true, now we consider possible conditions
  if (subcond0) {
    cond.assign (subcond0);
    if (subcond1)
      cond.op_or (subcond1);
  }
  else if (subcond1) {
    cond.assign (subcond1);
  }
  return true;
}

PCE_Type PCE_And::type () const {
  return arg (0)->type ();
}

void PCE_And::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types_equal (err, "&&");
  sem_args (err, context);
  _possible_types = (JoinPointLoc::join_point_type)
    (arg(0)->possible_types () & arg(1)->possible_types ());
}

bool PCE_And::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                        Binding &binding, Condition &cond) {
  Condition subcond0, subcond1;
  bool subresult0, subresult1;
  subresult0 = arg (0)->evaluate (jpl, context, binding, subcond0);
  bool pseudo_true0 = context.is_pseudo_true ();
  context.pseudo_true (false);
  subresult1 = arg (1)->evaluate (jpl, context, binding, subcond1);
  bool pseudo_true1 = context.is_pseudo_true ();
  context.pseudo_true (pseudo_true0 && pseudo_true1);
  
  // if any subresult was false the conjunction is false, too
  if (!(subresult0 && subresult1))
    return false;
  // both subresult were true, now we consider possible conditions
  if (subcond0) {
    cond.assign (subcond0);
    if (subcond1)
      cond.op_and (subcond1);
  }
  else if (subcond1) {
    cond.assign (subcond1);
  }
  return true;
}

PCE_Type PCE_Not::type () const {
  return arg (0)->type ();
}

void PCE_Not::semantics (ErrorStream &err, PointCutContext &context) {
  check_arg_types (err, "!", type ());
  sem_args (err, context);
  if (type () == PCE_NAME)
    _possible_types = JoinPointLoc::Name;
  else if (type () == PCE_CODE)
    _possible_types = JoinPointLoc::Code;
  else
    _possible_types = JoinPointLoc::None;
}

bool PCE_Not::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                        Binding &binding, Condition &cond) {
  Condition subcond;
  bool subresult = arg (0)->evaluate (jpl, context, binding, subcond);
  // if the argument was true because we evaluate a pseudo call return true
  if (context.is_pseudo_true ())
    return true;
  // if the subexpression evaluates to an unconditional 'true' or false the
  // result must be inverted
  if (!subcond)
    return !subresult;
  // otherwise the result is 'true' but the condition must be negated
  else {
    cond.assign (subcond);
    cond.op_not ();
    return true;
  }
}

PCE_Type PCE_Named::type () const {
  return arg (0)->type ();
}

void PCE_Named::semantics (ErrorStream &err, PointCutContext &context) {
  ArgSet new_arg_bindings; // must be defined here and not in a local scope

  check_arg_types (err, "pointcut", type ());
  
  if (!node ()) {
    // root node of an expression, no call
    for (unsigned a = 0; a < context.func ()->Arguments (); a++)
      new_arg_bindings.append (context.func ()->Argument (a));
  }
  else {
    CT_ExprList *args = ((CT_CallExpr*)node ())->Arguments ();
    for (int s = 0; s < args->Entries (); s++) {
      CTree *arg = args->Entry (s);

      // check if the current argument node type is not a SimpleName
      if (arg->NodeName () != CT_SimpleName::NodeId ()) {
	err << sev_error << node ()->token ()->location ()
	    << "pointcut argument " << s << " must be an identifier" 
	    << endMessage;
	new_arg_bindings.append (0);
	continue;
      }

      // Argument must be a formal, lookup in the class database
      const char *name = ((CT_SimpleName*)arg)->Object ()->Name ();
      // get the argument binding
      const CArgumentInfo *arginfo = get_binding (name, err, context);;
      // .. and add it to the next binding context
      new_arg_bindings.append (arginfo);
    }
  }

  // set the current function pointer for context var lookups
  CFunctionInfo *saved_func = context.func (_func);

  // push the argument binding
  context.arg_bindings ().push (&new_arg_bindings);

  // analyze referenced expression tree
  sem_args (err, context);

  // pop the argument binding from the stack again
  context.arg_bindings ().pop ();

  context.func (saved_func); // restore old function pointer
  
  _possible_types = (JoinPointLoc::join_point_type)arg(0)->possible_types ();
}

bool PCE_Named::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                          Binding &binding, Condition &cond) {
  return arg (0)->evaluate (jpl, context, binding, cond);
}

PCE_Type PCE_ContextVar::type () const {
  return PCE_NAME;
}

void PCE_ContextVar::semantics (ErrorStream &err, 
				PointCutContext &context) {
  if (!(_bound_arg  = get_binding (_name, err, context)))
    return;
  _check_type = _bound_arg->TypeInfo ();

  if (context.in_that () || context.in_target ()) {
    if (_check_type->isPointer () && _check_type->BaseType ()->is_void ()) {
      // "void*" matches any class!
      _check_type = 0;
    }
    else {
      if (_check_type->isPointer () || _check_type->isAddress ())
      	_check_type = _check_type->BaseType ();
      if (!_check_type->isClass ()) {
        err << sev_error << node ()->token ()->location ()
            << "argument of 'that' or 'target' must be a class, pointer to "
            << "class, or reference to class" << endMessage;
      }
    }
    _bound_to = context.in_that () ? CV_THAT : CV_TARGET;
  }
  else if (context.in_result ()) {
    _bound_to = CV_RESULT;
  }
  else {
    _bound_to = CV_ARG;
    _arg = context.get_arg ();
  }
  if (_bound_to == CV_ARG || _bound_to == CV_RESULT)
    _possible_types = JoinPointLoc::Type;
  else if (_bound_to == CV_TARGET || _bound_to == CV_THAT)
    _possible_types = (JoinPointLoc::join_point_type)
      (JoinPointLoc::Class|JoinPointLoc::Aspect);
  else
    _possible_types = JoinPointLoc::None;
}

bool PCE_ContextVar::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                               Binding &binding, Condition &cond) {

  assert (((_bound_to == CV_ARG || _bound_to == CV_RESULT) &&
           jpl.type () == JoinPointLoc::Type) ||
	        ((_bound_to == CV_TARGET || _bound_to == CV_THAT) && 
	         (jpl.type () & (JoinPointLoc::Class | JoinPointLoc::Aspect))));

  // determine the current type
  CTypeInfo *curr_type = 0;
  if (jpl.type () == JoinPointLoc::Type)
    curr_type = TI_Type::of ((JPL_Type&)jpl)->type_info ();
  else if (jpl.type () == JoinPointLoc::Class ||
           jpl.type () == JoinPointLoc::Aspect)
    curr_type = ((TI_Class*)((JPL_Class&)jpl).transform_info ())->class_info ()->TypeInfo ();

  assert (curr_type);
  if (!curr_type)
    return false;
    
  // check if the current type matches
  if (_check_type && *_check_type != *curr_type)
    return false;

  // do the binding
  binding._used = true;
  switch (_bound_to) {
  case CV_THAT:
    binding._this = (CArgumentInfo*)_bound_arg;
    break;
  case CV_TARGET:
    binding._target = (CArgumentInfo*)_bound_arg;
    break;
  case CV_RESULT:
    binding._result = (CArgumentInfo*)_bound_arg;
    break;
  case CV_ARG:
    binding._args[_arg] = (CArgumentInfo*)_bound_arg;
    break;
  }
  return true;
}

void PCE_ContextVar::mangle_type_check (ostream &out) {
  _check_type->Mangled (out);
}


PCE_Type PCE_Match::type () const {
  return PCE_NAME;
}

void PCE_Match::semantics (ErrorStream &err, PointCutContext &context) {
  _match_expr.parse (err, node ()->token ()->location (), _str.c_str ());
  if (_match_expr.is_function ())
    _possible_types = JoinPointLoc::Function;
  else if (_match_expr.is_type ())
    _possible_types = (JoinPointLoc::join_point_type)
      (JoinPointLoc::Class|JoinPointLoc::Aspect);
  else
    _possible_types = JoinPointLoc::None;
}

bool PCE_Match::evaluate (JoinPointLoc &jpl, PointCutContext &context,
                          Binding &binding, Condition &cond) {
  assert (!_match_expr.error ());
  
  if (jpl.type () == JoinPointLoc::Type) {
    if (_match_expr.is_type ()) {
      JPL_Type &jpl_type = (JPL_Type&)jpl;
      return _match_expr.matches ((MatchSignature &)jpl_type.match_signature ());
    }
    return false;    
  }
  else if ((jpl.type () & JoinPointLoc::Name) != 0) {
    JPL_Name &jpl_name = (JPL_Name&)jpl;
    if ((jpl.type () == JoinPointLoc::Function && _match_expr.is_function ()) ||
      ((jpl.type () & (JoinPointLoc::Class | JoinPointLoc::Aspect) &&
      _match_expr.is_type ()))) {
      return _match_expr.matches ((MatchSignature &)jpl_name.match_signature ());
    }
    return false;
  }
  return false;
}

void PCE_Match::mangle_type_check (ostream &out) {
  const char *str = _str.c_str ();
  while (*str) {
    switch (*str) {
      case ':' : out << "___S___"; break;
      case '%' : out << "___A___"; break;
      case '(' : out << "___L___"; break;
      case ')' : out << "___R___"; break;
      case ',': out << "___C___"; break;
      default: out << *str;
    }
    str++;
  }
}

