// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __CObjectInfo_h__
#define __CObjectInfo_h__

// base class of all semantic infos
// knows the type of the info, the name of the entity it
// represents, its qualified name (on demand), its type,
// its object linkage, its storage duration, its class
// member access level (protection), and the specifiers 
// that were used at the declaration of the entity,
// knows the source file location of the entity's declaration,
// the corresponding syntax tree node, the class database 
// it belongs to, and other semantic info objects that refer
// to the same entity

#include "Puma/CSourceInfo.h"
#include "Puma/CProtection.h"
#include "Puma/CStorage.h"
#include "Puma/CLinkage.h"
#include "Puma/CLanguage.h"
#include "Puma/CSpecifiers.h"
#include "Puma/Array.h"
#include "Puma/DString.h"

namespace Puma {


class CRecord;
class CStructure;
class CFileInfo;
class CClassInfo;
class CBaseClassInfo;
class CMemberAliasInfo;
class CTypeInfo;
class CScopeInfo;
class CLabelInfo;
class CUnionInfo;
class CLocalScope;
class CEnumInfo;
class CUsingInfo;
class CEnumeratorInfo;
class CTypedefInfo;
class CFunctionInfo;
class CArgumentInfo;
class CAttributeInfo;
class CTemplateParamInfo;
class CTemplateInfo;
class CNamespaceInfo;
class CClassInstance;
class CUnionInstance;
class CFctInstance;
class CClassDatabase;
class CTree;
class CT_ExprList;
class CTemplateInstance;

class CObjectInfo {
protected:
  enum ObjectId {
    FILE_INFO, 
    UNION_INFO, 
    CLASS_INFO,
    BASECLASS_INFO,
    MEMBERALIAS_INFO,
    ENUM_INFO, 
    TYPEDEF_INFO, 
    FUNCTION_INFO, 
    LABEL_INFO,
    ENUMERATOR_INFO,
    ATTRIBUTE_INFO, 
    TEMPLATE_PARAM_INFO,
    TEMPLATE_INFO,
    CLASS_INSTANCE_INFO,
    UNION_INSTANCE_INFO,
    FCT_INSTANCE_INFO,
    ARGUMENT_INFO, 
    LOCAL_INFO, 
    NAMESPACE_INFO,
    USING_INFO
  };

  bool               _DeleteMembersOnly;
  
private:
  DString            _Name;
  const char        *_QualName;         // Full qualified name
  bool              _abs, _tdef;        // Flags that were used for QualName()
  CTypeInfo         *_TypeInfo;
  CObjectInfo       *_BaseObject;       // corresponding object of base class
  CObjectInfo       *_Next;             // next linked object
  CObjectInfo       *_Prev;             // previous linked object
  CClassDatabase    *_ClassDB;
  CTree             *_Tree;             // corresponding syntax tree
  ObjectId           _Id;               // object type
  CSpecifiers        _Specifiers;
  CSourceInfo        _SourceInfo;       // location in source file
  CProtection::Type  _Protection;
  CLinkage::Type     _Linkage;
  CStorage::Type     _Storage;            
  CLanguage          _Language;

protected:
  CStructure        *_QualScope;        // != 0 only in CAttributeInfo, CFunctionInfo, and CRecord
  CStructure        *_AssignedScope;    // -real- scope of friend classes
  Array<CStructure*> _Registered;         
  
public: 
  ~CObjectInfo ();
  
  // compares the addresses of this and all linked objects
  // to determine whether two objects denote the same entity
  bool operator ==(const CObjectInfo &) const;
  bool operator !=(const CObjectInfo &) const;

  // ask the object info type
  CObjectInfo        *ObjectInfo () const;
  CLabelInfo         *LabelInfo () const;
  CMemberAliasInfo   *MemberAliasInfo () const; 
  CBaseClassInfo     *BaseClassInfo () const;
  CUsingInfo         *UsingInfo () const;
  CTypedefInfo       *TypedefInfo () const;
  CArgumentInfo      *ArgumentInfo () const;
  CAttributeInfo     *AttributeInfo () const;
  CTemplateParamInfo *TemplateParamInfo () const; 
  CStructure         *Structure () const; 
  CFileInfo          *FileInfo () const;  
  CRecord            *Record () const;        
  CLocalScope        *LocalScope () const;
  CScopeInfo         *ScopeInfo () const; 
  CClassInfo         *ClassInfo () const;
  CUnionInfo         *UnionInfo () const;
  CEnumInfo          *EnumInfo () const;
  CFunctionInfo      *FunctionInfo () const;
  CNamespaceInfo     *NamespaceInfo () const;
  CEnumeratorInfo    *EnumeratorInfo () const;
  CTemplateInfo      *TemplateInfo () const; 
  CClassInstance     *ClassInstance () const; 
  CFctInstance       *FctInstance () const; 
  CUnionInstance     *UnionInstance () const; 
  CTemplateInstance  *TemplateInstance () const;
  CRecord            *ClassScope () const;

  // Get ...
  ObjectId            Id () const;
  const DString&      Name () const; 
  const char         *QualName (bool abs = false, // created on demand
                                bool tdef = false);
  CObjectInfo        *DefObject () const;         // info of the definition
  CTypeInfo          *TypeInfo () const;
  CScopeInfo         *Scope () const;             // enclosing scope
  CStructure         *QualifiedScope () const;    // scope of qualified names
  CSourceInfo        *SourceInfo () const;        // location in source file
  CClassDatabase     *ClassDB () const; 
  CTree              *Tree () const;
  CObjectInfo        *NextObject () const;        // next linked object
  CObjectInfo        *PrevObject () const;        // previous linked object
  bool                isAnonymous () const;       // has only a private name?
  bool                isTemplate () const;
  bool                isTemplateInstance () const;
  bool                isBuiltin () const;         // has no tree?
  bool                isClassMember () const;     // is method or data member?
  CTemplateInfo      *Template () const;          // template info of template
  CObjectInfo        *BaseObject () const;        // base class object; not yet set
  CT_ExprList        *Init () const;              // get initializer (if any)
  CProtection::Type   Protection () const;        // member access level
  CLinkage::Type      Linkage () const;           // object linkage
  CStorage::Type      Storage () const;           // storage duration
  const CLanguage    &Language () const;          // entity encoding language
  CLanguage          &Language ();                // entity encoding language
  bool                isVirtual () const;         // defined virtual?
  bool                isStatic () const;          // defined static?
  bool                isExtern () const;          // defined extern?
  bool                isMutable () const;         // defined mutable?
  bool                isRegister () const;        // defined register?
  bool                isExplicit () const;        // defined explicit?
  bool                isInline () const;          // defined inline?
  bool                isAuto () const;            // defined auto?
  CStructure          *AssignedScope () const; 
  bool                isRegistered (const CStructure*) const;

  // Set ...
  void                Name (const char* s) { Name(DString(s==0?"":s)); }
  void                Name (const DString&);
  void                TypeInfo (CTypeInfo *);
  void                BaseObject (CObjectInfo *);
  void                Protection (CProtection::Type);
  void                Linkage (CLinkage::Type);
  void                Storage (CStorage::Type);
  void                FileInfo (CFileInfo *);
  void                Tree (CTree *);    
  void                ClassDB (CClassDatabase *);
  void                DeleteMembersOnly ();
  void                NextObject (CObjectInfo *); 
  void                PrevObject (CObjectInfo *); 
  void                Unlink (); 
  void                Register (CStructure *);
  void                Unregister (CStructure *);
  void                isVirtual (bool);
  void                isStatic (bool);
  void                isExtern (bool);
  void                isMutable (bool);
  void                isRegister (bool);
  void                isExplicit (bool);
  void                isInline (bool);
  void                isAuto (bool);
  void                AssignedScope (CStructure *);

protected:  
  CObjectInfo (ObjectId);

  void CleanUp ();
};

inline CObjectInfo::CObjectInfo (CObjectInfo::ObjectId id) :
  _DeleteMembersOnly (false),
  _Name              (),
  _QualName          ((const char*)0),
  _TypeInfo          ((CTypeInfo*)0),
  _BaseObject        ((CObjectInfo*)0),
  _Next              ((CObjectInfo*)this),
  _Prev              ((CObjectInfo*)this),
  _ClassDB           ((CClassDatabase*)0),
  _Tree              ((CTree*)0),
  _Id                (id),
  _Protection        (CProtection::PROT_NONE), 
  _Linkage           (CLinkage::LINK_NONE),
  _Storage           (CStorage::CLASS_NONE),
  _QualScope         ((CStructure*)0),
  _AssignedScope     ((CStructure*)0)
 {}

inline CObjectInfo::ObjectId CObjectInfo::Id () const
 { return _Id; }

inline bool CObjectInfo::operator !=(const CObjectInfo &info) const
 { return ! (*this == info); }

inline const DString& CObjectInfo::Name () const 
 { return _Name; }
inline CTypeInfo *CObjectInfo::TypeInfo () const
 { return _TypeInfo; }
inline CSourceInfo *CObjectInfo::SourceInfo () const
 { return (CSourceInfo*)&_SourceInfo; }
inline CProtection::Type CObjectInfo::Protection () const
 { return _Protection; }
inline CLinkage::Type CObjectInfo::Linkage () const
 { return _Linkage; }
inline CStorage::Type CObjectInfo::Storage () const
 { return _Storage; }
inline CLanguage &CObjectInfo::Language ()
 { return _Language; }
inline const CLanguage &CObjectInfo::Language () const
 { return _Language; }
inline CObjectInfo *CObjectInfo::BaseObject () const
 { return _BaseObject; }
inline CTree *CObjectInfo::Tree () const 
 { return _Tree; }
inline CClassDatabase *CObjectInfo::ClassDB () const
 { return _ClassDB; }
inline CT_ExprList *CObjectInfo::Init () const 
 { return (CT_ExprList*)0; }

inline void CObjectInfo::Protection (CProtection::Type s)
 { _Protection = s; }
inline void CObjectInfo::Linkage (CLinkage::Type s)
 { _Linkage = s; }
inline void CObjectInfo::Storage (CStorage::Type s)
 { _Storage = s; }
inline void CObjectInfo::BaseObject (CObjectInfo *info)
 { _BaseObject = info; }
inline void CObjectInfo::Tree (CTree *t)    
 { _Tree = t; }
inline void CObjectInfo::ClassDB (CClassDatabase *db) 
 { _ClassDB = db; }
inline void CObjectInfo::DeleteMembersOnly () 
 { _DeleteMembersOnly = true; }

inline void CObjectInfo::Register (CStructure *scope)
 { _Registered.append (scope); }

inline bool CObjectInfo::isAnonymous () const 
 { return (Name ().empty () || *Name ().c_str () == '%'); }
inline bool CObjectInfo::isBuiltin () const
 { return ! (Tree ()); }
inline bool CObjectInfo::isTemplateInstance () const 
 { return (ClassInstance () || UnionInstance () || FctInstance ()); }

inline bool CObjectInfo::isVirtual () const
 { return _Specifiers == CSpecifiers::SPEC_VIRTUAL; }
inline bool CObjectInfo::isStatic () const
 { return _Specifiers == CSpecifiers::SPEC_STATIC; }
inline bool CObjectInfo::isExtern () const
 { return _Specifiers == CSpecifiers::SPEC_EXTERN; }
inline bool CObjectInfo::isMutable () const
 { return _Specifiers == CSpecifiers::SPEC_MUTABLE; }
inline bool CObjectInfo::isRegister () const
 { return _Specifiers == CSpecifiers::SPEC_REGISTER; }
inline bool CObjectInfo::isExplicit () const
 { return _Specifiers == CSpecifiers::SPEC_EXPLICIT; }
inline bool CObjectInfo::isInline () const
 { return _Specifiers == CSpecifiers::SPEC_INLINE; }
inline bool CObjectInfo::isAuto () const
 { return _Specifiers == CSpecifiers::SPEC_AUTO; }

inline void CObjectInfo::isVirtual (bool v)
 { if (v) _Specifiers += CSpecifiers::SPEC_VIRTUAL; 
   else   _Specifiers -= CSpecifiers::SPEC_VIRTUAL; }
inline void CObjectInfo::isStatic (bool v)
 { if (v) _Specifiers += CSpecifiers::SPEC_STATIC; 
   else   _Specifiers -= CSpecifiers::SPEC_STATIC; }
inline void CObjectInfo::isExtern (bool v)
 { if (v) _Specifiers += CSpecifiers::SPEC_EXTERN; 
   else   _Specifiers -= CSpecifiers::SPEC_EXTERN; }
inline void CObjectInfo::isMutable (bool v)
 { if (v) _Specifiers += CSpecifiers::SPEC_MUTABLE; 
   else   _Specifiers -= CSpecifiers::SPEC_MUTABLE; }
inline void CObjectInfo::isRegister (bool v)
 { if (v) _Specifiers += CSpecifiers::SPEC_REGISTER; 
   else   _Specifiers -= CSpecifiers::SPEC_REGISTER; }
inline void CObjectInfo::isExplicit (bool v)
 { if (v) _Specifiers += CSpecifiers::SPEC_EXPLICIT; 
   else   _Specifiers -= CSpecifiers::SPEC_EXPLICIT; }
inline void CObjectInfo::isInline (bool v)
 { if (v) _Specifiers += CSpecifiers::SPEC_INLINE; 
   else   _Specifiers -= CSpecifiers::SPEC_INLINE; }
inline void CObjectInfo::isAuto (bool v)
 { if (v) _Specifiers += CSpecifiers::SPEC_AUTO; 
   else   _Specifiers -= CSpecifiers::SPEC_AUTO; }

// link objects

inline CObjectInfo *CObjectInfo::NextObject () const
 { return _Next; }
inline CObjectInfo *CObjectInfo::PrevObject () const
 { return _Prev; }

inline CMemberAliasInfo *CObjectInfo::MemberAliasInfo () const
 { return _Id == MEMBERALIAS_INFO ? (CMemberAliasInfo*)this : (CMemberAliasInfo*)0; }
inline CBaseClassInfo *CObjectInfo::BaseClassInfo () const
 { return _Id == BASECLASS_INFO ? (CBaseClassInfo*)this : (CBaseClassInfo*)0; }
inline CClassInfo *CObjectInfo::ClassInfo () const
 { return _Id == CLASS_INFO || _Id == CLASS_INSTANCE_INFO ? (CClassInfo*)this : (CClassInfo*)0; }
inline CUnionInfo *CObjectInfo::UnionInfo () const
 { return _Id == UNION_INFO || _Id == UNION_INSTANCE_INFO ? (CUnionInfo*)this : (CUnionInfo*)0; }
inline CEnumInfo *CObjectInfo::EnumInfo () const
 { return _Id == ENUM_INFO ? (CEnumInfo*)this : (CEnumInfo*)0; }
inline CUsingInfo *CObjectInfo::UsingInfo () const
 { return _Id == USING_INFO ? (CUsingInfo*)this : (CUsingInfo*)0; }
inline CEnumeratorInfo *CObjectInfo::EnumeratorInfo () const
 { return _Id == ENUMERATOR_INFO ? (CEnumeratorInfo*)this : (CEnumeratorInfo*)0; }
inline CTypedefInfo *CObjectInfo::TypedefInfo () const
 { return _Id == TYPEDEF_INFO ? (CTypedefInfo*)this : (CTypedefInfo*)0; }
inline CFunctionInfo *CObjectInfo::FunctionInfo () const
 { return _Id == FUNCTION_INFO || _Id == FCT_INSTANCE_INFO ? (CFunctionInfo*)this : (CFunctionInfo*)0; }
inline CArgumentInfo *CObjectInfo::ArgumentInfo () const
 { return _Id == ARGUMENT_INFO ? (CArgumentInfo*)this : (CArgumentInfo*)0; }
inline CAttributeInfo *CObjectInfo::AttributeInfo () const
 { return _Id == ATTRIBUTE_INFO || _Id == ENUMERATOR_INFO ? (CAttributeInfo*)this : (CAttributeInfo*)0; }
inline CNamespaceInfo *CObjectInfo::NamespaceInfo () const
 { return _Id == NAMESPACE_INFO || _Id == FILE_INFO ? (CNamespaceInfo*)this : (CNamespaceInfo*)0; }
inline CLabelInfo *CObjectInfo::LabelInfo () const
 { return _Id == LABEL_INFO ? (CLabelInfo*)this : (CLabelInfo*)0; }
inline CFileInfo *CObjectInfo::FileInfo () const
 { return _Id == FILE_INFO ? (CFileInfo*)this : (CFileInfo*)0; }
inline CStructure *CObjectInfo::Structure () const
 { return NamespaceInfo () || ClassInfo () || UnionInfo () || 
          FunctionInfo () || TemplateInfo () || LocalScope () ? 
          (CStructure*)this : (CStructure*)0; }
inline CRecord *CObjectInfo::Record () const
 { return ClassInfo () || UnionInfo () ? (CRecord*)this : (CRecord*)0; }
inline CLocalScope *CObjectInfo::LocalScope () const
 { return _Id == LOCAL_INFO ? (CLocalScope*)this : (CLocalScope*)0; }
inline CTemplateParamInfo *CObjectInfo::TemplateParamInfo () const
 { return _Id == TEMPLATE_PARAM_INFO ? (CTemplateParamInfo*)this : (CTemplateParamInfo*)0; }
inline CTemplateInfo *CObjectInfo::TemplateInfo () const
 { return _Id == TEMPLATE_INFO ? (CTemplateInfo*)this : (CTemplateInfo*)0; }
inline CScopeInfo *CObjectInfo::ScopeInfo () const
 { return /*_Id == SCOPE_INFO ||*/ Structure () ? (CScopeInfo*)this : (CScopeInfo*)0; }
inline CObjectInfo *CObjectInfo::ObjectInfo () const
 { return (CObjectInfo*)this; }
inline CClassInstance *CObjectInfo::ClassInstance () const
 { return _Id == CLASS_INSTANCE_INFO ? (CClassInstance*)this : (CClassInstance*)0; }
inline CUnionInstance *CObjectInfo::UnionInstance () const
 { return _Id == UNION_INSTANCE_INFO ? (CUnionInstance*)this : (CUnionInstance*)0; }
inline CFctInstance *CObjectInfo::FctInstance () const
 { return _Id == FCT_INSTANCE_INFO ? (CFctInstance*)this : (CFctInstance*)0; }

inline CStructure *CObjectInfo::QualifiedScope () const { 
  return _QualScope;
}

inline void CObjectInfo::AssignedScope (CStructure *s)
 { _AssignedScope = s; }
inline CStructure *CObjectInfo::AssignedScope () const
 { return _AssignedScope; } 


} // namespace Puma

#endif /* __CObjectInfo_h__ */
