// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __Syntax_h__
#define __Syntax_h__

#include "Puma/Token.h"
#include "Puma/Builder.h"
//#include "Puma/Semantic.h"
#include "Puma/ErrorSink.h"
#include "Puma/TokenProvider.h"

namespace Puma {

class Semantic;
class Config;
class CTree;

class Syntax {
  Token *_problem_token;
  bool _have_error;

  Builder &_builder;
  Semantic &_semantic;

public:
  TokenProvider *token_provider;

public: 
  struct State : public TokenProvider::State {
    State () {}
    State (int) {}
    State (const TokenProvider::State &s) : TokenProvider::State (s) {}
  };

protected:
  Syntax (Builder &, Semantic &);
  virtual ~Syntax () {}

public:
  CTree *run (TokenProvider &);
  template <class T> CTree *run (TokenProvider &, CTree *(T::*)());
  virtual void configure (Config &) {}
  TokenProvider *provider () const { return token_provider; }
  Token *problem () const;
  bool error () const;
  bool look_ahead (int, unsigned = 1);
  bool look_ahead (int*, unsigned = 1);
  inline int look_ahead () const;
  inline bool consume ();

protected:
  template <class T> bool parse (CTree *(T::*)());
  template <class T> bool seq (CTree *(T::*)());
  template <class T> bool seq (bool (T::*)());
  template <class T> bool list (CTree *(T::*)(), int, bool = false);
  template <class T> bool list (CTree *(T::*)(), int*, bool = false);
  template <class T> bool list (bool (T::*)(), int, bool = false);
  template <class T> bool list (bool (T::*)(), int*, bool = false);
  template <class T> bool catch_error (CTree *(T::*)(), const char*, int*, int*);
  bool parse (int);
  bool parse (int *);
  bool parse_token (int);
  bool opt (bool) const;

protected:
  Builder &builder () const;
  Semantic &semantic () const;

protected:
  // initial parse rule
  virtual CTree *trans_unit ();

  // handle a compiler directive token
  virtual void handle_directive ();

protected:
  State save_state ();
  void forget_state ();
  void restore_state ();
  void restore_state (State);
  void set_state (State);

  bool accept (CTree *, State);

  Token *locate_token ();
  void skip ();
  void skip_block (int, int);
  void skip_curly_block ();
  void skip_round_block ();
  void parse_block (int, int);
  void parse_curly_block ();
  void parse_round_block ();
  bool skip (int, bool = true);
  bool skip (int *, bool = true);
  bool is_in (int, int *) const;
};

inline Syntax::Syntax (Builder &b, Semantic &s) : 
  _problem_token ((Token*)0), 
  _have_error (false),
  _builder (b),
  _semantic (s) {
}

template <class T> 
CTree *Syntax::run (TokenProvider &tp, CTree *(T::*rule)()) {
  TokenProvider *tp_save = token_provider;
  token_provider = &tp;
  _have_error = false;
  _problem_token = (Token*)0;
  locate_token ();
  CTree *result = parse (rule) ? builder ().Top () : (CTree*)0;
  if (result) builder ().Pop ();
  token_provider = tp_save;
  return result;
}


inline int Syntax::look_ahead () const {
  Token *token = token_provider->current ();
  return token ? token->type () : 0;
}

inline bool Syntax::consume () {
  Token *t = token_provider->current ();
  _problem_token = (Token*)0;
  token_provider->next ();
  locate_token ();
  builder ().Push (builder ().token (t));
  return true;
}
 

// Workaround for (old) Puma overload resolution problem
#ifndef __puma
inline bool Syntax::parse (int token_type) { return parse_token (token_type); }
#endif 

inline bool Syntax::opt (bool) const { return true; }
inline bool Syntax::error () const { return _have_error; }
inline Token *Syntax::problem () const { return _problem_token; }
inline Builder &Syntax::builder () const { return _builder; }
inline Semantic &Syntax::semantic () const { return _semantic; }

inline CTree *Syntax::trans_unit () { return (CTree*)0; }

inline void Syntax::handle_directive () {
  while (token_provider->current () &&
         token_provider->current ()->is_directive ()) 
    token_provider->next ();
}

#ifndef __puma

template <class T> 
inline bool Syntax::parse (CTree *(T::*rule)()) {
  State s = save_state ();
  return accept ((((T*)this)->*rule) (), s);
}

template <class T> 
inline bool Syntax::seq (CTree *(T::*rule)()) {
  if (! parse (rule))
    return false;
  while (parse (rule));
  return true;
}

template <class T> 
inline bool Syntax::seq ( bool (T::*rule)()) {
  if (!(((T*)this)->*rule) ())
    return false;
  while ((((T*)this)->*rule) ());
  return true;
}

template <class T> 
inline bool Syntax::list (CTree *(T::*rule)(), int token_type, bool end_sep) {
  if (! parse (rule))
    return false;
  while (parse (token_type))
    if (! parse (rule))
      return end_sep ? true : false;
  return true;
}

template <class T> 
inline bool Syntax::list (bool (T::*rule)(), int token_type, bool end_sep) {
  if (! (((T*)this)->*rule) ())
    return false;
  while (parse (token_type))
    if (! (((T*)this)->*rule) ())
      return end_sep ? true : false;
  return true;
}

template <class T> 
inline bool Syntax::list (CTree *(T::*rule)(), int *token_type_set, bool end_sep) {
  if (! parse (rule))
    return false;
  while (parse (token_type_set))
    if (! parse (rule))
      return end_sep ? true : false;
  return true;
}

template <class T> 
inline bool Syntax::list (bool (T::*rule)(), int *token_type_set, bool end_sep) {
  if (! (((T*)this)->*rule) ())
    return false;
  while (parse (token_type_set))
    if (! (((T*)this)->*rule) ())
      return end_sep ? true : false;
  return true;
}

template <class T> 
bool Syntax::catch_error (CTree *(T::*rule)(), const char *msg,
                          int *finish_tokens, int *skip_tokens) {
  Token *current_pos, *token;
  int index;
  
  current_pos = token_provider->current ();
  if (! current_pos || is_in (current_pos->type (), finish_tokens))
    return false;

  index = ((ErrorCollector&)builder ().err ()).index ();
  if (parse (rule))
    return true;
    
  _have_error = true;
  token = problem () ? problem () : current_pos;
  builder ().Push (builder ().error ());

  // if there is no detailed error message generate a standard message
  if (index == ((ErrorCollector&)builder ().err ()).index ())
    builder ().err () << sev_error << token->location () << msg
      << " near token `" << token->text () << "'" << endMessage;

  skip (skip_tokens, false);
  return true;
}

#endif /* __puma */


} // namespace Puma

#endif /* __Syntax_h__ */
