// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/CCSyntax.h"
#include "Puma/CCSemantic.h"
#include "Puma/CCBuilder.h"
#include "Puma/CTokens.h"
#include "Puma/CTree.h"

namespace Puma {


CCSyntax::CCSyntax (CCBuilder &b, CCSemantic &s) : 
  CSyntax (b, s), _skip_bodies (SKIP_BODIES_NONE) {
}

CCBuilder &CCSyntax::builder () const { 
  return (CCBuilder&)Syntax::builder (); 
}

CCSemantic &CCSyntax::semantic () const { 
  return (CCSemantic&)Syntax::semantic (); 
}

void CCSyntax::configure (Config &config) {
  // configure the C syntax
  CSyntax::configure (config);
  
  // configure C++ syntax options
  if (! config.Option ("--real-instances"))
    _skip_bodies |= SKIP_BODIES_TPL;
  if (config.Option ("--skip-bodies-all"))
    _skip_bodies |= SKIP_BODIES_ALL;
  else if (config.Option ("--skip-bodies-tpl"))
    _skip_bodies |= SKIP_BODIES_TPL;
  else if (config.Option ("--skip-bodies-non-prj"))
    _skip_bodies |= SKIP_BODIES_NON_PRJ;
  else if (config.Option ("--skip-bodies-non-prim"))
    _skip_bodies |= SKIP_BODIES_NON_PRIM;
  
  // FIRST set initialization  
  init_namespace_def ();   
  init_explicit_instantiation (); 
  init_explicit_specialization ();
  init_qual_id ();
  init_unqual_id ();
  init_access_spec ();
}

void CCSyntax::init_prim_types () {
  CSyntax::init_prim_types ();
  _prim_types.set (TOK_BOOL);
}

void CCSyntax::get_search_scope (SearchScope &sc) {
  sc.scope      = semantic ().search_scope;
  sc.last_scope = semantic ().last_search_scope;
  sc.dep        = semantic ().dep_nested_name;
}

void CCSyntax::set_search_scope (SearchScope &sc) {
  semantic ().search_scope      = sc.scope;
  semantic ().last_search_scope = sc.last_scope;
  semantic ().dep_nested_name   = sc.dep;
}

void CCSyntax::skip_fct_try_block () {
  static int stop[] = { TOK_OPEN_ROUND, 0 };
  skip_ctor_init ();
  skip_fct_body ();
  
  // skip handler sequence
  while (look_ahead (TOK_CATCH)) {
    skip (stop, false);    // CATCH
    skip_round_block ();   // (...)
    skip_curly_block ();   // {...}
  }
}

void CCSyntax::skip_ctor_init () {
  static int stop_tokens[] = { TOK_OPEN_CURLY, 0 };
  skip (stop_tokens, false);
}

void CCSyntax::skip_fct_body () {
  skip_curly_block ();
}

void CCSyntax::skip_param_init () {
  static int stop_tokens[] = { TOK_COMMA, TOK_CLOSE_ROUND, TOK_ELLIPSIS, 0 };
  skip (stop_tokens, false);
}

void CCSyntax::skip_const_expr () {
  static int stop_tokens[] = { TOK_COMMA, TOK_SEMI_COLON, 0 };
  skip (stop_tokens, false);
}

void CCSyntax::skip_const_init () {
  static int stop_tokens[] = { TOK_COMMA, TOK_SEMI_COLON, 0 };
  skip (stop_tokens, false);
}

bool CCSyntax::is_fct_def () {
  State s = token_provider->get_state ();
  bool result = false, braces = false;
  Token *current;
  int token;

  while ((current = token_provider->current ())) {
    token = current->type ();
    
    // only declaration
    if (token == TOK_SEMI_COLON || 
        token == TOK_COMMA ||
        token == TOK_TYPEDEF ||
        // TODO: checking the AspectC++ tokens here is really bad style
        token == TOK_ADVICE ||
        token == TOK_POINTCUT ||
        token == TOK_SLICE ||
        // --
        (! braces && token == TOK_ASSIGN)) {
      break;
    // operator name
    } else if (token == TOK_OPERATOR) {
      skip (TOK_OPEN_ROUND, false);
    // function `try' block, function body..
    } else if (token == TOK_TRY || (braces && 
               (token == TOK_OPEN_CURLY || token == TOK_COLON))) {
      result = true;
      break;
    // class definition in function return type
    } else if (token == TOK_OPEN_CURLY && ! braces) {
      skip_curly_block ();
    // member initialization list
    } else if (token == TOK_COLON && braces) {
      skip (TOK_OPEN_CURLY, false);
    // array delimiter
    } else if (token == TOK_OPEN_SQUARE) {
      skip_block (TOK_OPEN_SQUARE, TOK_CLOSE_SQUARE);
    // template-id
    } else if (token == TOK_LESS) {
      skip_block (TOK_LESS, TOK_GREATER);
    // function parameter list
    } else if (token == TOK_OPEN_ROUND) {
      skip_round_block ();
      braces = true;
    // throw expression
    } else if (token == TOK_THROW) {
      skip ();
      skip_round_block (); 
    // asm declaration specifier
    } else if (token == TOK_ASM) {
      skip ();
      skip_round_block ();
    // names etc.
    } else {
      token_provider->next ();
      locate_token ();
    }
  }

  token_provider->set_state (s);
  return result;
}

bool CCSyntax::is_nested (State first) {
  int token, braces, curlies, squares;
  Token *current;

  if (! first)
    return false;
    
  State s = token_provider->get_state ();
  token_provider->set_state (first);

  braces = curlies = squares = 0;
  while ((current = token_provider->current ())) {
    token = current->type ();

    if (token == TOK_OPEN_ROUND)
      braces++;
    else if (token == TOK_CLOSE_ROUND)
      braces--;
    else if (token == TOK_OPEN_CURLY) 
      curlies++;
    else if (token == TOK_CLOSE_CURLY) 
      curlies--;
    else if (token == TOK_OPEN_SQUARE) 
      squares++;
    else if (token == TOK_CLOSE_SQUARE) 
      squares--;

    if (token_provider->get_state () == s)
      break;
        
    skip ();
  }

  token_provider->set_state (s);
  return (braces + curlies + squares) > 0;
}

bool CCSyntax::is_nested_name () {
  // result cache!
  static Token *last_token = 0;
  static bool last_result = false;
  
  Token *current = token_provider->current ();
  if (! current)
    return false;

  // return the last result if we check the same token
  if (last_token == current)
    return last_result;
  
  last_token  = current;
  last_result = false;   
  int token, depth = 0;
  State s = token_provider->get_state ();

  // keyword `template'?
  if (current->type () == TOK_TEMPLATE) 
    skip ();

  // identifier?
  current = token_provider->current ();
  if (current && current->type () == TOK_ID) {
    skip ();

    // template id?
    if (token_provider->current () &&
        token_provider->current ()->type () == TOK_LESS) {
      skip ();
      
      while ((current = token_provider->current ())) {
        token = current->type ();

        if (token == TOK_SEMI_COLON   || token == TOK_CLOSE_ROUND ||
            token == TOK_CLOSE_SQUARE || token == TOK_CLOSE_CURLY)
          break;
        else if (token == TOK_OPEN_ROUND) {
          skip_round_block ();
          continue;
        } else if (token == TOK_OPEN_CURLY) { 
          skip_curly_block ();
          continue;
        } else if (token == TOK_OPEN_SQUARE) {
          skip_block (TOK_OPEN_SQUARE, TOK_CLOSE_SQUARE);
          continue;
        } else if (token == TOK_LESS)
          depth++;
        else if (token == TOK_GREATER) {
          if (depth == 0) {
            skip ();
            break;
          }
          depth--;
        }
        
        skip ();
      }
    }

    // scope operator => nested name
    if (token_provider->current () &&
        token_provider->current ()->type () == TOK_COLON_COLON)
      last_result = true;
  }

  token_provider->set_state (s);
  return last_result;
}

bool CCSyntax::is_class_def () {
  // result cache!
  static Token *last_token = 0;
  static bool last_result = false;
  
  Token *current = token_provider->current ();
  if (! current)
    return false;

  // return the last result if we check the same token
  if (last_token == current)
    return last_result;
  
  last_token = current;
  last_result = false;
  int token;
  State s = token_provider->get_state ();

  while ((current = token_provider->current ())) {
    token = current->type ();

    if (token == TOK_SEMI_COLON) {
      break;
    } else if (token == TOK_OPEN_CURLY) {
      last_result = true;
      break;
    } else if (token == TOK_OPEN_ROUND) {
      skip_round_block ();
      continue;
    } else if (token == TOK_OPEN_SQUARE) {
      skip_block (TOK_OPEN_SQUARE, TOK_CLOSE_SQUARE);
      continue;
    } 
        
    skip ();
  }

  token_provider->set_state (s);
  return last_result;
}

/*****************************************************************************/
/*                                                                           */
/*                           C + +   G r a m m a r                           */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* This grammar is derived from:                                             */
/*                                                                           */
/*                           International Standard                          */
/*                            ISO/IEC 9899-1999(E)                           */
/*                          Second edition 1999-04-01                        */
/*                          Programming languages - C                        */
/*                                                                           */
/*                                    and                                    */
/*                                                                           */
/*                           International Standard                          */
/*                            ISO/IEC 14882:1998(E)                          */
/*                          First edition 1998-09-01                         */
/*                         Programming languages - C++                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                           A . 1   K e y w o r d s                         */
/*                                                                           */
/*****************************************************************************/

void CCSyntax::init_class_name () {
  init_class_template_id ();
  _class_name_1 = _template_id_1;
  _class_name_1.set (TOK_ID);
}

CTree *CCSyntax::rule_class_name () { 
  // 1: template_id
  // 1: ID
  return (template_id () ||
          parse (TOK_ID)) ? 
    semantic ().class_name () : (CTree*)0; 
}

bool CCSyntax::class_name () {
  return predict_1 (_class_name_1) && parse (&CCSyntax::rule_class_name);
}

void CCSyntax::init_enum_name () {
  _enum_name_1.set (TOK_ID);
}

CTree *CCSyntax::rule_enum_name () { 
  // 1: ID
  consume ();
  return semantic ().enum_name ();
}

bool CCSyntax::enum_name () {
  return look_ahead () == TOK_ID && parse (&CCSyntax::rule_enum_name);
}


CTree *CCSyntax::template_name () { 
  // 1: ID
  return parse (TOK_ID) ? 
    semantic ().template_name () : (CTree*)0; 
}


CTree *CCSyntax::class_template_name () { 
  // 1: ID
  return parse (TOK_ID) ? 
    semantic ().template_name (true) : (CTree*)0; 
}


void CCSyntax::init_namespace_name () {
  init_original_ns_name ();
  init_namespace_alias ();
  _namespace_name_1 = _original_ns_name_1;
  _namespace_name_1 |= _namespace_alias_1;
}

CTree *CCSyntax::rule_namespace_name () {
  // 1: original_ns_name
  // 1: namespace_alias
  return (original_ns_name () ||
          namespace_alias ()) ? 
    builder ().namespace_name () : (CTree*)0; 
}

bool CCSyntax::namespace_name () {
  return predict_1 (_namespace_name_1) &&
    parse (&CCSyntax::rule_namespace_name);
}


void CCSyntax::init_original_ns_name () {
  _original_ns_name_1.set (TOK_ID);
}

CTree *CCSyntax::rule_original_ns_name () {
  // 1: ID
  consume ();
  return semantic ().original_ns_name (); 
}

bool CCSyntax::original_ns_name () {
  return look_ahead () == TOK_ID && parse (&CCSyntax::rule_original_ns_name);
}


void CCSyntax::init_namespace_alias () {
  _namespace_alias_1.set (TOK_ID);
}

CTree *CCSyntax::rule_namespace_alias () {
  // 1: ID
  consume ();
  return semantic ().namespace_alias (); 
}

bool CCSyntax::namespace_alias () {
  return look_ahead () == TOK_ID && parse (&CCSyntax::rule_namespace_alias);
}

/*****************************************************************************/
/*                                                                           */
/*                      A . 4   E x p r e s s i o n s                        */
/*                                                                           */
/*****************************************************************************/

void CCSyntax::init_prim_expr () {
  CSyntax::init_prim_expr ();
  _prim_expr_1.set (TOK_THIS);
}

CTree *CCSyntax::rule_prim_expr () {
  // 1: literal
  // 1: THIS
  // 1: id_expr
  // 3: (  expr  )
  return (literal () || 
          parse (TOK_THIS) || 
          id_expr () ||
          (parse (TOK_OPEN_ROUND) && 
           parse (&CCSyntax::expr) && 
           parse (TOK_CLOSE_ROUND))) ? 
    builder ().prim_expr () : (CTree*)0; 
}

bool CCSyntax::prim_expr () {
  return predict_1 (_prim_expr_1) && parse (&CCSyntax::rule_prim_expr);
}

void CCSyntax::init_id_expr () {
  // no call to CSyntax::init_id_expr()! This is completely different
  init_qual_id ();
  init_unqual_id ();
  _id_expr_1 = _qual_id_1;
  _id_expr_1 |= _unqual_id_1;
}

CTree *CCSyntax::rule_id_expr () {
  // 1: qual_id
  // 1: unqual_id
  return (qual_id () || 
          unqual_id ()) ? 
    builder ().id_expr () : (CTree*)0; 
}

bool CCSyntax::id_expr () {
  return predict_1 (_id_expr_1) && parse (&CCSyntax::rule_id_expr);
}

void CCSyntax::init_unqual_id () {
  init_identifier ();
  init_oper_fct_id ();
  init_conv_fct_id ();
  init_template_id ();
  _unqual_id_1 = _identifier_1;
  _unqual_id_1 |= _oper_fct_id_1;
  _unqual_id_1 |= _conv_fct_id_1;
  _unqual_id_1 |= _template_id_1;
  _unqual_id_1.set (TOK_TILDE);
}

CTree *CCSyntax::rule_unqual_id () { 
  // 1: identifier
  // 1: oper_fct_id
  // 1: conv_fct_id
  // 1: template_id
  // 2: ~  class_name
  return (template_id () ||
          identifier () ||
          oper_fct_id () ||
          conv_fct_id () ||
          (parse (TOK_TILDE) && class_name ())) ? 
    builder ().unqual_id () : (CTree*)0; 
}

bool CCSyntax::unqual_id () {
  return predict_1 (_unqual_id_1) && parse (&CCSyntax::rule_unqual_id);
}

void CCSyntax::init_qual_id () {
  init_colon_colon ();
  init_nested_name_spec ();
  _qual_id_1 = _colon_colon_1;
  _qual_id_1 |= _nested_name_spec_1;
}

CTree *CCSyntax::rule_qual_id () { 
  // 2: colon_colon  identifier
  // 2: colon_colon  oper_fct_id
  // 2: colon_colon  template_id
  // 2: nested_name_spec  unqual_id
  // 3: nested_name_spec  template_key  unqual_id
  // 3: colon_colon  nested_name_spec  unqual_id
  // 4: colon_colon  nested_name_spec  template_key  unqual_id
  bool root = colon_colon ();
  bool nested = nested_name_spec ();
  bool ok = (nested ? 
             (template_key (), 
              unqual_id ()) :
             (root &&
              (template_id () ||
               identifier () ||
               oper_fct_id ())));
  if (root || nested)
    semantic ().reset_search_scope ();
  return ok ? builder ().qual_id () : (CTree*)0; 
}

bool CCSyntax::qual_id () {
  return predict_1 (_qual_id_1) && parse (&CCSyntax::rule_qual_id);
}

void CCSyntax::init_colon_colon () {
  _colon_colon_1.set (TOK_COLON_COLON);
}

CTree *CCSyntax::rule_colon_colon () { 
  // 1: ::
  return parse (TOK_COLON_COLON) ? 
    semantic ().colon_colon () : (CTree*)0;
}

bool CCSyntax::colon_colon () {
  return look_ahead () == TOK_COLON_COLON &&
    parse (&CCSyntax::rule_colon_colon);
}


void CCSyntax::init_nested_name_spec () {
  init_nested_name_spec1 ();
  _nested_name_spec_1 = _nested_name_spec1_1;
}

CTree *CCSyntax::rule_nested_name_spec () { 
  // 1+: nested_name_spec1..
  semantic ().enter_nested_name ();
  bool parsed = seq (&CCSyntax::nested_name_spec1);
  semantic ().leave_nested_name ();
  return parsed ? semantic ().nested_name_spec () : (CTree*)0;
}

bool CCSyntax::nested_name_spec () {
  return is_nested_name () && parse (&CCSyntax::rule_nested_name_spec);
}

void CCSyntax::init_nested_name_spec1 () {
  init_class_or_ns_name (),
  init_template_key ();
  _nested_name_spec1_1 = _class_or_ns_name_1;
  _nested_name_spec1_1 |= _template_key_1;
}

CTree *CCSyntax::rule_nested_name_spec1 () { 
  // 2: class_or_ns_name  ::
  // 3: template_key  class_or_ns_name  ::
  return ((template_key (), class_or_ns_name ()) && parse (TOK_COLON_COLON)) ?
    semantic ().nested_name_spec1 () : (CTree*)0;
}

bool CCSyntax::nested_name_spec1 () {
  return is_nested_name () && parse (&CCSyntax::rule_nested_name_spec1);
}


void CCSyntax::init_class_or_ns_name () {
  init_class_name ();
  init_namespace_name ();
  _class_or_ns_name_1 = _class_name_1;
  _class_or_ns_name_1 |= _namespace_name_1;
}

CTree *CCSyntax::rule_class_or_ns_name () { 
  // 1: class_name
  return (class_name () ||
          namespace_name ()) ? 
    builder ().class_or_ns_name () : (CTree*)0; 
}

bool CCSyntax::class_or_ns_name () {
  return predict_1 (_class_or_ns_name_1) &&
    parse (&CCSyntax::rule_class_or_ns_name);
}

CTree *CCSyntax::postfix_expr () { 
  // 1 : postfix_expr2
  // 1+: postfix_expr2  postfix_expr1..
  if (! parse (&CCSyntax::postfix_expr2))
    return (CTree*)0;

  CTree *result = builder ().postfix_expr ();
  semantic ().enter_postfix_expr ();
  semantic ().postfix_expr (result);
  while (postfix_expr1 ()) {
    result = builder ().postfix_expr (result);
    semantic ().postfix_expr (result);
  }
  semantic ().leave_postfix_expr ();
  return result;
};

CTree *CCSyntax::postfix_expr2 () { 
  // 1 : cmpd_literal
  // 1 : prim_expr
  // 3 : simple_type_spec  (  )
  // 4 : simple_type_spec  (  expr_list  )
  // 4 : TYPEID  (  expr  )
  // 4 : TYPEID  (  type_id  )
  // 5 : TYPENAME  nested_name_spec  identifier  (  )
  // 5 : TYPENAME  nested_name_spec  template_id  (  )
  // 6 : TYPENAME  nested_name_spec  identifier  (  expr_list  )
  // 6 : TYPENAME  nested_name_spec  template_id  (  expr_list  )
  // 6 : TYPENAME  nested_name_spec  template_key  template_id  (  )
  // 6 : TYPENAME  ::  nested_name_spec  identifier  (  )
  // 6 : TYPENAME  ::  nested_name_spec  template_id  (  )
  // 7 : CONST_CAST  <  type_id  >  (  expr  )
  // 7 : REINT_CAST  <  type_id  >  (  expr  )
  // 7 : STATIC_CAST  <  type_id  >  (  expr  )
  // 7 : DYNAMIC_CAST  <  type_id  >  (  expr  )
  // 7 : TYPENAME  nested_name_spec  template_key  template_id  (  expr_list  )
  // 7 : TYPENAME  ::  nested_name_spec  identifier  (  expr_list  )
  // 7 : TYPENAME  ::  nested_name_spec  template_id  (  expr_list  )
  // 7 : TYPENAME  ::  nested_name_spec  template_key  template_id  (  )
  // 8 : TYPENAME  ::  nested_name_spec  template_key  template_id  (  expr_list  )
  static int any_new_cast[] = { TOK_DYN_CAST, TOK_REINT_CAST, 
                                TOK_STAT_CAST, TOK_CONST_CAST, 0 };  
  // lookahead: check if this is a simple_type_spec (...)
  State s = save_state ();
  bool simple_ts = (simple_type_spec () && parse (TOK_OPEN_ROUND));
  restore_state (s);
  // new-style casts
  if (parse (any_new_cast)) {
    return (parse (TOK_LESS) && parse (&CCSyntax::type_id) && 
            parse (TOK_GREATER) && 
            parse (TOK_OPEN_ROUND) && parse (&CCSyntax::expr) && 
            parse (TOK_CLOSE_ROUND)) ?
      semantic ().postfix_expr2 () : (CTree*)0;
  // construct expression
  } else if (parse (TOK_TYPENAME)) {
    return (parse (TOK_COLON_COLON), nested_name_spec () &&
            (template_key () ? template_id () :
             template_id () || identifier ()) &&
            semantic ().reset_search_scope () &&
            parse (TOK_OPEN_ROUND) && 
            (parse (&CCSyntax::expr_list), parse (TOK_CLOSE_ROUND))) ?
      semantic ().postfix_expr2 () : (CTree*)0;
  // construct expression
  } else if (simple_ts) {
    return ((simple_type_spec (), parse (TOK_OPEN_ROUND), 
             parse (&CCSyntax::expr_list), parse (TOK_CLOSE_ROUND))) ?
      semantic ().postfix_expr2 () : (CTree*)0;
  // member access expressions
  } else if (cmpd_literal () || prim_expr ()) {
    return semantic ().postfix_expr2 ();
  }
  // typeid (RTTI)
  else if (parse (TOK_TYPEID)) {
    return (parse (TOK_OPEN_ROUND) &&
             (parse (&CCSyntax::expr) || parse (&CCSyntax::type_id)) &&
             parse (TOK_CLOSE_ROUND)) ?
      semantic ().postfix_expr2 () : (CTree*)0;
  }
  return (CTree*)0;
}

void CCSyntax::init_postfix_expr1 () {
  CSyntax::init_postfix_expr1 ();
}

CTree *CCSyntax::rule_postfix_expr1 () { 
  // 1: --
  // 1: ++
  // 2: .  id_expr
  // 2: ->  id_expr
  // 2: .  pseudo_dtor_name
  // 2: ->  pseudo_dtor_name
  // 2: (  )
  // 3: .  template_key  id_expr
  // 3: ->  template_key  id_expr
  // 3: (  expr_list  )
  // 3: [  expr  ]
  bool member_access = (look_ahead (TOK_DOT) || look_ahead (TOK_PTS));
  if (member_access)
    semantic ().enter_member_access ();
  bool success = 
    (parse (TOK_OPEN_ROUND) ?
      (parse (&CCSyntax::expr_list), parse (TOK_CLOSE_ROUND)) :
     parse (TOK_OPEN_SQUARE) ?
      (parse (&CCSyntax::expr) && parse (TOK_CLOSE_SQUARE)) :
     (parse (TOK_DECR) || parse (TOK_INCR) ||
     ((parse (TOK_DOT) || parse (TOK_PTS)) && 
      (parse (&CCSyntax::pseudo_dtor_name) || 
       (template_key (), id_expr ())))));
  if (member_access)
    semantic ().leave_member_access ();
  return success ?
    semantic ().postfix_expr1 () : (CTree*)0;
}

bool CCSyntax::postfix_expr1 () {
  return predict_1 (_postfix_expr1_1) && parse (&CCSyntax::rule_postfix_expr1);
}

CTree *CCSyntax::pseudo_dtor_name () {
  // 2: ~  type_name
  // 3: ::  ~  type_name
  // 3: nested_name_spec  ~  type_name
  // 4: ::  nested_name_spec  ~  type_name
  // 4: type_name  ::  ~  type_name
  // 5: ::  type_name  ::  ~  type_name
  // 5: nested_name_spec  type_name  ::  ~  type_name
  // 6: ::  nested_name_spec  type_name  ::  ~  type_name
  // 6: nested_name_spec  template_key  template_id  ::  ~  type_name
  // 7: ::  nested_name_spec  template_key  template_id  ::  ~  type_name
  bool ok = ((colon_colon (), nested_name_spec (), 
             (template_key () ? template_id () : type_name ())) && 
             ! parse (TOK_COLON_COLON));
  semantic ().reset_search_scope ();
  if (! ok) 
    return (CTree*)0;
  return (parse (TOK_TILDE) && type_name ()) ?
    semantic ().pseudo_dtor_name () : (CTree*)0;
}

CTree *CCSyntax::unary_expr () { 
  // 1: new_expr
  // 1: delete_expr
  // 1: postfix_expr
  // 2: any_unary_op  cast_expr
  // 2: SIZEOF  unary_expr
  // 4: SIZEOF  unary_expr1
  static int any_unary_op[] = { TOK_AND, TOK_MUL, TOK_PLUS, TOK_MINUS, 
                                TOK_TILDE, TOK_NOT, TOK_DECR, TOK_INCR, 0 };
  return (parse (any_unary_op) ?
          parse (&CCSyntax::cast_expr) :
          parse (TOK_SIZEOF) ?
          (parse (&CCSyntax::unary_expr1) || parse
           (&CCSyntax::unary_expr)) :
          (parse (&CCSyntax::new_expr) ||
           parse (&CCSyntax::delete_expr) ||
           parse (&CCSyntax::postfix_expr))) ?
    builder ().unary_expr () : (CTree*)0;
}

CTree *CCSyntax::new_expr () { 
  // 2: NEW  new_type_id
  // 3: NEW  new_type_id  new_init
  // 3: NEW  new_placement  new_type_id
  // 3: ::  NEW  new_type_id
  // 4: ::  NEW  new_type_id  new_init
  // 4: ::  NEW  new_placement  new_type_id
  // 4: NEW  new_placement  new_type_id  new_init
  // 4: NEW  (  type_id  )
  // 5: NEW  (  type_id  )  new_init
  // 5: NEW  new_placement  (  type_id  )
  // 5: ::  NEW  new_placement  new_type_id  new_init
  // 5: ::  NEW  (  type_id  )
  // 6: ::  NEW  (  type_id  )  new_init
  // 6: NEW  new_placement  (  type_id  )  new_init
  // 6: ::  NEW  new_placement  (  type_id  )
  // 7: ::  NEW  new_placement  (  type_id  )  new_init
  return ((parse (TOK_COLON_COLON), parse (TOK_NEW)) && 
          opt (parse (&CCSyntax::new_placement)) && 
          (parse (&CCSyntax::new_type_id) ||
           (parse (TOK_OPEN_ROUND) && 
            parse (&CCSyntax::type_id) &&
            parse (TOK_CLOSE_ROUND))) &&
          opt (parse (&CCSyntax::new_init))) ?
    builder ().new_expr () : (CTree*)0;
}

CTree *CCSyntax::new_placement () { 
  // 3: (  expr_list  )
  return (parse (TOK_OPEN_ROUND) && 
          parse (&CCSyntax::expr_list) &&
          parse (TOK_CLOSE_ROUND)) ?
    builder ().new_placement () : (CTree*)0;
}

CTree *CCSyntax::new_type_id () { 
  // 2: type_spec_seq  new_declarator
  // 2: type_spec_seq  private_name
  bool success = parse (&CCSyntax::type_spec_seq) && 
                 (parse (&CCSyntax::new_declarator) ||
                  parse (&CCSyntax::private_name));
  semantic ().finish_declarator ();
  return success ? 
    semantic ().introduce_named_type () : (CTree*)0;
}

CTree *CCSyntax::new_declarator () { 
  // 1 : direct_new_declarator
  // 1+: ptr_operator..
  // 2+: ptr_operator..  direct_new_declarator
  return (parse (&CCSyntax::direct_new_declarator) ||
          (seq (&CCSyntax::ptr_operator) &&
           opt (parse (&CCSyntax::direct_new_declarator)))) ?
    semantic ().abst_declarator () : (CTree*)0;
}

CTree *CCSyntax::direct_new_declarator () { 
  // 3 : [  expr  ]
  // 4+: [  expr  ]  direct_new_declarator1...
  return ((parse (TOK_OPEN_SQUARE) && parse (&CCSyntax::expr) &&
           parse (TOK_CLOSE_SQUARE)) &&  
          opt (seq (&CCSyntax::direct_new_declarator1))) ? 
    semantic ().direct_new_declarator () : (CTree*)0;
}

CTree *CCSyntax::direct_new_declarator1 () { 
  // 3: [  const_expr  ]
  return (parse (TOK_OPEN_SQUARE) && parse (&CCSyntax::const_expr) &&
          parse (TOK_CLOSE_SQUARE)) ?
    builder ().direct_new_declarator1 () : (CTree*)0;
}

CTree *CCSyntax::new_init () { 
  // 2: (  )
  // 3: (  expr_list  )
  return (parse (TOK_OPEN_ROUND) && (parse (&CCSyntax::expr_list),
          parse (TOK_CLOSE_ROUND))) ?
    builder ().new_init () : (CTree*)0;
}

CTree *CCSyntax::delete_expr () { 
  // 2: DELETE  cast_expr
  // 3: ::  DELETE  cast_expr
  // 4: DELETE  [  ]  cast_expr
  // 5: ::  DELETE  [  ]  cast_expr
  return ((parse (TOK_COLON_COLON), parse (TOK_DELETE)) &&
          (parse (TOK_OPEN_SQUARE) ? parse (TOK_CLOSE_SQUARE) : true) && 
          parse (&CCSyntax::cast_expr)) ?
    builder ().delete_expr () : (CTree*)0;
}

CTree *CCSyntax::pm_expr () { 
  // 1 : cast_expr
  // 3+: cast_expr  any_pm_op  cast_expr ..
  static int any_pm_op[] = { TOK_DOT_STAR, TOK_PTS_STAR, 0 };
  return list (&CCSyntax::cast_expr, any_pm_op) ? 
    builder ().pm_expr () : (CTree*)0;
}

CTree *CCSyntax::mul_expr () { 
  // 1 : pm_expr
  // 3+: pm_expr  any_mul_op  pm_expr ..
  static int any_mul_op[] = { TOK_MUL, TOK_DIV, TOK_MODULO, 0 };
  return list (&CCSyntax::pm_expr, any_mul_op) ? 
    builder ().mul_expr () : (CTree*)0;
}

CTree *CCSyntax::rel_expr () { 
  // 1 : shift_expr
  // 3+: shift_expr  any_rel_op  shift_expr ...
  static int any_rel_op[] = { TOK_LESS, TOK_GREATER, TOK_LEQ, TOK_GEQ, 0 };
  if (! parse (&CCSyntax::shift_expr))
    return (CTree*)0;
  while (true) {
    // �14.2.3 check if this is the end of a template argument list
    if (look_ahead (TOK_GREATER)) { 
      if ((semantic ().in_template_arg_list () &&
           ! is_nested (semantic ().in_template_arg_list ())) ||
          (semantic ().in_template_param_list () &&
           ! is_nested (semantic ().in_template_param_list ())))
      break;
    }
    if (! parse (any_rel_op))
      break;
    if (! parse (&CCSyntax::shift_expr))
      return (CTree*)0;
  }
  return builder ().rel_expr ();
}

CTree *CCSyntax::cond_expr () {
  // 1: log_or_expr
  // 5: log_or_expr  ?  expr  :  ass_expr
  return (parse (&CCSyntax::log_or_expr) && 
          (parse (TOK_QUESTION) ? (parse (&CCSyntax::expr) && 
           parse (TOK_COLON) && parse (&CCSyntax::ass_expr)) : true)) ?
    builder ().cond_expr () : (CTree*)0;
}

CTree *CCSyntax::ass_expr () { 
  // 1 : cond_expr
  // 1 : throw_expr
  // 2+: ass_expr1..  cond_expr
  // 2+: ass_expr1..  throw_expr
  semantic ().enter_expr ();
  bool parsed = (seq (&CCSyntax::ass_expr1), 
                 (parse (&CCSyntax::throw_expr) || parse (&CCSyntax::cond_expr)));
  semantic ().leave_expr ();
  return parsed ?
    builder ().ass_expr () : (CTree*)0;
}

CTree *CCSyntax::ass_expr1 () { 
  // 2: log_or_expr  any_ass_op
  static int any_ass_op[] = { TOK_ASSIGN, TOK_MUL_EQ, TOK_DIV_EQ, TOK_MOD_EQ, 
                              TOK_ADD_EQ, TOK_SUB_EQ, TOK_RSH_EQ, TOK_LSH_EQ,
                              TOK_AND_EQ, TOK_XOR_EQ, TOK_IOR_EQ, 0 };
  return (is_ass_expr () && parse (&CCSyntax::log_or_expr) && parse (any_ass_op)) ? 
    builder ().ass_expr1 () : (CTree*)0;
}

CTree *CCSyntax::const_expr () { 
  // 1: cond_expr
  semantic ().enter_expr ();
  bool parsed = parse (&CCSyntax::cond_expr); 
  semantic ().leave_expr ();
  return parsed ?
    semantic ().const_expr () : (CTree*)0;
}

/*****************************************************************************/
/*                                                                           */
/*                      A . 5   S t a t e m e n t s                          */
/*                                                                           */
/*****************************************************************************/

CTree *CCSyntax::stmt () { 
  // 1: label_stmt
  // 1: decl_stmt
  // 1: expr_stmt
  // 1: cmpd_stmt
  // 1: select_stmt
  // 1: iter_stmt
  // 1: jump_stmt
  // 1: try_block
  return (parse (&CCSyntax::label_stmt) ||
          parse (&CCSyntax::decl_stmt) ||
          parse (&CCSyntax::expr_stmt) ||
          parse (&CCSyntax::cmpd_stmt) ||
          parse (&CCSyntax::select_stmt) ||
          parse (&CCSyntax::iter_stmt) ||
          parse (&CCSyntax::jump_stmt) ||
          parse (&CCSyntax::try_block)) ? 
    builder ().stmt () : (CTree*)0; 
}
           
CTree *CCSyntax::stmt_seq () {
  // 1+: stmt..
  static int skip[] = { TOK_SEMI_COLON, TOK_CLOSE_CURLY, 0 };
  static int finish[] = { TOK_CLOSE_CURLY, 0 };
  if (! catch_error (&CCSyntax::stmt, "invalid statement", finish, skip))
    return (CTree*)0;
  while (catch_error (&CCSyntax::stmt, "invalid statement", finish, skip));
  return builder ().stmt_seq ();
}

CTree *CCSyntax::sub_stmt () {
  // 1: stmt
  // a substatement that is not a compound-statement is as if it 
  // would be a compound-statement containing the original statement;
  // in this case it implicitly defines a local scope
  if (! look_ahead (TOK_OPEN_CURLY))
    semantic ().enter_local_scope (); // enter local (block) scope
  return parse (&CCSyntax::stmt) ?
    builder ().sub_stmt () : (CTree*)0; 
}

CTree *CCSyntax::condition () {
  // 1: condition1
  // 1: expr
  return (parse (&CCSyntax::condition1) || parse (&CCSyntax::expr)) ? 
    builder ().condition () : (CTree*)0;
}

CTree *CCSyntax::condition1 () {
  // 4: type_spec_seq  declarator  =  ass_expr
  bool success = (parse (&CCSyntax::type_spec_seq) && 
                  parse (&CCSyntax::declarator));
  semantic ().finish_declarator ();
  if (! success)
    return (CTree*)0;
  CTree *cond = semantic ().condition (); // introduce name
  return (parse (TOK_ASSIGN), parse (&CCSyntax::ass_expr), cond) ?
    builder ().condition (cond) : (CTree*)0;
}

CTree *CCSyntax::decl_stmt () {
  // 1: block_decl
  return parse (&CCSyntax::block_decl) ?
    builder ().decl_stmt () : (CTree*)0;
}

/*****************************************************************************/
/*                                                                           */
/*                     A . 6   D e c l a r a t i o n s                       */
/*                                                                           */
/*****************************************************************************/

CTree *CCSyntax::decl () { 
  // 1: block_decl
  // 1: fct_def
  // 1: explicit_specialization
  // 1: template_decl
  // 1: explicit_instantiation
  // 1: linkage_spec
  // 1: namespace_def
  // 3: EXTERN  str_literal  decl
  State s = save_state ();
  bool extern_decl = (parse (TOK_EXTERN) && str_literal () && 
                      ! look_ahead (TOK_OPEN_CURLY));
  if (extern_decl)
    semantic ().enter_extern_decl ();
  restore_state (s);
  bool parsed = ((extern_decl ? 
                  parse (TOK_EXTERN) && str_literal () : true) &&
                 (explicit_specialization () ||
                  parse (&CCSyntax::template_decl) ||
                  explicit_instantiation () ||
                  namespace_def () ||
                  parse (&CCSyntax::linkage_spec) ||
                  (is_fct_def () ? 
		   (parse (&CCSyntax::fct_def) ||
		    parse (&CCSyntax::block_decl)) :
                   (parse (&CCSyntax::block_decl) ||
                    parse (&CCSyntax::fct_def))))); 
  if (extern_decl)
    semantic ().leave_extern_decl ();
  return parsed ? 
    builder ().decl () : (CTree*)0;
}

CTree *CCSyntax::block_decl () { 
  // 1: simple_decl
  // 1: asm_def
  // 1: ns_alias_def
  // 1: using_decl
  // 1: using_directive
  return (parse (&CCSyntax::simple_decl) ||
          parse (&CCSyntax::asm_def) ||
          parse (&CCSyntax::ns_alias_def) ||
          parse (&CCSyntax::using_directive) ||
          parse (&CCSyntax::using_decl)) ? 
    builder ().block_decl () : (CTree*)0;
}

CTree *CCSyntax::simple_decl () { 
  // 1: ;
  // 2: decl_spec_seq  ;
  // 3: decl_spec_seq  init_declarator_list  ;
  CTree *obj_decl;
  if ((obj_decl = CSyntax::simple_decl ()))
    semantic ().obj_decl (obj_decl);
  return obj_decl;
}

void CCSyntax::init_misc_spec () {
  CSyntax::init_misc_spec ();
  _misc_spec_1.set (TOK_FRIEND);
}

void CCSyntax::init_storage_class_spec () {
  CSyntax::init_storage_class_spec ();
  _storage_class_spec_1.set (TOK_MUTABLE);
}

void CCSyntax::init_fct_spec () {
  CSyntax::init_fct_spec ();
  _fct_spec_1.set (TOK_VIRTUAL);
  _fct_spec_1.set (TOK_EXPLICIT);
}

void CCSyntax::init_simple_type_spec () {
  init_prim_types ();
  init_type_name ();
  init_nested_name_spec ();
  init_colon_colon ();
  _simple_type_spec_1 = _prim_types;
  _simple_type_spec_1 |= _type_name_1;
  _simple_type_spec_1 |= _nested_name_spec_1;
  _simple_type_spec_1 |= _colon_colon_1;
}

CTree *CCSyntax::rule_simple_type_spec () { 
  // 1: prim_types
  // 1: type_name
  // 2: nested_name_spec  type_name
  // 2: ::  type_name
  // 3: ::  nested_name_spec  type_name
  // 3: nested_name_spec  template_key  template_id
  // 4: ::  nested_name_spec  template_key  template_id
  bool ok = (_prim_types[look_ahead ()] && consume ()) ||
     (colon_colon (),
      ((nested_name_spec (), 
        (template_key () ? class_template_id () : type_name ()))));
  semantic ().reset_search_scope (true);
  return ok ? semantic ().simple_type_spec () : (CTree*)0;
}   

bool CCSyntax::simple_type_spec () {
  return predict_1 (_simple_type_spec_1) &&
         parse (&CCSyntax::rule_simple_type_spec);
}

void CCSyntax::init_type_name () {
  init_class_name ();
  init_enum_name ();
  init_typedef_name ();
  _type_name_1 = _class_name_1;
  _type_name_1 |= _enum_name_1;
  _type_name_1 |= _typedef_name_1;
}

CTree *CCSyntax::rule_type_name () { 
  // 1: class_name
  // 1: enum_name
  // 1: typedef_name
  return (class_name () ||
          enum_name () || 
          typedef_name ()) ?
    builder ().type_name () : (CTree*)0;
}   

bool CCSyntax::type_name () {
  return predict_1 (_type_name_1) && parse (&CCSyntax::rule_type_name);
}

void CCSyntax::init_elaborated_type_spec () {
  init_class_key ();
  _elaborated_type_spec_1 = _class_key_1;
  _elaborated_type_spec_1.set (TOK_ENUM);
  _elaborated_type_spec_1.set (TOK_TYPENAME);
}

CTree *CCSyntax::rule_elaborated_type_spec () {
  // 2: class_key  identifier
  // 2: ENUM  identifier
  // 2: TYPENAME  template_id
  // 3: class_key  ::  identifier
  // 3: ENUM  ::  identifier
  // 3: class_key  nested_name_spec  identifier
  // 3: ENUM  nested_name_spec  identifier
  // 3: TYPENAME  nested_name_spec  identifier
  // 3: TYPENAME  nested_name_spec  template_id
  // 4: class_key  ::  nested_name_spec  identifier
  // 4: ENUM  ::  nested_name_spec  identifier
  // 4: TYPENAME  nested_name_spec  template_key  template_id
  // 4: TYPENAME  ::  nested_name_spec  identifier
  // 4: TYPENAME  ::  nested_name_spec  template_id
  // 5: TYPENAME  ::  nested_name_spec  template_key  template_id
  bool ok =
    ((class_key () || parse (TOK_ENUM)) ?
      (colon_colon (), nested_name_spec (), 
       template_id () || identifier ()) :
     (parse (TOK_TYPENAME) && 
      (colon_colon (), nested_name_spec (),
      (template_key () ? template_id () :
       template_id () || identifier ())))); 
//      (parse (&CCSyntax::colon_colon), parse (&CCSyntax::nested_name_spec)) &&
//      (parse (&CCSyntax::template_key) ? parse (&CCSyntax::template_id) :
//       parse (&CCSyntax::template_id) || parse (&CCSyntax::identifier)))); 
  semantic ().reset_search_scope ();
  return ok ? semantic ().introduce_tag () : (CTree*)0;
}

bool CCSyntax::elaborated_type_spec () {
  return predict_1 (_elaborated_type_spec_1) &&
         parse (&CCSyntax::rule_elaborated_type_spec);
}

CTree *CCSyntax::enumerator_list () { 
  // 1 : enumerator_def
  // 3+: enumerator_def  ,  enumerator_def ..
  return opt (list (&CCSyntax::enumerator_def, TOK_COMMA, true)) ?
    builder ().enumerator_list () : (CTree*)0;
}

CTree *CCSyntax::enumerator_def () { 
  // 1: enumerator
  // 3: enumerator  =  const_expr
  return (parse (&CSyntax::enumerator) && 
          (parse (TOK_ASSIGN) ? parse (&CSyntax::const_expr) : true)) ?
    semantic ().enumerator_def () : (CTree*)0;
}

CTree *CCSyntax::linkage_spec () {
  // 4: EXTERN  str_literal  {  }
  // 5: EXTERN  str_literal  {  decl_seq  }
  if (! (parse (TOK_EXTERN) && str_literal () && look_ahead (TOK_OPEN_CURLY)))
    return (CTree*)0;
  semantic ().enter_extern_decl ();
  bool parsed = (parse (TOK_OPEN_CURLY) && (seq (&CCSyntax::decl_seq), parse (TOK_CLOSE_CURLY)));
  semantic ().leave_extern_decl ();
  return parsed ? 
    builder ().linkage_spec () : (CTree*)0;
}

/*****************************************************************************/
/*                                                                           */
/*                    A . 6 . 1   N a m e s p a c e s                        */
/*                                                                           */
/*****************************************************************************/

void CCSyntax::init_namespace_def () {
  init_named_ns_def ();
  init_unnamed_ns_def ();
  _namespace_def_1 = _named_ns_def_1;
  _namespace_def_1 |= _unnamed_ns_def_1;
}

CTree *CCSyntax::rule_namespace_def () {
  // 1: named_ns_def
  // 1: unnamed_ns_def
  return (named_ns_def () || unnamed_ns_def ()) ?
    builder ().namespace_def () : (CTree*)0;
}

bool CCSyntax::namespace_def () {
  return predict_1 (_namespace_def_1) && parse (&CCSyntax::rule_namespace_def);
}

void CCSyntax::init_named_ns_def () {
  init_original_ns_def ();
  init_extension_ns_def ();
  _named_ns_def_1 = _original_ns_def_1;
  _named_ns_def_1 |= _extension_ns_def_1;
}

CTree *CCSyntax::rule_named_ns_def () {
  // 1: original_ns_def
  // 1: extension_ns_def
  return (extension_ns_def () || original_ns_def ()) ?
    builder ().named_ns_def () : (CTree*)0;
}

bool CCSyntax::named_ns_def () {
  return predict_1 (_named_ns_def_1) && parse (&CCSyntax::rule_named_ns_def);
}

void CCSyntax::init_original_ns_def () {
  _original_ns_def_1.set (TOK_NAMESPACE);
}

CTree *CCSyntax::rule_original_ns_def () {
  // 5: NAMESPACE  identifier  {  namespace_body  }
  if (!(consume () && identifier () && parse (TOK_OPEN_CURLY)))
    return (CTree*)0;
  CTree *ns = semantic ().introduce_orig_ns ();
  if (! ns)
    return (CTree*)0;
  return (parse (&CCSyntax::namespace_body) && parse (TOK_CLOSE_CURLY)) ?
    semantic ().namespace_def (ns) : semantic ().namespace_def_err (ns);
}

bool CCSyntax::original_ns_def () {
  return look_ahead (TOK_NAMESPACE) && parse (&CCSyntax::rule_original_ns_def);
}
  

void CCSyntax::init_extension_ns_def () {
  _extension_ns_def_1.set (TOK_NAMESPACE);
}

CTree *CCSyntax::rule_extension_ns_def () {
  // 5: NAMESPACE  original_ns_name  {  namespace_body  }
  if (!(consume () && original_ns_name () && parse (TOK_OPEN_CURLY)))
    return (CTree*)0;
  CTree *ns = semantic ().introduce_ns_extension ();
  if (! ns)
    return (CTree*)0;
  return (parse (&CCSyntax::namespace_body) && parse (TOK_CLOSE_CURLY)) ?
    semantic ().namespace_def (ns) : semantic ().namespace_def_err (ns);
}

bool CCSyntax::extension_ns_def () {
  return look_ahead (TOK_NAMESPACE) &&
          parse (&CCSyntax::rule_extension_ns_def);
}

  
void CCSyntax::init_unnamed_ns_def () {
  _unnamed_ns_def_1.set (TOK_NAMESPACE);
}

CTree *CCSyntax::rule_unnamed_ns_def () {
  // 5: NAMESPACE  private_name  {  namespace_body  }
  if (! (consume () && parse (&CCSyntax::private_name) &&
         parse (TOK_OPEN_CURLY)))
    return (CTree*)0; 
  CTree *ns = semantic ().introduce_unnamed_ns ();
  if (! ns)
    return (CTree*)0;
  return (parse (&CCSyntax::namespace_body) && parse (TOK_CLOSE_CURLY)) ?
    semantic ().namespace_def (ns) : semantic ().namespace_def_err (ns);
}

bool CCSyntax::unnamed_ns_def () {
  return look_ahead (TOK_NAMESPACE) && parse (&CCSyntax::rule_unnamed_ns_def);
}


CTree *CCSyntax::namespace_body () {
  // 1: decl_seq
  return opt (parse (&CCSyntax::decl_seq)) ?
    builder ().namespace_body () : (CTree*)0;
}

CTree *CCSyntax::ns_alias_def () {
  // 5: NAMESPACE  identifier  =  qual_ns_spec  ;
  return (parse (TOK_NAMESPACE) && identifier () &&
          parse (TOK_ASSIGN) && parse (&CCSyntax::qual_ns_spec) &&
          parse (TOK_SEMI_COLON)) ?
    semantic ().introduce_ns_alias () : (CTree*)0;
}

CTree *CCSyntax::qual_ns_spec () {
  // 1: namespace_name
  // 2: nested_name_spec  namespace_name
  // 2: colon_colon  namespace_name
  // 3: colon_colon  nested_name_spec  namespace_name
  bool ok = (colon_colon (), nested_name_spec (), namespace_name ());
  semantic ().reset_search_scope ();
  return ok ? builder ().qual_ns_spec () : (CTree*)0; 
}

CTree *CCSyntax::using_decl () {
  // 4: USING  ::  unqual_id  ;
  // 4: USING  nested_name_spec  unqual_id  ;
  // 5: USING  ::  nested_name_spec  unqual_id  ;
  // 5: USING  TYPENAME  nested_name_spec  unqual_id  ;
  // 6: USING  TYPENAME  ::  nested_name_spec  unqual_id  ;
  if (! parse (TOK_USING))
    return (CTree*)0;
  bool is_type = parse (TOK_TYPENAME);
  bool is_root = colon_colon ();
  bool ok = ((nested_name_spec () || (! is_type && is_root)) && 
             unqual_id () && parse (TOK_SEMI_COLON));
  semantic ().reset_search_scope ();
  return ok ? semantic ().using_decl () : (CTree*)0;
}

CTree *CCSyntax::using_directive () {
  // 4: USING  NAMESPACE  namespace_name  ;
  // 5: USING  NAMESPACE  ::  namespace_name  ;
  // 5: USING  NAMESPACE  nested_name_spec  namespace_name  ;
  // 6: USING  NAMESPACE  ::  nested_name_spec  namespace_name  ;
  bool ok = 
    (parse (TOK_USING) && parse (TOK_NAMESPACE) && 
     (colon_colon (), nested_name_spec (),
      (namespace_name () && parse (TOK_SEMI_COLON))));
  semantic ().reset_search_scope ();
  return ok ? semantic ().using_directive () : (CTree*)0;
}

/*****************************************************************************/
/*                                                                           */
/*                     A . 7   D e c l a r a t o r s                         */
/*                                                                           */
/*****************************************************************************/

CTree *CCSyntax::init_declarator () { 
  CTree *id = CSyntax::init_declarator ();
  // leave possibly entered class scope
  semantic ().finish_init_declarator ();
  return id;
}

void CCSyntax::init_direct_declarator1 () {
  CSyntax::init_direct_declarator1 ();
}

CTree *CCSyntax::rule_direct_declarator1 () { 
  // 3: [  array_delim  ]
  // 3: (  param_decl_clause  )
  // 4: (  param_decl_clause  )  cv_qual_seq
  // 4: (  param_decl_clause  )  exception_spec
  // 5: (  param_decl_clause  )  cv_qual_seq  exception_spec
  return (parse (TOK_OPEN_ROUND) ?  
           (parse (&CCSyntax::param_decl_clause) && 
            parse (TOK_CLOSE_ROUND) &&
            opt (cv_qual_seq ()) &&
            opt (parse (&CCSyntax::exception_spec))) : 
          (parse (TOK_OPEN_SQUARE) && 
           parse (&CCSyntax::array_delim) &&
           parse (TOK_CLOSE_SQUARE))) ?
    builder ().direct_declarator1 () : (CTree*)0;
}

bool CCSyntax::direct_declarator1 () {
  return predict_1 (_direct_declarator1_1) &&
    parse (&CCSyntax::rule_direct_declarator1);
}

CTree *CCSyntax::array_delim () { 
  // 0:
  // 1: const_expr
  CStructure *scope;
  scope = semantic ().array_delim ();
  parse (&CCSyntax::const_expr);
  semantic ().array_delim (scope);
  return builder ().array_delim ();
}

CTree *CCSyntax::ptr_operator () { 
  // 1: &
  // 1: *
  // 2: *  cv_qual_seq
  // 2: nested_name_spec  *
  // 3: nested_name_spec  *  cv_qual_seq
  // 3: ::  nested_name_spec  *
  // 4: ::  nested_name_spec  *  cv_qual_seq
  bool ok =              // Win __cdecl hack :(
    ((parse (TOK_AND) && opt (cv_qual_seq ())) ||
     ((colon_colon () ? 
       nested_name_spec () : 
      opt (nested_name_spec ())) &&
       parse (TOK_MUL) && opt (cv_qual_seq ())));
  semantic ().reset_search_scope ();
  return ok ? builder ().ptr_operator () : (CTree*)0;
}

void CCSyntax::init_cv_qual () {
  // CSyntax::init_cv_qual is not called, because TOK_RESTRICT is not wanted!
  _cv_qual_1.set (TOK_CONST);
  _cv_qual_1.set (TOK_VOLATILE);
}

CTree *CCSyntax::declarator_id () { 
  // 1: id_expr
  // 1: type_name
  // 2: nested_name_spec  type_name
  // 2: ::  type_name
  // 3: ::  nested_name_spec  type_name
  semantic ().enter_entity_name ();
  bool is_type_name = true;
  bool ok =
    ((colon_colon (), nested_name_spec (), type_name ()) ||
     ((is_type_name = false), id_expr ()));
  semantic ().reset_search_scope ();
  semantic ().leave_entity_name ();
  return ok ? semantic ().declarator_id (is_type_name) : (CTree*)0;
}

CTree *CCSyntax::direct_abst_declarator () { 
  // 1+: direct_declarator1...
  // 3 : ( abst_declarator )
  // 3+: ( abst_declarator )  direct_declarator1...
  return ((direct_declarator1 () ||
           (parse (TOK_OPEN_ROUND) && 
            parse (&CCSyntax::abst_declarator) &&
            parse (TOK_CLOSE_ROUND))) &&  
          opt (seq (&CCSyntax::direct_declarator1))) ? 
    semantic ().direct_abst_declarator () : (CTree*)0;
}

CTree *CCSyntax::param_decl_clause () { 
  // 0:
  // 1: ...
  // 1: param_decl_list
  // 3: param_decl_list  ... 
  semantic ().enter_param_decl_clause (); // enter function prototype scope
  if (!look_ahead (TOK_CLOSE_ROUND)) {
    if (!look_ahead (TOK_ELLIPSIS))
      parse (&CCSyntax::param_decl_list); 
    parse (TOK_ELLIPSIS);
  }
  semantic ().leave_param_decl_clause ();
  return semantic ().param_decl_clause ();
}

CTree *CCSyntax::param_decl () { 
  // 1: param_decl1 
  // 3: param_decl1  =  ass_expr
  if (! parse (&CCSyntax::param_decl1))
    return (CTree*)0;
  semantic ().finish_decl ();
  // parse param_init later due to potential use of names of members
  // that could not be resolved till the end of the class definition 
  if (semantic ().in_class_def () && look_ahead (TOK_ASSIGN)) { 
    CTree *pd = builder ().param_decl ();
    semantic ().add_rule (&CCSyntax::param_init, pd,
                          &CCSyntax::skip_param_init);
    return pd;
  // not in class definition, so parse as usual
  } else {
    parse (&CCSyntax::param_init);
    return builder ().param_decl ();
  }
}

CTree *CCSyntax::param_init () { 
  // 2: =  ass_expr
  return (parse (TOK_ASSIGN) && parse (&CCSyntax::ass_expr)) ? 
    builder ().param_init () : (CTree*)0;
}

CTree *CCSyntax::fct_def () { 
  // 2: declarator  fct_body
  // 2: declarator  fct_try_block
  // 3: declarator  ctor_init  fct_body
  // 3: decl_spec_seq  declarator  fct_body
  // 3: decl_spec_seq  declarator  fct_try_block
  // 4: decl_spec_seq  declarator  ctor_init  fct_body
  if (! ((parse (&CCSyntax::decl_spec_seq) || 
          semantic ().empty_decl_spec_seq ()), 
         semantic ().reset_search_scope (true),
         parse (&CCSyntax::declarator))) {
    semantic ().finish_declarator ();
    return (CTree*)0;
  }

  // introduce name and check scope; functions shall be 
  // defined only in namespace or class scope
  if (! semantic ().introduce_function ())
    return (CTree*)0;

  // skip function bodies on demand
  bool skip_fct_bodies = false;
  if ((_skip_bodies & SKIP_BODIES_ALL) ||
      ((_skip_bodies & SKIP_BODIES_TPL) && semantic ().skip_tpl_fct_bodies ()) ||
      ((_skip_bodies & SKIP_BODIES_NON_PRJ) && semantic ().non_project_loc ()) ||
      ((_skip_bodies & SKIP_BODIES_NON_PRIM) && semantic ().non_primary_loc ()))
    skip_fct_bodies = true;

  // parse fct_body and ctor_init later due to potential use of names 
  // of members that could not be resolved till the end of the class 
  // definition
  if (semantic ().in_class_def ()) { 
    CTree *fd = semantic ().finish_fct_def ();
    if (look_ahead (TOK_TRY)) 
      semantic ().add_rule (&CCSyntax::fct_try_block, fd,
                            &CCSyntax::skip_fct_try_block);
    else {
      if (look_ahead (TOK_COLON)) 
        semantic ().add_rule (&CCSyntax::ctor_init, fd, 
                              &CCSyntax::skip_ctor_init);
      else if (look_ahead (TOK_ASSIGN)) 
        semantic ().add_rule (&CCSyntax::pure_spec, fd, 
                              &CCSyntax::skip_ctor_init);
      if (look_ahead (TOK_OPEN_CURLY)) {
        semantic ().add_rule (skip_fct_bodies ? &CCSyntax::skipped_fct_body :
          &CCSyntax::fct_body, fd, &CCSyntax::skip_fct_body);
      } else {
        delete fd;
        fd = (CTree*)0;
      }
    }
    return fd;
  // not in class definition, so parse as usual
  } else
    return (parse (&CCSyntax::fct_try_block) ||
            ((parse (&CCSyntax::ctor_init) || parse (&CCSyntax::pure_spec)), 
             parse (skip_fct_bodies ? &CCSyntax::skipped_fct_body :
                    &CCSyntax::fct_body))) ? 
      semantic ().finish_fct_def () : (CTree*)0;
}

CTree *CCSyntax::skipped_fct_body () {
  static int stop_tokens[] = { TOK_CLOSE_CURLY, 0 };

  if (!parse (TOK_OPEN_CURLY)) return (CTree*)0;
  semantic ().enter_local_scope (); // enter local (block) scope
  skip (stop_tokens, false);
  if (!parse (TOK_CLOSE_CURLY)) return (CTree*)0;
  return semantic ().cmpd_stmt ();
}

CTree *CCSyntax::fct_body () { 
  // 1: cmpd_stmt
  return parse (&CCSyntax::cmpd_stmt) ? 
    builder ().fct_body () : (CTree*)0;
}

CTree *CCSyntax::init () { 
  // 2: =  init_clause 
  // 3: (  expr_list  )
  return (parse (TOK_ASSIGN) ? parse (&CCSyntax::init_clause) : 
          (parse (TOK_OPEN_ROUND) && parse (&CCSyntax::expr_list) &&
           parse (TOK_CLOSE_ROUND))) ?
    builder ().init () : (CTree*)0;
}

CTree *CCSyntax::init_clause () { 
  // 1: ass_expr
  // 2: {  }
  // 3: {  init_list  }
  return (parse (&CCSyntax::ass_expr) ||
          (parse (TOK_OPEN_CURLY) && 
           (parse (&CCSyntax::init_list), parse (TOK_CLOSE_CURLY)))) ?
    builder ().init_clause () : (CTree*)0;
}

/*****************************************************************************/
/*                                                                           */
/*                          A . 8   C l a s s e s                            */
/*                                                                           */
/*****************************************************************************/

CTree *CCSyntax::rule_class_spec () { 
  // 3: class_head  {  }
  // 4: class_head  {  member_spec  }
  if (! (is_class_def () && class_head () && look_ahead (TOK_OPEN_CURLY)))
    return (CTree*)0; 
  semantic ().reenter_class_scope ();
  return (consume () &&
          (parse (&CCSyntax::member_spec), parse (TOK_CLOSE_CURLY))) ? 
    semantic ().class_spec () : semantic ().class_spec_err ();
}

bool CCSyntax::class_spec () {
  return predict_1 (_class_spec_1) && parse (&CCSyntax::rule_class_spec);
}

CTree *CCSyntax::rule_class_head () { 
  // 1: class_key
  // 2: class_key  identifier
  // 2: class_key  template_id
  // 2: class_key  base_clause
  // 3: class_key  identifier  base_clause
  // 3: class_key  template_id  base_clause
  // 3: class_key  nested_name_spec  identifier
  // 3: class_key  nested_name_spec  template_id
  // 4: class_key  nested_name_spec  identifier  base_clause
  // 4: class_key  nested_name_spec  template_id  base_clause
  semantic ().enter_entity_name ();
  bool parsed = 
    (class_key () && 
     (nested_name_spec () ? 
      (template_id () || identifier ()) :
      (template_id () || identifier () || parse (&CCSyntax::private_name))));
  semantic ().reset_search_scope ();
  semantic ().leave_entity_name ();
  // introduce new class in current scope only if a
  // base class list or the class body follows
  if (parsed && (look_ahead (TOK_OPEN_CURLY) || look_ahead (TOK_COLON))) {
    CTree *cs = semantic ().introduce_class ();
    if (cs) {
      parse (&CCSyntax::base_clause);
      semantic ().add_base_classes (cs);
      return cs;
    }
  } 
  return (CTree*)0;
}

bool CCSyntax::class_head () {
  return predict_1 (_class_head_1) && parse (&CCSyntax::rule_class_head);
}

void CCSyntax::init_class_key () {
  CSyntax::init_class_key ();
  _class_key_1.set (TOK_CLASS);
}

CTree *CCSyntax::member_decl () { 
  // 1: member_decl1
  // 1: fct_def
  // 1: using_decl
  // 1: member_template_decl
  // 1: access_decl
  // 2: access_spec  :
  if (access_spec ()) 
    if (parse (TOK_COLON)) 
      return semantic ().access_spec ();
    else 
      return (CTree*)0;
  if (! (parse (&CCSyntax::using_decl) ||
         parse (&CCSyntax::access_decl) ||
         parse (&CCSyntax::member_template_decl) ||
         (is_fct_def () ? parse (&CCSyntax::fct_def) :
          parse (&CCSyntax::member_decl1) /*||
          parse (&CCSyntax::fct_def)*/))) 
    return (CTree*)0;
  semantic ().finish_decl ();
  return builder ().member_decl (); 
}

CTree *CCSyntax::member_decl1 () { 
  // 1: ;
  // 2: decl_spec_seq  ;
  // 2: member_declarator_list  ;
  // 3: decl_spec_seq  member_declarator_list  ;
  return ((parse (&CCSyntax::decl_spec_seq) || 
           semantic ().empty_decl_spec_seq ()), 
          parse (&CCSyntax::member_declarator_list), 
          parse (TOK_SEMI_COLON)) ?
    semantic ().member_decl1 () : (CTree*)0;
}

CTree *CCSyntax::access_decl () { 
  // 3: nested_name_spec  unqual_id  ;
  // 4: nested_name_spec  template_key  unqual_id  ;
  // 4: ::  nested_name_spec  unqual_id  ;
  // 5: ::  nested_name_spec  template_key  unqual_id  ;
  bool ok =  
    ((colon_colon (), nested_name_spec ()) && 
     (template_key (), 
      unqual_id ()) && 
     parse (TOK_SEMI_COLON));
  semantic ().reset_search_scope ();
  return ok ? semantic ().access_decl () : (CTree*)0;
}

CTree *CCSyntax::member_declarator () { 
  // 1: declarator
  // 2: declarator  pure_spec
  // 2: declarator  const_init
  // 3: identifier  :  const_expr
  // 3: private_name  :  const_expr
  CT_InitDeclarator *md;

  // lookahead: check if this is a bitfield declarator
  State s = save_state ();
  bool bitfield = (identifier (), parse (TOK_COLON));
  restore_state (s);

  // parse const_expr and const_init later due to potential use of names 
  // of members that could not be resolved till the end of the class 
  // definition
  if (semantic ().in_class_def () && ! bitfield) { 
    if (parse (&CCSyntax::declarator)) {
      // optional extensions
      parse (&CCSyntax::init_declarator_ext);
      // constant initializer
      if (! parse (&CCSyntax::pure_spec) && look_ahead (TOK_ASSIGN)) {
        md = (CT_InitDeclarator*)semantic ().introduce_member ();
        // initializers of const members are evaluated immediately
        if (md && md->Object () && md->Object ()->TypeInfo ()->isConst ()) {
          if (parse (&CCSyntax::const_init))
            md->Initializer (builder ().get_node (builder ().nodes ()-1));
        // initializers of non-const members are evaluated later
        } else if (md)
          semantic ().add_rule (&CCSyntax::const_init, md, 
                                &CCSyntax::skip_const_init);
      // pure specifier; always is "0"
      } else
        md = (CT_InitDeclarator*)semantic ().introduce_member ();
      return md;
    }
  // not in class definition scope, so parse as usual
  } else {
    if (bitfield) {
      if ((identifier () ||
           parse (&CCSyntax::private_name)) && parse (TOK_COLON) && 
          parse (&CCSyntax::const_expr))
        return semantic ().introduce_member ();
    } else {
      if (parse (&CCSyntax::declarator) &&
	  opt (parse (&CCSyntax::init_declarator_ext)) &&
          opt (parse (&CCSyntax::pure_spec) || 
               parse (&CCSyntax::const_init)))
        return semantic ().introduce_member ();
    }
  }
  return semantic ().member_declarator_err ();
}

CTree *CCSyntax::pure_spec () { 
  // 2: =  0
  return (parse (TOK_ASSIGN) && parse (TOK_ZERO_VAL)) ?
    semantic ().pure_spec () : (CTree*)0;
}

CTree *CCSyntax::const_init () { 
  // 2: =  const_expr
  return (parse (TOK_ASSIGN) && parse (&CCSyntax::const_expr)) ?
    semantic ().const_init () : (CTree*)0;
}

/*****************************************************************************/
/*                                                                           */
/*                   A . 9   D e r i v e d  c l a s s e s                    */
/*                                                                           */
/*****************************************************************************/

CTree *CCSyntax::base_clause () { 
  // 2: :  base_spec_list
  return (parse (TOK_COLON) && parse (&CCSyntax::base_spec_list)) ?
    builder ().base_clause () : (CTree*)0;
}

CTree *CCSyntax::base_spec_list () { 
  // 1 : base_spec
  // 3+: base_spec  ,  base_spec ..
  return list (&CCSyntax::base_spec, TOK_COMMA) ?
    builder ().base_spec_list () : (CTree*)0;
}

CTree *CCSyntax::base_spec () { 
  // 1: class_name
  // 2: ::  class_name
  // 2: VIRTUAL  class_name
  // 2: access_spec  class_name
  // 2: nested_name_spec  class_name
  // 3: ::  nested_name_spec  class_name
  // 3: VIRTUAL  ::  class_name
  // 3: access_spec  ::  class_name
  // 3: VIRTUAL  nested_name_spec  class_name
  // 3: access_spec  nested_name_spec  class_name
  // 3: VIRTUAL  access_spec  class_name
  // 3: access_spec  VIRTUAL  class_name
  // 4: VIRTUAL  ::  nested_name_spec  class_name
  // 4: access_spec  ::  nested_name_spec  class_name
  // 4: VIRTUAL  access_spec  nested_name_spec  class_name
  // 4: access_spec  VIRTUAL  nested_name_spec  class_name
  // 4: VIRTUAL  access_spec  ::  class_name
  // 4: access_spec  VIRTUAL  ::  class_name
  // 5: VIRTUAL  access_spec  ::  nested_name_spec  class_name
  // 5: access_spec  VIRTUAL  ::  nested_name_spec  class_name
  bool ok = false;
  if (parse (TOK_VIRTUAL) ? opt (access_spec ()) :
      (access_spec () ? opt (parse (TOK_VIRTUAL)) : true)) {
    colon_colon ();
    nested_name_spec ();
    semantic ().enter_base_spec ();
    ok = class_name ();
    semantic ().leave_base_spec ();
  }
  semantic ().reset_search_scope ();
  return ok ? builder ().base_spec () : (CTree*)0;
}

void CCSyntax::init_access_spec () {
  _access_spec_1.set (TOK_PRIVATE);
  _access_spec_1.set (TOK_PROTECTED);
  _access_spec_1.set (TOK_PUBLIC);
}

CTree *CCSyntax::rule_access_spec () { 
  // 1: PUBLIC
  // 1: PRIVATE
  // 1: PROTECTED
  consume (); // already checked by look ahead
  return builder ().access_spec ();
}

bool CCSyntax::access_spec () {
  return predict_1 (_access_spec_1) && parse (&CCSyntax::rule_access_spec);
}

/*****************************************************************************/
/*                                                                           */
/*        A . 1 0   S p e c i a l  m e m b e r  f u n c t i o n s            */
/*                                                                           */
/*****************************************************************************/

void CCSyntax::init_conv_fct_id () {
  _conv_fct_id_1.set (TOK_OPERATOR);
}

CTree *CCSyntax::rule_conv_fct_id () { 
  // 2: OPERATOR  conv_type_id
  consume (); // TOK_OPERATOR checked during prediction
  SearchScope sc;
  get_search_scope (sc);
  semantic ().reset_search_scope ();
  bool parsed = parse (&CCSyntax::conv_type_id);
  set_search_scope (sc);
  return parsed ?
    builder ().conv_fct_id () : (CTree*)0;
}

bool CCSyntax::conv_fct_id () {
  return look_ahead (TOK_OPERATOR) && parse (&CCSyntax::rule_conv_fct_id);
}

CTree *CCSyntax::conv_type_id () { 
  // 2: type_spec_seq  conv_declarator
  // 2: type_spec_seq  private_name
  bool success;
  semantic ().enter_conv_type_id ();
  success = parse (&CCSyntax::type_spec_seq);
  semantic ().leave_conv_type_id ();
  if (! success)
    return (CTree*)0;
  success = (parse (&CCSyntax::conv_declarator) ||
             parse (&CCSyntax::private_name));
  semantic ().finish_declarator ();
  return success ? 
    semantic ().introduce_named_type () : (CTree*)0;
}

CTree *CCSyntax::conv_declarator () { 
  // 1+: ptr_operator..  
  return seq (&CCSyntax::ptr_operator) ?
    semantic ().abst_declarator () : (CTree*)0;
}

CTree *CCSyntax::ctor_init () { 
  // 2: :  mem_init_list
  return (parse (TOK_COLON) && parse (&CCSyntax::mem_init_list)) ?
    semantic ().ctor_init () : (CTree*)0;
}

CTree *CCSyntax::mem_init_list () { 
  // 1 : mem_init
  // 3+: mem_init  ,  mem_init ..
  return list (&CCSyntax::mem_init, TOK_COMMA) ?
    builder ().mem_init_list () : (CTree*)0;
}

CTree *CCSyntax::mem_init () { 
  // 3: mem_init_id  (  )
  // 4: mem_init_id  (  expr_list  )
  return (parse (&CCSyntax::mem_init_id) && parse (TOK_OPEN_ROUND) && 
          (parse (&CCSyntax::expr_list), parse (TOK_CLOSE_ROUND))) ?
    builder ().mem_init () : (CTree*)0;
}

CTree *CCSyntax::mem_init_id () { 
  // 1: identifier
  // 1: class_name
  // 2: nested_name_spec  class_name
  // 2: ::  class_name
  // 3: ::  nested_name_spec  class_name
  bool ok = 
    ((colon_colon (), nested_name_spec (), 
      class_name ()) ||
     identifier ());
  semantic ().reset_search_scope ();
  return ok ? builder ().mem_init_id () : (CTree*)0;
}

/*****************************************************************************/
/*                                                                           */
/*                     A . 1 1   O v e r l o a d i n g                       */
/*                                                                           */
/*****************************************************************************/

void CCSyntax::init_oper_fct_id () { 
  _oper_fct_id_1.set (TOK_OPERATOR);
}

CTree *CCSyntax::rule_oper_fct_id () { 
  // 2: OPERATOR  any_operator
  // 2: OPERATOR  NEW
  // 2: OPERATOR  DELETE
  // 3: OPERATOR  (  )
  // 3: OPERATOR  [  ]
  // 4: OPERATOR  NEW  [  ]
  // 4: OPERATOR  DELETE  [  ]
  static int any_operator[] = { 
    TOK_PLUS, TOK_MINUS, TOK_MUL, TOK_DIV, TOK_MODULO, TOK_ROOF, 
    TOK_AND, TOK_OR, TOK_TILDE, TOK_NOT, TOK_LESS, TOK_GREATER, 
    TOK_EQL, TOK_GEQ, TOK_LEQ, TOK_NEQ, TOK_AND_AND, TOK_OR_OR, 
    TOK_LSH, TOK_RSH, TOK_DECR, TOK_INCR, TOK_ASSIGN, TOK_COMMA, 
    TOK_ADD_EQ, TOK_SUB_EQ, TOK_MUL_EQ, TOK_DIV_EQ, TOK_MOD_EQ, 
    TOK_AND_EQ, TOK_IOR_EQ, TOK_LSH_EQ, TOK_RSH_EQ, TOK_XOR_EQ,
    TOK_PTS_STAR, TOK_PTS, 0 };
  return (parse (TOK_OPERATOR) && 
          ((parse (TOK_NEW) || parse (TOK_DELETE)) ? 
            (parse (TOK_OPEN_SQUARE) ? parse (TOK_CLOSE_SQUARE) : true) :
           parse (TOK_OPEN_ROUND) ? parse (TOK_CLOSE_ROUND) : 
           parse (TOK_OPEN_SQUARE) ? parse (TOK_CLOSE_SQUARE) : 
           parse (any_operator))) ?
    builder ().oper_fct_id () : (CTree*)0;
}

bool CCSyntax::oper_fct_id () {
  return predict_1 (_oper_fct_id_1) && parse (&CCSyntax::rule_oper_fct_id);
}

/*****************************************************************************/
/*                                                                           */
/*                       A . 1 2   T e m p l a t e s                         */
/*                                                                           */
/*****************************************************************************/

void CCSyntax::init_template_key () {
  _template_key_1.set (TOK_TEMPLATE);
}

CTree *CCSyntax::rule_template_key () {
  // 1: TEMPLATE
  consume ();
  return semantic ().template_key ();
}

bool CCSyntax::template_key () {
  return look_ahead () == TOK_TEMPLATE && parse (&CCSyntax::rule_template_key);
}


CTree *CCSyntax::template_decl () { 
  // 5: TEMPLATE  <  template_param_list  >  decl
  // 6: EXPORT  TEMPLATE  <  template_param_list  >  decl
  if (! ((parse (TOK_EXPORT), parse (TOK_TEMPLATE)) && parse (TOK_LESS) && 
          parse (&CCSyntax::template_param_list) && parse (TOK_GREATER)))
    return (CTree*)0;
  semantic ().enter_template_decl ();
  parse (&CCSyntax::decl);
  return semantic ().template_decl ();
}

CTree *CCSyntax::member_template_decl () { 
  // 5: TEMPLATE  <  template_param_list  >  member_decl
  // 6: EXPORT  TEMPLATE  <  template_param_list  >  member_decl
  if (! ((parse (TOK_EXPORT), parse (TOK_TEMPLATE)) && parse (TOK_LESS) && 
          parse (&CCSyntax::template_param_list) && parse (TOK_GREATER)))
    return (CTree*)0;
  semantic ().enter_template_decl ();
  parse (&CCSyntax::member_decl);
  return semantic ().template_decl ();
}

CTree *CCSyntax::template_param_list () { 
  // 1 : template_param
  // 3+: template_param  ,  template_param ..
  semantic ().enter_template_param_list (token_provider->get_state ());
  bool success = list (&CCSyntax::template_param, TOK_COMMA);
  semantic ().leave_template_param_list ();
  return success ?
    semantic ().template_param_list () : (CTree*)0;
}

CTree *CCSyntax::template_param () { 
  // 1: type_param
  // 1: non_type_param
  return (parse (&CCSyntax::type_param) || 
          parse (&CCSyntax::non_type_param)) ?
    semantic ().template_param () : (CTree*)0;
}

CTree *CCSyntax::type_param () { 
  // 1: CLASS
  // 1: TYPENAME
  // 2: CLASS  identifier
  // 2: TYPENAME  identifier
  // 3: CLASS  =  type_id
  // 3: TYPENAME  =  type_id
  // 4: CLASS  identifier  =  type_id
  // 4: TYPENAME  identifier  =  type_id
  // 5: TEMPLATE  <  template_param_list  >  CLASS 
  // 6: TEMPLATE  <  template_param_list  >  CLASS  identifier
  // 7: TEMPLATE  <  template_param_list  >  CLASS  =  id_expr
  // 8: TEMPLATE  <  template_param_list  >  CLASS  identifier  =  id_expr
  return ((parse (TOK_CLASS) || parse (TOK_TYPENAME)) ?
           (identifier () || parse (&CCSyntax::private_name), 
            parse (TOK_ASSIGN) ? parse (&CCSyntax::type_id) : true) :
          parse (TOK_TEMPLATE) && 
           parse (TOK_LESS) && parse (&CCSyntax::template_param_list) && 
           parse (TOK_GREATER) && parse (TOK_CLASS) && 
           (identifier (), 
            parse (TOK_ASSIGN) ? 
             (semantic ().enter_expr (), 
              (id_expr () ? semantic ().leave_expr () :
               (semantic ().leave_expr (), false))) : true)) ?
    semantic ().introduce_type_param () : (CTree*)0;
}

CTree *CCSyntax::non_type_param () { 
  // 1: decl_spec_seq 
  // 2: decl_spec_seq  abst_declarator
  // 2: decl_spec_seq  declarator
  // 2: decl_spec_seq  param_init
  // 3: decl_spec_seq  abst_declarator  param_init
  // 3: decl_spec_seq  declarator  param_init
  if (! (parse (&CCSyntax::decl_spec_seq) && 
         (parse (&CCSyntax::declarator) || 
          parse (&CCSyntax::abst_declarator) ||
          parse (&CCSyntax::private_name))))
    return (CTree*)0;
  CTree *result = semantic ().introduce_non_type_param ();
  if (! result)
    return (CTree*)0;
  parse (&CCSyntax::param_init);
  return builder ().non_type_param (result);
}

void CCSyntax::init_template_id () {
  init_oper_fct_id ();
  init_conv_fct_id ();
  _template_id_1.set (TOK_ID);
  _template_id_1 |= _oper_fct_id_1;
  _template_id_1 |= _conv_fct_id_1;
}

CTree *CCSyntax::rule_template_id () { 
  // 3: template_name  <  >
  // 4: template_name  <  template_arg_list  >
  if (! ((parse (&CCSyntax::template_name) ||
          oper_fct_id () ||
          conv_fct_id ()) && parse (TOK_LESS)))
    return (CTree*)0;
  SearchScope sc;
  get_search_scope (sc);
  semantic ().reset_search_scope ();
  semantic ().enter_entity_name (false);
  parse (&CCSyntax::template_arg_list);
  semantic ().leave_entity_name ();
  set_search_scope (sc);
  return parse (TOK_GREATER) ?
    semantic ().introduce_template_instance () : (CTree*)0;
}

bool CCSyntax::template_id () {
  return predict_1 (_template_id_1) && parse (&CCSyntax::rule_template_id);
}

void CCSyntax::init_class_template_id () {
  _template_id_1.set (TOK_ID);
}

CTree *CCSyntax::rule_class_template_id () { 
  // 3: template_name  <  >
  // 4: template_name  <  template_arg_list  >
  if (! (parse (&CCSyntax::class_template_name) && parse (TOK_LESS)))
    return (CTree*)0;
  SearchScope sc;
  get_search_scope (sc);
  semantic ().reset_search_scope ();
  semantic ().enter_entity_name (false);
  parse (&CCSyntax::template_arg_list);
  semantic ().leave_entity_name ();
  set_search_scope (sc);
  return parse (TOK_GREATER) ?
    semantic ().introduce_template_instance () : (CTree*)0;
}

bool CCSyntax::class_template_id () {
  return predict_1 (_template_id_1) && parse (&CCSyntax::rule_class_template_id);
}

CTree *CCSyntax::template_arg_list () { 
  // 1 : template_arg
  // 3+: template_arg  ,  template_arg ..
  semantic ().enter_template_arg_list (token_provider->get_state ());
  bool success = list (&CCSyntax::template_arg, TOK_COMMA);
  semantic ().leave_template_arg_list ();
  return success ?
    builder ().template_arg_list () : (CTree*)0;
}

CTree *CCSyntax::template_arg () { 
  // 1: ass_expr
  // 1: type_id
  // 1: id_expr
  if (! (parse (&CCSyntax::type_id) || parse (&CCSyntax::ass_expr))) {
    semantic ().enter_expr ();
    bool parsed = id_expr ();
    semantic ().leave_expr ();
    if (! parsed)
      return (CTree*)0;
  }
  return builder ().template_arg ();
}

void CCSyntax::init_explicit_instantiation () { 
  _explicit_instantiation_1.set (TOK_TEMPLATE);
  _explicit_instantiation_1.set (TOK_EXTERN);
}

CTree *CCSyntax::rule_explicit_instantiation () { 
  // 2: TEMPLATE  decl
  // 3: EXTERN  TEMPLATE  decl  => GCC EXTENSION!!!
  if (! (parse (TOK_EXTERN), parse (TOK_TEMPLATE)))
    return (CTree*)0;
  semantic ().enter_template_param_list (0);
  return parse (&CCSyntax::decl) ?
    semantic ().explicit_instantiation () : (CTree*)0;
}

bool CCSyntax::explicit_instantiation () {
  return predict_1 (_explicit_instantiation_1) &&
         parse (&CCSyntax::rule_explicit_instantiation);
}

void CCSyntax::init_explicit_specialization () {
  _explicit_specialization_1.set (TOK_TEMPLATE);
}

CTree *CCSyntax::rule_explicit_specialization () { 
  // 4: TEMPLATE  <  >  decl
  if (! (consume () && parse (TOK_LESS) && parse (TOK_GREATER)))
    return (CTree*)0;
  semantic ().enter_template_param_list (0);
  return parse (&CCSyntax::decl) ?
    semantic ().explicit_specialization () : (CTree*)0;
}

bool CCSyntax::explicit_specialization () {
  return predict_1 (_explicit_specialization_1) &&
         parse (&CCSyntax::rule_explicit_specialization);
}

/*****************************************************************************/
/*                                                                           */
/*                A . 1 3  E x c e p t i o n  h a n d l i n g                */
/*                                                                           */
/*****************************************************************************/

CTree *CCSyntax::try_block () { 
  // 3+: TRY  cmpd_stmt  handler_seq
  return (parse (TOK_TRY) && parse (&CCSyntax::cmpd_stmt) && 
          parse (&CCSyntax::handler_seq)) ?
    builder ().try_block () : (CTree*)0;
}

CTree *CCSyntax::fct_try_block () { 
  // 3+: TRY  fct_body  handler_seq
  // 4+: TRY  ctor_init  fct_body  handler_seq
  bool skip_fct_bodies = false;
  if ((_skip_bodies & SKIP_BODIES_ALL) ||
      ((_skip_bodies & SKIP_BODIES_TPL) && semantic ().skip_tpl_fct_bodies ()) ||
      ((_skip_bodies & SKIP_BODIES_NON_PRJ) && semantic ().non_project_loc ()) ||
      ((_skip_bodies & SKIP_BODIES_NON_PRIM) && semantic ().non_primary_loc ()))
    skip_fct_bodies = true;
  return (parse (TOK_TRY) && opt (parse (&CCSyntax::ctor_init)) && 
          parse (skip_fct_bodies ? 
            &CCSyntax::skipped_fct_body : &CCSyntax::fct_body) && 
          parse (&CCSyntax::handler_seq)) ?
    builder ().fct_try_block () : (CTree*)0;
}

CTree *CCSyntax::handler_seq () { 
  // 1+: handler..
  return seq (&CCSyntax::handler) ?
    builder ().handler_seq () : (CTree*)0;
}

CTree *CCSyntax::handler () { 
  // 5: CATCH  (  exception_decl  )  cmpd_stmt
  if (! parse (TOK_CATCH))
    return (CTree*)0;
  semantic ().enter_local_scope (); // enter local (block) scope
  return (parse (TOK_OPEN_ROUND) && parse (&CCSyntax::exception_decl) && 
          parse (TOK_CLOSE_ROUND) && parse (&CCSyntax::cmpd_stmt)) ?
    semantic ().handler () : (CTree*)0;
}

CTree *CCSyntax::exception_decl () { 
  // 1: ...
  // 1: type_spec_seq 
  // 2: type_spec_seq  abst_declarator
  // 2: type_spec_seq  declarator
  if (parse (TOK_ELLIPSIS))
    return builder ().exception_decl ();
  bool parsed = (parse (&CCSyntax::type_spec_seq) && 
                 (parse (&CCSyntax::declarator) || 
                  parse (&CCSyntax::abst_declarator) ||
                  parse (&CCSyntax::private_name)));
  semantic ().finish_declarator ();
  return parsed ?
    semantic ().introduce_exception () : (CTree*)0;
}

CTree *CCSyntax::throw_expr () { 
  // 1: THROW
  // 2: THROW  ass_expr
  return (parse (TOK_THROW) && opt (parse (&CCSyntax::ass_expr))) ?
    builder ().throw_expr () : (CTree*)0;
}

CTree *CCSyntax::exception_spec () { 
  // 3: THROW  (  )
  // 4: THROW  (  ...  )              // MS VC++ extension :(
  // 4: THROW  (  type_id_list  )
  if (! (parse (TOK_THROW) && parse (TOK_OPEN_ROUND)))
    return (CTree*)0;
  semantic ().enter_exception_spec ();
  if ((parse (TOK_ELLIPSIS) || parse (&CCSyntax::type_id_list), 
       parse (TOK_CLOSE_ROUND)))
    return semantic ().exception_spec ();
  semantic ().leave_exception_spec ();
  return (CTree*)0;
}

CTree *CCSyntax::type_id_list () { 
  // 1 : type_id
  // 3+: type_id  ,  type_id ..
  return list (&CCSyntax::type_id, TOK_COMMA) ? 
    builder ().type_id_list () : (CTree*)0;
}


} // namespace Puma
