// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/Token.h"
#include "Puma/CTokens.h"
#include "Puma/CCommentTokens.h"
#include "Puma/PreMacro.h"
#include "Puma/PreAnswer.h"
#include "Puma/PreTree.h"
#include "Puma/ErrorStream.h"
#include "Puma/UnitManager.h"
#include "Puma/PrePredicate.h"
#include "Puma/PreTreeNodes.h"
#include "Puma/PreLevelState.h"
#include "Puma/PreParserState.h"
#include "Puma/PreMacroManager.h"
#include "Puma/PreFileIncluder.h"
#include "Puma/PreprocessorParser.h"
#include "Puma/PrePredicateManager.h"

#include <list>
#include <string>
using namespace std;

namespace Puma {


// Get the next token to parse.
Token *PreprocessorParser::next () { 
  return parseToken (); 
}


// Free the preprocessor syntax tree.
void PreprocessorParser::freeSyntaxTree (PreTree *node) {
  int i;
    
  if (! node) return;
    
  for (i = 0; i < node->sons (); i++) 
    freeSyntaxTree (node->son (i));
    
  for (i = 0; i < node->daughters (); i++)
    delete node->daughter (i);
    
  delete node;
}


// Free the preprocessor syntax tree.
void PreprocessorParser::freeSyntaxTree () { 
  freeSyntaxTree (_parserState->syntaxTree); 
  _parserState->syntaxTree = (PreTree*)0; 
}


// Invoke the parse process.
void PreprocessorParser::parse () {
  // Do not parse more than one time.
  if (_parserState->syntaxTree) 
    return;
    
  Token *token, *last = 0;

  // Print the result of the parse process if not in silent mode.
  while ((token = parseToken ()))
    if (! _silentMode) {
      if (last && 
          ((last->type () == TOK_CCOMMENT && 
            token->type () != TOK_CCOMMENT) ||
           (last->location ().filename ().name () !=
            token->location ().filename ().name ())))
        *_out << "\n";
        
      last = token;
      *_out << token->text () << flush;
    }
}


// Configure the preprocessor.
void PreprocessorParser::configure (const Config &c, bool process_includes) {
  unsigned i;
    
  _scanner.configure (c);

  for (i = 0; i < c.Options (); i++) {
    const ConfOption *o = c.Option (i);
    if (!strcmp (o->Name (), "--gnu") || !strcmp (o->Name (), "--gnu-2.95")) {
      supportGNU (true);
    } else if (! strcmp (o->Name (), "--lang-c")) {
      undefMacro ("__cplusplus");
    } else if (! strcmp (o->Name (), "-D")) {
      if (o->Arguments () < 1) continue;
      undefMacro (o->Argument (0));
      if (o->Arguments () == 2)
        defMacro (o->Argument (0), o->Argument (1));
      else
        defMacro (o->Argument (0));
    } else if (! strcmp (o->Name (), "-U")) {
      if (o->Arguments () != 1) continue;
      undefMacro (o->Argument (0));
    } else if (! strcmp (o->Name (), "-I")) {
      if (o->Arguments () != 1) continue;
      addInclPath (o->Argument (0));
    } else if (! strcmp (o->Name (), "-A")) {
      if (o->Arguments () != 2) continue;
      defPredicate (o->Argument (0), o->Argument (1));
    } else if (! strcmp (o->Name (), "--lock-macro")) {
      if (o->Arguments () < 1) continue;
      undefMacro (o->Argument (0));
      if (o->Arguments () == 2)
        defMacro (o->Argument (0), o->Argument (1), PreMacro::LOCK_MACRO);
      else
        defMacro (o->Argument (0), 0, PreMacro::LOCK_MACRO);
    } else if (! strcmp (o->Name (), "--inhibit-macro")) {
      if (o->Arguments () != 1) continue;
      defMacro (o->Argument (0), 0, PreMacro::INHIBIT_MACRO);
    } 
  }

  if (process_includes) {
    CUnit *forced_includes = new CUnit (*parserState ()->err);
    forced_includes->name ("<forced-includes>");
    _locals->addUnit (forced_includes);
    for (i = 0; i < c.Options (); i++) {
      const ConfOption *o = c.Option (i);
      if (! strcmp (o->Name (), "--include")) {
        if (o->Arguments () != 1) continue;
        const char *filename = o->Argument (0);
        *forced_includes << "#include ";
        if (filename[0] == '\"' || filename[0] == '<')
          *forced_includes << filename;
        else
          *forced_includes << "\"" << filename << "\"";
        *forced_includes << endl;
//        Unit *unit = fileIncluder ()->handleIncludeOption (o->Argument (0));
      }
    }
    *forced_includes << endu;
    fileIncluder ()->pushOnStack (forced_includes);
  }
}


// Parse a macro and return the name and the list of arguments
bool parseMacro (const char *name,
                 string &macro_name, list<string> &args) {
  if (!name) return false;
  
  string work = name;
  string::size_type arg_pos = work.find_first_of ("(");
  // arguments found?
  if (arg_pos != string::npos) {
    // check if there are characters in front of '('
    if (arg_pos == 0) return false; // syntax error
    macro_name = work.substr (0, arg_pos);
    work = work.substr (arg_pos + 1); // the argument list
    while ((arg_pos = work.find_first_of (",)")) != string::npos) {
      string arg = work.substr (0, arg_pos);
      if (arg[0] == ' ' || arg[arg.length () - 1] == ' ')
        return false;
      args.push_back (arg);
      work = work.substr (arg_pos + 1); // the argument list
    }
  }
  else {
    // no arguments
    macro_name = work;
  }
  return true;
}
                                     

// Define a new macro.
void PreprocessorParser::defMacro (const char *name, const char *body,
                                   char flags) const {
  string macro_name;
  list<string> args;

  // first handle 'name' which might also contain arguments
  if (!parseMacro (name, macro_name, args))
    return; // some message would be better here!
  
  if (! body) 
    body = "1";

  if (args.empty ()) {
    _macroManager->addMacro (new PreMacro (macro_name.c_str (), body, flags));
  }
  else {
    PreMacro::MacroArgs *macro_args = new PreMacro::MacroArgs ();
    for (list<string>::iterator i = args.begin (); i != args.end (); ++i)
      macro_args->append (i->c_str ());
    _macroManager->addMacro (new PreMacro (macro_name.c_str (), macro_args,
                                           body, flags));
  }
}


// Undefine a macro.
void PreprocessorParser::undefMacro (const char *name) const { 
  string macro_name;
  list<string> dummy_args;

  // handle 'name' which might also contain arguments
  if (!parseMacro (name, macro_name, dummy_args))
    return; // some message would be better here!

  _macroManager->removeMacro (macro_name.c_str ()); 
}


// Define a new predicate.
void PreprocessorParser::defPredicate (const char *name, const char *answer) const {
  if (! name || ! answer) 
    return;
    
  PrePredicate *predicate;
    
  if ((predicate = _predicateManager->getPredicate (name)))
    predicate->addAnswer (answer);
  else {
    predicate = new PrePredicate (name, answer);
    _predicateManager->addPredicate (predicate);
  }
}


// Undefine a predicate.
void PreprocessorParser::undefPredicate (const char *name) const { 
  _predicateManager->removePredicate (name); 
}


// Return the preprocessor syntax tree.
PreTree *PreprocessorParser::syntaxTree () const { 
  return _parserState->syntaxTree; 
}


// Add a new include path.
void PreprocessorParser::addInclPath (const char *path) const { 
  _fileIncluder->addIncludePath (path); 
}


void PreprocessorParser::supportGNU (bool v) { 
  _support_gnu = v; 
  if (_support_gnu) {
    if (! _macroManager->getMacro ("__GNUC__"))
      defMacro ("__GNUC__", "2");
    if (! _macroManager->getMacro ("__GNUC_MINOR__"))
      defMacro ("__GNUC_MINOR__", "8");
    if (! _macroManager->getMacro ("_GLIBCPP_EXTERN_TEMPLATE"))
      defMacro ("_GLIBCPP_EXTERN_TEMPLATE", "0");
    if (! _macroManager->getMacro ("__null"))
      defMacro ("__null", "0");
    if (! _macroManager->getMacro ("__extension__"))
      defMacro ("__extension__", " ");
    if (! _macroManager->getMacro ("_WCHAR_T_"))
      defMacro ("_WCHAR_T_", " ");
    if (! _macroManager->getMacro ("__attribute__")) {
      PreMacro::MacroArgs *args = new PreMacro::MacroArgs ();
      args->append ("par1");
      _macroManager->addMacro (new PreMacro ("__attribute__", args, " "));
    }
//     if (! _macroManager->getMacro ("__asm__")) {
//       PreMacro::MacroArgs *args = new PreMacro::MacroArgs ();
//       args->append ("par1");
//       _macroManager->addMacro (new PreMacro ("__asm__", args, " ",
//                                              PreMacro::VAR_ARG_MACRO));
//     }
    if (! _macroManager->getMacro ("__builtin_va_arg")) {
      PreMacro::MacroArgs *args = new PreMacro::MacroArgs ();
      args->append ("par1");
      args->append ("par2");
      _macroManager->addMacro (new PreMacro ("__builtin_va_arg", args,
        "((par2)__builtin_va_arg_helper(par1))"));
    }
      
  }
}


} // namespace Puma
