// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __CClassDatabase_h__
#define __CClassDatabase_h__

#include "Puma/CScopeInfo.h"
#include "Puma/CProject.h"
#include <iostream>
#include <map>
using namespace std;

namespace Puma {


class CObjectInfo;
class CClassInfo;
class CUnionInfo;
class CEnumInfo;
class CTypedefInfo;
class CTemplateParamInfo;
class CTypeInfo;
class CFileInfo;
class Token;

/** \file 
 *  Database of all semantic objects (infos) of a project. 
 *  \see Puma::CClassDatabase */

/** \class CClassDatabase CClassDatabase.h Puma/infos/CClassDatabase.h
 *  A CClassDatabase object is a database of all semantic objects
 *  (infos) of a project. */
class CClassDatabase {
  
  Array<CObjectInfo*> _Classes;    // index of classes
  Array<CObjectInfo*> _Unions;     // index of unions
  Array<CObjectInfo*> _Enums;      // index of enums
  Array<CObjectInfo*> _Typedefs;   // index of typedefs
  Array<CObjectInfo*> _Functions;  // index of functions
  Array<CObjectInfo*> _Files;      // index of files

  CProject *_Project; // associated project
  
  multimap<int,CFunctionInfo*> _builtin_ops; // map for built-in operator lookup

public:
  /** Construct a class database object.
   *  \param prj The corresponding project.
   *  \param size Not used. */
  CClassDatabase (CProject &prj, int size = 997);
  /** Destruct the class database. Also destroys all infos
   *  in the database. */
  virtual ~CClassDatabase ();

  // access to the database entries /////////////////////////////
  
  /** Get the number of semantic object infos. */
  unsigned ObjectInfos () const;
  /** Get the number of class infos. */
  unsigned ClassInfos () const;
  /** Get the number of union infos. */
  unsigned UnionInfos () const;
  /** Get the number of enumeration infos. */
  unsigned EnumInfos () const;
  /** Get the number of typedef infos. */
  unsigned TypedefInfos () const;
  /** Get the number of function infos. */
  unsigned FunctionInfos () const;
  /** Get the number of on-demand generated built-in operator functions. */
  unsigned FileInfos () const;
  /** Get the n-th semantic object info.
   *  \param n The number of the info.
   *  \return The info or \c NULL if \e n is invalid. */
  CObjectInfo *ObjectInfo (unsigned n) const;
  /** Get the n-th class info.
   *  \param n The number of the info.
   *  \return The info or \c NULL if \e n is invalid. */
  CClassInfo *ClassInfo (unsigned n) const;
  /** Get the n-th union info.
   *  \param n The number of the info.
   *  \return The info or \c NULL if \e n is invalid. */
  CUnionInfo *UnionInfo (unsigned n) const;
  /** Get the n-th enumeration info.
   *  \param n The number of the info.
   *  \return The info or \c NULL if \e n is invalid. */
  CEnumInfo *EnumInfo (unsigned n) const;
  /** Get the n-th typedef info.
   *  \param n The number of the info.
   *  \return The info or \c NULL if \e n is invalid. */
  CTypedefInfo *TypedefInfo (unsigned n) const;
  /** Get the n-th function info.
   *  \param n The number of the info.
   *  \return The info or \c NULL if \e n is invalid. */
  CFunctionInfo *FunctionInfo (unsigned n) const;
  /** Get the n-th on-demand generated built-in operator info.
   *  \param n The number of the info.
   *  \return The info or \c NULL if \e n is invalid. */
  CFileInfo *FileInfo (unsigned n) const;

  /** Get the semantic object info identified by the given
   *  source code position. 
   *  \param pos The position in the source code. 
   *  \return The info or \c NULL if the info could not be identified. */
  CObjectInfo *ObjectInfo (Token *pos) const; 
  /** Get the semantic object info identified by the given
   *  source code position.
   *  \param pos The position in the source code. 
   *  \return The info or \c NULL if the info could not be identified. */
  CObjectInfo *ObjectInfo (CT_Token *pos) const; 

public:
  // Administration /////////////////////////////////////////////
  
  /** Insert a new semantic object into the database.
   *  \param info The semantic object info to insert. */
  void Insert (CObjectInfo *info);
  /** Remove the given semantic object from the database.
   *  \param info The semantic object info to remove. */
  void Remove (CObjectInfo *info);
  /** Get a function info object for a built-in operator.
   *  \param name Operator name.
   *  \param tok Operator token type, used as a key for fast lookup.
   *  \param rtype result type of the operator. Not used for lookup,
   *         only for object creation.
   *  \param t0 first argument type.
   *  \param t1 second argument type or \c NULL.
   *  \return on-demand generated operator function object. */
  CFunctionInfo *BuiltinOperator (const char *name, int tok, CTypeInfo *rtype,
    CTypeInfo *t0, CTypeInfo *t1);
  /** Dump the contents of the database. The dump is indented as
   *  tree corresponding to the nesting of the semantic objects.
   *  \param out The output stream to dump to.
   *  \param depth The max. depth in the tree to be dumped. */
  void Dump (ostream &out, int depth = 0) const;

  /** Get the project info delivered to the constructor of the database. */
  CProject *Project () const;

private:
  void CreateParameter (CFunctionInfo *fi, CTypeInfo *type) const;

  void Dump (ostream &, CStructure *, int, int) const;
  void DumpType (ostream &, CObjectInfo *, int) const;
  void DumpUsing (ostream &, CUsingInfo *, int) const;
  void DumpFunction (ostream &, CFunctionInfo *, int, int) const;
  void DumpAttribute (ostream &, CAttributeInfo *, int) const;
  void DumpNamespace (ostream &, CNamespaceInfo *, int) const;
  void DumpQualities (ostream &, CObjectInfo *) const;
  void DumpScopeName (ostream &, CStructure *) const;
  void DumpLocalScope (ostream &, CObjectInfo *, int) const;
  void DumpTemplateParam (ostream &, CTemplateParamInfo *, int) const;
  void indent (ostream &, int) const;

//  unsigned Infos (const char *, unsigned) const;
//  CObjectInfo *Info (const char *, unsigned, unsigned) const;
};

inline CClassDatabase::CClassDatabase (CProject &p, int size) :
  _Project (&p)
 {}

inline unsigned CClassDatabase::ObjectInfos () const
 { return ClassInfos () + UnionInfos () + EnumInfos () + 
          TypedefInfos () + FunctionInfos () + FileInfos (); }
inline unsigned CClassDatabase::ClassInfos () const
 { return _Classes.length (); }
inline unsigned CClassDatabase::UnionInfos () const
 { return _Unions.length (); }
inline unsigned CClassDatabase::EnumInfos () const
 { return _Enums.length (); }
inline unsigned CClassDatabase::TypedefInfos () const
 { return _Typedefs.length (); }
inline unsigned CClassDatabase::FunctionInfos () const
 { return _Functions.length (); }
inline unsigned CClassDatabase::FileInfos () const
 { return _Files.length (); }

inline CClassInfo *CClassDatabase::ClassInfo (unsigned i) const
 { return (CClassInfo*)_Classes.lookup (i); }
inline CUnionInfo *CClassDatabase::UnionInfo (unsigned i) const
 { return (CUnionInfo*)_Unions.lookup (i); }
inline CEnumInfo *CClassDatabase::EnumInfo (unsigned i) const
 { return (CEnumInfo*)_Enums.lookup (i); }
inline CTypedefInfo *CClassDatabase::TypedefInfo (unsigned i) const
 { return (CTypedefInfo*)_Typedefs.lookup (i); }
inline CFunctionInfo *CClassDatabase::FunctionInfo (unsigned i) const
 { return (CFunctionInfo*)_Functions.lookup (i); }
inline CFileInfo *CClassDatabase::FileInfo (unsigned i) const
 { return (CFileInfo*)_Files.lookup (i); }

inline CProject *CClassDatabase::Project () const
 { return _Project; }


} // namespace Puma

#endif /* __CClassDatabase_h__ */
