/** \file src/misc.cc */
/*
 * This file is part of assoGiate,
 * an editor of the file types database for GNOME.
 *
 * Copyright (C) 2007 Kevin Daughtridge <kevin@kdau.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "private.hh"
#include "misc.hh"

#include <gtkmm/box.h>
#include <gtkmm/combobox.h>
#include <gtkmm/entry.h>
#include <gtkmm/label.h>
#include <gtkmm/liststore.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/stock.h>
#include <gtkmm/treestore.h>
#include <libgnomevfsmm/utils.h>

/******************************************************************************/
namespace assoGiate {
/******************************************************************************/

void
die_on_database_load_error(const MimeDatabaseLoadError& e) throw()
{
	Gtk::MessageDialog err(_("Could not load file types database"), false,
		Gtk::MESSAGE_ERROR, Gtk::BUTTONS_NONE, true);
	err.add_button(Gtk::Stock::QUIT, Gtk::RESPONSE_CLOSE);
	err.set_secondary_text(compose::ucompose(_("The file types database could "
		"not be loaded or reloaded. File Types Editor must quit.\n\n(%1)"),
		e.what()));
	err.run();
	std::exit(EXIT_FAILURE);
}

void
show_database_update_warning(const MimeDatabaseUpdateError& e,
	Gtk::Window* parent) throw()
{
	static bool shown = false;
	if (!shown) {
		Gtk::MessageDialog warn(_("Could not update information files"), false,
			Gtk::MESSAGE_WARNING, Gtk::BUTTONS_CLOSE, true);
		if (parent != NULL) warn.set_transient_for(*parent);
		warn.set_secondary_text(compose::ucompose(_("Information files, which "
			"are based on the file types database, could not be updated. Your "
			"modifications might not take effect. File Types Editor might try "
			"to update these files again, but this warning will not be "
			"repeated.\n\n(%1)"), e.what()));
		warn.run();
		shown = true;
	}
}

void
activate_url(Gtk::Window& source, const ustring& url) throw()
{
	try {
		Gnome::Vfs::url_show(url);
	} catch (Glib::Exception& e) {
		Gtk::MessageDialog warn(_("Could not open URL"), false,
			Gtk::MESSAGE_WARNING, Gtk::BUTTONS_CLOSE, true);
		warn.set_transient_for(source);
		warn.set_secondary_text(compose::ucompose(_("The URL <%1> could not be "
			"opened. %2"), url, e.what()));
		warn.run();
	}
}

void
activate_email(Gtk::Window& source, const ustring& address) throw()
{	activate_url(source, "mailto:" + address); }

void
activate_help(Gtk::Window& source, const ustring& link) throw()
{
	try {
		std::vector<std::string> path;
		path.push_back(DATADIR);
		path.push_back("gnome");
		path.push_back("help");
		path.push_back(PACKAGE);

		if (!Glib::file_test(Glib::build_filename(path),
			Glib::FILE_TEST_IS_DIR))
			throw Glib::FileError(Glib::FileError::NO_SUCH_ENTITY,
				_("The manual does not exist."));

		path.push_back(ustring());
		path.push_back(PACKAGE ".xml");

		for (const gchar* const *i = g_get_language_names(); *i; ++i) {//unwrapd
			if (ustring(*i).find('.') != ustring::npos) continue;
			path[4] = *i;
			if (Glib::file_test(Glib::build_filename(path),
				Glib::FILE_TEST_EXISTS))
				break;
			path[4].clear();
		}

		if (path[4].empty())
			throw Glib::FileError(Glib::FileError::NO_SUCH_ENTITY,
				_("The manual is not available in an appropriate language."));

		ustring url = "ghelp://" + Glib::build_filename(path);
		if (!link.empty()) url += "?" + link;
		Gnome::Vfs::url_show(url);

	} catch (Glib::Exception& e) {
		Gtk::MessageDialog warn(_("Could not display help"), false,
			Gtk::MESSAGE_WARNING, Gtk::BUTTONS_CLOSE, true);
		warn.set_transient_for(source);
		warn.set_secondary_text(e.what());
		warn.run();
	}
}

ustring
get_location_name(Location locations) throw()
{
	bool standard = locations & SYSTEM_STANDARD || locations & USER_STANDARD,
		system = locations & SYSTEM_OVERRIDE,
		user = locations & USER_OVERRIDE;
		
	if (standard && system && user)
		return _("Standard, system, and user databases");
	else if (standard && system)
		return _("Standard and system databases");
	else if (standard && user)
		return _("Standard and user databases");
	else if (system && user)
		return _("System and user databases");
	else if (standard)
		return _("Standard database");
	else if (system)
		return _("System database");
	else if (user)
		return _("User database");
	else
		return _("None");
}

/******************************************************************************/
/* class InfoTable                                                            */
/******************************************************************************/

InfoTable::InfoTable()
:	Gtk::Table(1, 2, false), m_row(0)
{
	set_col_spacings(12);
	set_row_spacings(6);
}

void
InfoTable::add_item(const ustring& text, Gtk::Widget& item)
{	add_item(text, item, item); }

void
InfoTable::add_item(const ustring& text, Gtk::Widget& item, Gtk::Widget& target)
{
	Gtk::Label *label =
		new Gtk::Label(text, Gtk::ALIGN_LEFT, Gtk::ALIGN_CENTER, true);
	label->set_mnemonic_widget(target);
	label->show();
	
	add_item(*Gtk::manage(label), item);
}

void
InfoTable::add_item(Gtk::Widget& label, Gtk::Widget& item)
{
	attach(label, 0, 1, m_row, m_row + 1,
		Gtk::SHRINK | Gtk::FILL, Gtk::FILL);
	attach(item, 1, 2, m_row, m_row + 1,
		Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
	++m_row;
}

/******************************************************************************/
/* class AddItemDialog                                                        */
/******************************************************************************/

AddItemDialog::AddItemDialog(Gtk::Window* parent, const ustring& title)
:	Gtk::Dialog(title, false, false), m_table(), m_add(NULL)
{
	if (parent != NULL) set_transient_for(*parent);

	m_table.set_border_width(6);
	get_vbox()->pack_start(m_table, false, false);

	add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	m_add = add_button(Gtk::Stock::ADD, Gtk::RESPONSE_OK);
	set_default_response(Gtk::RESPONSE_OK);

	show_all_children();
}

void
AddItemDialog::add_item(const ustring& text, Gtk::Widget& item, bool required)
{	add_item(text, item, item, required); }

void
AddItemDialog::add_item(const ustring& text, Gtk::Widget& item,
	Gtk::Widget& target, bool required)
{
	m_table.add_item(text, item, target);
	item.show();
	if (required)
		add_required(target);
}

void
AddItemDialog::add_required(Gtk::Widget& required)
{
	if cast(&required, Gtk::Entry, entry)
		entry->signal_changed().connect
			(sigc::mem_fun(*this, &AddItemDialog::update));
	else if cast(&required, Gtk::ComboBox, combo)
		combo->signal_changed().connect
			(sigc::mem_fun(*this, &AddItemDialog::update));
	else return;

	m_required.push_back(&required);
	update();
}

void
AddItemDialog::update()
{
	bool satisfied = true;
	FOREACH(std::list<Gtk::Widget*>, m_required, i)
		if cast(*i, Gtk::Entry, entry)
			satisfied &= !entry->get_text().empty();
		else if cast(*i, Gtk::ComboBox, combo)
			satisfied &= bool(combo->get_active());
		else satisfied = false;

	m_add->set_sensitive(satisfied);
}

/******************************************************************************/
/* class EditableColumnsBase                                                  */
/******************************************************************************/

EditableColumnsBase::EditableColumnsBase()
:	removable()
{	add(removable); }

EditableColumnsBase&
EditableColumnsBase::get()
{
	static EditableColumnsBase the;
	return the;
}

/******************************************************************************/
/* class EditableListView                                                     */
/******************************************************************************/

EditableListView::EditableListView(const RefPtr<Gtk::TreeModel>& store,
	const ustring& title)
:	Gtk::Frame(), m_view(store), m_bbox(Gtk::BUTTONBOX_END, 6),
	m_add(Gtk::Stock::ADD), m_remove(Gtk::Stock::REMOVE), s_add(), s_remove()
{
	Gtk::VBox *vbox = new Gtk::VBox(false, 6);
	add(*Gtk::manage(vbox));

	if (title.empty()) {
		set_shadow_type(Gtk::SHADOW_NONE);
		set_border_width(10);
	} else {
		Gtk::Label *label = new Gtk::Label(title, true);
		label->set_mnemonic_widget(m_view);
		set_label_widget(*Gtk::manage(label));
		
		vbox->set_border_width(6);
	}

	Gtk::ScrolledWindow *sw = new Gtk::ScrolledWindow();
	sw->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	sw->set_shadow_type(Gtk::SHADOW_IN);
	vbox->pack_start(*Gtk::manage(sw), true, true);
	
	m_view.set_size_request(-1, title.empty() ? 200 : 100);
	m_view.set_headers_visible(title.empty());
	m_view.set_headers_clickable(true);
	m_view.get_selection()->signal_changed().connect
		(sigc::mem_fun(*this, &EditableListView::on_selection_changed));
	sw->add(m_view);
	on_selection_changed();

	vbox->pack_start(m_bbox, false, false);

	m_add.signal_clicked().connect
		(sigc::mem_fun(*this, &EditableListView::on_do_add));
	m_bbox.pack_start(m_add, false, false);

	m_remove.signal_clicked().connect
		(sigc::mem_fun(*this, &EditableListView::on_do_remove));
	m_bbox.pack_start(m_remove, false, false);
}

sigc::signal<void>
EditableListView::signal_add()
{	return s_add; }

sigc::signal<void, Gtk::TreeIter>
EditableListView::signal_remove()
{	return s_remove; }

void
EditableListView::on_selection_changed()
{
	Gtk::TreeIter iter = m_view.get_selection()->get_selected();
	if (iter)
		m_remove.set_sensitive(iter->get_value
			(EditableColumnsBase::get().removable));
	else
		m_remove.set_sensitive(false);
}

void
EditableListView::on_do_add()
{	s_add.emit(); }

void
EditableListView::on_do_remove()
{
	Gtk::TreeIter iter = m_view.get_selection()->get_selected();
	if (iter && iter->get_value(EditableColumnsBase::get().removable)) {
		s_remove.emit(iter);
		if refptr_cast(m_view.get_model(), Gtk::ListStore, ls)
			ls->erase(iter);
		else if refptr_cast(m_view.get_model(), Gtk::TreeStore, ts)
			ts->erase(iter);
	}
}

} /* namespace assoGiate */
