/*
 * Chan_Misdn -- Channel Driver for Asterisk
 *
 * Interface to Asterisk
 *
 * Copyright (C) 2004, Christian Richter
 *
 * Christian Richter <crich@beronet.com>
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License
 */


#include <stdio.h>
#include <pthread.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include <asterisk/lock.h>

#include <asterisk/channel.h>
#include <asterisk/channel_pvt.h>
#include <asterisk/config.h>
#include <asterisk/logger.h>
#include <asterisk/module.h>
#include <asterisk/pbx.h>
#include <asterisk/options.h>
#include <asterisk/io.h>
#include <asterisk/frame.h>
#include <asterisk/translate.h>
#include <asterisk/cli.h>
#include <asterisk/musiconhold.h>
#include <asterisk/dsp.h>
#include <asterisk/translate.h>
#include <asterisk/config.h>
#include <asterisk/file.h>

#include "chan_misdn_if.h"
#include "te_lib.h"

//#define MISDN_DEBUG 1

#ifdef ASTERISK_STABLE
#define AST_CID_P(ast) ast->callerid
#define AST_BRIDGED_P(ast) ast->bridge
#else
#define AST_CID_P(ast) ast->cid.cid_num
#define AST_BRIDGED_P(ast) ast_bridged_channel(ast) 
#endif


#define MISDN_CHECK_DTMFS 

static char *desc = "Channel driver for mISDN Support (Bri/Pri)";
static char *type = "mISDN";
static char *tdesc = "This driver enables the asterisk to use hardware which is supported by the new  isdn4linux System mISDN";

static int usecnt =0;
static int writing =0;

static char default_context[255]="misdn";
static char default_msns[1024]="";
static char tracefile[512]="";
static int tracing =0 ;

/* Only alaw and mulaw is allowed for now */
static int prefformat =  AST_FORMAT_ALAW; // AST_FORMAT_SLINEAR ;  //AST_FORMAT_ULAW |


static ast_mutex_t usecnt_lock; 

int misdn_debug=0;

chan_list_t dummy_cl;

chan_list_t *cl_te=NULL;
pthread_mutex_t cl_te_lock;

typedef struct msn_list_s msn_list_t; 
struct msn_list_s {
  char *msn;
  msn_list_t *next;
};

static msn_list_t *msn_list=NULL;


typedef struct config_list_s config_list_t;
pthread_mutex_t config_mutex; 

struct config_list_s {
  int port;
  char *context;
  int immediate;
  int hold_allowed;
  msn_list_t *msn_list;
  config_list_t *next;
};

static config_list_t *config_list=NULL;


manager_te_t *mgr_te;
enum te_event_response_e
cb_te_events(manager_te_t *mgr, enum te_event_e event, bchannel_te_t *bc, void *user_data);


void cl_queue_chan(chan_list_t **list, chan_list_t *chan);
void cl_dequeue_chan(chan_list_t **list, chan_list_t *chan);
chan_list_t *cl_find_chan_by_bc_te(chan_list_t *list, bchannel_te_t *bc);
void * audio_thread( void * data);
void chan_misdn_log(char *tmpl, ...);
int check_ret (manager_te_t *mgr,bchannel_te_t * bc, enum te_event_e event, int ret);
static int start_bc_tones(bchannel_te_t *bc);
static int stop_bc_tones(bchannel_te_t *bc);
static void release_chan(bchannel_te_t *bc);
void misdn_reload_config ();

/*************** Helpers *****************/

static chan_list_t * get_chan_by_ast(struct ast_channel *ast)
{
  chan_list_t *tmp;
  
  for (tmp=cl_te; tmp; tmp = tmp->next) {
    if ( tmp->ast == ast ) return tmp;
  }
  
  return NULL;
}

static chan_list_t * get_chan_by_ast_name(char *name)
{
  chan_list_t *tmp;
  
  for (tmp=cl_te; tmp; tmp = tmp->next) {
    if ( tmp->ast  && strcmp(tmp->ast->name,name) == 0) return tmp;
  }
  
  return NULL;
}


static char *get_context(bchannel_te_t *bc) {
  config_list_t *config=config_list;
  
  for (;config; config=config->next) {
    if(config->port == bc->stack->port) return config->context;
  }
  
  return default_context;
}

static char* tone2str(bchannel_te_t *bc)
{
  static struct {
    char name[16];
    enum tone_e tone;
  } *tone, tone_buf[] = {
    {"NOTONE",TONE_NONE},
    {"DIAL",TONE_DIAL},
    {"BUSY",TONE_BUSY},
    {"ALERT",TONE_ALERTING},
    {"",TONE_NONE}
  };
  
  
  for (tone=&tone_buf[0]; tone->name[0]; tone++) {
    if (tone->tone == bc->tone) return tone->name;
  }
  return NULL;
}


static config_list_t *find_config_by_port(int port)
{
  config_list_t *config=config_list;
  
  for (;config; config=config->next) {
    if (config->port==port)
      return config;
  }
  return NULL;
}

static void print_bearer(bchannel_te_t *bc) 
{
  switch (bc->capability) {
  case INFO_CAPABILITY_SPEECH:
    if (misdn_debug>1) chan_misdn_log(" --> Bearer: Speech\n");
    break;
  case INFO_CAPABILITY_AUDIO:
    if (misdn_debug>1) chan_misdn_log(" --> Bearer: Audio\n");
    break;
  case INFO_CAPABILITY_DIGITAL_UNRESTRICTED:
    if (misdn_debug>1) chan_misdn_log(" --> Bearer: Unres Digital\n");
    break;
  case INFO_CAPABILITY_DIGITAL_RESTRICTED:
    if (misdn_debug>1) chan_misdn_log(" --> Bearer: Res Digital\n");
    break;
  default: 
    if (misdn_debug>1) chan_misdn_log(" --> Unknown Bearer\n");
    break;
  }
}
/*************** Helpers END *************/


/*** CLI HANDLING ***/
static int misdn_set_debug(int fd, int argc, char *argv[])
{
  if (argc != 4 )return RESULT_SHOWUSAGE; 
  
  misdn_debug = atoi(argv[3]);
  ast_cli(fd,"changing debug level to %d\n",misdn_debug);

  return 0;
}


static int misdn_clean_pid (int fd, int argc, char *argv[])
{
  manager_te_t *mgr = mgr_te;
  bchannel_te_t *bc;
  int pid;
  chan_list_t *ch; 
  
  if (argc != 4)
    return RESULT_SHOWUSAGE;
  
  pid = atoi(argv[3]);
  
  bc = manager_find_bc_by_pid(mgr, pid);
  if (bc) ch=cl_find_chan_by_bc_te(cl_te, bc);
  
  if (bc && bc->cr) {
    ast_cli(fd,"BChannel with pid %d found, cleaning it up..\n",pid); 
    manager_clean_bc(mgr, bc);

    
  } else {
    ast_cli(fd,"BChannel with pid %d not found .. restart * may help :(\n",pid); 
  }

  if (ch) {
    ast_queue_hangup(ch->ast);
  }

  return 0;
}


static int misdn_show_config (int fd, int argc, char *argv[])
{
  config_list_t *config=config_list;
  
  ast_cli(fd,"Config: %p\n",config_list); 
  
  for (;config; config=config->next) {
    msn_list_t *msn;
    
    ast_cli(fd,"* Port %d Context %s\n",config->port,config->context);
    for (msn=config->msn_list; msn; msn=msn->next) 
      ast_cli(fd, " --> msn %s\n",msn->msn);
  }

  return 0;
}

static int misdn_reload (int fd, int argc, char *argv[])
{
  ast_cli(fd, "Reloading mISDN Config\n");
  misdn_reload_config();
  
  return 0;
}

static int misdn_show_cl (int fd, int argc, char *argv[])
{
  chan_list_t *help=cl_te;
  ast_cli(fd,"Chan List: %p\n",cl_te); 
  
  for (;help; help=help->next) {
    bchannel_te_t *bc=help->bc_te;   
    struct ast_channel *ast=help->ast;
    if (misdn_debug > 1) ast_cli(fd, "Bc:%p Ast:%p\n", bc, ast);
    
    ast_cli(fd,
	    " --> Pid:%d Ch:%d Mode:%s Org:%s dad:%s oad:%s ctx:%s\n",
	    bc->cr?bc->cr->pid:-1, bc->channel,
	    bc->stack->mode==NT_MODE?"NT":"TE",
	    help->orginator == ORG_AST?"*":"I",
	    ast?ast->exten:NULL,
	    ast?AST_CID_P(ast):NULL,
	    ast?ast->context:NULL
	    );
  }
  
  return 0;
}


static int misdn_show_stacks (int fd, int argc, char *argv[])
{
  manager_te_t *mgr = mgr_te;
  
  stack_te_t *stack;
  ast_cli(fd, "BEGIN STACK_LIST:\n");
  for (stack=mgr->stack_list;
       stack;
       stack=stack->next ) {
    int i;
    ast_cli(fd, "* Stack Addr: Uid %x Port %d Type %s Prot. %s Link %s\n",stack->upper_id, stack->upper_id & IF_CONTRMASK, stack->mode==NT_MODE?"NT":"TE", stack->ptp?"PTP":"PMP", stack->l2link?"UP":"DOWN");
    for (i=0; i< stack->b_num; i++) {
      bchannel_te_t *mybc=&stack->bc_te[i];
      ast_cli(fd," --> bchan: addr %x channel %d pid %d cr %x tone %s\n", mybc->addr,mybc->channel, mybc->cr?mybc->cr->pid:-1,mybc->cr?mybc->cr->l3_id:-1 , tone2str(mybc));
    }
  }

  return 0;
}


static int misdn_send_display (int fd, int argc, char *argv[])
{
  char *channame; 
  char *msg; 
  
  if (argc != 5)
    return RESULT_SHOWUSAGE;
  
  channame = argv[3];
  msg = argv[4];

  ast_cli(fd, "Sending %s to %s\n",msg, channame);
  {
    chan_list_t *tmp;
    tmp=get_chan_by_ast_name(channame);
    
    if (tmp && tmp->bc_te) {
      strcpy(tmp->bc_te->display, msg);
      manager_te_send_event(mgr_te, tmp->bc_te, EVENT_INFORMATION);
    } else {
      ast_cli(fd,"No such channel %s\n",channame);
      return RESULT_FAILURE;
    }
  }

  return RESULT_SUCCESS ;
}

static struct ast_cli_entry cli_send_display =
  { {"misdn","send","display", NULL},
    misdn_send_display,
    "Sends Text to mISDN Channel", // Short description
    "Usage: misdn send display <channel> \"<msg>\" \n"
    "       Send <msg> to <channel> as Display Message\n"
    "       when channel is a mISDN channel\n"
  };



static struct ast_cli_entry cli_clean_pid =
  { {"misdn","clean","pid", NULL},
    misdn_clean_pid,
    "Cleans up a broken channel, should never happen but nobodys perfect", // Short description
    "Usage: misdn clean pid <pid>\n"
  };


static struct ast_cli_entry cli_show_config =
  { {"misdn","show","config", NULL},
    misdn_show_config,
    "Shows internal mISDN config, read from cfg-file", // Short description
    "Usage: misdn show config\n"
  };
 

static struct ast_cli_entry cli_reload =
  { {"misdn","reload", NULL},
    misdn_reload,
    "Reloads internal mISDN config, read from cfg-file", // Short description
    "Usage: misdn reload\n"
  };
 

static struct ast_cli_entry cli_show_cl =
  { {"misdn","show","channels", NULL},
    misdn_show_cl,
    "Shows internal mISDN chan_list", // Short description
    "Usage: misdn show channels\n"
  };

static struct ast_cli_entry cli_show_stacks =
  { {"misdn","show","stacks", NULL},
    misdn_show_stacks,
    "Shows internal mISDN stack_list", // Short description
    "Usage: misdn show stacks\n"
  };


static struct ast_cli_entry cli_set_debug =
  { {"misdn","set","debug", NULL},
    misdn_set_debug,
    "Sets Debuglevel of chan_misdn, at the moment, level={1,2}", // Short description
    "Usage: misdn set debug <level>\n"
  };
/*** CLI END ***/


/*****************************/
/*** AST Indications Start ***/
/*****************************/

static int misdn_call(struct ast_channel *ast, char *dest, int timeout)
{
  int port=1;
  char *prt; 
  char *ext;
  char dest_cp[128];
  char *tokb;
  int r;


  if (!dest) {
    chan_misdn_log( "! IND : CALL dad:%s tech:%s WITHOUT PORT, check extension.conf\n",ext,ast->name, ast->context);
    if (misdn_debug > 0) chan_misdn_log("* SEND: State Down\n");
      ast_setstate(ast, AST_STATE_DOWN);
    return -1;
  }
  
  strcpy(dest_cp, dest);

  prt=strtok_r(dest_cp,"/", &tokb);
  
  if (prt) {
    port=atoi(prt);
    ext=strtok_r(NULL,"/", &tokb);
    if (!ext) {
      chan_misdn_log( "! IND : CALL dad:%s tech:%s WITH WRONG ARGS, check extension.conf\n",ext,ast->name, ast->context);
      if (misdn_debug > 0) chan_misdn_log("* SEND: State Down\n");
      ast_setstate(ast, AST_STATE_DOWN);
      return -1;
    }
  } else {
    chan_misdn_log( "! IND : CALL dad:%s tech:%s WITHOUT PORT, check extension.conf\n",ext,ast->name, ast->context);
    if (misdn_debug > 0) chan_misdn_log("* SEND: State Down\n");
      ast_setstate(ast, AST_STATE_DOWN);
      return -1;
  }
  
  if (misdn_debug>0) chan_misdn_log( "* IND : CALL dad:%s tech:%s ctx:%s\n",ext,ast->name, ast->context);
  
  if ((ast->_state != AST_STATE_DOWN) && (ast->_state != AST_STATE_RESERVED)) {
    ast_log(LOG_WARNING, "misdn_call called on %s, neither down nor reserved\n", ast->name);
    return -1;
  }

  {
    config_list_t *config=find_config_by_port(port);
    
    if(config) {
      strcpy(ast->context, config->context);
      if (misdn_debug>1) chan_misdn_log("* --> Setting Context to %s\n",config->context);
    } 
  }
  
  {
    bchannel_te_t *newbc=NULL;
    
    newbc=manager_te_get_free_bc(mgr_te, port );
    if (!newbc) {
      if (misdn_debug > 0) chan_misdn_log("! No free channel chan %p\n",ast);
      if (misdn_debug > 0) chan_misdn_log("* SEND: State Down\n");
      ast_setstate(ast, AST_STATE_DOWN);
      return -1;
    }

    if (misdn_debug > 2) printf("* --> adding2newbc ext %s\n",ext);
    if (ext) strcpy(newbc->dad,ext);
    if (misdn_debug > 2) printf("* --> adding2newbc callerid %s\n",AST_CID_P(ast));
    if (AST_CID_P(ast)) strcpy(newbc->oad,AST_CID_P(ast));
    
    if (ast_test_flag(ast, AST_FLAG_DIGITAL)) {
      newbc->digital=1;
      if (misdn_debug>1) chan_misdn_log("* --> Call with flag Digital\n");
    }
    {
      chan_list_t *ch=malloc(sizeof(chan_list_t));
      if (!ch) { perror("malloc for chan_list"); exit(0);}
      
      memset(ch,0,sizeof(chan_list_t));
      ch->bc_te = newbc;
      ch->orginator=ORG_AST;
      ch->ast = ast;
      ast->pvt->pvt=ch;
      cl_queue_chan(&cl_te, ch) ;
      ch->state=MISDN_CALLING;
    }

    //newbc->channel=1;
    r=manager_te_send_event(mgr_te, newbc, EVENT_SETUP );
    
    //manager_bchannel_setup(newbc);
    
    if (check_ret(mgr_te, newbc, EVENT_SETUP, r) ) {
      chan_misdn_log("* --> Theres no Channel at the moment .. !\n");
      if (misdn_debug>0) chan_misdn_log("* SEND: State Down pid:%d\n",newbc->cr?newbc->cr->pid:-1);
      ast_setstate(ast, AST_STATE_DOWN);
      return -1;
    }
    
    if (misdn_debug>0) chan_misdn_log("* SEND: State Dialing pid:%d\n",newbc->cr?newbc->cr->pid:1);
    ast_setstate(ast, AST_STATE_DIALING);
    //ast_setstate(ast, AST_STATE_RING);
    return 0; 
  }
}


int misdn_answer(struct ast_channel *ast)
{
  chan_list_t *p;

  if (!ast || ! ast->pvt) return -1;
  
  p = ast->pvt->pvt;
  
  //printf("Got Answer\n");
  
  if (!p) {
    printf("Channel not connected ??\n");
    ast_queue_hangup(ast);
  }

  if (!p->bc_te) {
    printf("Got Answer, but theres no bc obj ??\n");

    ast_queue_hangup(ast);

  }

  if (misdn_debug > 0 ) chan_misdn_log("* IND : Answer pid:%d\n",p->bc_te->cr?p->bc_te->cr->pid:-1);
  p->state = MISDN_CONNECTED;
  
  manager_te_send_event(mgr_te, p->bc_te, EVENT_CONNECT);
  start_bc_tones(p->bc_te);
  
  
  return 0;
}

int misdn_digit(struct ast_channel *ast, char digit )
{
  chan_list_t *p;
  p = ast->pvt->pvt;
  if (misdn_debug>0) chan_misdn_log("* IND : Digit %c pid:%d\n",digit,p->bc_te->cr?p->bc_te->cr->pid:-1);
  
  if (p->bc_te) {
    bchannel_te_t *bc=p->bc_te;
    bc->info_dad[0]=digit;
    bc->info_dad[1]=0;
    manager_te_send_event(mgr_te, bc, EVENT_INFORMATION);
  }
  
  return 0;
}


int misdn_fixup(struct ast_channel *oldast, struct ast_channel *ast)
{
  chan_list_t *p;
  p = ast->pvt->pvt;
  if (misdn_debug>0) chan_misdn_log("* IND : Got Fixup\n");
  return 0;
}

int misdn_soption(struct ast_channel *ast, int opt, void *data, int len )
{
  chan_list_t *p;
  p = ast->pvt->pvt;
  if (misdn_debug>0) chan_misdn_log("* IND : Got option %d\n",opt);
  return 0;
}

int misdn_qoption(struct ast_channel *ast, int opt, void *data, int *len )
{
  chan_list_t *p;
  p = ast->pvt->pvt;
  if (misdn_debug>0) chan_misdn_log("* IND : Got Query option %d\n",opt);
  return 0;
}

int misdn_transfer (struct ast_channel *ast, char *dest)
{
  chan_list_t *p;
  p = ast->pvt->pvt;
  if (misdn_debug>0) chan_misdn_log("* IND : Got Transfer %s\n",dest);
  return 0;
}



int misdn_indication(struct ast_channel *ast, int cond)
{
  chan_list_t *p;
  p = ast->pvt->pvt;
  
  if (misdn_debug>2) chan_misdn_log("* IND : Got Indication from ast %p bc_te %p\n",ast,p->bc_te);
  
  switch (cond) {
  case AST_CONTROL_BUSY:
    if (misdn_debug>0) chan_misdn_log("* IND :\tbusy\n");
    
    //manager_send_tone();
    
    if (p) p->state=MISDN_BUSY;
    
    if (p && p->bc_te) {
      //manager_te_send_event(mgr_te, p->bc_te, EVENT_ALERTING);
      manager_send_tone(mgr_te, p->bc_te, TONE_BUSY);
      if (misdn_debug>0) chan_misdn_log("* SEND: State Busy pid:%d\n",p->bc_te->cr?p->bc_te->cr->pid:-1);
      ast_setstate(ast,AST_STATE_BUSY);
    }
    break;
  case AST_CONTROL_RING:
    if (p && p->bc_te) {
      if (misdn_debug>0)    chan_misdn_log("* IND :\tring pid:%d\n",p->bc_te->cr?p->bc_te->cr->pid:-1);
    }
    break;
  case AST_CONTROL_RINGING:
    if (p) p->state=MISDN_ALERTING;
    
    if (p->bc_te) {
      if (misdn_debug>0) chan_misdn_log("* IND :\tringing pid:%d\n",p->bc_te->cr?p->bc_te->cr->pid:-1);
      
      manager_te_send_event(mgr_te, p->bc_te, EVENT_ALERTING);
      manager_send_tone(mgr_te, p->bc_te, TONE_ALERTING);
      if (misdn_debug > 0 ) chan_misdn_log("* SEND: State Ring pid:%d\n",p->bc_te->cr?p->bc_te->cr->pid:-1);
      ast_setstate(ast,AST_STATE_RINGING);
    }
    //ast_playtones_start(ast, 0, "1111", 0);
    //tone_zone_play_tone(ast, ZT_TONE_RINGTONE);
    break;
  case AST_CONTROL_ANSWER:
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log("* IND :\tanswer pid:%d\n",p->bc_te->cr?p->bc_te->cr->pid:-1);
    }
    break;
  case AST_CONTROL_TAKEOFFHOOK:
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log("*\ttakeoffhook pid:%d\n",p->bc_te->cr?p->bc_te->cr->pid:-1);
    }
    break;
  case AST_CONTROL_OFFHOOK:
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log("*\toffhook pid:%d\n",p->bc_te->cr?p->bc_te->cr->pid:-1);
    }
    break; 
  case AST_CONTROL_FLASH:
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log("*\tflash pid:%d\n",p->bc_te->cr?p->bc_te->cr->pid:-1);
    }
    break;
  case AST_CONTROL_PROGRESS:
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log("* IND :\tprogress pid:%d\n",p->bc_te->cr?p->bc_te->cr->pid:-1);
    }
    if (p->bc_te && p->bc_te->stack->mode == NT_MODE) {
      //manager_te_send_event(mgr_te, p->bc_te, EVENT_PROCEEDING);
      //start_bc_tones(p->bc_te);
      //ast_setstate(ast,AST_STATE_RINGING);
    }
    
    break;
  case AST_CONTROL_CONGESTION:
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log("* IND :\tcongestion pid:%d\n",p->bc_te->cr?p->bc_te->cr->pid:-1);
    }
    break;
  case -1 :
    if (p && p->bc_te) {
      if (misdn_debug>0) chan_misdn_log("* IND :\t-1! pid:%d\n",p->bc_te->cr?p->bc_te->cr->pid:-1);
    } else {
      if (misdn_debug>0) chan_misdn_log("* IND :\t-1!\n");
    }
    //misdn_answer(ast);
    break;
  default:
    if (p && p->bc_te) {
      printf("* Unknown Indication:%d pid:%d\n",cond,p->bc_te->cr?p->bc_te->cr->pid:-1);
    } else {
      printf("* Unknown Indication :%d\n",cond);
    }
  }
  
  return 0;
}

static void misdn_destroy(chan_list_t *ch)
{
}


int misdn_hangup(struct ast_channel *ast)
{
  chan_list_t *p;
  bchannel_te_t *bc_te=NULL;
  
  p = ast->pvt->pvt;
  writing =0;
  ast_log(LOG_DEBUG, "misdn_hangup(%s)\n", ast->name);
  if (!ast->pvt->pvt) {
    // seems that we've cleaned everything up, .. so exiting 
    return 0;
  }
  
  bc_te=p->bc_te;
  ast->pvt->pvt = NULL;
  
  if (bc_te) {
    if (misdn_debug > 0) chan_misdn_log("* IND : HANGUP\tpid:%d ctx:%s dad:%s oad:%s\n",p->bc_te->cr?p->bc_te->cr->pid:-1, ast->context, ast->exten, AST_CID_P(ast));
    //manager_send_tone(mgr_te, bc_te, TONE_BUSY);
    switch (p->state) {
    case MISDN_DIALING:
      manager_send_tone(mgr_te, bc_te, TONE_BUSY);
      if (misdn_debug > 1) chan_misdn_log("* --> State Dialing\n");
      p->state=MISDN_CLEANING;
      bc_te->cause=16;
      //manager_te_send_event(mgr_te, bc_te, EVENT_RELEASE);
      //manager_te_send_event(mgr_te, bc_te, EVENT_RELEASE_COMPLETE);
      
      if (bc_te->stack->mode == NT_MODE)
	manager_te_send_event(mgr_te, bc_te, EVENT_RELEASE_COMPLETE);
      else
	manager_te_send_event(mgr_te, bc_te, EVENT_DISCONNECT);

      
      break;
    case MISDN_CALLING:
      if (misdn_debug > 1) chan_misdn_log("* --> State Calling\n");
      p->state=MISDN_CLEANING;
      if (bc_te->stack->mode == NT_MODE)
	manager_te_send_event(mgr_te, bc_te, EVENT_RELEASE_COMPLETE);
      else
	manager_te_send_event(mgr_te, bc_te, EVENT_DISCONNECT);
      break;
    case MISDN_ALERTING:
      if (misdn_debug > 1) chan_misdn_log("* --> State Alerting\n");
      bc_te->cause=16;
      if (p->orginator == ORG_AST) {
	p->state=MISDN_CLEANING;
	if (bc_te->stack->mode == NT_MODE) {
	  manager_te_send_event(mgr_te, bc_te, EVENT_DISCONNECT);
	  manager_te_send_event(mgr_te, bc_te, EVENT_RELEASE_COMPLETE);
	}
	else
	  manager_te_send_event(mgr_te, bc_te, EVENT_DISCONNECT);
      } else {
	manager_send_tone(mgr_te, bc_te, TONE_BUSY);
	p->state=MISDN_CLEANING;
	manager_te_send_event(mgr_te, bc_te, EVENT_DISCONNECT);
      }
      break;
    case MISDN_CONNECTED:
      // Alerting or Disconect
      if (misdn_debug > 1) chan_misdn_log("* --> State Connected\n");
      
      bc_te->cause=16;

#define MISDN_DISCONNECT_BUG_FIXED
#ifdef MISDN_DISCONNECT_BUG_FIXED
      if (bc_te->stack->mode == NT_MODE) {
	manager_send_tone(mgr_te, bc_te, TONE_BUSY);
      }
      manager_te_send_event(mgr_te, bc_te, EVENT_DISCONNECT);
#else
      
      if (bc_te->stack->mode == NT_MODE) {
	manager_send_tone(mgr_te, bc_te, TONE_BUSY);
	manager_te_send_event(mgr_te, bc_te, EVENT_RELEASE);
      }
      else {
	manager_te_send_event(mgr_te, bc_te, EVENT_DISCONNECT);
      }
#endif
      
      p->state=MISDN_CLEANING; //MISDN_HUNGUP_FROM_AST;
      break;

    case MISDN_CLEANING:
      break;
    default:
      // Alerting or Disconect
      if (misdn_debug > 1) chan_misdn_log("* --> State Default\n");
      
      stop_bc_tones(bc_te);
      bc_te->cause=16;
      if (bc_te->stack->mode == NT_MODE)
	manager_te_send_event(mgr_te, bc_te, EVENT_RELEASE);
      else
	manager_te_send_event(mgr_te, bc_te, EVENT_DISCONNECT);
      p->state=MISDN_CLEANING; //MISDN_HUNGUP_FROM_AST;
    }
    
    p->ast=NULL;
  }
  
  return 0;
}

struct ast_frame  *misdn_read(struct ast_channel *ast)
{
  static int i = 0 ;
  chan_list_t *p;
  p = ast->pvt->pvt;

  
  if (i==0) {
    printf("read caled\n");
    i=1;
  }
  
  return NULL;
}

int misdn_write(struct ast_channel *ast, struct ast_frame *frame)
{
  chan_list_t *p;
  int i  = 0;
  p = ast->pvt->pvt;

  if (p->bc_te && p->bc_te->tone != TONE_NONE)
    manager_send_tone(mgr_te,p->bc_te,TONE_NONE);
  
  //printf("tx2misdn\n");
  //if (ast->_state != AST_STATE_UP) return 0; 
#if MISDN_DEBUG
  {
    int i, max=5>frame->samples?frame->samples:5;
    
    printf("write2mISDN %p %d bytes: ", p, frame->samples);
    
    for (i=0; i<  max ; i++) printf("%2.2x ",((char*) frame->data)[i]);
    printf ("\n");
  }
#endif
  if (p->bc_te) {
    manager_flip_buf_bits(frame->data, frame->samples);
    i= manager_send_frame(p->bc_te, frame->data, frame->samples);
  }

  return 0;
}

int misdn_bridge (struct ast_channel *c0, struct ast_channel *c1, int flags,
		  struct ast_frame **fo, struct ast_channel **rc)
{
  chan_list_t *ch1,*ch2;
  
  return 1;
  
  ch1=get_chan_by_ast(c0);
  ch2=get_chan_by_ast(c1);
  
  if (ch1 && ch2 ) ;
  else
    return -1;
  
  return 1;
}

/** AST INDICATIONS END **/

static int start_bc_tones(bchannel_te_t *bc)
{
  manager_send_tone(mgr_te, bc ,TONE_NONE);
  manager_bchannel_activate(bc);

  return 0;
}

static int stop_bc_tones(bchannel_te_t *bc)
{
  if (bc) {
    //manager_send_tone(mgr_te, bc, TONE_NONE);
    manager_bchannel_deactivate(bc);
  }

  return 0;
}


struct ast_channel *misdn_new(chan_list_t *chlist, int state, char * name, char * context, char *exten, char *callerid)
{
  struct ast_channel *tmp;
  
  tmp = ast_channel_alloc(1);
  if (tmp) {
    //memset(tmp,0,sizeof(struct ast_channel));
    snprintf(tmp->name, sizeof(tmp->name), name);
    tmp->type = type;
    
    tmp->nativeformats = prefformat;
    tmp->pvt->rawreadformat = prefformat;
    tmp->pvt->rawwriteformat = prefformat;
    tmp->writeformat = prefformat;
    tmp->readformat = prefformat;
    tmp->priority=1;
    ast_setstate(tmp, state);
    if (state == AST_STATE_RING)
      tmp->rings = 1;
    else
      tmp->rings = 0;
    
    tmp->pvt->pvt = chlist;
    if (misdn_debug>1)chan_misdn_log("* NEW CHANNEL dad:%s oad:%s ctx:%s\n",exten,callerid, context);
    
    tmp->pvt->call = misdn_call;
    tmp->pvt->hangup = misdn_hangup;
    tmp->pvt->read = misdn_read;
    tmp->pvt->write = misdn_write;
    tmp->pvt->answer = misdn_answer;
    tmp->pvt->indicate = misdn_indication; 

    //tmp->pvt->bridge = misdn_bridge;
    
    tmp->pvt->fixup = misdn_fixup;
    tmp->pvt->setoption = misdn_soption;
    tmp->pvt->queryoption = misdn_qoption;
    tmp->pvt->transfer = misdn_transfer;
    

    tmp->pvt->send_digit = misdn_digit; 
    
    strncpy(tmp->context, context, sizeof(tmp->context)-1);
    strncpy(tmp->exten, exten,  sizeof(tmp->exten) - 1);
    if (callerid) 
      AST_CID_P(tmp)=strdup(callerid);
    strcpy(tmp->language, "");
    //i->owner = tmp;

    /*ast_mutex_lock(&usecnt_lock);
      usecnt++;
      ast_mutex_unlock(&usecnt_lock);
      ast_update_use_count(); */
    
  } else
    ast_log(LOG_WARNING, "Unable to allocate channel structure\n");
  return tmp;
}


static struct ast_channel *misdn_request(char *type, int format, void *data)
{
  chan_list_t *p;
  struct ast_channel *tmp = NULL;
  char buf[128];
  char buf2[128], *oad;
  char *tokb;
  
  sprintf(buf,"%s/%s",type,(char*)data);
  strcpy(buf2,data);
  oad=strtok_r(buf2,"/", &tokb);
  if (oad)
    oad=strtok_r(NULL,"/", &tokb);
  
  tmp = misdn_new( &dummy_cl, AST_STATE_DOWN, buf, "","",oad);
  
  if (!tmp)
    misdn_destroy(p);
  
  return tmp;
}



int chan_misdn_write_frm(chan_list_t * tmp, char * buf,  int len )
{
  struct ast_frame frame, *f=NULL;


  //if (tmp->state == MISDN_WAITING4DIGS ) return 0;
  //if (tmp->bc_te && tmp->bc_te->tone != TONE_NONE) return 0 ;
  
  //return 0;
  //printf("tx2*\n");
  
  
  frame.frametype  = AST_FRAME_VOICE;
  frame.subclass = prefformat;
  frame.datalen = len;
  frame.samples = len ;
  frame.mallocd =0 ;
  frame.offset= 0 ;
  frame.src = NULL;
  frame.data = buf ;


#ifdef MISDN_CHECK_DTMFS
  if (tmp->trans) 
    f=ast_translate(tmp->trans, &frame,0);
  else 
    printf("No T-Path found\n");
  
  if (tmp->dsp && f) {
    char c=ast_dsp_digitdetect(tmp->dsp, f);
    if (c) {
      int digit = 0;
      char buf[8];
      digit=ast_dsp_getdigits(tmp->dsp, buf, 1);
      
      if(digit) {
	// sending INFOS as DTMF-Frames :)
	struct ast_frame fr;
	fr.frametype = AST_FRAME_DTMF;
	fr.subclass = buf[0] ;
	fr.src=NULL;
	fr.data = NULL ;
	fr.datalen = 0;
	fr.samples = 0 ;
	fr.mallocd =0 ;
	fr.offset= 0 ;
	
	if (tmp->ast && tmp->ast->pvt) { 
	  if ( misdn_debug > 3) chan_misdn_log("* SEND: DTMF :%c\n",buf[0]);

	  ast_queue_frame(tmp->ast, &fr);

	}
      }
      //return 0;
    }
    ast_frfree (f);
  } else {
    printf(" Frame Not Transed\n");
  }

#endif

  if (tmp && tmp->ast && tmp->ast->pvt )  {
#if MISDN_DEBUG
    int i, max=5>len?len:5;
    
    printf("write2* %p %d bytes: ",tmp, len);
    
    for (i=0; i<  max ; i++) printf("%2.2x ",((char*) frame.data)[i]);
    printf ("\n");
#endif
    //if (tmp->ast->_state == AST_STATE_UP)

    ast_queue_frame(tmp->ast,&frame);

  } 

  return 0;
}

/** Channel Queue ***/
chan_list_t *cl_find_chan_by_bc_te(chan_list_t *list, bchannel_te_t *bc)
{
  chan_list_t *help=list;
  for (;help; help=help->next)
    if (help->bc_te == bc) return help;

  return NULL;
}

void cl_queue_chan(chan_list_t **list, chan_list_t *chan)
{
  if (misdn_debug>2) chan_misdn_log("* Queuing chan %p\n",chan);

#ifdef MISDN_CHECK_DTMFS
  chan->dsp = ast_dsp_new();
  if (chan->dsp) ast_dsp_set_features(chan->dsp, DSP_FEATURE_DTMF_DETECT);
  chan->trans=ast_translator_build_path(AST_FORMAT_SLINEAR, prefformat);
#endif
  
  pthread_mutex_lock(&cl_te_lock);
  if (!*list) {
    if (misdn_debug>2) chan_misdn_log("* List is empty so new chan is Listroot\n");
    *list = chan;
  } else {
    chan_list_t *help=*list;
    for (;help->next; help=help->next); 
    help->next=chan;
    if (misdn_debug> 2) chan_misdn_log("* just adding Chan to the End of List\n");
  }
  chan->next=NULL;
  pthread_mutex_unlock(&cl_te_lock);
}

void cl_dequeue_chan(chan_list_t **list, chan_list_t *chan) 
{
  
  if (misdn_debug>2)chan_misdn_log("* Dequeuing chan %p from List %p\n",chan, list);
  
#ifdef MISDN_CHECK_DTMFS
  if (chan->dsp) 
    ast_dsp_free(chan->dsp);
  if (chan->trans)
    ast_translator_free_path(chan->trans);
#endif


  pthread_mutex_lock(&cl_te_lock);
  if (!*list) {
    printf("List is empty ?\n");
    pthread_mutex_unlock(&cl_te_lock);
    return;
  }
  
  if (*list == chan) {
    if (misdn_debug>2) chan_misdn_log("* Its the first Chan\n");
    *list=(*list)->next;
    pthread_mutex_unlock(&cl_te_lock);
    return ;
  }
  
  {
    chan_list_t *help=*list;
    for (;help->next; help=help->next) {
      if (help->next == chan) {
	if (misdn_debug>2)chan_misdn_log("* Its not the first Chan\n");
	help->next=help->next->next;
	pthread_mutex_unlock(&cl_te_lock);
	return;
      }
    }
  }
  printf("Nothin dequed from chan list\n");
  pthread_mutex_unlock(&cl_te_lock);
}

/** Channel Queue End **/


/** Isdn asks us to release channel, pendant to misdn_hangup **/
static void release_chan(bchannel_te_t *bc) {
  chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
  
  if (ch) {
    if (ch->ast && ch->ast->pvt) {
      if (misdn_debug>1) chan_misdn_log("* REMOVE CHANNEL pid:%d ctx:%s dad:%s oad:%s\n",bc->cr?bc->cr->pid:-1, ch->ast->context, ch->ast->exten,AST_CID_P(ch->ast));
      //ast_setstate(ch->ast, AST_STATE_BUSY);
      if (misdn_debug>1) chan_misdn_log("* --> State Down\n");
      ast_setstate(ch->ast, AST_STATE_DOWN);
      ch->ast->pvt->pvt=NULL;
      
      switch(ch->state) {
      case MISDN_EXTCANTMATCH:
      case MISDN_WAITING4DIGS:
	{
	  if (misdn_debug > 1 ) chan_misdn_log("* --> State Wait4dig | ExtCantMatch\n");
	  ast_hangup(ch->ast);
	}
	break;
	
      case MISDN_CALLING:
      case MISDN_DIALING:
	if (misdn_debug > 1 ) chan_misdn_log("* --> In State Calling|Dialing\n");
	if (misdn_debug > 0 ) chan_misdn_log("* --> Queue Hangup\n");

	ast_queue_control(ch->ast, AST_CONTROL_HANGUP);

	break;
      case MISDN_CLEANING:
	// this state comes out of ast so we mustnt call a ast function !
	if (misdn_debug > 1 ) chan_misdn_log("* --> In StateCleaning\n");
	break;
	
      default:
	if (misdn_debug > 1 ) chan_misdn_log("* --> In State Default\n");
	if (misdn_debug > 0 ) chan_misdn_log("* --> Queue Hangup\n");
	
	
	if (ch->ast && ch->ast->pvt) {
	  

	  ast_queue_hangup(ch->ast);

	}

      }
    }
    cl_dequeue_chan(&cl_te, ch);
    free(ch);
  } else {
    // chan is already cleaned, so exiting 
  }
  
}
/*** release end **/

int is_msn_valid(bchannel_te_t *bc)
{
  msn_list_t *msn=msn_list;
  
  config_list_t *config; 

  if (misdn_debug > 2) chan_misdn_log("Locking Config Mutex\n");
  
  pthread_mutex_lock(&config_mutex);
  config = find_config_by_port(bc->stack->port);
  if (misdn_debug > 2) chan_misdn_log("UnLocking Config Mutex\n");
  pthread_mutex_unlock(&config_mutex);
  
  if (!config) return 0;
  
  if (misdn_debug > 2) chan_misdn_log("Locking Config Mutex\n");
  pthread_mutex_lock(&config_mutex);
  for(msn=config->msn_list;
      msn;
      msn=msn->next) {
    
    if (!bc->dad || ! msn->msn) continue;
    
    if (!strcmp(bc->dad,msn->msn)) {
      if (misdn_debug > 2) chan_misdn_log("UnLocking Config Mutex\n");
      pthread_mutex_unlock(&config_mutex);
      return 1;
    }
    if (msn->msn[0]=='*') {
      if (misdn_debug > 2) chan_misdn_log("UnLocking Config Mutex\n");
      pthread_mutex_unlock(&config_mutex);
      return 1;
    }
  }

  if (misdn_debug > 2) chan_misdn_log("UnLocking Config Mutex\n");
  pthread_mutex_unlock(&config_mutex);
  
  return 0;
}

/** State Machine for sending ISDN Messages to mISDN **/

int check_ret_te (manager_te_t *mgr,bchannel_te_t * bc, enum te_event_e event, int ret)
{
  switch (event) {
  case EVENT_SETUP:
    switch(ret) {
    case -ENOCHAN:
      return 1;
    default:
      break;
    }
    break;
  default:
    break;
  }
  return 0;
}

int check_ret (manager_te_t *mgr,bchannel_te_t * bc, enum te_event_e event, int ret)
{
if (bc->stack->mode == TE_MODE) return check_ret_te(mgr, bc, event, ret);
  
 switch (event) {
 case EVENT_SETUP_ACKNOWLEDGE:
   
   switch(ret) {
   case -ENOCHAN:
     manager_te_send_event(mgr, bc, EVENT_RELEASE_COMPLETE);
     return 1;
     break;
   }
   break;
 case EVENT_SETUP:
   switch(ret) {
   case -ENOCHAN:
     return 1;
     break;
   }
   break;
 default:
   break;
 }
 
 return 0;
}

/** Send mISDN State Machine End **/

void do_immediate_setup(bchannel_te_t *bc,chan_list_t *ch , struct ast_channel *ast)
{
  char predial[256]="";
  char *p = predial;
  
  struct ast_frame fr;
  
  strncpy(predial, ast->exten, sizeof(predial) -1 );
  
  ch->state=MISDN_DIALING;

  //manager_bchannel_setup(bc);
  
  if (bc->stack->mode == NT_MODE) {
    int ret; 
    ret = manager_te_send_event(mgr_te, bc, EVENT_SETUP_ACKNOWLEDGE );
  } else {
    int ret;
    ret = manager_te_send_event(mgr_te, bc, EVENT_SETUP_ACKNOWLEDGE );
  }

  
  stop_bc_tones(bc);
  
  if (misdn_debug >0)
    chan_misdn_log("* Starting Ast ctx:%s dad:%s oad:%s with 's' extension\n", ast->context, ast->exten, AST_CID_P(ast));
  
  strcpy(ast->exten,"s");
  
  if (ast_pbx_start(ast)<0) {
    ast=NULL;
    manager_send_tone(mgr_te,bc,TONE_BUSY);
    if (bc->stack->mode == NT_MODE)
      manager_te_send_event(mgr_te, bc, EVENT_RELEASE_COMPLETE );
    else
      manager_te_send_event(mgr_te, bc, EVENT_DISCONNECT );
  }
  
  
  while (strlen(p) ) {
    fr.frametype = AST_FRAME_DTMF;
    fr.subclass = *p ;
    fr.src=NULL;
    fr.data = NULL ;
    fr.datalen = 0;
    fr.samples = 0 ;
    fr.mallocd =0 ;
    fr.offset= 0 ;
    
    if (ch->ast && ch->ast->pvt) {

      ast_queue_frame(ch->ast, &fr);

      
    }
    p++;
  }
}

enum te_event_response_e
cb_te_events(manager_te_t *mgr, enum te_event_e event, bchannel_te_t *bc, void *user_data)
{
  
  if (event != EVENT_BCHAN_DATA) { // Debug Only Non-Bchan
    if (misdn_debug >0) chan_misdn_log("I IND :%s\tpid:%d\tmode:%s\taddr:%x\n",manager_isdn_get_info( event),bc->cr?bc->cr->pid:-1, bc->stack->mode==NT_MODE?"NT":"TE", bc->addr);
    
    if (misdn_debug >1) chan_misdn_log(" --> dad: %s oad %s channel %d port %d\n",bc->dad,bc->oad,bc->channel, bc->stack->port);
  }
  
  switch (event) {
  case EVENT_INFORMATION:
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      if (ch->state == MISDN_WAITING4DIGS) {
	// Ok, incomplete Setup, waiting till extension exists
	strcat(bc->dad,bc->info_dad);
	if (!ch->ast) { printf("ERROR: Infos without ast-obj??\n");return RESPONSE_ERR;}
	strcpy(ch->ast->exten, bc->dad);
	
	if(!ast_canmatch_extension(ch->ast, get_context(bc), bc->dad, 1, bc->oad)) {
	  chan_misdn_log("Extension can never match, so disconnecting\n");
	  manager_send_tone(mgr,bc,TONE_BUSY);
	  ch->state=MISDN_EXTCANTMATCH;
	  if (bc->stack->mode == NT_MODE)
	    manager_te_send_event(mgr, bc, EVENT_RELEASE_COMPLETE );
	  else
	    manager_te_send_event(mgr, bc, EVENT_DISCONNECT );
	  break;
	}
	
	if (ast_exists_extension(ch->ast, get_context(bc), bc->dad, 1, bc->oad)) {
	  ch->state=MISDN_DIALING;
	  
	  //manager_send_tone(mgr, bc,TONE_NONE);
	  stop_bc_tones(bc);
	  if (misdn_debug >0) chan_misdn_log("* Starting Ast ctx:%s dad:%s oad:%s\n", ch->ast->context, ch->ast->exten, AST_CID_P(ch->ast));
	  if (ast_pbx_start(ch->ast)<0) {
	    manager_send_tone(mgr,bc,TONE_BUSY);
	    if (bc->stack->mode == NT_MODE)
	      manager_te_send_event(mgr, bc, EVENT_RELEASE_COMPLETE );
	    else
	      manager_te_send_event(mgr, bc, EVENT_DISCONNECT );
	  }
	}
	
      } else {
	// sending INFOS as DTMF-Frames :)
	struct ast_frame fr;
	fr.frametype = AST_FRAME_DTMF;
	fr.subclass = bc->info_dad[0] ;
	fr.src=NULL;
	fr.data = NULL ;
	fr.datalen = 0;
	fr.samples = 0 ;
	fr.mallocd =0 ;
	fr.offset= 0 ;
	
	if (ch->ast && ch->ast->pvt) {

	  ast_queue_frame(ch->ast, &fr);

	}
	else
	  printf("Ast Channel doesn't exist anymore @ ast_queue_frame??\n");
      }
    }
    break;
  case EVENT_SETUP:
    
    if (bc->stack->mode == TE_MODE && ! is_msn_valid(bc)) {
      chan_misdn_log("--> Ignoring Call, its not in our MSN List\n");
      return RESPONSE_IGNORE_SETUP; // Ignore MSNs which are not in our List
    }
    
    print_bearer(bc);
    
    {
      chan_list_t *ch=malloc(sizeof(chan_list_t));
      struct ast_channel *chan;
      char name[128];
      if (!ch) { perror("malloc for chan_list"); exit(0);}
      memset(ch,0,sizeof(chan_list_t));
      ch->bc_te = bc;
      ch->orginator = ORG_MISDN;
      
      if (strlen(bc->oad))
	sprintf(name,"mISDN/%d/%s",bc->stack->port,bc->oad);
      else
	sprintf(name,"mISDN/%d",bc->stack->port);
      
      //manager_bchannel_setup(bc);      
      //chan=misdn_new(ch, AST_STATE_DOWN,name ,get_context(bc), bc->dad, bc->oad);
      chan=misdn_new(ch, AST_STATE_RING,name ,get_context(bc), bc->dad, bc->oad);
      
      ch->ast = chan;
      
      if (bc->digital) {
	ast_set_flag(chan,AST_FLAG_DIGITAL);
	chan_misdn_log("--> Setting Digital\n");
      }
      
      /** queue new chan **/
      cl_queue_chan(&cl_te, ch) ;
      //printf("chan queued ind list %p\n",cl_te);
      

      /*
	added support for s extension hope it will help those poor cretains
	which haven't overlap dial.
      */
      {
	config_list_t *config = find_config_by_port(bc->stack->port);
	if ( config &&  config->immediate) {
	  do_immediate_setup(bc, ch , chan);
	  break;
	}
	
      }
      
      if(!ast_canmatch_extension(ch->ast, get_context(bc), bc->dad, 1, bc->oad)) {
	chan_misdn_log("Extension can never match, so disconnecting\n");
	manager_send_tone(mgr,bc,TONE_BUSY);
	ch->state=MISDN_EXTCANTMATCH;
	
	if (bc->stack->mode == NT_MODE)
	  manager_te_send_event(mgr, bc, EVENT_RELEASE_COMPLETE );
	else
	  manager_te_send_event(mgr, bc, EVENT_DISCONNECT );
	break;
      }
      
      if (ast_exists_extension(ch->ast, get_context(bc), bc->dad, 1, bc->oad)) {
	ch->state=MISDN_DIALING;

	if (bc->stack->mode == NT_MODE) {
	  int ret; 
	  ret = manager_te_send_event(mgr, bc, EVENT_SETUP_ACKNOWLEDGE );
	  //ret = manager_te_send_event(mgr, bc, EVENT_PROCEEDING );
	  //if ( check_ret (mgr, bc, EVENT_PROCEEDING, ret)) return RESPONSE_OK;
	} else {
	  int ret;
	  ret = manager_te_send_event(mgr, bc, EVENT_SETUP_ACKNOWLEDGE );
	  //ret= manager_te_send_event(mgr, bc, EVENT_PROCEEDING );
	  //if ( check_ret (mgr, bc, EVENT_PROCEEDING, ret)) return RESPONSE_OK;
	}
	
	//manager_bchannel_setup(bc);
	
	//stop_bc_tones(bc);
	///manager_send_tone(mgr, bc,TONE_NONE);
	if (misdn_debug >0) chan_misdn_log("* Starting Ast ctx:%s dad:%s oad:%s\n", chan->context, chan->exten, AST_CID_P(chan));
	if (ast_pbx_start(chan)<0) {
	  chan=NULL;
	  manager_send_tone(mgr,bc,TONE_BUSY);
	  if (bc->stack->mode == NT_MODE)
	    manager_te_send_event(mgr, bc, EVENT_RELEASE_COMPLETE );
	  else
	    manager_te_send_event(mgr, bc, EVENT_DISCONNECT );
	}
      } else {
	int ret= manager_te_send_event(mgr, bc, EVENT_SETUP_ACKNOWLEDGE );
	if ( check_ret (mgr, bc, EVENT_SETUP_ACKNOWLEDGE, ret)) {
	  manager_te_send_event(mgr,bc,EVENT_RELEASE_COMPLETE);
	}
	// send tone to phone :)
	//manager_bchannel_setup(bc);
	
	manager_send_tone(mgr, bc,TONE_DIAL);
	
	ch->state=MISDN_WAITING4DIGS;
      }
      
      //manager_te_send_event(mgr,bc,EVENT_PROCEEDING);
    }
    break;
  case EVENT_SETUP_ACKNOWLEDGE:
    if (bc->stack->mode == TE_MODE ) {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      if (ch) {
	//should i do something here?
      }
    }
    //manager_bchannel_setup(bc);
    
    break;
  case EVENT_PROCEEDING:
    break;
  case EVENT_PROGRESS:
    if (bc->stack->mode == TE_MODE ) {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      if (ch && ch->ast) {
	start_bc_tones(bc);  // Only
	if (misdn_debug >0) chan_misdn_log("* SEND: Queue Progress pid:%d\n", bc->cr?bc->cr->pid:-1);

	ast_queue_control(ch->ast, AST_CONTROL_PROGRESS);

	
      }
    }
    break;
    
  case EVENT_RELEASE:
    if (misdn_debug > 1) chan_misdn_log("--> cause %d\n",bc->cause);
    
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      
      if (ch ){
	
	if (ch && ch->ast) {
	  switch ( bc->cause) {
	  case 17: //user busy

	    if (misdn_debug >0) chan_misdn_log("* SEND: Queue Busy pid:%d\n", bc->cr?bc->cr->pid:-1);
	    break;
	  case -1:
	    /*
	      OK, it really sucks, this is a RELEASE from NT-Stack So we take
	      it and return easylie, It seems that we've send a DISCONNECT
	      before, so we should RELEASE_COMPLETE after that Disconnect
	      (looks like ALERTING State at misdn_hangup !!
	    */
	    return RESPONSE_OK;
	    break;
	  }
	}
	
	bc->cause=16;
	manager_te_send_event(mgr, bc, EVENT_RELEASE_COMPLETE );
	//manager_bchannel_cleanup(bc);
      }
    }
    break;
  case EVENT_RELEASE_COMPLETE:
    {
      if (misdn_debug > 1) chan_misdn_log("--> cause %d\n",bc->cause);
      
      stop_bc_tones(bc);
      release_chan(bc);
      //manager_bchannel_cleanup(bc);
    }
    break;
  case EVENT_ALERTING:
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      
      if (ch) ch->state = MISDN_ALERTING;
      
      if (ch && ch->ast) {
	if (misdn_debug>0) chan_misdn_log("* SEND: Queue Frame Ringing pid:%d\n", bc->cr?bc->cr->pid:-1);

	ast_queue_control(ch->ast, AST_CONTROL_RINGING);

	
      }
    }
    break;
  case EVENT_CONNECT:
    manager_te_send_event(mgr,bc,EVENT_CONNECT_ACKNOWLEDGE);
  case EVENT_CONNECT_ACKNOWLEDGE:
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      bc->state=STATE_CONNECTED;
      start_bc_tones(bc);
      if (ch && ch->ast) {
	ch->state = MISDN_CONNECTED;
	if (misdn_debug > 0) chan_misdn_log("* SEND: Queue Answer pid:%d\n", bc->cr?bc->cr->pid:-1);

	ast_queue_control(ch->ast, AST_CONTROL_ANSWER);

      }
    }
    break;
  case EVENT_DISCONNECT:
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      
      stop_bc_tones(bc);
      
      if (misdn_debug > 1) chan_misdn_log("--> cause %d\n",bc->cause);
      
      if (ch && ch->ast && ch->ast->pvt) {
	switch ( bc->cause) {
	case 17: //user busy
	  if (misdn_debug > 0) chan_misdn_log("* SEND: Queue Busy pid:%d\n", bc->cr?bc->cr->pid:-1);

	  ast_queue_control(ch->ast, AST_CONTROL_BUSY);

	  break;
	}
      }
      
      bc->cause=16;
      manager_te_send_event(mgr,bc,EVENT_RELEASE);
      //manager_te_send_event(mgr,bc,EVENT_RELEASE_COMPLETE);
      
    }
    break;
  case EVENT_BCHAN_DATA:
    {
      chan_list_t *tmp=cl_find_chan_by_bc_te(cl_te, bc);
#if 1
      
      //if (bc->stack->pri) printf("Got Bchan Data %x!\n",bc->addr);
      
      if (tmp) {
	manager_flip_buf_bits(bc->bframe, bc->bframe_len);
	chan_misdn_write_frm(tmp, bc->bframe, bc->bframe_len );
      }
#else
      len=bc->bframe_len;
      
      if (bc->bframe_len > ibuf_freecount(bc->astbuf)) {
	printf("sbuf overflow!\n");
	len=ibuf_freecount(bc->astbuf);
      }
      
      ibuf_memcpy_w(bc->astbuf, bc->bframe, len);
      if (bc->astbuf->rsem) sem_post(bc->astbuf->rsem);
#endif
    }
    break;
  case EVENT_TIMEOUT:
    break; //Ignore now ..
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      if (ch)
	switch (ch->state) {
	case MISDN_CALLING:
	  chan_misdn_log("GOT TIMOUT AT CALING pid:%d\n", bc->cr?bc->cr->pid:-1);
	  break;
	case MISDN_DIALING:
	  break;
	default:
	  manager_te_send_event(mgr,bc,EVENT_RELEASE_COMPLETE);
	}
    }
    break;
  case EVENT_CLEANUP:
    {
      stop_bc_tones(bc);
      release_chan(bc);
    }
    break;
    
    /***************************/
    /** Suplementary Services **/
    /***************************/
  case EVENT_HOLD:
    {
      chan_list_t *ch=cl_find_chan_by_bc_te(cl_te, bc);
      if (ch && ch->ast) {
	config_list_t * config = find_config_by_port(bc->stack->port);

	if (!config || !config->hold_allowed) {
	  chan_misdn_log("Hold not allowed on port %d\n", bc->stack->port);
	  manager_te_send_event(mgr, bc, EVENT_HOLD_REJECT);
	  break;
	}

	manager_te_send_event(mgr, bc, EVENT_HOLD_REJECT);
	break;
	
	if (AST_BRIDGED_P(ch->ast)){
	  
	  manager_send_tone(mgr_te, bc, TONE_NONE);
	  //stop_bc_tones(bc);
	  //printf("Mohing !!\n");
	  ast_stopstream(ch->ast);
	  ast_stopstream(AST_BRIDGED_P(ch->ast));
	  
	  //ast_streamfile(ch->ast->bridge, "tt-monkeys", NULL);
	  ast_moh_start(AST_BRIDGED_P(ch->ast), NULL);
	  
	  manager_te_send_event(mgr, bc, EVENT_HOLD_ACKNOWLEDGE);
	}
	else {
	  manager_te_send_event(mgr, bc, EVENT_HOLD_REJECT);
	  printf("We aren't bridged to anybody\n");
	}
      } else {
	manager_te_send_event(mgr, bc, EVENT_HOLD_REJECT);
	printf("No ast !\n");
      }
    } 
    break;
  default:
    break;
  }
  
  return RESPONSE_OK;
}

/** TE STUFF END **/

/******************************************
 *
 *   Asterisk Channel Endpoint END
 *
 *
*******************************************/

static msn_list_t *build_msn_list(char *msns)
{ //create MSN - Matrix
  char *tok, *tokb;
  msn_list_t *msn=NULL, *list;
  char *mymsns=strdup(msns);

  if (msns) {
    for (tok=strtok_r(mymsns,",\n",&tokb);
	 tok;
	 tok=strtok_r(NULL,",\n",&tokb)) {

      if (!msn) {
	msn=malloc(sizeof(msn_list_t));
	list=msn;
	if (!msn) {perror("mallocing msn list"); exit(1); }

	msn->msn=strdup(tok);
	msn->next=NULL;
      } else {
	msn->next=malloc(sizeof(msn_list_t));
	msn->next->msn=strdup(tok);
	msn->next->next=NULL;
	msn=msn->next;
      }
    }
  } else {
    free(mymsns);
    return NULL;
  }
  
  free(mymsns);
  return list;
}

static void my_build_config(char *cat, struct ast_variable *v)
{
  char ports[255]="";
  char context[255]="";
  char msns[255]="";
  char *prt, *tokb;
  int immediate = 0;
  int hold_allowed = 0;
  
  if (!v || !cat) return ;
  
  while(v) {
    if (!strcasecmp(v->name, "ports"))
      strncpy(ports,v->value,sizeof(ports)-1);
    if (!strcasecmp(v->name, "context"))
      strncpy(context,v->value,sizeof(context)-1);
    if (!strcasecmp(v->name, "msns"))
      strncpy(msns,v->value,sizeof(msns)-1);
    if (!strcasecmp(v->name, "immediate") && ast_true(v->value))
      immediate=1;
    if (!strcasecmp(v->name, "hold_allowed") && ast_true(v->value))
      hold_allowed=1;
    v=v->next;
  }
  
  if (!strlen(ports)) {
    printf("MISDN ERROR: you've forgot to set the ports in the misdn configfile @%s\n",cat);
    exit(1);
  }

  
  {
    for (prt=strtok_r(ports,",\n",&tokb);
	 prt;
	 prt=strtok_r(NULL,",\n",&tokb)) {
      config_list_t *element, *help ; 
      
      if (config_list) {
	for (help=config_list;
	     help->next;
	     help=help->next);
	help->next=malloc(sizeof(config_list_t));
	element=help->next;
      } else {
	element=malloc(sizeof(config_list_t));
	config_list=element;
      }
      
      if (!element) { perror("malloc in build_config"); exit(1); }
      
      element->port=atoi(prt);
      element->context=strdup(context);
      element->immediate=immediate;
      element->hold_allowed=hold_allowed;
      
      if (strlen(msns)){
	element->msn_list=build_msn_list(msns);
      } else {
	element->msn_list=NULL;
      }
      element->next=NULL;
    }
  }
  
  
  
}

static void get_ports_from_cfg(char *ports)
{
  config_list_t *help;
  
  strcpy(ports,"");

  if (misdn_debug > 2) chan_misdn_log("Locking Config Mutex\n");
  pthread_mutex_lock(&config_mutex);
  if (config_list) {
    for (help=config_list;
	 help;
	 help=help->next) {
      char prt[16]="";
      sprintf(prt,"%d,",help->port);
      strcat(ports,prt);
    }
  }
  if (misdn_debug > 2) chan_misdn_log("UnLocking Config Mutex\n");
  pthread_mutex_unlock(&config_mutex);
  
  ports[strlen(ports)-1]=0;
}


static void destroy_config()
{
  config_list_t *help, *t1;

  for (help=config_list;
       help; ) {
    t1=help->next;
    
    if (help->msn_list) {
      msn_list_t *msn_help;

      for (msn_help=help->msn_list;
	   msn_help; ) {
	msn_list_t *t=msn_help->next;
	if (msn_help->msn) free(msn_help->msn);
	free (msn_help);
	msn_help=t;
      }
    }

    if (help->context) free(help->context);
    free(help);
    help=t1;
  }
}

void misdn_reload_config () {
  char ports[256];
  struct ast_config *cfg;
  struct ast_variable *v;
  char config[]="misdn.conf";


  if (misdn_debug>2) chan_misdn_log("Locking config_mutex\n");
  pthread_mutex_lock(&config_mutex);
  
  
  if (config_list) destroy_config();
  config_list=NULL;

  if ((cfg = ast_load(config))) {
    v = ast_variable_browse(cfg, "general");
    while(v) {
      if (!strcasecmp(v->name, "ports"))
	strncpy(ports,v->value,sizeof(ports)-1);
      if (!strcasecmp(v->name, "context"))
	strncpy(default_context,v->value,sizeof(default_context)-1);
      if (!strcasecmp(v->name, "msns"))
	strncpy(default_msns,v->value,sizeof(default_msns)-1);
      if (!strcasecmp(v->name, "debug"))
	misdn_debug=atoi(v->value);
      if (!strcasecmp(v->name, "tracefile")) {
	tracing=1;
	strncpy(tracefile,v->value,sizeof(tracefile)-1);
      }
      
      v=v->next;
    }
  } 
  
  {
    char *cat;
    cat = ast_category_browse(cfg, NULL);
    while(cat) {
      if (strcasecmp(cat,"general")) {
	struct ast_variable *v=ast_variable_browse(cfg,cat);
	my_build_config (cat,v );
      }
      cat=ast_category_browse(cfg,cat);
    }
  }
  ast_destroy(cfg);

  if (misdn_debug>2) chan_misdn_log("UnLocking config_mutex\n");
  pthread_mutex_unlock(&config_mutex);
  
}

int load_module()
{
  char ports[256]="";
  int i = mISDN_open();
  if (i<0) {
    perror("Open of mISDN Failed\n");
    exit(1);
  }
  mISDN_close(i);
  
  misdn_reload_config();
  
  if (ast_channel_register(type, tdesc, prefformat, misdn_request)) {
    ast_log(LOG_ERROR, "Unable to register channel class %s\n", type);
    unload_module();
    return -1;
  }

  ast_cli_register(&cli_send_display);
  ast_cli_register(&cli_show_cl);
  ast_cli_register(&cli_show_config);
  ast_cli_register(&cli_show_stacks);
  ast_cli_register(&cli_set_debug);
  ast_cli_register(&cli_clean_pid);
  ast_cli_register(&cli_reload);
  
  pthread_mutex_init(&cl_te_lock, NULL);
  
  msg_init();
  
  get_ports_from_cfg(ports);
  {
    mgr_te=manager_te_init(ports, cb_te_events, NULL);
    if (!mgr_te) printf("No te ports initialized\n");
  }
  
  return 0;
}



int unload_module()
{
  /* First, take us out of the channel loop */
  ast_channel_unregister(type);
  destroy_config();
  return 0;
}

int usecount()
{
  int res;
  ast_mutex_lock(&usecnt_lock);
  res = usecnt;
  ast_mutex_unlock(&usecnt_lock);
  return res;
}

char *description()
{
  return desc;
}

char *key()
{
  return ASTERISK_GPL_KEY;
}

void chan_misdn_log(char *tmpl, ...)
{
  va_list ap;
  char buf[1024]; //maxbuflog
  
  va_start(ap, tmpl);
  
  vsprintf( buf, tmpl, ap );
  
  va_end(ap);

  ast_console_puts(buf);
  
  if (tracing) {
    time_t tm = time(NULL);
    char *tmp=ctime(&tm),*p;
    FILE *fp= fopen(tracefile, "a");

    p=strchr(tmp,'\n');
    if (p) *p=':';
    
    if (!fp) {
      ast_console_puts("Error opening Tracefile");
      ast_console_puts(strerror(errno));
      ast_console_puts("\n");
      return ;
    }
    
    fputs(tmp,fp);
    fputs(" ", fp);
    fputs(buf, fp);

    fclose(fp);
  }
}



int reload(void) {
  chan_misdn_log("Reloading mISDN Config\n");
  misdn_reload_config();
  
  return 0;
}

