# -*- coding: utf-8 -*-

# ==============================================================================
# COPYRIGHT (C) 1991 - 2003  EDF R&D                  WWW.CODE-ASTER.ORG
# THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
# IT UNDER THE TERMS OF THE GNU GENERAL PUBLIC LICENSE AS PUBLISHED BY
# THE FREE SOFTWARE FOUNDATION; EITHER VERSION 2 OF THE LICENSE, OR
# (AT YOUR OPTION) ANY LATER VERSION.
#
# THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
# WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
# MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
# GENERAL PUBLIC LICENSE FOR MORE DETAILS.
#
# YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE
# ALONG WITH THIS PROGRAM; IF NOT, WRITE TO EDF R&D CODE_ASTER,
#    1 AVENUE DU GENERAL DE GAULLE, 92141 CLAMART CEDEX, FRANCE.
# ==============================================================================

"""
Definition of ASTER_CALCUL class and its derivated.
"""

import os
import sys
import re
import getpass
import time

from asrun.installation       import aster_root, confdir
from asrun.i18n               import _
from asrun.mystring           import print3
from asrun.job                import nodepara, Func_actu, Del
from asrun.profil             import ASTER_PROFIL
from asrun.build              import ASTER_BUILD
from asrun.execution          import build_export
from asrun.profile_modifier   import ModifierFactory, SEPAR
from asrun.common_func        import get_tmpname
from asrun.utils              import find_command, YES_VALUES, NO_VALUES


msg_pb_version = u"""######################################################################

WARNING : ASTK Server version %s differs from
          ASTK Client version %s.
          An update from Server or Client is necessary.
          This execution may fail ! 
          If it works results should be correct.

######################################################################
"""

msg_info = u"""###############################################
           Client name : %(mcli)s
              Username : %(ucli)s
%(sep)s
           Server name : %(serv)s
              Username : %(user)s
                  Node : %(node)s
              Platform : %(plt)s
%(sep)s
    Code_Aster version : %(vers)s
%(sep)s
            Time (min) : %(tpsjob)s
           Memory (MB) : %(vmem).1f
  Number of processors : %(nbp)s   (OpenMP)
       Number of nodes : %(mpi_nbnoeud)s   (MPI)
  Number of processors : %(mpi_nbcpu)s   (MPI)
%(sep)s
            Debug mode : %(dbg)s"""

msg_classe  = u" specified batch queue : %s"
msg_depart  = u"            start time : %s"
msg_consbtc = u"            BTC script : %s"
msg_vers  = u"""   Version ASTK Server : %s
   Version ASTK Client : %s
###############################################
"""


class ASTER_CALCUL:
   """
   This class read user's profile and (if needed) call as_run through or not
   a terminal, or just write a btc file...
   """
   _supported_services     = ('study', 'parametric_study', 'meshtool', 'stanley',
                              'convbase', 'distribution', 'exectool')
   _available_services_ref = ('study',                     'meshtool', 'stanley',
                              'convbase',                 'exectool')

   def __init__(self, run, filename=None, prof=None, pid=None):
      """
      Initializations
      """
      assert filename or prof, 'none of (filename, prof) provided!'
      self.run = run
      if pid is None:
         self.pid = self.run['num_job']
      else:
         self.pid = pid
      self.jobid = -1
      self.queue = "unknown"

      if prof is not None:
         self.prof = prof
      else:
         # ----- profile filename
         fprof = get_tmpname(self.run, self.run['tmp_user'], basename='profil_astk')
         self.run.ToDelete(fprof)
         kret = self.run.Copy(fprof, filename, niverr='<F>_PROFILE_COPY')
         self.prof = ASTER_PROFIL(fprof, self.run)
      if self.prof['nomjob'][0] == '':
         self.prof['nomjob'] = 'unnamed'

      # ----- decode service called
      self.decode_special_service()

      # attributes
      self.dict_info   = None
      self.as_exec_ref = os.path.join(aster_root, 'ASTK', 'ASTK_SERV', 'bin', 'as_exec_ref')
      self.nodepara    = nodepara
      self.diag        = '?'

   def error(self, msg):
      """
      Print an error msg and exit.
      """
      self.run.Mess(msg, '<F>_ERROR')

   def decode_special_service(self):
      """
      Return the profile modified for the "special" service.
      """
      self.serv = self.prof['special'][0].split(SEPAR)[0]
      if self.serv == '':
         if self.prof['distrib'][0] != '':
            self.serv = 'distribution'
         elif self.prof['exectool'][0] != '':
            self.serv = 'exectool'
         elif self.prof['parent'][0] == 'parametric':
            self.serv = 'parametric_study'
         else:
            self.serv = 'study'
      self.run.DBG("service name : %s" % self.serv)
      if self.serv not in self._supported_services:
         self.error(_(u'Unknown service : %s') % self.serv)
      elif self.serv in ('study', 'parametric_study'):
         return
      
      modifier = ModifierFactory(self.serv, self.prof, self.run)
      modifier.modify()
      self.prof =  modifier.return_profile()

   def on_mach_ref(self):
      """
      Are we on reference server ?
      """
      return os.path.exists(self.as_exec_ref) and os.access(self.as_exec_ref, os.X_OK)

   def flash(self, typ, num_job=None):
      """
      If typ='o', return something like .../flasheur/nomjob.o1234
      """
      assert self.prof is not None
      if num_job is None:
         num_job = self.jobid
      return os.path.join(self.run['flasheur'],
            '%s.%s%s' % (self.prof['nomjob'][0], typ, num_job))

   def build_dict_info(self, opts):
      """
      Build a dictionnary grouping all parameters.
      """
      sep = "-----------------------------------------------"
      mode = self.prof['mode'][0]
      node = self.prof['noeud'][0] or self.prof['serveur'][0]
      self.prof.update_content()
      self.dict_info = {
         'sep'                   : sep,
         'export'                : self.prof.filename,
         'mcli'                  : self.prof['mclient'][0],
         'ucli'                  : self.prof['uclient'][0],
         'serv'                  : self.prof['serveur'][0],
         'user'                  : self.prof['username'][0],
         'mode'                  : mode,
         'node'                  : node,
         'plt'                   : self.run['plate-forme'],
         'vers'                  : self.prof['version'][0],
         'tpsjob'                : self.prof['tpsjob'][0],
         'vmem'                  : float(self.prof['memjob'][0] or 0.) / 1024.,
         'nbp'                   : self.prof['ncpus'][0],
         'mpi_nbnoeud'           : self.prof['mpi_nbnoeud'][0],
         'mpi_nbcpu'             : self.prof['mpi_nbcpu'][0],
         'dbg'                   : self.prof['debug'][0],
         'prof_content'          : self.prof.content,
         'nomjob'                : self.prof['nomjob'][0],
         'nomjob_'               : self.flash('', ''),
         'nomjob_p'              : self.flash('p', '$num_job'),
         'sbin'                  : os.path.join(aster_root, 'bin'),
         'who'                   : getpass.getuser(),
         'opts'                  : opts,
         'remote_shell_protocol' : self.run['remote_shell_protocol'],
         'remote_copy_protocol'  : self.run['remote_copy_protocol'],
      }
      
      if self.prof['srv_dbg'][0] in YES_VALUES:
         self.dict_info['opts'] += ' --debug'
      if self.prof['srv_verb'][0] in YES_VALUES:
         self.dict_info['opts'] += ' --verbose'
      
      # rep_trav from profile or config(_nodename)
      rep_trav = self.prof['rep_trav'][0]
      if rep_trav == '':
         rep_trav = self.nodepara(self.run, node, 'rep_trav')
      self.dict_info['rep_trav'] = rep_trav
      # set message using previous content
      self.dict_info['message'] = self.message()

   def message(self):
      """
      Format information message.
      """
      # No "' in ASTK_MESSAGE !
      ASTK_MESSAGE = []

      # check client and server versions
      serv_vers = self.run.__version__
      try:
         client_vers = self.prof['origine'][0].split()[1]
      except Exception, msg:
         self.run.DBG('Error : unexpected "origine" value :', self.prof['origine'][0])
         client_vers = ''
      if serv_vers != client_vers:
         ASTK_MESSAGE.append(msg_pb_version % (serv_vers, client_vers))

      ASTK_MESSAGE.append(msg_info % self.dict_info)

      if self.prof['classe'][0]:
         ASTK_MESSAGE.append(msg_classe % self.prof['classe'][0])
      if self.prof['depart'][0]:
         ASTK_MESSAGE.append(msg_depart % self.prof['depart'][0])
      ASTK_MESSAGE.append(self.dict_info['sep'])

      if self.prof['consbtc'][0] not in NO_VALUES:
         msg = "generated"
      else:
         msg = "provided by user"
      ASTK_MESSAGE.append(msg_consbtc % msg)
      ASTK_MESSAGE.append(self.dict_info['sep'])

      ASTK_MESSAGE.append(msg_vers % (serv_vers, client_vers))

      return os.linesep.join(ASTK_MESSAGE)

   def get_jobid(self, txt):
      """
      Decode a such string 'JOBID=  232564  QUEUE=  q4G_1h'.
      """
      mat = re.search('JOBID[ ]*=[ ]*([0-9]+) *QUEUE *= +(.*)', txt)
      if mat is not None:
         jobid, queue = mat.groups()
         queue = queue.split()[0]
      else:
         jobid, queue = '?', 'unkwown'
      return jobid, queue

   def consbtc(self, fbtc):
      """
      Write btc file.
      """
      assert type(self.dict_info) is dict
      
      mode = self.prof['mode'][0]

      if self.prof['display'][0]:
         str_display = 'DISPLAY=%s ; export DISPLAY' % self.prof['display'][0]
      else:
         str_display = ''
      aster_profile = os.path.join(confdir, 'profile.sh')
      if os.path.exists(aster_profile):
         str_aster_profile = '. %s' % aster_profile
      else:
         str_aster_profile = ''
      if mode == 'interactif':
         m = 'i'
         str_pid = 'num_job=%s' % self.pid
      else:
         m = 'b'
         # ne conserver que le numero (par ex: 12345@node12 => 12345)
         str_pid = """num_job=`echo $%s | awk '{inv=$0; sub("^[0-9]+","",inv); sub(inv,"",$0); print $0;}'`""" % self.run['batch_jid']

      btc = r"""#!/bin/sh

%(str_display)s
%(str_aster_profile)s
%(str_pid)s

# copie du .export dans le flasheur et rep_trav
cat << EOFEXPORT > %(nomjob_p)s
%(prof_content)s
EOFEXPORT

# on redéfinit car déjà arrivé en dhcp ou quand on change de noeud...
LOGNAME=%(who)s
export LOGNAME

# message d'info
printf %(message)r

# lance l'exec
%(sbin)s/as_run %(opts)s %(nomjob_p)s \
      --num_job=$num_job \
      --remote_shell_protocol=%(remote_shell_protocol)s \
      --remote_copy_protocol=%(remote_copy_protocol)s \
      | tee %(rep_trav)s/%(nomjob)s.$num_job.fort.6.%(m)s

# diagnostic
egrep -- '--- DIAGNOSTIC JOB :' %(rep_trav)s/%(nomjob)s.$num_job.fort.6.%(m)s | awk '{print $5}' > %(nomjob_)s%(m)s$num_job
\rm -f %(rep_trav)s/%(nomjob)s.$num_job.fort.6.%(m)s

\rm -rf %(fbtc)s

"""
      dict_btc = self.dict_info.copy()
      dict_btc.update({
         'str_display'        : str_display,
         'str_aster_profile'  : str_aster_profile,
         'str_pid'            : str_pid,
         'm'                  : m,
         'fbtc'               : fbtc,
      })

      open(fbtc, 'w').write(btc % dict_btc)
      os.chmod(fbtc, 0755)

   def soumbtc_batch(self, fbtc):
      """
      Run btc script in batch mode.
      """
      from asrun.batch import BatchSystemFactory
      scheduler = BatchSystemFactory(self.run, self.prof, fbtc)
      iret, jobid, queue = scheduler.start()
      if iret != 0:
         print3(_(u"Error during submitting job !"))
         iret = 5
      if jobid == '':
         print3(_(u"Unexpected jobid"))
         iret = 5
      return iret, jobid, queue

   def soumbtc_interactif(self, fbtc):
      """
      Run btc in interactive mode.
      """
      self.jobid = self.pid
      # commandes
      cmd_batch      = '%(fbtc)s 1> %(output)s 2> %(error)s'
      cmd_interactif = '%(fbtc)s 2> %(error)s | tee %(output)s'

      node = self.dict_info['node']
      dico = {
         'fbtc'   : fbtc,
         'output' : self.flash('o'),
         'error'  : self.flash('e'),
      }
      # follow output or not
      xterm = self.prof['xterm'][0]
      if xterm == '' and self.prof['follow_output'][0] in YES_VALUES:
         xterm   = self.run['terminal']
         display = self.prof['display'][0] or ':0.0'
         xterm = xterm.replace('@D', display)
      
      if self.prof['depart'][0] != '' or xterm == '':
         cmd = cmd_batch % dico
      else:
         cmd = cmd_interactif % dico

      # delayed start
      if self.prof['depart'][0] != '':
         cmd = "echo '%s' | at %s" % (cmd, self.prof['depart'][0])
      elif xterm != '':
         if re.search('@E', xterm) == None:
            xterm = xterm + ' -e @E'
         cmd = xterm.replace('@E', '/bin/sh -c "%s"' %  cmd)
      
      # run on another node
      distant = self.run.GetHostName().split('.')[0] != node.split('.')[0]
      
      if not distant:
         # check xterm command
         if xterm != '':
            term = xterm.split()[0]
            if not os.access(term, os.X_OK):
               print3(_(u"Not an executable : %s") % term)
               return 7, '', 'unknown'
      else:
         # check node connection
         iret, output = self.run.Shell('echo hello', mach=node)
         if output.find('hello') < 0:
            print3(output)
            print3(_(u"Connection failure to %s (from %s)") % (node, self.prof['serveur'][0]))
            return 6, '', 'unknown'

      kret, output = self.run.Shell(cmd, mach=node, bg=True)

      return 0, self.jobid, 'interactif'

   def start(self, options=''):
      """
      Go !
      """
      # ----- copy and read .export, build dict for formatting
      self.build_dict_info(options)

      jn = self.pid
      self.mode = self.prof['mode'][0]
      self.name = self.prof['nomjob'][0]
      
      # export file is not necessary in interactive mode
      if self.mode == 'batch':
         self.prof.WriteExportTo(self.prof.filename)
         self.run.DBG('profile written into :', self.prof.filename)

      # ----- batch + machref, call as_exec_ref
      if self.mode == 'batch' and self.serv in self._available_services_ref and self.on_mach_ref():
         fbtc = get_tmpname(self.run, self.run['tmp_user'], basename=self.prof['nomjob'][0] + '.u')
         os.environ['ASTK_MESSAGE'] = repr(self.dict_info['message'])
         
         cmd = self.as_exec_ref + ' ' + self.prof.filename + ' ' + fbtc
         iret, output = self.run.Shell(cmd, follow_output=True)
         
         self.run.DBG('OUTPUT of as_exec_ref :', output, all=True)
         
         self.jobid, self.queue = self.get_jobid(output)
         if os.path.exists(fbtc):
            jret = self.run.Rename(fbtc, self.flash('u'))
         jret = self.run.Copy(self.flash('p'), self.prof.filename)
         return iret, output

      # ----- consbtc ?
      fbtc = os.path.join(self.run['flasheur'], 'btc.%s' % jn)
      if self.prof['consbtc'][0] in NO_VALUES and not 'make_env' in self.prof['actions']:
         fbtc0 = self.prof.Get('D', 'btc')[0]['path']
         iret = self.run.Copy(fbtc, fbtc0)
      else:
         self.consbtc(fbtc)

      # ----- soumbtc ?
      iret = 0
      if self.prof['soumbtc'][0] not in NO_VALUES:
         if self.mode == 'interactif':
            iret, self.jobid, self.queue = self.soumbtc_interactif(fbtc)
         else:
            iret, self.jobid, self.queue = self.soumbtc_batch(fbtc)

         # copy fbtc into flasheur/
         jret = self.run.Copy(self.flash('u'), fbtc)

      # faut-il recopier le btc vers le client
      res_fbtc = self.prof.Get('R', 'btc')
      if len(res_fbtc) > 0:
         res_fbtc = res_fbtc[0]['path']
         self.run.Copy(res_fbtc, fbtc)

      return iret, ''

   def get_state(self):
      """
      Return current state of the job.
      """
      return Func_actu(self.run, self.jobid, self.name, self.mode)

   def wait(self, refresh_delay=1.):
      """
      Wait for job completion.
      """
      state = '_'
      while state != 'ENDED':
         time.sleep(refresh_delay)
         res = self.get_state()
         state, diag = res[0:2]
      self.diag = diag

   def get_diag(self):
      """
      Return diagnostic of the execution based on the output file.
      """
      res = self.get_state()
      state, diag = res[0:2]
      res = [diag, 0., 0., 0., 0.]
      self.diag = diag
      if state=='ENDED' and os.path.exists(self.flash('o')):
         txt = open(self.flash('o'), 'r').read()
         mat = re.search('%s +([0-9\.]+) +([0-9\.]+) +([0-9\.]+) +([0-9\.]+)' % 'Total', txt)
         if mat is not None:
            res = res[:1] + [float(v) for v in mat.groups()]
      return res

   def kill(self):
      """
      Kill the job (if it is running) and delete of its files.
      """
      Del(self.run, self.jobid, self.name, self.mode, signal='KILL')

   def request(self, key):
      """
      Return the value to request.
      """
      if key == 'cpu':
         res = max(int(self.prof['ncpus'][0] or 1), int(self.prof['mpi_nbcpu'][0] or 1))
      elif key == 'mem':
         res = float(self.prof['memjob'][0] or 0.) / 1024.
      else:
         res = 0
      return res

   def on_host(self, serv, host):
      """
      Change submission serv/host.
      """
      if serv is not None:
         self.prof['serveur'] = serv
      if host is not None:
         self.prof['noeud'] = host



class ASTER_TESTCASE(ASTER_CALCUL):
   """
   Derivation for a testcase.
   """
   def __init__(self, run, test, filename=None, prof=None, pid=None, **kwargs):
      """
      Initializations
      """
      ASTER_CALCUL.__init__(self, run, filename, prof, pid)
      self.testcase = test
      self.param = kwargs.copy()
      self.prof = self.change_profile()

   def change_profile(self):
      """
      Prepare profile object.
      """
      # initialize the profile
      ptest = init_profil_from(self.run, self.prof.copy())
      ptest.filename = get_tmpname(self.run, self.run['tmp_user'],
                                   basename=self.testcase + '.export', pid=self.pid)
      self.run.DBG('profile filename set to : ', ptest.filename)
      del ptest['xterm']
      del ptest['follow_output']
      del ptest['rep_trav']
      del ptest['detr_rep_trav']
      
      # update with the export to run the test
      prt = build_export(self.run, self.param['conf'], self.param['REPREF'],
                         self.param['reptest'], self.testcase, self.param['resutest'])
      prt['nomjob'] = self.testcase
      ptest.update(prt)
      
      # apply facmtps
      try:
         ptest['tpsjob'] = int(ptest['tpsjob'][0])  * self.param['facmtps']
      except Exception:
         pass
      try:
         ptest.args['tpmax'] = int(ptest.args['tpmax'])  * self.param['facmtps']
      except Exception:
         pass
      
      ptest.update_content()
      self.run.DBG('Profile of testcase : %s' % self.testcase, ptest.content, all=True)
      return ptest

   def clean_results(self):
      """
      Remove all result files.
      """
      erre = self.prof.Get('R', 'erre')[0]['path']
      resu = self.prof.Get('R', 'resu')[0]['path']
      mess = self.prof.Get('R', 'mess')[0]['path']
      for f in (erre, resu, mess):
         self.run.Delete(f)



class ASTER_PARAMETRIC(ASTER_CALCUL):
   """
   Derivation for a parametric study :
      - change all "comm" files
      - change repe_out into resudir/calc_000i
   """
   def __init__(self, run, label, filename=None, prof=None, pid=None, **kwargs):
      """
      Initializations. Required arguments : resudir, keywords, values + prof['repe']
      """
      if prof is not None:
         prof['parent'] = 'parametric'
      ASTER_CALCUL.__init__(self, run, filename, prof, pid)
      self.label = label
      self.resudir = kwargs['resudir']
      try:
         os.makedirs(os.path.join(self.resudir, label))
      except OSError:
         pass
      self.values   = kwargs['values'].copy()
      self.keywords = kwargs['keywords'].copy()
      self.prof = self.change_profile()

   def change_profile(self):
      """
      Prepare profile object.
      """
      prof = self.prof.copy()
      prof.filename = get_tmpname(self.run, self.run['tmp_user'],
                                  basename=self.label + '.export', pid=self.pid)
      self.run.DBG('profile filename set to : ', prof.filename)
      prof['actions'] = 'make_etude'
      prof['nomjob']  = '%s-%s' % (self.prof['nomjob'][0], self.label)
      # delete unused entries
      assert prof.Get('R', typ='repe')
      del prof['xterm']
      del prof['follow_output']
      del prof['rep_trav']
      del prof['detr_rep_trav']
      prof.Del('D', typ='distr')
      compress = prof.Get('R', typ='repe')[0]['compr']
      prof.Del('R', typ='repe')
      # add repe_out
      prof.Set('R', {
         'type'  : 'repe', 'isrep' : True, 'ul' : 0,
         'path'  : os.path.join(self.resudir, self.label, 'REPE_OUT'),
         'compr' : compress,
      })
      # add base or bhdf as data
      type_base, compress = prof.get_base('D')
      if type_base:
         path = prof.Get('D', typ=type_base)[0]['path']
         prof.Del('D', typ=type_base)
         prof.Set('D', {
         'type'  : type_base, 'isrep' : True, 'ul' : 0,
         'path'  : os.path.join(path, self.label, type_base),
         'compr' : compress,
         })
      # add base or bhdf as result
      type_base, compress = prof.get_base('R')
      if type_base:
         prof.Del('R', typ=type_base)
         prof.Set('R', {
         'type'  : type_base, 'isrep' : True, 'ul' : 0,
         'path'  : os.path.join(self.resudir, self.label, type_base),
         'compr' : compress,
         })
      # change comm files
      lcomm = prof.Get('D', typ='comm')
      prof.Del('D', typ='comm')
      for i, df in enumerate(lcomm):
         fcom = os.path.join(self.resudir, self.label, 'command_%d.comm' % i)
         dest = fcom
         if df['compr']:
            dest = dest + '.gz'
         kret = self.run.Copy(dest, df['path'], niverr='<E>_COPY_ERROR')
         if kret == 0 and df['compr']:
            kret, f = self.run.Gunzip(dest, niverr='<E>_DECOMPRESSION')
         df.update({ 'compr' : False, 'path' : fcom })
         prof.Set('D', df)
         self.change_comm_files(dest)
      return prof
      
   def change_comm_files(self, filename):
      """
      Change parameter definition in comm file.
      """
      assert os.path.exists(filename), filename
      content = open(filename, 'r').read()
      for para, pval in self.values.items():
         exp = re.compile('^( *)(%s *=.*)$' % para, re.MULTILINE)
         content = exp.sub('\g<1>%s = %s' % (para, pval), content)
      # insert commands at the beginning of each calculation
      if self.keywords.get('UNITE_PRE_CALCUL') or self.keywords.get('PRE_CALCUL'):
         ideb, jdeb = find_command(content, 'DEBUT')
         if self.keywords.get('PRE_CALCUL'):
            insert = self.keywords['PRE_CALCUL']
         if self.keywords.get('UNITE_PRE_CALCUL'):
            insert = """INCLUDE(UNITE=%s)""" % self.keywords['UNITE_PRE_CALCUL']
         content = os.linesep.join([content[:jdeb+1], insert, content[jdeb+1:]])
      # insert commands at the end of each calculation
      if self.keywords.get('UNITE_POST_CALCUL') or self.keywords.get('POST_CALCUL'):
         ifin, jfin = find_command(content, 'FIN')
         if self.keywords.get('POST_CALCUL'):
            insert = self.keywords['POST_CALCUL']
         if self.keywords.get('UNITE_POST_CALCUL'):
            insert = """INCLUDE(UNITE=%s)""" % self.keywords['UNITE_POST_CALCUL']
         content = os.linesep.join([content[:ifin], insert, content[ifin:]])

      open(filename, 'w').write(content)


def init_profil_from(run, prof, keep_surch=True):
   """
   Initialize an empty profile from another one.
   """
   # prepare the profile
   ptest = prof.copy()
   ptest.data = []
   if keep_surch:
      for data in prof.data:
         if data['type'] in ('exec', 'ele', 'cmde', 'conf', 'py'):
            ptest.data.append(data.copy())
   ptest.resu = []
   ptest['actions'] = 'make_etude'
   # machine
   if ptest['origine'][0] == '':
      ptest['origine'] = 'ASTK %s' % run.__version__
   ptest['uclient'], ptest['mclient'] = run.system.getuser_host()
   if ptest['serveur'][0] == '':
      ptest['serveur'] = ptest['mclient']
   if ptest['node'][0] == '':
      ptest['node'] = ptest['mclient']
   if ptest['username'][0] == '':
      ptest['username'] = ptest['uclient']
   for k in ('ncpus', 'mpi_nbnoeud', 'mpi_nbcpu'):
      if ptest[k][0] == '':
         ptest[k] = 1
   return ptest


