# -*- coding: utf-8 -*-

# ==============================================================================
# COPYRIGHT (C) 1991 - 2003  EDF R&D                  WWW.CODE-ASTER.ORG
# THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
# IT UNDER THE TERMS OF THE GNU GENERAL PUBLIC LICENSE AS PUBLISHED BY
# THE FREE SOFTWARE FOUNDATION; EITHER VERSION 2 OF THE LICENSE, OR
# (AT YOUR OPTION) ANY LATER VERSION.
#
# THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
# WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
# MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
# GENERAL PUBLIC LICENSE FOR MORE DETAILS.
#
# YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE
# ALONG WITH THIS PROGRAM; IF NOT, WRITE TO EDF R&D CODE_ASTER,
#    1 AVENUE DU GENERAL DE GAULLE, 92141 CLAMART CEDEX, FRANCE.
# ==============================================================================

"""
Definition of ASTER_PROFIL class.
"""

import os
import os.path as osp
import re

from asrun.common.i18n import _
from asrun.common.utils import Enum
from asrun.common.sysutils import FileName, local_user, local_full_host
from asrun.mystring import print3, split_endlines, ufmt

MODES = Enum("INTERACTIF", "BATCH")


class ASTER_PROFIL:
    """Class to read and parse an exported ASTK profile.
    Attributes :
        _filename : absolute filename of the '.export' file
        _content  : content of the corresponding '.export' file
        param     : value of each parameters (type dict) (each param is a list)
        args      : arguments (type dict)
        data      : list of the files/directories which are datas
        resu      : list of the files/directories which are results
            (each of data/resu is a dict of : path, ul, compr, type, isrep)
    """
    def __init__(self, filename=None, run=None):
        """filename : filename of the '.export' file to read
        run   : ASTER_RUN object (optional)
        """
        # initialisations
        self.param = {}
        self.args = {}
        self.data = []
        self.resu = []
        self.agla = []
        self._filename = None
        self.set_filename(filename)
        self.verbose = False
        self.debug = False
        self._content = None
        # ----- reference to ASTER_RUN object which manages the execution
        self.run = run
        if run != None:
            self.verbose = run['verbose']
            self.debug   = run['debug']
        if filename != None:
            # check if file exists
            if not osp.isfile(filename):
                self._mess(ufmt(_(u'file not found : %s'), filename), '<A>_ALARM')
            else:
                # read the file
                f = open(filename, 'r')
                content = f.read()
                f.close()
                self.parse(content)
        
        if self.debug:
            print3('<DBG> <init> ASTER_PROFIL :')
            print3(self)

    def __repr__(self):
        """Pretty print content of the profile
        """
        fmt = ' %-10s = %s'
        fmt2 = ' %-6s %2s %1s %-s %1s'
        txt = []
        txt.append(fmt % ('Filename', self._filename))
        for comm, d in ('Parameters :', self.param), ('Arguments :', self.args):
            txt.append(comm)
            sorted_keys = d.keys()
            sorted_keys.sort()
            for key in sorted_keys:
                txt.append(fmt % (key, d[key]))
        for comm, l in ('Data :', self.data), ('Result :', self.resu):
            txt.append(comm)
            for d in l:
                c = ' '
                if d['compr']:
                    c = 'C'
                rep = ' '
                if d['isrep']:
                    rep = 'R'
                txt.append(fmt2 % (d['type'], d['ul'], c, d['path'], rep))
        txt.append('Agla fields :')
        for val in self.agla:
            txt.append(' '.join(val))
        return os.linesep.join(txt)

    def __getitem__(self, key):
        """Return the value of parameter 'key', or '' if not exists
        (so never raise KeyError exception).
        """
        if self.param.has_key(key):
            return self.param[key]
        else:
            return [""]

    def get_version_path(self):
        """Return the path of the version used.
        """
        label = self['version'][0]
        if label == "":
            path = ""
        elif not self.run:
            path = label
        else:
            path = self.run.get_version_path(label)
        return path
        
    
    def Get(self, DR, typ):
        """Return the list of datas (DR='D') or results (DR='R') or both (DR='DR')
        of the type 'typ'
        """
        lr = []
        if   DR == 'D':
            what = self.data
        elif DR == 'R':
            what = self.resu
        else:
            what = self.data + self.resu
        for entry in what:
            if entry['type'] == typ:
                lr.append(entry)
        return lr

    def get_base(self, DR):
        """If there is a base or bhdf in profile in datas/results, return
        the type and if it's compressed or not.
        """
        type_base, compress = None, None
        base = self.Get(DR, typ='base')
        if base:
            type_base = 'base'
            compress  = base[0]['compr']
        else:
            base = self.Get(DR, typ='bhdf')
            if base:
                type_base = 'bhdf'
                compress  = base[0]['compr']
        return type_base, compress

    def get_jobname(self):
        """Return job name."""
        return self["nomjob"][0]
    
    def Set(self, DR, dico):
        """Add an entry defined in `dico` in .data/.resu.
        """
        if   DR == 'D':
            l_what = [self.data]
        elif DR == 'R':
            l_what = [self.resu]
        else:
            l_what = [self.data, self.resu]
        l_k = ('type', 'isrep', 'path', 'ul', 'compr')
        for k in l_k:
            if not dico.has_key(k):
                self._mess(_(u"key '%s' missing") % k, '<F>_PROGRAM_ERROR')
        dico['ul'] = str(dico['ul'])
        for what in l_what:
            what.append(dico)
        # update content
        self.update_content()

    def Del(self, DR, typ):
        """Delete entries of type 'typ' in .data/.resu.
        """
        lobj = []
        if DR.find('D') > -1:
            for entry in self.data:
                if entry['type'] != typ:
                    lobj.append(entry)
            self.data = lobj
        lobj = []
        if DR.find('R') > -1:
            for entry in self.resu:
                if entry['type'] != typ:
                    lobj.append(entry)
            self.resu = lobj

    def __setitem__(self, p, v):
        """Add the parameter 'p' with the value 'v'
        """
        if self.debug and self.param.has_key(p):
            print3('<DBG> (AS_PROFIL.setitem) force param['+p+'] = ', v)
        if type(v) not in (list, tuple):
            v = [v,]
        self.param[p] = v

    def __delitem__(self, p):
        """Delete the parameter 'p'
        """
        if self.param.has_key(p):
            del self.param[p]

    def _mess(self, msg, cod='', store=False):
        """Just print a message
        """
        if hasattr(self.run, 'Mess'):
            self.run.Mess(msg, cod, store)
        else:
            print3('%-18s %s' % (cod, msg))

    def parse(self, content):
        """Extract fields of config from 'content'
        """
        for l in split_endlines(content):
            if not re.search('^[ ]*#', l):
                spl = l.split()
                typ = ''
                if len(spl) > 0:
                    typ = spl[0]
                    if typ not in ('A', 'P', 'F', 'R', 'N'):
                        self._mess(_(u'unexpected type : %s') % typ, '<A>_ALARM')
                else:
                    continue
                if len(spl) >= 3:
                    if typ == 'P':
                        if self.param.has_key(spl[1]):
                            self.param[spl[1]].append(' '.join(spl[2:]))
                        else:
                            self.param[spl[1]] = [' '.join(spl[2:])]
                    elif typ == 'A':
                        self.args[spl[1]] = ' '.join(spl[2:])
                    elif typ == 'F' or typ == 'R':
                        if len(spl) >= 5:
                            dico = {
                                'type'  : spl[1],
                                'path'  : spl[2],
                                'isrep' : typ == 'R',
                                'ul'    : spl[4],
                                'compr' : spl[3].find('C') > -1,
                            }
                            if spl[3].find('D')>-1:
                                self.data.append(dico)
                            if spl[3].find('R')>-1:
                                self.resu.append(dico)
                        else:
                            self._mess(_(u'fields missing on line : %s') % l, '<A>_ALARM')
                    elif typ == 'N':
                        # just store agla fields
                        self.agla.append(spl)
                elif len(spl) >= 2:
                    typ = spl[0]
                    if typ == 'A':
                        self.args[spl[1]] = ''
        self.update_content()

    def copy(self):
        """Return a copy of the profile
        """
        newp = self.__class__(None, self.run)
        for attr in ('_filename', '_content'):
            setattr(newp, attr, getattr(self, attr))
        for attr in ('args', 'param'):
            setattr(newp, attr, getattr(self, attr).copy())
        for attr in ('data', 'resu'):
            for dico in getattr(self, attr):
                getattr(newp, attr).append(dico.copy())
        for val in self.agla:
            newp.agla.append(val[:])
        return newp
        
    def update(self, other):
        """Update the profile using values from 'other' :
                - replace params and args
                - add datas and results
        """
        self.param.update(other.param)
        self.args.update(other.args)
        self.data.extend(other.data)
        self.resu.extend(other.resu)
        self.agla.extend(other.agla)

    def update_content(self):
        """Fill 'content' attribute.
        """
        txt = []
        for p, dico in ('P', self.param), ('A', self.args):
            sorted_keys = dico.keys()
            sorted_keys.sort()
            for key in sorted_keys:
                l_val = dico[key]
                if type(l_val) not in (list, tuple):
                    l_val = [l_val,]
                for v in l_val:
                    txt.append(' '.join([p, key, str(v)]))
        for dr, l_val in ('D', self.data), ('R', self.resu):
            for d in l_val:
                c = ' '
                if d['compr']:
                    c = 'C'
                fr = 'F'
                if d['isrep']:
                    fr = 'R'
                txt.append(' '.join([fr, d['type'], d['path'], dr+c, str(d['ul'])]))
        for val in self.agla:
            txt.append(' '.join(val))
        txt.append('')
        self._content = os.linesep.join(txt)

    def get_filename(self):
        """Return filename of profile.
        """
        return self._filename

    def set_filename(self, filename):
        """Change filename of profile.
        """
        self._filename = filename

    def get_content(self):
        """Return the content of the profile.
        """
        self.update_content()
        return self._content

    def WriteExportTo(self, fich, dbg=False):
        """Write the export file represents this profile.
        """
        self.set_filename(self._filename or fich)
        self.update_content()
        if self.debug or dbg:
            print3('<DBG> <-- content of "%s"' % fich)
            print3(self._content)
            print3('<DBG> end of file -->')
        try:
            open(fich, 'w').write(self._content)
        except IOError:
            self._mess(ufmt(_(u'No write access to %s'), fich), '<F>_ERROR')
        except Exception:
            self._mess(ufmt(_(u'Can not write export file : %s'), fich), '<F>_ERROR')

    def add_param_from_dict(self, dpara):
        """Add parameters and arguments to a profile.
        """
        self.args['memjeveux']      = dpara['memjeveux']
        self.args['tpmax']          = dpara['tps_job']
        if dpara['memjeveux_stat'] != 0:
            self.args['memjeveux_stat'] = dpara['memjeveux_stat']
        new_para = {}
        for key, val in dpara.items():
            if self.args.has_key(key):
                continue
            if key == 'tps_job':
                key = 'tpsjob'
                val = val / 60.
            elif key == 'mem_job':
                key = 'memjob'
                val = val * 1024
            if val != 0:
                new_para[key] = [val,]
        self.param.update(new_para)

    def add_default_parameters(self):
        """Add default values in parameters to make the profile immediately runnable.
        """
        if self.run:
            self['version'] = self.run['aster_vers']
            self['origine'] = 'ASTK %s' % self.run.__version__
        
        # add default values
        self['actions'] = "make_etude"
        self['consbtc'] = "oui"
        self['soumbtc'] = "oui"
        self['mem_aster'] = 100
        self['ncpus'] = 1
        self['mpi_nbcpu'] = 1
        self['mpi_nbnoeud'] = 1

        self.set_param_time(3600*24)
        self.set_param_memory(2048)
        self.set_running_mode(MODES.INTERACTIF)

    def set_param_time(self, tsec):
        """Set time parameter in seconds."""
        self['tpsjob'] = int(1.0 * tsec / 60)
        self.args['tpmax'] = tsec

    def set_param_memory(self, memory):
        """Set memory parameter in MB."""
        if self.run and re.search('64$', self.run['plate-forme']):
            facW = 8
        else:
            facW = 4
        self['memjob'] = memory * 1024
        self.args['memjeveux'] = 1.0 * memory / facW

    def from_remote_server(self, ignore_types=None):
        """Change local pathnames to be visible from a remote server.
        Do not change files/dirs of ignore_types."""
        for l_val in (self.data, self.resu):
            for dico in l_val:
                fname = FileName(dico['path'])
                fname.user = local_user
                fname.host = local_full_host
                dico['path'] = fname.repr()

    def set_running_mode(self, mode):
        """Set the running mode"""
        # string assignement is used by the OM module
        if type(mode) is str:
            mode = getattr(MODES, mode.upper(), -1)
        if not MODES.exists(mode):
            raise ValueError("not a valid mode : %s" % mode)
        self["mode"] = MODES.get_id(mode).lower()

    def get_timeout(self):
        """Return timeout value from profile.
        """
        val = self['tpsjob'][0]
        try:
            timeout = int(float(val) * 60)
        except Exception, msg:
            raise Exception("%s\nException : %s" % (val, msg))
        return timeout


