#!/usr/bin/env python
# -*- coding: utf-8 -*-

import time
import os.path as osp
import shutil
import unittest

from common import dict_conf, execcmd, tmpdir
from data   import study_export

from asrun.run    import AsRunFactory
from asrun.client import CLIENT, SERVER_CONF
from asrun.calcul import parse_submission_result
from asrun.common.utils import YES_VALUES, NO_VALUES
from asrun.common.rcfile import read_rcfile



class TestClientFunctions(unittest.TestCase):

    def test01_client(self):
        # local astkrc directory for test
        rcdir = osp.join(tmpdir, 'astkrc')
        if osp.exists(rcdir):
            shutil.rmtree(rcdir)

        run = AsRunFactory(rcdir=rcdir)
        client = CLIENT(rcdir, run=run)

        # test the default configuration (rcdir does not exist)
        prefs = client.get_user_preferences()
        assert type(prefs) is dict
        assert len(prefs) == 11     # number of fields in default prefs file
        assert prefs.has_key('nom_domaine')

        # before the refresh the values are not set 
        client.init_server_config()
        assert client.get_server_rcinfo() is None
        servers = client.get_server_list()
        assert len(servers) == 1
        cfg = client.get_server_config(servers[0])
        if cfg['nom_complet'].find('?') > -1:
            # not configured: '?FULL_SERVER_NAME?'
            cfg['nom_complet'] = ''
        assert cfg.get('batch') is None
        assert cfg.get('interactif_memmax') is None
        
        # ask for a refresh
        client.refresh_server_config()

        servers = client.get_server_list()
        assert len(servers) == 1
        cfg = client.get_server_config(servers[0])
        assert cfg['batch'] == "non"
        assert type(cfg['interactif_memmax']) in (int, long)
        assert cfg['interactif_memmax'] > 1

        # test a real configuration
        shutil.copy(osp.join(dict_conf['DATA'], "config_serveurs"), rcdir)
        client.init_server_config(force=True)
        assert client.get_server_rcinfo() == '1.1'

        servers = client.get_server_list()
        assert len(servers) == 7
        for serv in servers:
            cfg = client.get_server_config(serv)
            assert cfg['etat'] in ('on', 'off')
        assert "cli70cx" in servers
        cfg = client.get_server_config("cli70cx")
        assert cfg["rep_serv"].find("ASTK_SERV") < 0, cfg["rep_serv"]
        all_cfg = client.get_dict_server_config()
        assert all_cfg["cli70cx"]["interactif"] in (YES_VALUES + NO_VALUES)

        # fails outside EDF network
        iret, output = run.Shell("ping -c 1 -W 2 mailhost.edf.fr")
        if iret == 0:
            client.init_server_config(refresh=True)

        client.save_server_config()


    def test01_module_aster(self):
        # Test the usage done by the Code_Aster module :
        #   - compare remote_xxxx_protocol
        from pprint import pprint
        rcdir = osp.join(tmpdir, 'astkrc')
        #rcdir = ".astkrc"
        if osp.exists(rcdir):
            shutil.rmtree(rcdir)

        run = AsRunFactory(rcdir=rcdir)
        client = CLIENT(rcdir, run=run)
        prefs = client.get_user_preferences()
        # should be removed when config/prefs will be merged or proto imposed by server
        dconf = {}
        read_rcfile(client.rcfile('config'), dconf)
        for type, defv in (('shell', 'SSH'),
                           ('copy', 'SCP')):
            key = 'remote_%s_protocol' % type
            proto = _get_protocol(prefs, type, defv, optional=dconf)
            client.set_user_preferences(key, proto)
        client.save_user_preferences()
        # test saving
        prefs = client.get_user_preferences(force=True)
        assert prefs.get(key) is not None and prefs[key] == proto, prefs.get(key)


    def test02_call_server(self):
        export = osp.join(tmpdir, "call_server.export")
        open(export, "w").write(study_export % dict_conf)
        cmd = dict_conf["as_run"] + ["--call_server", export]
        iret, out = execcmd(cmd, "server.6", return_output=True)
        nomjob = "call_server_cmdline"
        assert iret == 0
        jobid, queue = parse_submission_result(out)
        assert jobid.strip() != ""
        assert jobid.strip().isdigit()
        time.sleep(4)
        cmd = dict_conf["as_run"] + ["--del", jobid, "study", "interactif"]
        iret = execcmd(cmd, "server.7")
        assert iret == 0


def _get_protocol(main_dict, type, default=None, optional=None):
    """Return the protocol found in config dicts."""
    assert type in ('shell', 'copy'), type
    key = 'remote_%s_protocol' % type.lower()
    main = main_dict.get(key)
    optv = main
    if optional is not None:
        optv = optional.get(key)
    #if main != optv:
        #print "'%s' differs '%s' (prefs) != '%s' (config)" % (key, main, optv)
    proto = main or optv or default
    print "shell protocol: '%s' (prefs), '%s' (config), '%s' (default) => '%s'" \
        % (main, optv, default, proto)
    return proto


if __name__ == "__main__":
    unittest.main()
