/*
 * atanks - obliterate each other with oversize weapons
 * Copyright (C) 2003  Thomas Hudson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * */

#include <time.h>
#include "player.h"
#include "globaldata.h"
#include "files.h"

GLOBALDATA::GLOBALDATA ():dataDir(NULL),configDir(NULL),updates(NULL),lastUpdates(NULL),allPlayers(NULL),
    players(NULL),currTank(NULL),SOUND(NULL),saved_game_list(NULL)
{
  tank_status = (char *)calloc(128, sizeof(char));
  if (!tank_status)
    {
      perror ( (char *)"globaldata.cpp: Failed allocating memory for tank_status in GLOBALDATA::GLOBALDATA");
      exit (1);
    }

  initialise ();
  language = LANGUAGE_ENGLISH;
  sound = 1.0;
  name_above_tank = TRUE;
  colourDepth = 16;
  screenWidth = DEFAULT_SCREEN_WIDTH;
  screenHeight = DEFAULT_SCREEN_HEIGHT;
  width_override = height_override = 0;
  temp_screenWidth = screenWidth;
  temp_screenHeight = screenHeight;
  halfWidth = screenWidth / 2;
  halfHeight = screenHeight / 2;
  frames_per_second = FRAMES_PER_SECOND;
  numPermanentPlayers = 10;
  violent_death = FALSE;
#ifndef DOS
  cacheCirclesBG = 1;
#endif
#ifdef DOS
  cacheCirclesBG = 0;
#endif
  ditherGradients = 1;
  detailedLandscape = 0;
  detailedSky = 0;
  colour_theme = 0.0;
  startmoney = 15000;
  turntype = TURN_RANDOM;
  skipComputerPlay = SKIP_HUMANS_DEAD;
  dataDir = (char *)DATA_DIR;
  os_mouse = 1.0;
  full_screen = false;
  interest = 1.25;
  scoreHitUnit = 75;
  scoreSelfHit = 0;
  scoreUnitDestroyBonus = 5000;
  scoreUnitSelfDestroy = 0;
  scoreRoundWinBonus = 10000;
  sellpercent = 0.80;
  game_name[0] = '\0';
  load_game = 0.0;
  campaign_mode = 0.0;
  saved_game_index = 0;
  saved_game_list = NULL;
  max_fire_time = 0.0;
  close_button_pressed = false;

  updates = new BOX[MAXUPDATES];
  if (!updates)
    {
      perror ( (char *)"globaldata.cc: Failed allocating memory for updates in GLOBALDATA::GLOBALDATA");
      exit (1);
    }
  lastUpdates = new BOX[MAXUPDATES];
  if (!lastUpdates)
    {
      perror ( (char *)"globaldata.cc: Failed allocating memory for lastUpdates in GLOBALDATA::GLOBALDATA");
      exit (1);
    }
  updateCount = 0;
  lastUpdatesCount = 0;

  players = new PLAYER*[MAXPLAYERS];
  if (!players)
    {
      perror ( (char *)"globaldata.cc: Failed allocating memory for players in GLOBALDATA::GLOBALDATA");
      exit (1);
    }
  numPlayers = 0;
  rounds = 5;

  allPlayers = (PLAYER**)malloc (sizeof (PLAYER*));
  if (!allPlayers)
    {
      fprintf (stderr, "Failed allocating memory for players in globaldata.cc\n");
      exit (1);
    }

  for (int count = 0; count < 360; count++)
    {
      slope[count][0] = sin (count / (180 / PI));
      slope[count][1] = cos (count / (180 / PI));
    }
  slope[270][1] = 0;
  configDir = NULL;
  bIsGameLoaded = true;
  bIsBoxed = false;
  iHumanLessRounds = -1;
  dMaxVelocity = 0.0;	// Will be set in game()
#ifdef DEBUG_AIM_SHOW
  bASD = false; // will be set to true after the first drawing of the map
#endif
}

/*
This function saves the global data to a text file. If all goes
well, TRUE is returned, on error, FALSE is returned.
-- Jesse

New method to save to binary file. The old one is below!
-- Sven
*/
int GLOBALDATA::saveToFile(ofstream &ofsFile, bool aIsSaveGame)
{
  bool bResult = true;

  // First: Write Global identifier:
  bResult = pushData(FILEPART_GLOBALS, ofsFile);

  if (bResult)
    {
      // Now changed screen dimensions can be applied for the next load
      screenWidth = (int) temp_screenWidth;
      screenHeight = (int) temp_screenHeight;
    }

  // Now we can push everything into the file:
  // numPlayers isn't saved, it's dynamic!
  if (bResult) bResult = pushData("Dither Gradient   ", ditherGradients,        ofsFile);
  if (bResult) bResult = pushData("Detail Landscape  ", detailedLandscape,      ofsFile);
  if (bResult) bResult = pushData("Detail Sky        ", detailedSky,            ofsFile);
  if (bResult) bResult = pushData("Interest          ", interest,               ofsFile);
  if (bResult) bResult = pushData("Score Round Win   ", scoreRoundWinBonus,     ofsFile);
  if (bResult) bResult = pushData("Score Unit Hit    ", scoreHitUnit,           ofsFile);
  if (bResult) bResult = pushData("Score Unit Destroy", scoreUnitDestroyBonus,  ofsFile);
  if (bResult) bResult = pushData("Score Self Destroy", scoreUnitSelfDestroy,   ofsFile);
  if (bResult) bResult = pushData("Sell Ratio        ", sellpercent,            ofsFile);
  if (!aIsSaveGame)
    {
      // The following options are not saved in save games:
      if (bResult) bResult = pushData("Start Money       ", startmoney,             ofsFile);
      if (bResult) bResult = pushData("Turn Type         ", turntype,               ofsFile);
      if (bResult) bResult = pushData("Skip Computer Play", skipComputerPlay,       ofsFile);
      if (bResult) bResult = pushData("Sound             ", sound,                  ofsFile);
      if (bResult) bResult = pushData("Screen Width      ", screenWidth,            ofsFile);
      if (bResult) bResult = pushData("Screen Height     ", screenHeight,           ofsFile);
      if (bResult) bResult = pushData("OS Mouse          ", os_mouse,               ofsFile);
      if (bResult) bResult = pushData("Permanent Players ", numPermanentPlayers,    ofsFile);
      if (bResult) bResult = pushData("Language          ", language,               ofsFile);
      if (bResult) bResult = pushData("Color Theme       ", colour_theme,           ofsFile);
      if (bResult) bResult = pushData("Frames Per Second ", frames_per_second,      ofsFile);
      if (bResult) bResult = pushData("Violent Death     ", violent_death,          ofsFile);
      if (bResult) bResult = pushData("Max Fire Time     ", max_fire_time,          ofsFile);
    }
  if (bResult) bResult = pushData(FILEPART_ENDSECT, ofsFile);
  return(bResult);
}

int GLOBALDATA::saveToFile_Text( FILE *file)
{
  if (! file) return FALSE;


  screenWidth = (int) temp_screenWidth;
  screenHeight = (int) temp_screenHeight;

  fprintf (file, "*GLOBAL*\n");

  fprintf (file, "NUMPLAYERS=%d\n", numPlayers);
  fprintf (file, "ROUNDS=%f\n", rounds);
  fprintf (file, "DITHER=%f\n", ditherGradients);
  fprintf (file, "DETAILEDSKY=%f\n", detailedSky);
  fprintf (file, "DETAILEDLAND=%f\n", detailedLandscape);
  fprintf (file, "STARTMONEY=%f\n", startmoney);
  fprintf (file, "TURNTYPE=%f\n", turntype);
  fprintf (file, "INTEREST=%f\n", interest);
  fprintf (file, "SCOREROUNDWINBONUS=%f\n", scoreRoundWinBonus);
  fprintf (file, "SCOREHITUNIT=%f\n", scoreHitUnit);
  fprintf (file, "SCOREUNITDESTROYBONUS=%f\n", scoreUnitDestroyBonus);
  fprintf (file, "SCOREUNITSELFDESTROY=%f\n", scoreUnitSelfDestroy);
  fprintf (file, "ACCELERATEDAI=%f\n", skipComputerPlay);
  fprintf (file, "SELLPERCENT=%f\n", sellpercent);
  fprintf (file, "ENABLESOUND=%f\n", sound);
  fprintf (file, "SCREENWIDTH=%d\n", screenWidth);
  fprintf (file, "SCREENHEIGHT=%d\n", screenHeight);
  fprintf (file, "OSMOUSE=%f\n", os_mouse);
  fprintf (file, "NUMPERMANENTPLAYERS=%d\n", numPermanentPlayers);
  fprintf (file, "LANGUAGE=%f\n", language);
  fprintf (file, "COLOURTHEME=%f\n", colour_theme);
  fprintf (file, "FRAMES=%f\n", frames_per_second);
  fprintf (file, "VIOLENTDEATH=%f\n", violent_death);
  fprintf (file, "MAXFIRETIME=%f\n", max_fire_time);
  fprintf (file, "***\n");
  return TRUE;
}

/*
This function loads global settings from a text
file. The function returns TRUE on success and FALSE if
any erors are encountered.
-- Jesse

Changed to binary mode, being the counterpart to loadFromFile()
The original methid is below.
-- Sven
*/
int GLOBALDATA::loadFromFile(ifstream &ifsFile)
{
  char chArea[NAME_LENGTH] = "";
  double dData = 0.0;
  bool bResult = true;
  bool bIsFinished = false;
  bool bIsData = false;

  double sound_bookmark = 1.0;
  if (! sound)
    sound_bookmark = sound;

  while (!bIsFinished && (bResult = popData(chArea, bIsData, ifsFile)))
    {
      if (!bIsData && !strcmp(chArea, FILEPART_ENDSECT))      bIsFinished  = true;
      else if (!strcmp(chArea, "Dither Gradient   ")) bResult = popData(ditherGradients,        ifsFile);
      else if (!strcmp(chArea, "Detail Landscape  ")) bResult = popData(detailedLandscape,      ifsFile);
      else if (!strcmp(chArea, "Detail Sky        ")) bResult = popData(detailedSky,            ifsFile);
      else if (!strcmp(chArea, "Interest          ")) bResult = popData(interest,               ifsFile);
      else if (!strcmp(chArea, "Score Round Win   ")) bResult = popData(scoreRoundWinBonus,     ifsFile);
      else if (!strcmp(chArea, "Score Unit Hit    ")) bResult = popData(scoreHitUnit,           ifsFile);
      else if (!strcmp(chArea, "Score Unit Destroy")) bResult = popData(scoreUnitDestroyBonus,  ifsFile);
      else if (!strcmp(chArea, "Score Self Destroy")) bResult = popData(scoreUnitSelfDestroy,   ifsFile);
      else if (!strcmp(chArea, "Sell Ratio        ")) bResult = popData(sellpercent,            ifsFile);
      else if (!strcmp(chArea, "Start Money       ")) bResult = popData(startmoney,             ifsFile);
      else if (!strcmp(chArea, "Turn Type         ")) bResult = popData(turntype,               ifsFile);
      else if (!strcmp(chArea, "Skip Computer Play")) bResult = popData(skipComputerPlay,       ifsFile);
      else if (!strcmp(chArea, "Sound             ")) bResult = popData(sound,                  ifsFile);
      else if (!strcmp(chArea, "Screen Width      ")) bResult = popData(screenWidth,            ifsFile);
      else if (!strcmp(chArea, "Screen Height     ")) bResult = popData(screenHeight,           ifsFile);
      else if (!strcmp(chArea, "OS Mouse          ")) bResult = popData(os_mouse,               ifsFile);
      else if (!strcmp(chArea, "Permanent Players ")) bResult = popData(numPermanentPlayers,    ifsFile);
      else if (!strcmp(chArea, "Language          ")) bResult = popData(language,               ifsFile);
      else if (!strcmp(chArea, "Color Theme       ")) bResult = popData(colour_theme,           ifsFile);
      else if (!strcmp(chArea, "Frames Per Second ")) bResult = popData(frames_per_second,      ifsFile);
      else if (!strcmp(chArea, "Violent Death     ")) bResult = popData(violent_death,          ifsFile);
      else if (!strcmp(chArea, "Max Fire Time     ")) bResult = popData(max_fire_time,          ifsFile);
      else if (!bIsFinished && bIsData)
        bResult = popData(dData, ifsFile); // Something we do not know (any more/yet), so dump it!
    }

  if (! sound_bookmark)
    sound = sound_bookmark;

  if (width_override)
    screenWidth = width_override;
  if (height_override)
    screenHeight = height_override;

  halfWidth = screenWidth / 2;
  halfHeight = screenHeight / 2;

  if (skipComputerPlay > SKIP_HUMANS_DEAD)
    skipComputerPlay = SKIP_HUMANS_DEAD;

  return(bResult);
}

int GLOBALDATA::loadFromFile_Text (FILE *file)
{
  char line[MAX_CONFIG_LINE];
  int equal_position, line_length;
  char field[MAX_CONFIG_LINE], value[MAX_CONFIG_LINE];
  char *result = NULL;
  bool done = false;
  double sound_bookmark = 1.0;

  if (! sound)
    sound_bookmark = sound;

 // read until we hit line "*ENV*" or "***" or EOF
  do
    {
      result = fgets(line, MAX_CONFIG_LINE, file);
      if (! result)     // eof
        return FALSE;
      if (! strncmp(line, (char *)"***", 3) )     // end of record
        return FALSE;
    }
  while ( strncmp(line, (char *)"*GLOBAL*", 5) );     // read until we hit new record

  while ( (result) && (!done) )
    {
      // read a line
      memset(line, '\0', MAX_CONFIG_LINE);
      result = fgets(line, MAX_CONFIG_LINE, file);
      if (result)
        {
          // if we hit end of the record, stop
          if (! strncmp(line, (char *)"***", 3) )
            {
              return TRUE;
            }
          // find equal sign
          line_length = strlen(line);
          // strip newline character
          if ( line[line_length - 1] == '\n')
            {
              line[line_length - 1] = '\0';
              line_length--;
            }
          equal_position = 1;
          while ( ( equal_position < line_length) && (line[equal_position] != '=') )
            equal_position++;
          // make sure we have valid equal sign

          if ( equal_position <= line_length )
            {
              // seperate field from value
              memset(field, '\0', MAX_CONFIG_LINE);
              memset(value, '\0', MAX_CONFIG_LINE);
              strncpy(field, line, equal_position);
              strcpy(value, & (line[equal_position + 1]));
              if (! strcasecmp(field, "numplayers") )
                sscanf(value, "%d", &numPlayers);
              else if (! strcasecmp(field, "rounds") )
                sscanf(value, "%lf", &rounds);
              else if (! strcasecmp(field, "dither"))
                sscanf(value, "%lf", &ditherGradients);
              else if (! strcasecmp(field, "detailedsky"))
                sscanf(value, "%lf", &detailedSky);
              else if (! strcasecmp(field, "detailedland"))
                sscanf(value, "%lf", &detailedLandscape);
              else if (! strcasecmp(field, "startmoney"))
                sscanf(value, "%lf", &startmoney);
              else if (! strcasecmp(field, "turntype"))
                sscanf(value, "%lf", &turntype);
              else if (! strcasecmp(field, "interest"))
                sscanf(value, "%lf", &interest);
              else if (! strcasecmp(field, "scoreroundwinbonus"))
                sscanf(value, "%lf", &scoreRoundWinBonus);
              else if (! strcasecmp(field, "scorehitunit"))
                sscanf(value, "%lf", &scoreHitUnit);
              else if (! strcasecmp(field, "scoreunitdestroybonus"))
                sscanf(value, "%lf", &scoreUnitDestroyBonus);
              else if (! strcasecmp(field, "scoreunitselfdestroy"))
                sscanf(value, "%lf", &scoreUnitSelfDestroy);
              else if (! strcasecmp(field, "acceleratedai"))
                sscanf(value, "%lf", &skipComputerPlay);
              else if (! strcasecmp(field, "sellpercent"))
                sscanf(value, "%lf", &sellpercent);
              else if (! strcasecmp(field, "enablesound"))
                sscanf(value, "%lf", &sound);
              else if (! strcasecmp(field, "screenwidth"))
                sscanf(value, "%d", &screenWidth);
              else if (! strcasecmp(field, "screenheight"))
                sscanf(value, "%d", &screenHeight);
              else if (! strcasecmp(field, "OSMOUSE"))
                sscanf(value, "%lf", &os_mouse);
              else if (! strcasecmp(field, "numpermanentplayers"))
                sscanf(value, "%d", &numPermanentPlayers);
              else if (! strcasecmp(field, "language") )
                sscanf(value, "%lf", &language);
              else if (! strcasecmp(field, "colourtheme") )
                sscanf(value, "%lf", &colour_theme);
              else if (! strcasecmp(field, "frames") )
                sscanf(value, "%lf", &frames_per_second);
              else if (! strcasecmp(field, "violentdeath") )
                sscanf(value, "%lf", &violent_death);
              else if (! strcasecmp(field, "maxfiretime") )
                sscanf(value, "%lf", &max_fire_time);
            }    // end of found field=value line

        }     // end of read a line properly
    }     // end of while not done
  if (! sound_bookmark)
    sound = sound_bookmark;

  if (width_override)
    screenWidth = width_override;
  if (height_override)
    screenHeight = height_override;

  halfWidth = screenWidth / 2;
  halfHeight = screenHeight / 2;

  if (skipComputerPlay > SKIP_HUMANS_DEAD)
    skipComputerPlay = SKIP_HUMANS_DEAD;

  return TRUE;
}





void GLOBALDATA::initialise ()
{
  numTanks = 0;
}

void GLOBALDATA::addPlayer (PLAYER *player)
{
  if (numPlayers < MAXPLAYERS)
    {
      players[numPlayers] = player;
      numPlayers++;
      if ((int)player->type == HUMAN_PLAYER)
        {
          numHumanPlayers++;
          computerPlayersOnly = FALSE;
        }
    }
}

void GLOBALDATA::removePlayer (PLAYER *player)
{
  int fromCount = 0;
  int toCount = -1;

  if ((int)player->type == HUMAN_PLAYER)
    {
      numHumanPlayers--;
      if (numHumanPlayers == 0)
        {
          computerPlayersOnly = TRUE;
        }
    }

  while (fromCount < numPlayers)
    {
      if (player != players[fromCount])
        {
          if ((toCount >= 0) && (fromCount > toCount))
            {
              players[toCount]    = players[fromCount];
              players[fromCount]  = NULL;
              toCount++;
            }
        }
      else
        // Position found, now move the remaining players down!
        toCount = fromCount;
      fromCount++;
    }
  numPlayers--;
}

PLAYER *GLOBALDATA::getNextPlayer (int *playerCount)
{
  (*playerCount)++;
  if (*playerCount >= numPlayers)
    *playerCount = 0;
  return (players[*playerCount]);
}

PLAYER *GLOBALDATA::createNewPlayer (ENVIRONMENT *env)
{
  PLAYER **reallocatedPlayers;
  PLAYER *player;

  reallocatedPlayers = (PLAYER**)realloc (allPlayers, sizeof (PLAYER*) * (numPermanentPlayers + 1));
  if (reallocatedPlayers != NULL)
    {
      allPlayers = reallocatedPlayers;
    }
  else
    {
      perror ( (char *)"atanks.cc: Failed allocating memory for reallocatedPlayers in GLOBALDATA::createNewPlayer");
      exit (1);
    }
  player = new PLAYER (this, env);
  if (!player)
    {
      perror ( (char *)"globaldata.cc: Failed allocating memory for player in GLOBALDATA::createNewPlayer");
      exit (1);
    }
  allPlayers[numPermanentPlayers] = player;
  numPermanentPlayers++;

  return (player);
}

void GLOBALDATA::destroyPlayer (PLAYER *player)
{
  int fromCount = 0;
  int toCount = 0;

  for (; fromCount < numPermanentPlayers; fromCount++)
    {
      if (allPlayers[fromCount] != player)
        {
          allPlayers[toCount] = allPlayers[fromCount];
          toCount++;
        }
    }
  numPermanentPlayers--;
}



// This function returns the path to the
// config directory used by Atanks
char *GLOBALDATA::Get_Config_Path()
{
  char *my_config_dir;
  char *homedir;

  // figure out file name
  homedir = getenv(HOME_DIR);
  if (! homedir)
    homedir = (char *)".";
  my_config_dir = (char *) calloc( strlen(homedir) + 24,
                                   sizeof(char) );
  if (! my_config_dir)
    return NULL;

  sprintf (my_config_dir, "%s/.atanks", homedir);
  return my_config_dir;

}



// Thisa function checks to see if one full second has passed since the
// last time the function was called.
// The function returns true if time has passed. The function
// returns false if time hasn't passed or it was unable to tell
// how much time has passed.
bool GLOBALDATA::Check_Time_Changed()
{
  static time_t last_second = 0;
  time_t current_second;

  current_second = time(NULL);
  if ( current_second == last_second )
    return false;

  // time has changed
  last_second = current_second;
  return true;
}

