/*
 * rawwrite.c: write contents of a file to a raw disk
 *
 * Copyright (C) 1997 Roman Hodek <Roman.Hodek@informatik.uni-erlangen.de>
 *
 * This program is free software.  You can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation: either version 2 or
 * (at your option) any later version.
 *
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <osbind.h>

#define SECTORSIZE	512
#define DISKBUF		(18*SECTORSIZE)

char buf[DISKBUF];

char *bioserrs[20] = {
	"No error",
	"General, unspecified error",
	"Drive not ready",
	"Unknown command",
	"CRC error",
	"Bad request",
	"Seek error",
	"Unknown media",
	"Sector not found",
	"Out of paper",
	"Write fault",
	"Read fault",
	"General error",
	"Write-protected medium",
	"Medium changed",
	"Unknown device",
	"Bad sectors on format",
	"Insert other disk",
	"Insert disk!",
	"Device not responding"
};

char *bioserr( long err )
{
	if (-err > 0 && -err < sizeof(bioserrs)/sizeof(*bioserrs))
		return( bioserrs[-err] );
	else
		return "";
}

	
void usage( void )
{
	fprintf( stderr, "Usage: rawwrite file a:|b:\n" );
	exit( 1 );
}


int main( int argc, char *argv[] )
{
	FILE *f;
	int dstdev, sector, non_whole;
	long err;
	unsigned long nread, nsect;
	
	if (argc != 3)
		usage();

	dstdev = tolower(argv[2][0]);
	if ((dstdev != 'a' && dstdev != 'b') ||
		!(argv[2][1] == 0 || (argv[2][1] == ':' && argv[2][2] == 0)))
		usage();
	dstdev -= 'a';

	if (!(f = fopen( argv[1], "rb" ))) {
		perror( argv[1] );
		exit( 1 );
	}

	sector = non_whole = 0;
	while( (nread = fread( buf, 1, DISKBUF, f )) > 0) {
		if (non_whole) {
			fprintf( stderr, "Internal error: read returned "
					 "non-sector-aligned chunks\n" );
			exit( 1 );
		}
		nsect = (nread+SECTORSIZE-1)/SECTORSIZE;
		non_whole = nread % SECTORSIZE;
		if ((err = Rwabs( 3, buf, nsect, sector, dstdev ))) {
			fprintf( stderr, "Rwabs error #%ld: %s\n", err, bioserr(err) );
			exit( 1 );
		}
		sector += nsect;
		printf( "%5d sectors written\r", sector );
		fflush( stdout );
	}
	if (nread < 0) {
		perror( "read" );
		exit( 1 );
	}

	printf( "%5d sectors ", sector );
	if (non_whole) {
		printf( "+ %d bytes ", non_whole );
	}
	printf( "written.\n" );
	return( 0 );
}
