# -*- mode: python; coding: utf-8 -*-

from __future__ import with_statement
import os, sys
import time
import string
import subprocess
import types
import logging
import threading
import ConfigParser
import logging

Log = logging.getLogger('atheist')

from atheist.const import *


# Minimal modificaciton from the standard SMTPHandler from python2.6
class SMTPHandler(logging.Handler):
    """
    A handler class which sends an SMTP email for each logging event.
    """
    def __init__(self, mailhost, fromaddr, toaddrs, subject, credentials=None):
        """
        Initialize the handler.

        Initialize the instance with the from and to addresses and subject
        line of the email. To specify a non-standard SMTP port, use the
        (host, port) tuple format for the mailhost argument. To specify
        authentication credentials, supply a (username, password) tuple
        for the credentials argument.
        """
        logging.Handler.__init__(self)
        if type(mailhost) == types.TupleType:
            self.mailhost, self.mailport = mailhost
        else:
            self.mailhost, self.mailport = mailhost, None
        if type(credentials) == types.TupleType:
            self.username, self.password = credentials
        else:
            self.username = None
        self.fromaddr = fromaddr
        if type(toaddrs) == types.StringType:
            toaddrs = [toaddrs]
        self.toaddrs = toaddrs
        self.subject = subject

    def getSubject(self, record):
        """
        Determine the subject for the email.

        If you want to specify a subject line which is record-dependent,
        override this method.
        """
        return self.subject

    weekdayname = ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun']

    monthname = [None,
                 'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
                 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']

    def date_time(self):
        """
        Return the current date and time formatted for a MIME header.
        Needed for Python 1.5.2 (no email package available)
        """
        year, month, day, hh, mm, ss, wd, y, z = time.gmtime(time.time())
        s = "%s, %02d %3s %4d %02d:%02d:%02d GMT" % (
                self.weekdayname[wd],
                day, self.monthname[month], year,
                hh, mm, ss)
        return s

    def emit(self, record):
        """
        Emit a record.

        Format the record and send it to the specified addressees.
        """
        try:
            import smtplib
            try:
                from email.utils import formatdate
            except ImportError:
                formatdate = self.date_time
            port = self.mailport
            if not port: port = smtplib.SMTP_PORT

            logging.debug("SMTP Connecting: %s:%s" % (self.mailhost, port))

            smtp = smtplib.SMTP(self.mailhost, port)
            msg = self.format(record)
            msg = "From: %s\r\nTo: %s\r\nSubject: %s\r\nDate: %s\r\n\r\n%s" % (
                            self.fromaddr,
                            string.join(self.toaddrs, ","),
                            self.getSubject(record),
                            formatdate(), msg)

            smtp.ehlo()
            smtp.starttls()
            smtp.ehlo()

            if self.username:
                smtp.login(self.username, self.password)
            smtp.sendmail(self.fromaddr, self.toaddrs, msg)
            smtp.close()
        except (KeyboardInterrupt, SystemExit):
            raise
        except:
            self.handleError(record)



class JabberHandler(logging.Handler):

    def __init__(self, account, to_id):

        self.xmpp = __import__('xmpp')
        logging.Handler.__init__(self)

        self.from_id, password = account
        self.to_id = to_id

        jid = self.xmpp.protocol.JID(self.from_id)
        self.user = jid.getNode()
        self.server = jid.getDomain()

        self.log = logging.getLogger("JabberHandler")

        self.log.debug("Connecting %s@%s" % (self.user, self.server))

        self.conn = self.xmpp.Client(self.server, debug=[])
        conres = self.conn.connect()

        if not conres:
            self.log.error("Unable to connect to server %s!" % self.server)
            return

        if conres != 'tls':
            self.log.warning("Unable to estabilish secure connection - TLS failed!")
        authres = self.conn.auth(self.user,
                                 password,
                                 resource=self.server)

        if not authres:
            self.log.error("Unable to authorize on %s - check login/password." % self.server)
            return

        if authres != 'sasl':
            self.log.warning("Unable to perform SASL auth os %s. Old authentication method used!" % self.server)

#        self.conn.sendInitPresence(requestRoster=0)


    def emit(self, record):
#        try:
        mid = self.conn.send(\
            self.xmpp.protocol.Message(to   = self.to_id,
                                  body = self.format(record)))
#        except:
#            self.handleError(record)


# Singleton pattern
class Singleton(type):
    def __init__(cls, name, bases, dct):
        cls.__instance = None
        type.__init__(cls, name, bases, dct)

    def __call__(cls, *args, **kw):
        if cls.__instance is None:
            cls.__instance = type.__call__(cls, *args, **kw)
        return cls.__instance


def ellipsis(param, width=40, just=False, char=u'…'):
    """
    >>> ellipsis("this is a sentence", width=10, char='_')
    u'this is a_'
    >>> ellipsis("this is a ", width=12, just=True, char='_')
    u'this is a   '
    """
    width = int(width)

    if not param: param = ''
    cad = str(param).strip()
    retval = cad.split('\n')[0][:width-1]
    retval = unicode(retval, 'utf-8')
    if retval != cad: retval += char
    if just: retval = retval.ljust(width)
    return retval

def high(val):
    """
    >>> high('foo')
    '\\x1b[1mfoo\\x1b[m'
    """
    return HIGH + val + NORM



def count(n, unit):
    """
    >>> count(3, 'tick')
    '3 ticks'
    >>> count(1, 'dog')
    '1 dog'
    """
    return "%s %s%s" % (n, unit, 's' if n>1 else '')

def run_cmd(cmd):
    devnull = open('/dev/null', 'w')
    ps = subprocess.Popen(cmd, shell=True,
                          stdout=subprocess.PIPE, stderr=devnull)
    output = ps.communicate()[0]
    devnull.close()
    ps.stdout.close()
    return ps.returncode, output


def remove_dups(val):
    """
    >>> remove_dups([1,2,3,1,2])
    [1, 2, 3]
    >>> remove_dups([1,1,2,1,2,1])
    [1, 2]
    """
    retval = []
    for x in val:
        if x not in retval: retval.append(x)
    return retval


class Compath:
    def __init__(self, root=None):
        if root is None: root = os.getcwd()
        self.root = root

    def __call__(self, path=None):
        if path is None:
            path = self.root

        return path.replace(self.root, '.')

compath = Compath()


class ProgressBar:
    def __init__(self, max_val=100, width=80, label='', disable=False):
        self.max = max_val       # total item amount
        self.label = label       # prefix for the bar
        self.width = min(100, width - 34)  # bar width
        self.step = max_val/100
        self.val = 0             # current item amount
        self.blk = 0
        self.pct = 0             # completed percentage

        if disable or max_val<4:
            self._render = lambda x:None
            self.clean = lambda:None

        self.tinit = time.time()
        self.eta = ''

    def inc(self, val=1, cad=''):
        self.val += val
        self.blk = self.val * self.width / self.max
        self.pct = self.val * 100 / self.max
        if self.val > 3:
            self.eta = "%2.1fs" % (1.1 * (((self.max  - self.val) * (time.time() - self.tinit)) / self.val))
        self._render(cad)

    def _render(self, ustr):
        cad = ':%4s [ %s' % (self.label[:4], self.blk * '#')
        cad += (self.width - self.blk) * '-'
        cad += ' ] %s/%s (%3d%%) %s %s\r' % \
            (str(self.val).rjust(len(str(self.max))),
             self.max, self.pct, self.eta,
             ellipsis(ustr))

        sys.__stdout__.write(cad)
        sys.__stdout__.flush()

    def clean(self):
        print (' ' * (self.width + 30 + len(self.label))) + '\r',
        sys.__stdout__.flush()


def print_exc(exc):
    for line in exc.split('\n'):
        if line.strip(): logging.error('| ' + line)


class ThreadFunc(threading.Thread):
    def __init__(self, function, args=(), kargs={}, start=True):
        threading.Thread.__init__(self, name=function.__name__)
        self.function = function
        self.args = args
        self.kargs = kargs
        self.active = threading.Event()
        self.active.set()
        if start:
            self.start()

    def cancel(self):
        print '*' * 20
        Log.debug("ThreadFunc.cancel '%s'" % self.function.__name__)
        self.active.clear()

    def run(self):
        self.function(*self.args, **self.kargs)


class SortedDict(dict):
    '''A fixed-position dictionary. The keys will be stored on a list in the
    same order as are inserted.'''

    def __init__(self, other={}):
        self.__keylist = []
        self.update(other)

    def __getitem__(self, key):
        return dict.__getitem__(self, key)

    def __setitem__(self, key, value):
        dict.__setitem__(self, key, value)
        if key in self.__keylist:
            self.__keylist.remove(key)
        self.__keylist.append(key)

    def update(self, other):
        for k,v in other.items():
            self[k] = v

    def keys(self):
        return self.__keylist

    def values(self):
        return [self[k] for k in self.__keylist]

    def items(self):
        return [(k, self[k]) for k in self.__keylist]

    def __iter__(self):
        return self.__keylist.__iter__()

    def clear(self):
        self.__keylist = []
        dict.clear(self)


class Record(object):
    "kargs are automatic attributes"

    def __init__(self, **kargs):
        self._attrs = kargs.keys()

        for k,v in kargs.items():
            setattr(self, k, v)

    def __str__(self):
        retval = ""
        for k in self._attrs:
            retval += "%s:'%s' " % (k, getattr(self, k))

        return "<Record %s>" % retval


class IniParser(ConfigParser.ConfigParser):

    def __init__(self, fname):
        self.fname = fname
        ConfigParser.ConfigParser.__init__(self)
        self.read(fname)


    def get_item(self, path, default=None):
        try:
            if '.' in path:
                sect, name = path.split('.')
            else:
                sect = 'DEFAULT'
                name = key

            retval = self.get(sect, name)

            if not retval.strip():
                Log.warning("File '%s': key '%s' contains nothing" % \
                                 (self.fname, path))

            return retval

        except ConfigParser.NoOptionError, e:
            if default is None: raise
            return default

    def get_sec(self, sec_name):
        try:
            return Record(**dict(self.items(sec_name)))
        except ConfigParser.NoSectionError, e:
            Log.critical("A '%s' section is required in your '%s'." % \
                              (sec_name, self.fname))
            sys.exit(1)

    def __getitem__(self, path):
        try:
            return self.get_item(path)
        except  (ConfigParser.NoOptionError, ConfigParser.NoSectionError), e:
            Log.critical("In file '%s'. %s" % (self.fname, e))
            sys.exit()




# from http://code.activestate.com/recipes/203871/
class ThreadPool:

    class JoiningEx: pass

    """ Flexible thread pool class.  Creates a pool of threads, then
    accepts tasks that will be dispatched to the next available
    thread """

    def __init__(self, num_workers):
        """Initialize the thread pool with num_workers workers """

        assert num_workers > 0
        self.__workers = []
        self.__resizeLock = threading.Lock()
        self.__taskLock = threading.Condition(threading.Lock())
        self.__tasks = []
        self.__isJoining = False
        self.resize(num_workers)


    def resize(self, newsize):
        """ public method to set the current pool size """

        if self.__isJoining:
            raise ThreadPool.JoiningEx()

        with self.__resizeLock:
            self.__resize(newsize)

        return True


    def __resize(self, newsize):
        """Set the current pool size, spawning or terminating threads
        if necessary.  Internal use only; assumes the resizing lock is
        held."""

        diff = newsize - len(self.__workers)

        # If we need to grow the pool, do so
        for i in range(diff):
            self.__workers.append(ThreadPool.Worker(self))

        # If we need to shrink the pool, do so
        for i in range(-diff):
            thread = self.__workers.pop()
            thread.stop = True


    def __len__(self):
        """Return the number of threads in the pool."""

        with self.__resizeLock:
            return len(self.__workers)


    def add(self, task, args=None, callback=None):
        """Insert a task into the queue.  task must be callable;
        args and taskCallback can be None."""

        assert callable(task)

        if self.__isJoining:
            raise ThreadPool.JoiningEx()

        with self.__taskLock:
            self.__tasks.append((task, args, callback))
            self.__taskLock.notify()
            return True


    def nextTask(self):
        """ Retrieve the next task from the task queue.  For use
        only by ThreadPoolWorker objects contained in the pool """

        with self.__taskLock:
            while not self.__tasks:
                if self.__isJoining:
                    raise ThreadPool.JoiningEx()

                self.__taskLock.wait()

            assert self.__tasks
            return self.__tasks.pop(0)


    def join(self, waitForTasks=True, waitForThreads=True):
        """ Clear the task queue and terminate all pooled threads,
        optionally allowing the tasks and threads to finish """

#        Log.info("pool joining")

        self.__isJoining = True  # prevent more task queueing

        if waitForTasks:
            while self.__tasks:
                time.sleep(0.1)

        with self.__resizeLock:
            if waitForThreads:
                with self.__taskLock:
                    self.__taskLock.notifyAll()

                for t in self.__workers:
                    t.join()

            # ready to reuse
            del self.__workers[:]
            self.__isJoining = False

#        Log.info("pool joining end")


    class Worker(threading.Thread):
        """ Pooled thread class """

        def __init__(self, pool):
            """ Initialize the thread and remember the pool. """

            threading.Thread.__init__(self)
            self.__pool = pool
            self.stop = False
            self.start()


        def run(self):
            """ Until told to quit, retrieve the next task and execute
            it, calling the callback if any.  """

            while not self.stop:
                try:
                    cmd, args, callback = self.__pool.nextTask()
                except ThreadPool.JoiningEx:
                    break

                logging.debug("thread %s taken %s" % (self, cmd))

                result = cmd(*args)
                if callback:
                    callback(result)


def merge(*input):
    """
    >>> merge([1,2], [2,4], [5, 6])
    [1, 2, 2, 4, 5, 6]
    >>> merge([[1,2], [2,4]])
    [[1, 2], [2, 4]]
    >>> merge(*[[1,2], [2,4]])
    [1, 2, 2, 4]
    """
    return reduce(list.__add__, input, list())

def merge_uniq(*input):
    """
    >>> merge_uniq([1,2], [2,4], [5, 6])
    [1, 2, 4, 5, 6]
    >>> merge_uniq([1,2])
    [1, 2]
    >>> merge_uniq(*[[1,2], [2,4]])
    [1, 2, 4]
    """
    return list(set(merge(*input)))


def get_supercls(cls):
    return [cls] + merge(*[get_supercls(x) for x in cls.__bases__])
