/****************************************************************
**
** Attal : Lords of Doom
**
** fightUnit.cpp
** Manage Units
**
** Version : $Id: fightUnit.cpp,v 1.31 2006/05/21 14:47:58 lusum Exp $
**
** Author(s) : Pascal Audoux - Sardi Carlo
**
** Date : 28/12/2000
**
** Licence :
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/

#include "fightUnit.h"

// generic include files
#include <math.h>

// include files for QT
#include <QBrush>
#include <QPainter>
#include <QRect>

// application specific include files
#include "libCommon/attalSettings.h"
#include "libCommon/dataTheme.h"
#include "libCommon/log.h"

#include "libFight/fightCell.h"

#include "libClient/imageTheme.h"

#include "conf.h"

extern DataTheme DataTheme;
extern ImageTheme ImageTheme;

/** rtti number for GraphicalFightUnit class */
const int FightUnit::RTTI = 2234;


FightUnit::FightUnit( Q3Canvas * canvas )
: Q3CanvasSprite( (*ImageTheme.creatures[0])[0], canvas ), GenericFightUnit()
{
	_cpt = 0;
	_isMoving = false;
	_fifo.setAutoDelete( false );
 	collisions( true );
 	setFrame( 0 );
	setZ( CAN_F_UNIT );
	show();
}

FightUnit::FightUnit( Q3Canvas * canvas, GenericFightUnit * unit, bool mirror )
	: Q3CanvasSprite( (*ImageTheme.creatures[0])[0], canvas ),
	  GenericFightUnit( *unit )
{
	_cpt = 0;
	_fifo.setAutoDelete( false );
	_isMoving = false;
	_haveToStop = false;
	_destroyed = false;
	collisions( true );
	setFrame( 0 );
	setZ( CAN_F_UNIT );
	show();
	Q3CanvasPixmapArray * sequence 	= (*ImageTheme.creatures[DataTheme.creatures.computeSequence( unit )])[ (mirror  ^ getCreature ()->isAnimated()) ? 1 : 0 ];
	//Q3CanvasPixmapArray * sequence = (*ImageTheme.creatures[computeSequence( unit )])[ (mirror) ? 1 : 0 ];

	if( sequence ) {
		setSequence( sequence );
	}

	_lookingToRight = mirror;
}

FightUnit::~FightUnit()
{

}

void FightUnit::initCreatureForMove( GraphicalFightCell *cell )
{
	Creature* creature = getCreature ();
	uint race = creature->getRace ();
	uint level = creature->getLevel ();
	bool lookingToRight = isLookingToRight ();
	Q3CanvasPixmapArray frameArray;

	CreaturePixmap* basePixs = ImageTheme.getCreature ( race , level );
	setSequence ( (*basePixs)[ lookingToRight?0:1 ] );

	setFrame ( creature->getFirstAnimationFrame( Moving ) );

	QRect endRect = cell->boundingRect ();

	QRect startRect = endRect;

	if( _moving ) {
		startRect = _moving->boundingRect();
		double dx = double ( endRect.x() - startRect.x() ) / 10.0;
		double dy = double ( endRect.y() - startRect.y() ) / 10.0;
		setVelocity( dx, dy );
	} else {
		//setX( endRect.x () );
		//setY( endRect.y () );
		_cpt = 10;
		setVelocity( 0, 0 );
	}
}

/*!

*/

int FightUnit::hit( int nb )
{
	int ret = GenericFightUnit::hit( nb );
	if( _number == 0 ) {
		setDestroyed( true );
	}

	return ret;
}


void FightUnit::goTo( FightCell * cell )
{
	if( _currentFightCell ) {
		_currentFightCell->setUnit( NULL );
	}

	AttalSettings::FightSettings fsettings = AttalSettings::getInstance()->getFightSettings();

	Creature* creature = getCreature ();
	uint race = creature->getRace ();
	uint level = creature->getLevel ();

	if( fsettings.isAnimationEnabled ) {
		if( DataTheme.creatures.at( race , level )->isAnimated() ) {
			_currentAnimationType = Moving;
			_isMoving = true;
			canvas()->setAdvancePeriod ( 100 );
			if( creature->getNumFrames() > 0 ) {
				_cpt = 10;
				setAnimated ( true );
				if( _fifo.count() == 0 ) {
					_moving = (FightCell*)getCell();

				}
				_fifo.enqueue( cell );
			}
		} else {
			setPosition( cell );
		}
	} else {
		setPosition( cell );
	}

	GenericFightUnit::goTo( (GenericFightCell *)cell );
	cell->setUnit( (GenericFightUnit *)this );
}

void FightUnit::setPosition( FightCell * cell )
{
	if( _currentFightCell ) {
		_currentFightCell->setUnit( NULL );
	}

	Creature * creature = getCreature();
	QRect rect = cell->boundingRect();

	move( rect.x() + creature->getXOffset() - boundingRect().width() + 2 * rect.width(), rect.y() + ( rect.height() ) - boundingRect().height() + creature->getYOffset() );

	setZ( CAN_F_UNIT + cell->getRow() );
	canvas()->update();

	GenericFightUnit::goTo( (GenericFightCell *)cell );
	cell->setUnit( (GenericFightUnit *)this );
}

void FightUnit::advance( int stage )
{
	Creature* creature = getCreature ();
	uint race = creature->getRace ();
	uint level = creature->getLevel ();
	if( DataTheme.creatures.at( race , level )->isAnimated() ) {
		if( stage == 1 ) {
			setFrame ( frame() + 1 );

			Creature* creature = getCreature ();

			switch ( _currentAnimationType )
			{
			case Moving:

				if( (uint)frame() >= creature->getLastAnimationFrame( Moving ) ) {
					setFrame( creature->getFirstAnimationFrame( Moving ) );
				}
				_cpt++;
				if( _cpt > 10 ) {
					_cpt = 0;
					if( _fifo.count() > 0 ) {
						FightCell * temp = _fifo.dequeue();
						initCreatureForMove( temp );
						_moving = temp;
					} else {
						setFrame( creature->getFirstAnimationFrame( Moving ) );
						if( _moving ) {
							setPosition( _moving );
						}
						setVelocity( 0, 0 );
						_moving = 0;
						setAnimated( false );
						setActive( false );
						if( _haveToStop ) {
							_socket->sendFightUnitEndMove();
							_haveToStop = false;
						}
						_isMoving = false;
					}
				}
				break;
			case Shooting:
				if( (uint)frame() >= creature->getLastAnimationFrame( Shooting ) ) {
					setFrame( creature->getFirstAnimationFrame( Moving ) );
					setAnimated ( false );
				}
				_isMoving = false;
				setPosition( (FightCell*)this->getCell() );
				break;
			case Defending:
				if( (uint)frame() >= creature->getLastAnimationFrame( Defending ) ) {
					setFrame( creature->getFirstAnimationFrame( Moving ) );
					setAnimated ( false );
				}
				_isMoving = false;
				setPosition( (FightCell*)this->getCell() );
				break;
			case Fighting:
				if( (uint)frame() >= creature->getLastAnimationFrame( Fighting ) ) {
					setFrame( creature->getFirstAnimationFrame( Moving ) );
					setAnimated ( false );
				}
				_isMoving = false;
				setPosition( (FightCell*)this->getCell() );
				break;
			case Selecting:
				if( (uint)frame() >= creature->getLastAnimationFrame( Selecting ) ) {
					setFrame( creature->getFirstAnimationFrame( Selecting ) );
					//setAnimated ( false );
				}
				_isMoving = false;
			default: break;
			}
		} else {

		}

		Q3CanvasSprite::advance ( stage );
	} else {
		//logDD( "advance not animated %d %d", race, level );
		_isMoving = false;
	}
}

void FightUnit::draw ( QPainter& painter )
{
	Creature * creature = getCreature();
	Q3CanvasSprite::draw ( painter );

	int numItems = getNumber ();

	QRect r = boundingRect ();

	int x,y;
	if( numItems > 0 ) {
		if(!_destroyed) {
			x = r.width () - 30 -  creature->getXOffset();
			y = r.height () - 16 -  creature->getYOffset();
		} else {
			x = r.width () - 30;
			y = r.height () - 16;
		}
	} else {
		x = r.width () - 30;
		y = r.height () - 16;
	}

	if( ! _destroyed ) {
		QFont oldFont = painter.font ();
		QPen oldPen = painter.pen ();
		QBrush oldBrush = painter.brush ();
		QFont f = oldFont;
		QPen pen ( QColor ( 100, 255 , 255 ) );
		painter.setPen ( pen );
		QBrush br ( QColor ( 200, 150, 150 ) );
		painter.setBrush ( br );

		painter.drawRect ( r.left () + x, r.top () + y, 30, 16 );

		QPen fontPen ( QColor ( 120, 130, 240 ) );
		painter.setPen ( fontPen );

		f.setPixelSize ( 12 );
		painter.setFont ( f );

		QString s = QString::number ( numItems );
		painter.drawText ( r.left () + x  , r.top () + y  , 30, 16, Qt::AlignCenter, s );

		painter.setFont ( oldFont );
		painter.setPen ( oldPen );
		painter.setBrush ( oldBrush );
	}
}

Q3CanvasPixmap * FightUnit::imageAdvanced() const
{
	return image();
}

void FightUnit::setDestroyed( bool state )
{
	if( state ) {
		_destroyed = state;
		setSequence( ImageTheme.deadCreature );
		setFrame( 0 );
		setZ( CAN_F_UNIT_DEAD );
	}
}

void FightUnit::animate (CreatureAnimationType current)
{
	AttalSettings::FightSettings fsettings = AttalSettings::getInstance()->getFightSettings();

	Creature* creature = getCreature ();
	uint race = creature->getRace ();
	uint level = creature->getLevel ();
	//logDD("animate %d crea %d, race %d , level %d ", current, creature, race ,level);

	if( fsettings.isAnimationEnabled ) {
		if( DataTheme.creatures.at( race , level )->isAnimated() ) {
			_currentAnimationType = current;

			canvas()->setAdvancePeriod ( 200 );
			setAnimated ( true );

			setFrame ( creature->getFirstAnimationFrame( current ) );
		}
	}
}

bool FightUnit::hit( const QPoint & p ) const
{
	QImage image;
	image = this->image()->toImage();
	bool ret = false;
	int ix = p.x()-int(x());
	int iy = p.y()-int(y());
	    
	if ( image.valid( ix , iy ) ) {
		QRgb pixel = image.pixel( ix, iy );
		ret = ( qAlpha( pixel ) != 0 );
		
	}
	return ret;
}

void FightUnit::setActive( bool state ) 
{
	if(!getCreature()->isAnimated()) {
		if(state) {
			QImage ima = image()->toImage();
			ima.convertDepth( 32 );

			int w = ima.width();
			int h = ima.height();

			QImage img( w, h, 32 );
			img.setAlphaBuffer( TRUE );

			for ( int x = 0; x < w; x++ ) {
				for ( int y = 0; y < h/2; y++ ) {
					QRgb pixel = (x<w&y<h) ? ima.pixel( x, y ) : 0;
					img.setPixel( x, y, colorH( pixel ) );
				}
				for ( int y = h/2; y < h; y++ ) {
					QRgb pixel = (x<w&y<h) ? ima.pixel( x, y ) : 0;
					img.setPixel( x, y, pixel  );
				}
			}
			setFrame(1);
			image(1)->convertFromImage(img);
		} else {
			setFrame(0);
		}
		canvas()->update();
		update();
	} else {
		if(state) {
			animate(Selecting);
		} else {
			if(_currentAnimationType == Selecting) {
				setAnimated ( false );
			}
		}
	}
}

QRgb FightUnit::colorH( QRgb v )
{
	int ach = qAlpha(v);
	if ( ach == 255 )
		return v;

	//int r = std::min ( (int) (qRed(v)* 2 *255 /255),255);
	//int g = std::min ( (int) (qGreen(v) * 2 *255 /255),255);
	//int b = std::min ( (int) (qBlue(v) *2 *255 /255),255);

	return qRgba(255,255,255,ach);
}

