/****************************************************************
**
** Attal : Lords of Doom
**
** imageTheme.cpp
** manage images in theme
**
** Version : $Id: imageTheme.cpp,v 1.14 2004/04/04 17:34:27 tribunal2 Exp $
**
** Author(s) : Pascal Audoux
**
** Date : 10/06/2001
**
** Licence :
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/

#include "imageTheme.h"

// generic include files
// include files for QT
#include <qcanvas.h>
#include <qlist.h>
#include <qpixmap.h>
#include <qimage.h>
// application specific include files
#include "libCommon/dataTheme.h"
#include "libCommon/genericMapCreature.h"
#include "libCommon/log.h"
#include "libCommon/unit.h"

#include "libClient/attalSound.h"

ImageTheme ImageTheme;

extern DataTheme DataTheme;
extern QString IMAGE_PATH;

//
// ----- CreaturePixmap -----
//

CreaturePixmap::CreaturePixmap( QList<QPixmap> pix, QList<QPoint> hotspots )
{
	_pixmaps[0] = new QCanvasPixmapArray( pix, hotspots );

	QList<QPixmap> newList;
	for( uint i=0; i<pix.count(); i++ ) {
		QCanvasPixmap oldPix( *pix.at(i), *hotspots.at(i) );
		QCanvasPixmap * newPix;
		QImage newImg;
		if( !oldPix.isNull() ) {
			newImg = oldPix.convertToImage().mirror( true, false );
			newPix = new QCanvasPixmap( newImg );
			newList.append( newPix );
		} else {
			newList.append( new QPixmap() );
		}
	}
	_pixmaps[1] = new QCanvasPixmapArray( newList, hotspots );
}

CreaturePixmap::~CreaturePixmap()
{
	if( _pixmaps[0] ) {
		delete _pixmaps[0];
	}

	if( _pixmaps[1] ) {
		delete _pixmaps[1];
	}
}

QCanvasPixmapArray * CreaturePixmap::operator[]( uint i )
{
	return _pixmaps[i%2];
}

//
// ----- MapCreaturePixmap -----
//

MapCreaturePixmap::MapCreaturePixmap( QList<QPixmap> pix, QList<QPoint> hotspots )
{
	_pixmaps[0] = new QCanvasPixmapArray( pix, hotspots );

	QList<QPixmap> newList;
	for( uint i=0; i<pix.count(); i++ ) {
		QCanvasPixmap oldPix( *pix.at(i), *hotspots.at(i) );
		QCanvasPixmap * newPix;
		QImage newImg;
		if( !oldPix.isNull() ) {
			newImg = oldPix.convertToImage().mirror( true, false );
			newPix = new QCanvasPixmap( newImg );
			newList.append( newPix );
		} else {
			newList.append( new QPixmap() );
		}
	}
	_pixmaps[1] = new QCanvasPixmapArray( newList, hotspots );
}

MapCreaturePixmap::~MapCreaturePixmap()
{
	if( _pixmaps[0] ) {
		delete _pixmaps[0];
	}

	if( _pixmaps[1] ) {
		delete _pixmaps[1];
	}
}

QCanvasPixmapArray * MapCreaturePixmap::operator[]( uint i )
{
	return _pixmaps[i%2];
}


//
// ----- ImageTheme -----
//

ImageTheme::ImageTheme()
{
	cells = 0;
	lords = 0;
	bases = 0;
	buildings = 0;
	insideBuildings = 0;
	skills = 0;
	transition = 0;
	decoration = 0;
	creatures = 0;
	artefacts = 0;
	events = 0;
	mapCreatures = 0;
	_insideBase = 0;
	_artefactIcon = 0;
	_resourceIcon = 0;
	_resourceSmallIcon = 0;
	_flags = 0;
	_sound = 0;
}

ImageTheme::~ImageTheme()
{
	clear();
}

QPixmap * ImageTheme::getInsideBase( uint num )
{
	QPixmap * ret = 0;
	if( num < DataTheme.bases.count() ) {
		if( _insideBase[num] == 0 ) {
			QString name = IMAGE_PATH + "base/insideBase_" + QString::number( num ) + ".png";
			_insideBase[num] = new QPixmap( name );
		}
		ret = _insideBase[num];
	}

	return ret;
}

QPixmap * ImageTheme::getArtefactIcon( uint num )
{
	QPixmap * ret;
	if( num < DataTheme.artefacts.count() ) {
		if( _artefactIcon[num] == 0 ) {
			QString name = IMAGE_PATH + "artefacts/artefactIcon_" + QString::number( num ) + ".png";
			_artefactIcon[num] = new QPixmap( name );
		}
		ret = _artefactIcon[num];
	} else {
		logEE( "artefact %d pixmap not found", num );
		ret = new QPixmap();
	}

	return ret;
}

QPixmap * ImageTheme::getResourceIcon( uint num )
{
	QPixmap * ret;
	if( num < DataTheme.resources.count() ) {
		if( _resourceIcon[num] == 0 ) {
			QString name;
			name.sprintf( "ressources/ress_%02d.png", num + 1 );
			_resourceIcon[num] = new QPixmap( IMAGE_PATH + name );
		}
		ret = _resourceIcon[num];
	} else {
		logEE( "artefact %d pixmap not found", num );
		ret = new QPixmap();
	}

	return ret;
}

QPixmap * ImageTheme::getResourceSmallIcom( uint num )
{
	QPixmap * ret;
	if( num < DataTheme.resources.count() ) {
		if( _resourceSmallIcon[num] == 0 ) {
			QString name;
			name.sprintf( "ressources/smallResource_%02d.png", num + 1 );
			_resourceSmallIcon[num] = new QPixmap( IMAGE_PATH + name );
		}
		ret = _resourceSmallIcon[num];
	} else {
		logEE( "artefact %d pixmap not found", num );
		ret = new QPixmap();
	}

	return ret;
}

#define CRE_ARRAY( list1, list2 ) \
QList<QPixmap> list1; \
list1.setAutoDelete( true ); \
QList<QPoint> list2; \
list2.setAutoDelete( true )

#define ADD_PIX( list1, list2, name ) \
list1.append( new QPixmap( IMAGE_PATH + name ) ); \
list2.append( new QPoint( 0, 0 ) )

void ImageTheme::init()
{
	uint i;

	initSkills();
	initCreatures();
	initCells();
	initBuildings();
	initDecorations();
	initTransitions();
	initArtefacts();
	initResources();
	initBonus();
	initChest();
	initMapCreatures();
	initEvents();

	_flags = new QPixmap * [DataTheme.teams.count()];
	for( i = 0; i < DataTheme.teams.count(); i++ ) {
		_flags[i] = 0;
	}

	initLords();
	initBases();
	initSound();
}

void ImageTheme::save()
{

}

void ImageTheme::clear()
{
	if( cells ) {
		delete cells;
	}

	if( lords ) {
		delete lords;
	}
	
	if( bases ) {
		delete bases;
	}

	if( buildings ) {
		delete buildings;
	}

	if( insideBuildings ) {
		delete insideBuildings;
	}

	if( skills ) {
		delete skills;
	}

	if( transition ) {
		delete transition;
	}

	if( decoration ) {
		delete decoration;
	}

	if( creatures ) {
		int cpt = 0;
		uint i, j;
		uint nbRaces = DataTheme.countRace();
		uint nbCreatures;
		for( i = 0; i < nbRaces; i++ ) {
			nbCreatures = DataTheme.creatures.getRace( i )->count();
			for( j = 0; j < nbCreatures; j++ ) {
				delete creatures[cpt];
				cpt++;
				//XXX: ?? delete creatures;
			}
		}
		delete [] creatures;
	}

	if( artefacts ) {
		delete artefacts;
	}

	if( _artefactIcon ) {
		uint i;
		uint nbArtefacts = DataTheme.artefacts.count();
		for( i = 0; i < nbArtefacts; i++ ) {
			if( _artefactIcon[i] != 0 ) {
				delete _artefactIcon[i];
				_artefactIcon[i] = 0;
			}
		}
		delete [] _artefactIcon;
	}

	if( _resourceIcon ) {
		uint i;
		uint nbRes = DataTheme.resources.count();
		for( i = 0; i < nbRes; i++ ) {
			if( _resourceIcon[i] != 0 ) {
				delete _resourceIcon[i];
				_resourceIcon[i] = 0;
			}
		}
		delete [] _resourceIcon;
	}

	if( _resourceSmallIcon ) {
		uint i;
		uint nbRes = DataTheme.resources.count();
		for( i = 0; i < nbRes; i++ ) {
			if( _resourceSmallIcon[i] != 0 ) {
				delete _resourceSmallIcon[i];
				_resourceSmallIcon[i] = 0;
			}
		}
		delete [] _resourceSmallIcon;
	}

	if( bonus ) {
		/// XXX: improve delete of array
		delete bonus;
	}

	if( _insideBase ) {
		delete [] _insideBase;
	}

	if( _flags ) {
		delete [] _flags;
	}

	delete _sound;
}

QString ImageTheme::getPhotoCreature( GenericFightUnit * unit )
{
	QString pix = IMAGE_PATH + "units/face_" + QString::number( unit->getRace() ) + "_" + QString::number( unit->getLevel() ) + ".png";
	//logDD("Pixmap: %s", pix.latin1() );
	return  pix;
}

QPixmap * ImageTheme::getFlag( uint num )
{
	QPixmap * ret = 0;

	if( num < DataTheme.teams.count() ) {
		ret = _flags[num];
		if( ! ret ) {
			ret = new QPixmap( IMAGE_PATH + "players/flag_" + QString::number( num ) + ".png" );
		}
	}

	return ret;
}

QPixmap * ImageTheme::getLordPixmap( uint num )
{
	if( num <= DataTheme.lords.count() ) {
		if( ! _lordPixmap[num] ) {
			QString name;
			name.sprintf( "lords/lord_%03d.png", num );
			_lordPixmap[num] = new QPixmap( IMAGE_PATH + name );
		}
		return _lordPixmap[num];
	} else {
		logEE("There is not so many lords");
	}

	return 0;
}

QPixmap * ImageTheme::getLordSmallPixmap( uint num )
{
	if( num <= DataTheme.lords.count() ) {
		if( ! _lordSmallPixmap[num] ) {
			QString name;
			name.sprintf( "lords/smallLord_%03d.png", num );
			_lordSmallPixmap[num] = new QPixmap( IMAGE_PATH + name );
		}
		return _lordSmallPixmap[num];
	} else {
		logEE("There is not so many lords");
	}

	return 0;
}

QPixmap * ImageTheme::getBaseSmallPixmap( uint /*num*/ )
{
	/// XXX: TODO
	logEE( "not finished" );
	return _baseSmallPixmap[ 0 ];
}

void ImageTheme::initSkills()
{
	CRE_ARRAY( listPixmapSkill, listPointSkill );
	ADD_PIX( listPixmapSkill, listPointSkill, "technics/technic_1_1.png" );
	ADD_PIX( listPixmapSkill, listPointSkill, "technics/technic_1_2.png" );
	skills = new QCanvasPixmapArray( listPixmapSkill, listPointSkill );
}

void ImageTheme::initCells()
{
	uint i, j;
	QString name;

	cells = new QCanvasPixmapArray * [DataTheme.tiles.count()];

	CRE_ARRAY( listPixmapUnknown, listPointUnknown );
	ADD_PIX( listPixmapUnknown, listPointUnknown, "tiles/unknown.png" );
	cells[0] = new QCanvasPixmapArray( listPixmapUnknown, listPointUnknown );

	for( i = 1; i < DataTheme.tiles.count(); i++ ) {
		uint nbTiles = DataTheme.tiles.at( i )->getDiversificationNumber();
		CRE_ARRAY( listPixmapCell, listPointCell );

		for( j = 0; j < nbTiles; j++ ) {
			name.sprintf( "tiles/tile_%d_%d.png", i, j );
			ADD_PIX( listPixmapCell, listPointCell, name.latin1() );
		}
		cells[i] = new QCanvasPixmapArray( listPixmapCell, listPointCell );
	}


	CRE_ARRAY( listPixmapPathCell, listPointPathCell );
	for( i = 0; i < 33; i++ ) {
		QString name = "misc/arrow-" + QString::number( i ) + ".png";
		ADD_PIX( listPixmapPathCell, listPointPathCell, name.latin1() );
	}
	pathCells = new QCanvasPixmapArray( listPixmapPathCell, listPointPathCell );
}

void ImageTheme::initCreatures()
{
	uint i, j;
	int cpt = 0;
	creatures = new CreaturePixmap * [DataTheme.creatures.count()];
	for( i = 0; i < DataTheme.countRace(); i++ ) {
		for( j = 0; j < DataTheme.creatures.getRace( i )->count(); j++ ) {
			QList<QPixmap> listPixmapFightUnit;
			listPixmapFightUnit.setAutoDelete( true );
			QList<QPoint> listPointFightUnit;
			listPointFightUnit.setAutoDelete( true );

			if ( DataTheme.creatures.at (i , j)->isAnimated () )
			{
				for ( int fr = 0; fr < DataTheme.creatures.at ( i , j )->getNumFrames (); fr++ )
				{
					QString numFrame = "00";
					numFrame.sprintf ("%02d",fr + 1);
					QString fileName = "units/cre_" + QString::number( i ) + "_" + QString::number( j ) + "_" + numFrame + ".png";
					ADD_PIX( listPixmapFightUnit, listPointFightUnit, fileName );
				}
			}
			else
			{
			ADD_PIX( listPixmapFightUnit, listPointFightUnit, "units/cre_" + QString::number( i ) + "_" + QString::number( j ) + ".png" );
			}
			creatures[cpt] = new CreaturePixmap( listPixmapFightUnit , listPointFightUnit );
			cpt++;
		}
	}

	CRE_ARRAY( listPixmapDead, listPointDead );
	listPixmapDead.append( new QPixmap( IMAGE_PATH + "units/dead.png" ) ); \
	listPointDead.append( new QPoint( 0, -70 ) );
	deadCreature = new QCanvasPixmapArray( listPixmapDead, listPointDead );
}

void ImageTheme::initLords()
{
	uint nbLord = DataTheme.lords.count();
	uint i;

	CRE_ARRAY( listPixmapLord, listPointLord );
	ADD_PIX( listPixmapLord, listPointLord, "units/lord.png" );
	lords = new QCanvasPixmapArray( listPixmapLord , listPointLord );

	_lordPixmap = new QPixmap * [ nbLord ];
	for( i = 0; i < nbLord; i++ ) {
		_lordPixmap[i] = 0;
	}

	_lordSmallPixmap = new QPixmap * [ nbLord ];
	for( i = 0; i < nbLord; i++ ) {
		_lordSmallPixmap[i] = 0;
	}
}

void ImageTheme::initBases()
{
	uint i;
	
	_insideBase = new QPixmap * [DataTheme.bases.count()];
	for( i = 0; i < DataTheme.bases.count(); i++ ) {
		_insideBase[i] = 0;
	}

	CRE_ARRAY( listPixmapBase, listPointBase );
	for( i = 0; i < DataTheme.bases.count(); i++ ) {
		ADD_PIX( listPixmapBase, listPointBase, "base/base_" + QString::number( i ) + ".png" );
	}
	bases = new QCanvasPixmapArray( listPixmapBase, listPointBase );

	_baseSmallPixmap = new QPixmap * [1];
	_baseSmallPixmap[0] = new QPixmap( IMAGE_PATH + "base/smallBase.png" );
}

void ImageTheme::initArtefacts()
{
	uint i;
	CRE_ARRAY( listPixmapArtefact, listPointArtefact );
	for( i = 0; i < DataTheme.artefacts.count(); i++ ) {
		ADD_PIX( listPixmapArtefact, listPointArtefact, "artefacts/artefactItem_" + QString::number( i ) + ".png" );
	}
	artefacts = new QCanvasPixmapArray( listPixmapArtefact, listPointArtefact );

	_artefactIcon = new QPixmap * [DataTheme.artefacts.count()];
	for( i = 0; i < DataTheme.artefacts.count(); i++ ) {
		_artefactIcon[i] = 0;
	}
}

void ImageTheme::initResources()
{
	uint i;

	_resourceIcon = new QPixmap * [DataTheme.resources.count()];
	_resourceSmallIcon = new QPixmap * [DataTheme.resources.count()];
	for( i = 0; i < DataTheme.resources.count(); i++ ) {
		_resourceIcon[i] = 0;
		_resourceSmallIcon[i] = 0;
	}
}

void ImageTheme::initBonus()
{
	uint j;
	QString name;
	bonus = new QCanvasPixmapArray * [ 4 ];

	CRE_ARRAY( listPixmapBonus0, listPointBonus0 );
	for( j = 0; j < DataTheme.resources.count(); j++ ) {
		name.sprintf( "ressources/smallResource_%02d.png", j+1 );
		ADD_PIX( listPixmapBonus0, listPointBonus0, name );
	}
	bonus[0] = new QCanvasPixmapArray( listPixmapBonus0, listPointBonus0 );

	CRE_ARRAY( listPixmapBonus1, listPointBonus1 );
	name = "events/event1.png";
	ADD_PIX( listPixmapBonus1, listPointBonus1, name );
	bonus[1] = new QCanvasPixmapArray( listPixmapBonus1, listPointBonus1 );

	CRE_ARRAY( listPixmapBonus2, listPointBonus2 );
	name = "events/event1.png";
	ADD_PIX( listPixmapBonus2, listPointBonus2, name );
	bonus[2] = new QCanvasPixmapArray( listPixmapBonus2, listPointBonus2 );

	CRE_ARRAY( listPixmapBonus3, listPointBonus3 );
	name = "events/scroll.png";
	ADD_PIX( listPixmapBonus3, listPointBonus3, name );
	bonus[3] = new QCanvasPixmapArray( listPixmapBonus3, listPointBonus3 );
}

void ImageTheme::initChest()
{
	CRE_ARRAY( listPixmapChest, listPointChest );
	ADD_PIX( listPixmapChest, listPointChest, "events/chest.png" );
	chest = new QCanvasPixmapArray( listPixmapChest, listPointChest );
}

void ImageTheme::initBuildings()
{
	uint i, j;
	buildings = new QCanvasPixmapArray * [ DataTheme.buildings.count() ];
	for( i = 0; i < DataTheme.buildings.count(); i++ ) {
		CRE_ARRAY( listPixmapBuilding, listPointBuilding );
		for( j = 0; j < DataTheme.buildings.at( i )->getNbFrame(); j++ ) {
			QString name = "building/building_" + QString::number( i ) + "_" + QString::number( j ) + ".png";
			ADD_PIX( listPixmapBuilding, listPointBuilding, name );
		}
		buildings[i] = new QCanvasPixmapArray( listPixmapBuilding, listPointBuilding );
	}

	insideBuildings = new QCanvasPixmapArray * [ DataTheme.bases.count() ];
	for( i = 0; i < DataTheme.bases.count(); i++ ) {
		CRE_ARRAY( listPixmapInside, listPointInside );
		for( j = 0; j < DataTheme.bases.at( i )->getBuildingCount(); j++ ) {
			ADD_PIX( listPixmapInside, listPointInside, "base/inside_" + QString::number( i ) + "_" + QString::number( j ) + ".png" );
			ADD_PIX( listPixmapInside, listPointInside, "base/insideSel_" + QString::number( i ) + "_" + QString::number( j ) + ".png" );
		}
		insideBuildings[i] = new QCanvasPixmapArray( listPixmapInside, listPointInside );
	}
}

void ImageTheme::initTransitions()
{
	uint i, j;
	transition = new QCanvasPixmapArray * [ DataTheme.tiles.count() ];
	for( i = 1; i <= DataTheme.tiles.count(); i++ ) {
		CRE_ARRAY( listPixmapTrans, listPointTrans );
		for( j = 1; j <= NB_TRANSITION; j++ ) {
			QString name = "tiles/transi_" + QString::number( i ) + "_" + QString::number( j ) + ".png";
			ADD_PIX( listPixmapTrans, listPointTrans, name.latin1() );
		}
		transition[i-1] = new QCanvasPixmapArray( listPixmapTrans, listPointTrans );
	}
	
}

void ImageTheme::initDecorations()
{
	uint i;
	CRE_ARRAY( listPixmapDeco, listPointDeco );
	for( i = 1; i <= DataTheme.decorations.count(); i++ ) {
		ADD_PIX( listPixmapDeco, listPointDeco, "decor/decoration_" + QString::number( i ) + ".png" );
	}
	decoration = new QCanvasPixmapArray( listPixmapDeco, listPointDeco );
}

void ImageTheme::initEvents()
{
	uint i;
	events = new QCanvasPixmapArray * [ 3 ];
	QString fileName;

	CRE_ARRAY( listPixmapEvent, listPointEvent );
	fileName = "events/event1.png";
	ADD_PIX( listPixmapEvent, listPointEvent, fileName );
	events[0] = new QCanvasPixmapArray( listPixmapEvent, listPointEvent );

	CRE_ARRAY( listPixmapEvent2, listPointEvent2 );
	uint nbRes = DataTheme.resources.count();
	for( i = 0; i < nbRes; ++i ) {
		fileName.sprintf( "ressources/smallResource_%02d.png", i+1 );
		ADD_PIX( listPixmapEvent2, listPointEvent2, fileName );
	}
	events[1] = new QCanvasPixmapArray( listPixmapEvent2, listPointEvent2 );

	CRE_ARRAY( listPixmapEvent3, listPointEvent3 );
	fileName = "events/chest.png";
	ADD_PIX( listPixmapEvent3, listPointEvent3, fileName );
	events[2] = new QCanvasPixmapArray( listPixmapEvent3, listPointEvent3 );
}



void ImageTheme::initMapCreatures()
{
	uint i, j;

	mapCreatures = new MapCreaturePixmap ** [ DataTheme.countRace() ];
	for( i = 0; i < DataTheme.countRace(); i++ ) {
		mapCreatures[i] = new MapCreaturePixmap * [ DataTheme.creatures.getRace( i )->count() ];
		for( j = 0; j < DataTheme.creatures.getRace( i )->count(); j++ ) {
			CRE_ARRAY( listPixmapMapCreature, listPointMapCreature );
			QString name = "units/creMap_" + QString::number( i ) + "_" + QString::number( j ) + ".png";
			ADD_PIX( listPixmapMapCreature, listPointMapCreature, name );
			mapCreatures[i][j] = new MapCreaturePixmap( listPixmapMapCreature, listPointMapCreature );
		}
	}
}

QPixmap * ImageTheme::getMapCreature( uint race, uint level, bool rightLooking )
{
	return ( ( * ( mapCreatures[race][level] ) )[0] )->image( rightLooking?1:0 );
}

QPixmap * ImageTheme::getMapCreature( GenericMapCreature * creature, bool rightLooking )
{
	return getMapCreature( creature->getRace(), creature->getLevel(), rightLooking );
}

CreaturePixmap * ImageTheme::getCreature ( uint race, uint level )
{
	int nCnt = 0;
	for( uint i = 0; i < race; i++ ) {
		nCnt += DataTheme.creatures.getRace( i )->count();
	}

	nCnt += level;

	return ( creatures[nCnt] );
}

void ImageTheme::initSound()
{
	_sound = new AttalSound();
	_sound->init();
}

void ImageTheme::playMusicMap()
{
	_sound->playMusicMap();
}

void ImageTheme::playMusicBase( uint race )
{
	_sound->playMusicBase( race );
}

void ImageTheme::playMusicFight()
{
	_sound->playMusicFight();
}






