/****************************************************************
**
** Attal : Lords of Doom
**
** engine.cpp
** the game engine !
**
** Version : $Id: engine.cpp,v 1.69 2004/12/24 18:28:19 lusum Exp $
**
** Author(s) : Pascal Audoux - Sardi Carlo
**
** Date : 03/10/2000
**
** Licence :
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/

#include "engine.h"


// generic include files
#include <stdlib.h>
// include files for QT
#include <qapplication.h>
#include <qfile.h>
#include <qmessagebox.h>
#include <qtextstream.h>
#include <qxml.h>
// application specific includes
#include "libCommon/artefact.h"
#include "libCommon/artefactManager.h"
#include "libCommon/dataTheme.h"
#include "libCommon/define.h"
#include "libCommon/genericBonus.h"
#include "libCommon/genericEvent.h"
#include "libCommon/genericMapCreature.h"
#include "libCommon/genericInsideBuilding.h"
#include "libCommon/log.h"
#include "libCommon/unit.h"
#include "libCommon/priceMarket.h"

#include "libServer/campaign.h"
#include "libServer/parser.h"
#include "libServer/questManager.h"

#include "server/questionManager.h"

extern DataTheme DataTheme;
extern QString CAMPAIGN_PATH;

Engine::Engine( AttalServer * serv )
	:GameData()
{
	_counter=0;
	_currentPlayer = 0;
	_fight = 0;
	_server = serv;
	_state = NOT_PLAYING;
	_calendar = new Calendar();
	_isCreature = false;
	_question = new QuestionManager();
	_campaign = 0;
	_tmpPlay = 0;

	QObject::connect( _server, SIGNAL( sig_readEvent( int ) ), SLOT( slot_readSocket( int ) ) );
	QObject::connect( _server, SIGNAL( sig_newPlayer( AttalPlayerSocket * ) ), SLOT( slot_newPlayer( AttalPlayerSocket * ) ) );
	QObject::connect( _server, SIGNAL( sig_endConnection( QString ) ), SLOT( slot_endConnection( QString ) ) );
}

void Engine::slot_readSocket( int num )
{
	_server->getSocketData( num );

	switch( _state ) {
	case NOT_PLAYING:
		stateNotPlaying( num );
		break;
	case IN_GAME:
		stateInGame( num );
		break;
	case IN_QUESTION:
		stateInQuestion( num );
		break;
	case IN_FIGHT:
		stateInFight( num );
		break;
	}

	if( _server->isData( num ) ) {
		slot_readSocket( num );
	}
}

void Engine::slot_endConnection( QString name )
{
	for( uint i = 0; i < _players.count(); i++ ) {
		if( _players.at( i )->getName() == name ) {
			_players.remove( _players.at( i ) );
		}
	}
}

void Engine::handleMessage()
{
	QString msg;
	uchar d = _server->readChar();
	for( uint i = 0; i < d; i++ ) {
		msg.ref( i ) = _server->readChar();
	}
	_server->sendMessage( msg );
}

void Engine::stateNotPlaying( int num )
{
	uchar c = _server->getCla1();
	switch( c ) {
	case SO_MSG:
		handleMessage();
		break;
	case SO_CONNECT:
		switch( _server->getCla2() ) {
		case C_CONN_NAME:{
			QString res;
			/// XXX:  CHECK build of 'res'
			_players.at( num )->setNum( num );
			uint len = _server->readChar();
			for( uint i = 0; i < len; i++) {
				res.append( _server->readChar() );
			}
			for( uint j = 0; j < _players.count(); j++ ) {
				if( _players.at( j )->getName() == res ) {
					res+=QString("_%1").arg(j);
				}
			}

			_players.at( num )->setName( res );
			_server->sendConnectionId( num );
			_server->sendConnectionName( res , num);
			emit sig_newPlayer( _server->findSocket( _players.at( num ) ) );}
			break;
		case C_CONN_PLAYER:
			logEE( "Should not happen (Server : SO_CONNECT/C_CONN_PLAYER)" );
			break;
		case C_CONN_ID:
			logEE( "Should not happen (Server : SO_CONNECT/C_CONN_ID)" );
			break;
		case C_CONN_OK:
			logEE( "Should not happen (Server : SO_CONNECT/C_CONN_OK)" );
			break;
		}
		break;
	default:
		logEE( "Game not started...%d", c );
	}
}

void Engine::stateInGame( int num )
{
	if( num == _players.find( _currentPlayer ) ) {
		switch( _server->getCla1() ) {
		case SO_MSG:
			handleMessage();
			break;
		case SO_MVT:
			handleInGameMvt( num );
			break;
		case SO_QR:
			logEE( "Should not happen (Server : SO_QR/... state must be IN_QUESTION)" );
			break;
		case SO_TECHNIC:
			logDD( "Not yet implemented" );
			break;
		case SO_EXCH:
			handleInGameExchange();
			break;
		case SO_FIGHT:
			logEE( "Should not happen : state must be IN_FIGHT");
			break;
		case SO_GAME:
			logDD( "Not yet implemented" );
			break;
		case SO_MODIF:
			handleInGameModif();
			break;
		case SO_TURN:
			handleInGameTurn();
			break;
		case SO_CONNECT:
			switch( _server->getCla2() ) {
			case C_CONN_NAME:
				_players.at( num )->setNum( num );
				_players.at( num )->setName( "Player" );
				_server->sendConnectionId( num );
				break;
			case C_CONN_PLAYER:
				logEE( "Should not happen (Server : SO_CONNECT/C_CONN_PLAYER)" );
				break;
			case C_CONN_ID:
				logEE( "Should not happen (Server : SO_CONNECT/C_CONN_ID)" );
				break;
			case C_CONN_OK:
				logEE( "Should not happen (Server : SO_CONNECT/C_CONN_OK)" );
				break;
			}
			break;
		default:
			logEE( "Unknown socket_class from %d", num );
		}
	} else {
		switch( _server->getCla1() ) {
		case SO_MSG:
			handleMessage();
			break;
		default:
			logEE( "this player should not play now" );
			break;
		}
	}
}

void Engine::stateInQuestion( uint num )
{
	if( num == (uint) _players.find( _currentPlayer ) ) {
		switch( _server->getCla1() ) {
		case SO_MSG:
			handleMessage();
			break;
		case SO_QR:
			switch( _server->getCla2() ) {
			case C_QR_ANSWER:
				handleAnswer();
				break;
			default:
				logEE( "Should not happen (Server : SO_QR/...)" );
			}
			break;
		default:
			logEE( "Should not happen" );
		}
	}
}

void Engine::handleAnswer()
{
	switch( _server->getCla3() ) {
	case ANSWER_ENUM:{
		uchar choice = _server->readChar();
		if( choice == 0 ) {
			_currentPlayer->increaseResource( 0, 1000 );
			_server->sendPlayerResource( 0, _currentPlayer->getResource( 0 ) );
		} else	{
			switch( _question->getType() ) {
			case C_QR_CHEST: {
				GenericLord * lord = _question->getLord();
				if( lord ) {
					manageIncreaseExperience( lord, 2500 );
				} else {
					logEE( "Lord in _question should not be NULL" );
				}
				} break;
			default:
				logEE( "should not happen" );
			}
		}
		_state = IN_GAME;
		}
		break;
	case ANSWER_YESNO:
		switch( _question->getType() ) {
		case C_QR_CREATURE_JOIN:
			handleAnswerCreatureJoin();
			break;
		case C_QR_CREATURE_MERCENARY:
			handleAnswerCreatureMercenary();
			break;
		case C_QR_CREATURE_FLEE:
			handleAnswerCreatureFlee();
			break;
		default:
			logEE( "Should not happen" );
		}
	}
}

void Engine::handleAnswerCreatureJoin()
{
	bool accept = (bool) _server->readChar();
	GenericMapCreature * creature = _question->getCreature();
	GenericLord * lord = _question->getLord();
	
	uchar race = creature->getRace();
	uchar level = creature->getLevel();
	
	if( accept ) {
		_state = IN_GAME;
		GenericFightUnit * unit = 0;
		int place = -1;
				
		for( uint i = 0; i < MAX_UNIT; i++ ) {
			unit = lord->getUnit( i );
			
			if( unit ) {
				if( ( unit->getRace() == race ) && ( unit->getLevel() == level ) ) {
					place = i;
					break;
				}
			} else {
				place = i;
				break;
			}
		}
		
		if( place >= 0 ) {
			unit = lord->getUnit( place );
			
			if( ! unit ) {
				unit = new GenericFightUnit();
				unit->setCreature( race, level );
			}
			unit->addNumber( creature->getCreatureNumber() );
			lord->setUnit( place, unit );
			_server->updateUnit( lord, place );
		} else {
			_server->sendAskNone( tr( "You have no place for recruiting new creatures" ) );
		}
		GenericCell * cell = creature->getCell();
		_server->sendCreatureRemove( cell );
			
		cell->setCreature( 0 );
		_creatures.remove(creature);
	} else {
		if( creature->isFleeing() ) {
			_question->setType( C_QR_CREATURE_FLEE );
			_server->sendAskCreatureFlee( creature );
		} else {
			_state = IN_GAME;
			startFight( lord->getId(), _question->getCreature() );
			_isCreature = true;
		}
	}
}

void Engine::handleAnswerCreatureMercenary()
{
	bool accept = (bool) _server->readChar();
	GenericMapCreature * creature = _question->getCreature();
	GenericLord * lord = _question->getLord();
	
	uchar race = creature->getRace();
	uchar level = creature->getLevel();
	
	if( accept ) {
		_state = IN_GAME;
		GenericFightUnit * unit = 0;
		int place = -1;
				
		for( uint i = 0; i < MAX_UNIT; i++ ) {
			unit = lord->getUnit( i );
			
			if( unit ) {
				if( ( unit->getRace() == race ) && ( unit->getLevel() == level ) ) {
					place = i;
					break;
				}
			} else {
				place = i;
				break;
			}
		}
		
		if( place >= 0 ) {
			if( _currentPlayer->canBuy( creature->getCreature(), creature->getCreatureNumber() ) ) {
				_currentPlayer->buy( creature->getCreature(), creature->getCreatureNumber() );
				_server->sendPlayerResources( _currentPlayer );
				
				unit = lord->getUnit( place );
				
				if( ! unit ) {
					unit = new GenericFightUnit();
					unit->setCreature( race, level );
				}
				unit->addNumber( creature->getCreatureNumber() );
				lord->setUnit( place, unit );
				_server->updateUnit( lord, place );
			} else {
				_server->sendAskNone( tr( "You have not enough resources for recruiting this creatures" ) );
			}
		} else {
			_server->sendAskNone( tr( "You have no place for recruiting new creatures" ) );
		}
		GenericCell * cell = creature->getCell();
		_server->sendCreatureRemove( cell );
			
		cell->setCreature( 0 );
		_creatures.remove(creature);
	} else {
		GenericMapCreature * creature = _question->getCreature();
		
		if( creature->isFleeing() ) {
			_question->setType( C_QR_CREATURE_FLEE );
			_server->sendAskCreatureFlee( creature );
		} else {
			_state = IN_GAME;
			startFight( _question->getLord()->getId(), _question->getCreature() );
			_isCreature = true;
		}
	}
}

void Engine::handleAnswerCreatureFlee()
{
	bool accept = (bool) _server->readChar();
	
	if( accept ) {
		_state = IN_GAME;
		GenericMapCreature * creature = _question->getCreature();
		GenericCell * cell = creature->getCell();
		_server->sendCreatureRemove( cell );
			
		cell->setCreature( 0 );
		_creatures.remove(creature);
	} else {
		_state = IN_GAME;
		startFight( _question->getLord()->getId(), _question->getCreature() );
		_isCreature = true;
	}
}

void Engine::handleInGameMvt( int num )
{
	if( _server->getCla2() == C_MVT_ONE ) {
		uchar lord = _server->readChar();
		uint row = _server->readInt();
		uint col = _server->readInt();
		GenericCell * destCell = _map->at( row, col );
		GenericLord * movingLord = _currentPlayer->getLordById( lord );

		handleOneMove( movingLord, destCell, num );
	} else if( _server->getCla2() == C_MVT_MULTI ) {
		uchar lord = _server->readChar();
		uint nbCell = _server->readInt();
		uint i, row, col;
		GenericLord * movingLord = _currentPlayer->getLordById( lord );
		QPtrList<GenericCell> list;
		for( i = 0; i < nbCell; i++ ) {
			row = _server->readInt();
			col = _server->readInt();
			list.append( _map->at( row, col ) );
		}
		for( i = 0; i < nbCell; i++ ) {
			bool ret = handleOneMove( movingLord, list.at( i ) , num );
			if( ! ret ) {
				list.clear();
				break;
			}
		}
	} else {
		logEE( "Should not happen" );
	}
}

bool Engine::handleOneMove( GenericLord * movingLord, GenericCell * destCell, int num )
{
	/// XXX : check validity of mvt
	/// XXX : to improve : all clients concerned
	bool ret = false;

	uint lordRow = movingLord->getCell()->getRow();
	uint lordCol = movingLord->getCell()->getCol();
	uint row = destCell->getRow();
	uint col = destCell->getCol();

	if( ! ( ( lordRow   <= (row+1) ) &&
	    ( ( lordRow + 1 ) >= row ) &&
	    ( lordCol <= (col+1) ) &&
	    ( ( lordCol + 1 ) >= col ) ) ) {
	    	logEE( "Player should not request this mvt for lord (%d,%d) to (%d,%d)",
		lordRow, lordCol, row, col );
		return ret;
	}

	int cost = movingLord->computeCostMvt( destCell );
	if( cost > movingLord->getCharac( MOVE ) ) {
		logEE( "not enough mvt pt : %d < %d", movingLord->getCharac( MOVE ), cost );
		return ret;
	}

	if( destCell->getLord() ) {
		movingOnLord( movingLord, destCell );
	} else if( destCell->getEvent() ) {
		movingOnEvent( movingLord, destCell );
	} else if( destCell->getBuilding() ) {
		movingOnBuilding( movingLord, destCell, num );
	} else if( destCell->getBase() ) {
		movingOnBase( movingLord, destCell, num );
	} else if( destCell->getCreature() ) {
		movingOnCreature( movingLord, destCell );
	} else {
		movingOnFreeCell( movingLord, destCell, num );
		ret = true;
	}

	return ret;
}

void Engine::movingOnLord( GenericLord * movingLord, GenericCell * destCell )
{
	int cost = movingLord->computeCostMvt( destCell );

	if( destCell->getLord()->getOwner() == _currentPlayer ) {
		_server->sendLordExchange( movingLord, destCell->getLord() );
	} else {
		movingLord->decreaseBaseCharac( MOVE, cost );
		startFight( movingLord->getId(), destCell->getLord() );
		_isCreature = false;
	}
}

void Engine::movingOnEvent( GenericLord * movingLord, GenericCell * destCell )
{
	GenericEvent * event = destCell->getEvent();
	switch( event->getType() ) {
	case GenericEvent::EventArtefact:
		movingOnArtefact( movingLord, destCell );
		break;
	case GenericEvent::EventBonus:
		movingOnBonus( movingLord, destCell );
		break;
	case GenericEvent::EventChest:
		movingOnChest( movingLord, destCell );
		break;
	default:
		logEE( "Should not happen" );
		break;
	}
}

void Engine::movingOnArtefact( GenericLord * movingLord, GenericCell * destCell )
{
	GenericEvent * event = destCell->getEvent();
	movingLord->getArtefactManager()->addArtefact( event->getArtefact()->getType() );
	destCell->setEvent( 0 );
	_server->delEvent( event );
	_server->ownArtefact( movingLord->getArtefactManager()->getArtefactByType( event->getArtefact()->getType() ), _currentPlayer );

	_events.remove(event);
	//delete event;
}

void Engine::movingOnBonus( GenericLord * movingLord, GenericCell * destCell )
{
	GenericEvent * event = destCell->getEvent();
	GenericBonus * bonus = event->getBonus();
	destCell->setEvent( 0 );
	_server->delEvent( event );

	switch( bonus->getType() ) {
	case GenericBonus::BonusResource:
		movingOnBonusResource( bonus );
		break;
	case GenericBonus::BonusPrimSkill:
		movingOnBonusPrimSkill( movingLord, bonus );
		break;
	case GenericBonus::BonusSkill:
		break;
	case GenericBonus::BonusSpell:
		break;
	}

	_events.remove(event);
	//delete event;
}

void Engine::movingOnBonusResource( GenericBonus * bonus )
{
	uint typeResource = bonus->getParam( 0 );
	uint numberType = bonus->getParam( 1 );
	uint number = 0;

	if( numberType == 0 ) {
		number = bonus->getParam( 2 );
	} else if( numberType == 1 ) {
		uint nbVar = bonus->getParam( 2 );
		for( uint i = 0; i < nbVar; ++i ) {
			bonus->getParam( 3 + i );
		}
	} else {
		/// Not finished
		//uint min = bonus->getParam( 2 );
		//uint max = bonus->getParam( 3 );
		//uint step = bonus->getParam( 4 );
	}

	if( number > 0 ) {
		_currentPlayer->increaseResource( typeResource, number );
		_server->sendPlayerResource( typeResource, _currentPlayer->getResource( typeResource ) );
	}
}

void Engine::movingOnBonusPrimSkill( GenericLord * movingLord, GenericBonus * bonus )
{
	uint skill = bonus->getParam( 0 );
	uint value = bonus->getParam( 1 );
	LordCharac charac;

	switch( skill ) {
	case 0:
		charac = ATTACK;
		break;
	case 1:
		charac = DEFENSE;
		break;
	case 2:
		charac = POWER;
		break;
	case 3:
		charac = KNOWLEDGE;
		break;
	case 4:
		charac = MORALE;
		break;
	case 5:
		charac = LUCK;
		break;
	default:
		logEE( "Should not happen" );
		charac = ATTACK;
		break;
	}

	movingLord->increaseBaseCharac( charac, value );
	_server->sendLordCharac( movingLord->getOwner(), movingLord, charac );
}

void Engine::movingOnChest( GenericLord * movingLord, GenericCell * destCell )
{
	GenericEvent * event = destCell->getEvent();

	_question->setLord( movingLord );
	_question->setType( C_QR_CHEST );
	_server->sendAskChest();
	_state = IN_QUESTION;
	destCell->setEvent( 0 );
	_server->delEvent( event );
	_events.remove(event);
	//delete event;
}



void Engine::movingOnBuilding( GenericLord * movingLord, GenericCell * destCell, int num )
{
	uchar lord = movingLord->getId();
	uint i, j;

	// XXX: reread (cost computed ?...
	GenericBuilding * building = destCell->getBuilding();
	if( handleBuildingEnter( building, movingLord ) ) {
		movingLord->moveTo( destCell );
		if( ! _currentPlayer->hasBuilding( building ) ) {
			GenericPlayer * tempPlayer;
			for( i = 0; i < _players.count(); i++ ) {
				tempPlayer = _players.at( i );
				uint tmpnbuil = tempPlayer->numBuilding();
				for( j = 0; j < tmpnbuil; j++ ) {
					if( tempPlayer->getBuilding( j ) == building ) {
						tempPlayer->removeBuilding( j );
					}
				}
			}
			_currentPlayer->addBuilding( building );
			building->setOwner( _currentPlayer );
			for( i = 0; i < _players.count(); i++ ) {
				tempPlayer = _players.at( i );
				if( tempPlayer->canSee( destCell ) ) {
					_server->ownBuilding( building, tempPlayer );
				}
			}
		}
		destCell->setLord( movingLord );
		int row = movingLord->getCell()->getRow();
		int col = movingLord->getCell()->getCol();
		_server->sendMvt( _players.at( num ), lord, row, col );
	}
}

void Engine::movingOnBase( GenericLord * movingLord, GenericCell * destCell, int num )
{
	// XXX: test if we have to fight before...
	GenericBase * base = destCell->getBase();
	uchar lord = movingLord->getId();
	int cost = movingLord->computeCostMvt( destCell );

	if( ! _currentPlayer->hasBase( base ) ) {
		uint index;
		GenericPlayer * tempPlayer;
		if( base->getOwner() ) {
			base->getOwner()->removeBase( base );
		}
		_currentPlayer->addBase( base );
		base->setOwner( _currentPlayer );
		base->getPopLoss();
		for( index = 0; index < _players.count(); index++ ) {
			tempPlayer = _players.at( index );
			if( tempPlayer->canSee( destCell ) ) {
				updatePlayerPrices(tempPlayer);
				_server->ownBase( base, tempPlayer );
				uint nbRes = DataTheme.resources.count();
				int price;
				for( uint i = 0; i < nbRes; i++ ) {
					price = tempPlayer->getPriceMarket()->getResourcePrice( i );
					_server->sendPlayerPrice( i, price );
				}
			}
		}
	}
	movingLord->decreaseBaseCharac( MOVE, cost );
	movingLord->moveTo( destCell );
	destCell->setLord( movingLord );
	int row = movingLord->getCell()->getRow();
	int col = movingLord->getCell()->getCol();
	_server->sendMvt( _players.at( num ), lord, row, col );
}

void Engine::movingOnCreature( GenericLord * movingLord, GenericCell * destCell )
{
	//uchar lord = movingLord->getId();
	int cost = movingLord->computeCostMvt( destCell );
	
	movingLord->decreaseBaseCharac( MOVE, cost );

	GenericMapCreature * creature = destCell->getCreature();
	uint lordForce = movingLord->computeForceIndicator();
	uint creatureForce = creature->computeForceIndicator();
	
	if( creatureForce > 0 ) {
		double ratio = double( lordForce ) / double( creatureForce );
		
		CreatureAction action = computeCreatureAction( creature, ratio );
		
		switch( action ) {
		case CreatureJoin:
			_question->setType( C_QR_CREATURE_JOIN );
			_question->setLord( movingLord );
			_question->setCreature( creature );
			_state = IN_QUESTION;
			_server->sendAskCreatureJoin( creature );
			break;
		case CreatureMercenary:
			_question->setType( C_QR_CREATURE_MERCENARY );
			_question->setLord( movingLord );
			_question->setCreature( creature );
			_state = IN_QUESTION;
			_server->sendAskCreatureMercenary( creature );
			break;
		case CreatureFlee:
			_question->setType( C_QR_CREATURE_FLEE );
			_question->setLord( movingLord );
			_question->setCreature( creature );
			_state = IN_QUESTION;
			_server->sendAskCreatureFlee( creature );
			break;
		case CreatureFight:
			startFight( movingLord->getId(), creature );
			_isCreature = true;
			break;
		}	
	}
}

void Engine::movingOnFreeCell( GenericLord * movingLord, GenericCell * destCell, int num )
{
	uint i;
	int cost = movingLord->computeCostMvt( destCell );
	uchar lord = movingLord->getId();
	int lordRow = movingLord->getCell()->getRow();
	int lordCol = movingLord->getCell()->getCol();

	movingLord->decreaseBaseCharac( MOVE, cost );
	QPtrList<GenericCell> removedCells = _currentPlayer->removeLordVision( movingLord );
	movingLord->moveTo( destCell );
	QPtrList<GenericCell> addedCells = _currentPlayer->addLordVision( movingLord );
	updateMapVision( removedCells, addedCells );

	// XXX: test if enter in vision field of other players...
	uint nbPlayers = _players.count();
	GenericPlayer * tempPlayer;

	for( i = 0; i < nbPlayers; ++i ) {
		tempPlayer = _players.at( i );
		if( tempPlayer != _currentPlayer ) {
			if( ( ! tempPlayer->canSee( lordRow, lordCol ) )
				&& ( tempPlayer->canSee( destCell ) ) ) {
				_server->sendLordVisit( movingLord, tempPlayer, true );
			}
			if( ( tempPlayer->canSee( lordRow, lordCol ) )
				&& ( ! tempPlayer->canSee( destCell ) ) ) {
				_server->sendLordVisit( movingLord, tempPlayer, false );
			}
		}
	}
	destCell->setLord( movingLord );
	int row = movingLord->getCell()->getRow();
	int col = movingLord->getCell()->getCol();
	_server->sendMvt( _players.at( num ), lord, row, col );
}

void Engine::handleInGameExchange()
{
	switch( _server->getCla2() ) {
	case C_EXCH_UNIT:
		exchangeUnits();
		break;
	case C_EXCH_ARTEFACT:
		exchangeArtefact();
		break;
	case C_EXCH_BASEUNIT:
		exchangeBaseUnits();
		break;
	default:
		break;
	}
}

void Engine::exchangeUnits()
{
	uchar idLord1 = _server->readChar();
	uchar idUnit1 = _server->readChar();
	uchar idLord2 = _server->readChar();
	uchar idUnit2 = _server->readChar();
	GenericLord * lord1 = 0;
	GenericLord * lord2 = 0;
	
	if( idLord1 ) {
		lord1 = _currentPlayer->getLordById( idLord1 );
	}
	if( idLord2 ) {
		lord2 = _currentPlayer->getLordById( idLord2 );
	}

	/// XXX: check player of lord ?
	if( lord1 && lord2 ) {
		GenericFightUnit * unit1 = lord1->getUnit( idUnit1 );
		GenericFightUnit * unit2 = lord2->getUnit( idUnit2 );

		if( unit1 ) {
			if( unit2 ) {
				if( ( unit1->getRace() == unit2->getRace() ) &&
					unit1->getLevel() == unit2->getLevel() ) {
					unit2->addNumber( unit1->getNumber() );
					lord1->setUnit( idUnit1, 0 );
					delete unit1;
				} else {
					lord1->setUnit( idUnit1, unit2 );
					lord2->setUnit( idUnit2, unit1 );
				}
			} else {
				lord2->setUnit( idUnit2, unit1 );
				lord1->setUnit( idUnit1, 0 );
			}
			_server->sendExchangeUnit( lord1, idUnit1, lord2, idUnit2 );
		}
	}
}

void Engine::exchangeArtefact()
{
	uchar idLord1 = _server->readChar();
	int item = _server->readInt();
	uchar idLord2 = _server->readChar();
	
	GenericLord * lord1 = 0;
	GenericLord * lord2 = 0;
	
	if( idLord1 ) {
		lord1 = _currentPlayer->getLordById( idLord1 );
	}
	if( idLord2 ) {
		lord2 = _currentPlayer->getLordById( idLord2 );
	}
	
	if( lord1 && lord2 ) {
		ArtefactManager * manag1 = lord1->getArtefactManager();
		ArtefactManager * manag2 = lord2->getArtefactManager();
		
		GenericLordArtefact * artefact = manag1->getArtefact( item );
		
		manag1->removeArtefact( item );
		manag2->addArtefact( artefact );
		
		_server->sendExchangeArtefact( lord1, item, lord2 );
	}
}

void Engine::exchangeBaseUnits()
{
	uchar idBase = _server->readChar();
	uchar idUnit1 = _server->readChar();
	uchar idLord = _server->readChar();
	uchar idUnit2 = _server->readChar();

	bool noex = false;	
	GenericBase * base = _currentPlayer->getBaseById( idBase );
	GenericLord * lord = 0; 
	if(idLord && idLord<255){
		lord = _currentPlayer->getLordById( idLord );
	}
	GenericFightUnit * uni1 = 0;
	GenericFightUnit * uni2 = 0;

	if(base){
		if(idUnit1<=MAX_UNIT){
			uni1 = base->getUnit( idUnit1);
		}
		if(lord){
			if(idUnit2<=MAX_UNIT){
				uni2 = lord->getUnit( idUnit2 );
			}
			if( uni1 && uni2 ) {
				if( uni1->getCreature() != uni2->getCreature() ) {
					lord->setUnit( idUnit2, uni1 );
					base->setUnit( idUnit1, uni2 );
				} else {
					uni2->addNumber( uni1->getNumber() );
					base->setUnit( idUnit1, 0 );
					delete uni1;
				}
			} else if (!uni1) {
				if(lord->countUnits() > 1){
					lord->setUnit( idUnit2, 0 );
					base->setUnit( idUnit1, uni2 );
				} else {
					noex = true;
				}
			} else if (!uni2) {
				lord->setUnit( idUnit2, uni1 );
				base->setUnit( idUnit1, 0 );
			}
		} else {
			if(idUnit2<=MAX_UNIT){
				uni2 = base->getUnit( idUnit2);
			}
		if( uni1 && uni2 ) {
			if( uni1->getCreature() != uni2->getCreature() ) {
					base->setUnit( idUnit2, uni1 );
					base->setUnit( idUnit1, uni2 );
				} else {
					uni1->addNumber( uni2->getNumber() );
					base->setUnit( idUnit2, 0 );
					delete uni2;
				}
			} else {
				base->setUnit( idUnit2, uni1 );
				base->setUnit( idUnit1, 0 );
			}
		}	
		if(noex == false) 
			_server->sendExchangeBaseUnitCl(base,idUnit1,lord,idUnit2);
	}
}

void Engine::handleInGameModif()
{
	switch( _server->getCla2() ) {
	case C_MOD_MAP:
	case C_MOD_CELL:
	case C_MOD_PLAYER:
		logDD( "Not yet implemented" );
		break;
	case C_MOD_BASE:
		handleInGameModifBase();
		break;
	case C_MOD_BUILD:
		logDD( "Not yet implemented" );
		break;
	case C_MOD_LORD:
		handleInGameModifLord();
		break;
	}
}

void Engine::handleInGameModifBase()
{
	uint index;
	GenericPlayer * tempPlayer; 

	switch( _server->getCla3() ) {
	case C_BASE_BUILDING: {
		int id = _server->readChar();
		int building = _server->readChar();
		bool isBuy = _server->readChar();
	

		GenericBase * base = _currentPlayer->getBaseById( id );
		if(base) {
			GenericBaseModel * baseModel = DataTheme.bases.at( base->getRace() );
			InsideBuildingModel * buildingModel = baseModel->getBuildingModel( building );
			GenericInsideBuilding * tmpbuil = base->getBuildingByType( building );

			if(isBuy==true && !tmpbuil){
				if( base->canBuild() && _currentPlayer->canBuy( buildingModel ) ) {
					for( uint i = 0; i < DataTheme.resources.count(); i++ ) {
						if( buildingModel->getCost( i ) != 0 ) {
							_currentPlayer->decreaseResource( i, buildingModel->getCost( i ) );
							_server->sendPlayerResource( i, _currentPlayer->getResource( i ) );
						}
					}
					base->addBuilding( building );
					base->canBuild( false );
					for( index = 0; index < _players.count(); index++ ) {
						tempPlayer = _players.at( index );
						if( tempPlayer->canSee( base->getCell() ) ) {
							_server->updateBaseBuilding( base, base->getBuildingByType( building ), tempPlayer );
						}
					}
				}
			} else if(isBuy == false && tmpbuil) {
				if( base->canSell() && (DataTheme.bases.at( tmpbuil->getRace() )->getBuildingModel(tmpbuil->getLevel() )->getAction()->getType() !=INSIDE_VILLAGE)){
					for( uint i = 0; i < DataTheme.resources.count(); i++ ) {
						if( buildingModel->getCost( i ) != 0 ) {
							_currentPlayer->increaseResource( i, buildingModel->getCost( i )/2 );
							_server->sendPlayerResource( i, _currentPlayer->getResource( i ) );
						}
					}
					base->removeBuilding( tmpbuil );
					base->canSell( false );
					for( index = 0; index < _players.count(); index++ ) {
						tempPlayer = _players.at( index );
						if( tempPlayer->canSee( base->getCell() ) ) {
							_server->updateBaseBuilding( base , tmpbuil , tempPlayer);
						}
					}
				}
			}
		}

		} break;
	case C_BASE_UNIT: {
		int row = _server->readInt();
		int col = _server->readInt();
		if( _map->at( row, col )->getBase() ) {
			GenericBase * base = _map->at( row, col )->getBase();
			uchar race = _server->readChar();
			uchar level = _server->readChar();
			int number = _server->readInt();
			Creature * creature = DataTheme.creatures.at( race, level );
			if( base->getCreatureProduction( creature ) >= number  && number != 0) {
				if( _currentPlayer->canBuy( creature, number ) ) {
					_currentPlayer->buy( creature, number );
					base->addGarrison( creature, number );
					base->buyCreature( creature, number );
					_server->sendPlayerResources( _currentPlayer );
					_server->sendBaseUnit( base, creature, number );
				}
			}
		}
		} break;
	case C_BASE_MARKET: {
		//XXX : not full implemented
		int firstRes = _server->readInt();
		int secondRes = _server->readInt();
		int value = _server->readInt();
		int cupr;
		
		if(firstRes != secondRes){
			PriceMarket * realPrice = _currentPlayer->getPriceMarket();
			if(realPrice->getResourcePrice( firstRes ) >= realPrice->getResourcePrice( secondRes )){

				cupr= realPrice->getResourceInResource( firstRes,secondRes );

				//logDD("firstRes %d, secondRes %d, value %d, cupr %d",firstRes,secondRes,value, cupr);
				_currentPlayer->increaseResource( secondRes, value*cupr);
				_currentPlayer->decreaseResource( firstRes, value);
			} else {
				cupr= realPrice->getResourceInResource( secondRes,firstRes );

				//logDD("firstRes %d, secondRes %d, value %d, cupr",firstRes,secondRes,value, cupr);
				_currentPlayer->increaseResource( secondRes, value);
				_currentPlayer->decreaseResource( firstRes, value*cupr);

			}
			_server->sendPlayerResource( firstRes, _currentPlayer->getResource(firstRes));
			_server->sendPlayerResource( secondRes, _currentPlayer->getResource(secondRes));
		}
		} break;
	default:
		logDD( "Not yet implemented" );
		break;
	}
}

void Engine::handleInGameModifLord()
{
	switch( _server->getCla3() ) {
	case C_LORD_GARRISON: 
		handleInGameModifLordGarrison();
		break;
	case C_LORD_UNIT: 
		handleInGameModifLordUnit();
		break;
	default:
		logDD( "Not yet implemented (modif lord)" );
		break;
	}
}

void Engine::handleInGameModifLordGarrison()
{
	uchar idLord = _server->readChar();
	bool garrison = ( _server->readChar() == (uchar)1 );
	GenericLord * lord = _currentPlayer->getLordById( idLord );
	
	/// XXX: test if it is possible...
	if( lord && lord->getCell()->getBase() != 0 ) {
		lord->setVisible( !garrison );
		GenericBase * base = lord->getCell()->getBase();
		if( ( lord == base->getGarrisonLord() ) ||
		( lord == base->getVisitorLord() ) ) {
			if( ( garrison && ( lord == base->getVisitorLord() ) )
			|| ( ( lord == base->getGarrisonLord() ) && !garrison ) ) {
				base->exchangeLords();
				_server->setGarrison( lord, garrison );
			}
		} else if( ( base->getGarrisonLord() == 0 ) || ( base->getVisitorLord() == 0 ) ) {
			if( garrison ) {
				if( base->getGarrisonLord() ) {
					base->exchangeLords();
				}
				base->setGarrisonLord( lord );
				_server->setGarrison( lord, true );
			} else {
				if( base->getVisitorLord() ) {
					base->exchangeLords();
				}
				base->setVisitorLord( lord );
				_server->setGarrison( lord, false );
			}
		}
	}
}

void Engine::handleInGameModifLordUnit()
{
	uchar id = _server->readChar();
	uchar num = _server->readChar();
	uchar race = _server->readChar();
	uchar level = _server->readChar();
	uint nb = _server->readInt();
	uchar move = _server->readChar();
	int health = _server->readInt();
	
	GenericLord * lord = 0; 
	GenericFightUnit * unit = 0;
		
	if( id && id < 255 ) {
		lord = _currentPlayer->getLordById( id );
	}

	unit =lord->getUnit( num );
	if( ! unit ) {
		unit = new GenericFightUnit();
		unit->setCreature( race, level );
		unit->setMove( move );
		unit->setHealth( health );
	}
	if( unit->getNumber() < nb ) {
		/* no cheat possible */
		return;
	}

	if( nb == 0 ){
		if( lord->countUnits() > 1 ) {
			unit->setNumber( nb );
		}
	} else {
		unit->setNumber( nb );
	}

	lord->setUnit( num, unit );
	_server->updateUnit( lord, num );
	if( unit->getNumber() == 0 ){
		delete unit;
		unit = 0;
		lord->setUnit( num, unit );
	}	
}

bool Engine::handleBuildingEnter( GenericBuilding * building, GenericLord * lord )
{
	bool enter = false;
	QPtrList<Action> list;

	if( ! building->hasBeenVisited() ) {
		if( building->getCondition() ) {
		}
		enter = true;
		list = building->getActionList( Action::FIRSTTIME );

		if( list.isEmpty() ) {
			list = building->getActionList( Action::FIRSTTIMELORD );
		}
		if( list.isEmpty() ) {
			list = building->getActionList( Action::NEXTTIME );
		}
		for( uint i = 0; i < list.count(); i++ ) {
			handleBuildingAction( list.at( i ), lord );
		}
		building->enter( lord );
	} else if( ! building->hasBeenVisited( lord ) ) {
		if( building->getCondition() ) {
		}
		enter = true;
		list = building->getActionList( Action::FIRSTTIMELORD );
		if( list.isEmpty() ) {
			list = building->getActionList( Action::NEXTTIME );
		}
		for( uint i = 0; i < list.count(); i++ ) {
			handleBuildingAction( list.at( i ), lord );
		}
		building->enter( lord );
	} else {
		if( building->getCondition() ) {
		}
		enter = true;
		list = building->getActionList( Action::NEXTTIME );
		for( uint i = 0; i < list.count(); i++ ) {
			handleBuildingAction( list.at( i ), lord );
		}
		building->enter( lord );
	}
	return enter;
}

void Engine::handleBuildingAction( Action * action, GenericLord * lord )
{
	for( int i = 0; i < action->getElementaryNumber(); i++ ) {
		ElementaryAction * _elementary = action->getElementaryAction( i );

		switch( _elementary->getType() ) {
			case ElementaryAction::ATTACK:
				lord->increaseBaseCharac( ATTACK, _elementary->getCoeff() );
				_server->sendLordCharac( _currentPlayer, lord, ATTACK );
				break;
			case ElementaryAction::DEFENSE:
				lord->increaseBaseCharac( DEFENSE, _elementary->getCoeff() );
				_server->sendLordCharac( _currentPlayer, lord, DEFENSE );
				break;
			case ElementaryAction::POWER:
				lord->increaseBaseCharac( POWER, _elementary->getCoeff() );
				_server->sendLordCharac( _currentPlayer, lord, POWER );
				break;
			case ElementaryAction::KNOWLEDGE:
				lord->increaseBaseCharac( KNOWLEDGE, _elementary->getCoeff() );
				_server->sendLordCharac( _currentPlayer, lord, KNOWLEDGE );
				break;
			case ElementaryAction::MOVE:
				lord->increaseBaseCharac( MOVE, _elementary->getCoeff() );
				_server->sendLordCharac( _currentPlayer, lord, MOVE );
				break;
			case ElementaryAction::MAXMOVE:
				lord->increaseBaseCharac( MAXMOVE, _elementary->getCoeff() );
				_server->sendLordCharac( _currentPlayer, lord, MAXMOVE );
				break;
			case ElementaryAction::TECHNICPOINT:
				lord->increaseBaseCharac( TECHNICPOINT, _elementary->getCoeff() );
				_server->sendLordCharac( _currentPlayer, lord, TECHNICPOINT );
				break;
			case ElementaryAction::MAXTECHNICPOINT:
				lord->increaseBaseCharac( MAXTECHNICPOINT, _elementary->getCoeff() );
				_server->sendLordCharac( _currentPlayer, lord, MAXTECHNICPOINT );
				break;
			case ElementaryAction::MORALE:
				lord->increaseBaseCharac( MORALE, _elementary->getCoeff() );
				_server->sendLordCharac( _currentPlayer, lord, MORALE );
				break;
			case ElementaryAction::LUCK:
				lord->increaseBaseCharac( LUCK, _elementary->getCoeff() );
				_server->sendLordCharac( _currentPlayer, lord, LUCK );
				break;
			case ElementaryAction::VISION:
				lord->increaseBaseCharac( VISION, _elementary->getCoeff() );
				_server->sendLordCharac( _currentPlayer, lord, VISION );
				break;
			case ElementaryAction::EXPERIENCE:
				lord->increaseBaseCharac( EXPERIENCE, _elementary->getCoeff() );
				_server->sendLordCharac( _currentPlayer, lord, EXPERIENCE );
				/// XXX: check exp for level...
				break;
			case ElementaryAction::SKILL:
				break;
			case ElementaryAction::ARTEFACT:
				break;
			case ElementaryAction::UNIT:
				break;				
			default:
				break;
		}
	}
}

void Engine::handleInGameTurn()
{
	switch( _server->getCla2() ) {
		case C_TURN_PLAY:
			logEE( "Should not happen (Server : SO_TURN/C_TURN_PLAY)" );
			break;
		case C_TURN_LORD:{
			int type = _server->readInt();
			if( type == 1 ) {
				_counter++;
			}
			_server->sendLordPlTurn( _counter );
			}
			break;
		case C_TURN_PLORD:
			logEE( "Should not happen (Server : SO_TURN/C_TURN_PLORD)" );
			break;
		case C_TURN_END:
			nextPlayer();
			break;
	}
}

void Engine::stateInFight( int num )
{
	if( _fight ) {
		switch( _server->getCla1() ) {
		case SO_MSG:
			handleMessage();
			break;
		case SO_FIGHT:
			_fight->handleSocket( _players.at( num ) );
			break;
		default:
			logEE( "Should not happen" );
			break;
		}
	} else {
		logEE( "Should not happen, _fight == 0" );
	}
}

void Engine::slot_newPlayer( AttalPlayerSocket * player )
{
	_players.append( player->getPlayer() );
}

void Engine::reinit()
{
	_state = NOT_PLAYING;
	GameData::reinit();

	//XXX: to change !! (cf DataTheme)
	//_lords.clear();
	//_lords.init();
	// XXX: logEE( "lords not reinitialized" );

	_server->clear();
}

void Engine::startCampaign()
{
	if( _campaign ) {
		QString filename;
		uint nbScen = _campaign->getScenarioNumber();
		
		for( uint i = 0; i < nbScen; i++ ) {
			filename = _campaign->getScenario( i );
			if( loadGame( CAMPAIGN_PATH + filename , false) ) {
				startGame();
			}
		}
	}
}

void Engine::startGame()
{
	_isFinished = false;
	if( _players.count() > _server->getNbSocket() ) {
		return;
	}

	uint i, j;

	_state = IN_GAME;
	_server->startGame();
	_calendar->reinit();
	//_calendar->newDay();

	_server->sendSizeMap( _map->getHeight(), _map->getWidth() );

	for( i = 0; i < _players.count(); i++ ) {
		_currentPlayer = _players.at( i );
		_currentPlayer->setMap( _map );
		_currentPlayer->setNum( i );
		_currentPlayer->setAlive( true );

		_currentPlayer->initMapVision();
		_server->setCurrentPlayer( _currentPlayer );
		_server->sendGameInfoPlayer();
		for( j = 0; j < _currentPlayer->numLord(); j++ ) {
			QPtrList<GenericCell> removed;
			QPtrList<GenericCell> added = _currentPlayer->addLordVision( _currentPlayer->getLord( j ) );
			//updateMapVision( removed, added );
			//_server->newLord( _currentPlayer->getLord( j ) );

		}
		for( j = 0; j < _currentPlayer->numBase(); j++ ) {
			QPtrList<GenericCell> removed;
			QPtrList<GenericCell> added = _currentPlayer->addBaseVision( _currentPlayer->getBase( j ) );
			updateMapVision( removed, added );
			//_server->ownBase( _currentPlayer->getBase( j ), _currentPlayer );
		}

		for( j = 0; j < _currentPlayer->numBuilding(); j++ ) {
			QPtrList<GenericCell> removed;
			QPtrList<GenericCell> added = _currentPlayer->addBuildingVision( _currentPlayer->getBuilding( j ) );
			updateMapVision( removed, added );
			//_server->ownBuilding( _currentPlayer->getBuilding( j ), _currentPlayer );
		}

		for( j = 0; j < DataTheme.resources.count(); j++ ) {
			_server->sendPlayerResource( j, _currentPlayer->getResource( j ) );
		}

		for( uint col = 0; col < _map->getWidth(); ++col ) {
			for( uint row = 0; row < _map->getHeight(); ++row ) {
				if(_map->at( row, col )->getBase()){
					_map->at( row,col )->getBase()->getInitPopul();
				}
				if( _currentPlayer->canSee( row, col ) ) {
					updateCellVision( _map->at( row, col ) );
				}
			}
		}
		updatePlayerPrices(_currentPlayer);
		uint nbRes = DataTheme.resources.count();
		int price;
		for( uint i = 0; i < nbRes; i++ ) {
			price = _currentPlayer->getPriceMarket()->getResourcePrice( i );
			_server->sendPlayerPrice( i, price );
		}
	}
	_server->sendAskNone( getScenarioDescription(), true );
	
	newDay();
	_currentPlayer = _players.at( 0 );
	_server->setCurrentPlayer( _currentPlayer );
	_currentPlayer->newWeek();
	_currentPlayer->newTurn();
	_server->beginTurn();
	/// XXX: a good idea could be to set a game loop here !! (beware of not bypassing qt event loop)
	while( ! _isFinished ) {
		qApp->processEvents();
	}
}

void Engine::endGame()
{
	_isFinished = true;
	_state = NOT_PLAYING;
	_server->endGame();
}

void Engine::startFight()
{
	if( !_fight ) {
		_fight = new FightEngine( _server );
	}
	_state = IN_FIGHT;
	_fight->init( _players.at( 0 ), _players.at( 0 )->getLord( 0 ), _players.at( 1 ), _players.at( 1 )->getLord( 0 ) );
}

void Engine::startFight( int lordAttack, GenericLord *lordDefense )
{
	if( !_fight ) {
		_fight = new FightEngine( _server );
		QObject::connect( _fight, SIGNAL( sig_endFight( char ) ), SLOT( slot_endFight( char ) ) );
	}
	_fight->setDefendCell( lordDefense->getCell() );
	_state = IN_FIGHT;
	_fight->init( _currentPlayer, _currentPlayer->getLordById( lordAttack ), lordDefense->getOwner(), lordDefense );
}

void Engine::startFight( int lordAttack, GenericMapCreature * creature )
{
	if( !_fight ) {
		_fight = new FightEngine( _server );
		QObject::connect( _fight, SIGNAL( sig_endFight( char ) ), SLOT( slot_endFight( char ) ) );
	}
	_state = IN_FIGHT;
	_fight->setDefendCell( creature->getCell() );
	GenericFightUnit * units[MAX_UNIT];
	for( int i = 0; i < MAX_UNIT; i++ ) {
		if( creature->getStack( i ) > 0 ) {
			units[i] = new GenericFightUnit();
			units[i]->setCreature( creature->getCreature() );
			units[i]->setNumber( creature->getStack( i ) );
		} else {
			units[i] = 0;
		}
	}
	
	_fight->init( _currentPlayer, _currentPlayer->getLordById( lordAttack ), units, (GameData *)this );

	/// XXX: destroy units...
	logDD( "start fight finished" );
}

void Engine::slot_endFight( char result )
{
	//XXX: Handle result (flee, and so on...)
	GenericLord * loser, * winner;

	if( ( result & FIGHTER_DEFENSE_WIN ) == FIGHTER_DEFENSE_WIN ) {
		loser = _fight->getAttackLord();
		winner = _fight->getDefendLord();
	} else {
		loser = _fight->getDefendLord();
		winner = _fight->getAttackLord();
	}
	
	if( _isCreature ) {
		if( loser == _fight->getDefendLord() ) {
			GenericCell * cell = _fight->getDefendCell();
			_server->sendCreatureRemove( cell );
			GenericMapCreature * creature = cell->getCreature();
			cell->setCreature( 0 );
			_creatures.remove(creature);
			
			uint experience = _fight->getExperience( winner );
			manageIncreaseExperience( winner, experience );
		} else {
			_server->sendLordRemove( loser );
			loser->removeFromGame();
		}
	} else {
		_server->sendLordRemove( loser );
		loser->removeFromGame();
		uint experience = _fight->getExperience( winner );
		manageIncreaseExperience( winner, experience );
	}

	_state = IN_GAME;
	if(_fight){
		delete _fight;
	}
	_fight = 0;
}

void Engine::manageIncreaseExperience( GenericLord * lord, uint experience )
{
	uint currentLevel, nextLevelExp;
		
	currentLevel = DataTheme.lordExperience.computeLevelForExperience( lord->getCharac( EXPERIENCE ) );
		
	while( experience > 0 ) {
		if( currentLevel < DataTheme.lordExperience.getLevelNumber() ) {
			nextLevelExp = DataTheme.lordExperience.getLevel( currentLevel + 1 );
			nextLevelExp -= lord->getCharac( EXPERIENCE );
			
			if( experience >= nextLevelExp ) {
				lord->increaseBaseCharac( EXPERIENCE, experience - nextLevelExp );
				GenericLordModel * model = DataTheme.lords.at( lord->getId() );
				LordCharac charac = model->getCategory()->getRandomEvolution();
				lord->increaseBaseCharac( charac, 1 );
				_server->sendLordCharac( lord->getOwner(), lord, charac );
				currentLevel++;
				experience -= nextLevelExp;
			} else {
				lord->increaseBaseCharac( EXPERIENCE, experience );
				_server->sendLordCharac( lord->getOwner(), lord, EXPERIENCE );
				experience = 0;
			}				
		} else {
			experience = 0;
		}
	}
}

bool Engine::saveGame( const QString & filename )
{
    	QFile f( filename );

	if (! f.open(IO_WriteOnly) ) {
		logEE("Could not open file %s for writing\n", filename.latin1() );
		return false;
	}
	
	QTextStream ts( &f );
	GameData::save( &ts );
	f.close();
	
	return true;
}

void Engine::newDay()
{
	uint index;
	GenericPlayer * tempPlayer; 

	_calendar->newDay();
	if( _calendar->getDay() == 1 ) {
		// new week
		uint nbItems = _bases.count();
		for( uint i = 0; i < nbItems; ++i ) {
			_bases.at( i )->initCreatureProduction();
			_bases.at( i )->getPopGrowth();
			for( index = 0; index < _players.count(); index++ ) {
				tempPlayer = _players.at( index );
				if( tempPlayer->canSee( _bases.at( i )->getCell() ) ) {
					_server->sendBasePopulation( _bases.at( i ) , tempPlayer );
				}
			}
		}
		nbItems = _creatures.count();
		for( uint i = 0; i < nbItems; ++i ) {
			_creatures.at( i )->grow();
			_server->updateCreature( _creatures.at( i ) ); /// XXX:  we could update only if 'category' has changed...
		}
	}
}


void Engine::nextPlayer()
{
	if( ! enoughPlayers() ) {
		endGame();
		return;
	}

	checkMainQuest();
	_counter=0;
	uint num = _players.find( _currentPlayer );
	num++;
	if( num == _players.count() ) {
		_currentPlayer = _players.at( 0 );
		newDay();//_calendar->newDay();
	} else {
		_currentPlayer = _players.at( num );
	}
  
	/// XXX: could happen ?
	if( ! _currentPlayer->isAlive() ) {
		nextPlayer();
		return;
	}
	if( ( _currentPlayer->numBase() == 0 ) and ( _currentPlayer->numLord() == 0 ) ) {
		_currentPlayer->setAlive( false );
		_server->playerLose( _currentPlayer );
		nextPlayer();
		return;
	}
	_server->setCurrentPlayer( _currentPlayer );
	uint i;

	for( i = 0; i < _currentPlayer->numBuilding(); i++ ) {
		handleBuildingTurn( _currentPlayer, _currentPlayer->getBuilding( i ) );
	}
	for( i = 0; i < _currentPlayer->numBase(); i++ ) {
		handleBaseTurn( _currentPlayer, _currentPlayer->getBase( i ) );
	}
	for( i = 0; i < _currentPlayer->numLord(); i++ ) {
		handleCreatTurn( _currentPlayer, _currentPlayer->getLord( i ) );
	}
	_currentPlayer->newTurn();
	_server->beginTurn();
	checkMainQuest();
	//checkPlayerQuests();
}

void Engine::handleBuildingTurn( GenericPlayer * player, GenericBuilding * building )
{
	QPtrList<Action> list = building->getActionList( Action::DATE );
	for( uint i = 0; i < list.count(); i++ ) {
		handleBuildingAction( list.at( i ), player );
	}
}

void Engine::handleBaseTurn( GenericPlayer * player, GenericBase * base )
{
	
	for( uint i = 0; i < base->getBuildingCount(); i++ ) {
		GenericInsideBuilding * inbuil = base->getBuilding(i);
		for( uint j = 0; j < DataTheme.resources.count(); j++ ) {
			player->decreaseResource( j,inbuil->getMantCost(j));
			_server->sendPlayerResource( j, player->getResource(j)  );
		}
	}
	for( uint j = 0; j < DataTheme.resources.count(); j++ ) {
		player->increaseResource( j, base->collectRessource(j));
		_server->sendPlayerResource( j, player->getResource(j)  );
	}
}

void Engine::handleCreatTurn( GenericPlayer * player, GenericLord * lord )
{
	for( uint i = 0; i < MAX_UNIT; i++ ) {
		GenericFightUnit * unit = lord->getUnit(i);
		if( unit ) {
			for( uint j = 0; j < DataTheme.resources.count(); j++ ) {
				if(unit->getMantCost (j)) {
					int mant = ( unit->getMantCost( j ) * unit->getNumber() );
#if 0
					int ress = player->getResource(j);
					uint decr = (mant - ress) / unit->getMantCost( j ) ;
					decr = decr > 0 ? decr : 0;
					if (decr > 0) {
						decr = decr > unit->getNumber() ? unit->getNumber() : unit->getNumber()-decr;
						unit->setNumber(decr);
						lord->setUnit(i,unit);
					}
#endif	
					player->decreaseResource( j, mant );
					_server->sendPlayerResource( j, player->getResource( j )  );
				}
			}
		}
	}
	//_server->updateUnits(lord);
}

void Engine::handleBuildingAction( Action * action, GenericPlayer * player )
{
	int ress;
	for( int i = 0; i < action->getElementaryNumber(); i++ ) {
		ElementaryAction * elementary = action->getElementaryAction( i );
		switch( elementary->getType() ) {
			case ElementaryAction::RESSOURCE:
				player->setResource( elementary->getArg(), player->getResource( elementary->getArg() ) + elementary->getCoeff() );
				_server->sendPlayerResource( elementary->getArg(), player->getResource( elementary->getArg() ) );
				break;
			case ElementaryAction::RANDRESSOURCE:
				ress = getRandResource(elementary->getCoeff());
				player->setResource( elementary->getArg(), player->getResource( elementary->getArg() ) + ress );
				_server->sendPlayerResource( elementary->getArg(), player->getResource( elementary->getArg() ) );
				break;
			default:
				break;
		}
	}
}

bool Engine::loadCampaign( const QString & filename )
{
	bool ret = true;
	
	if( _campaign ) {
		delete _campaign;
	}
	_campaign = new Campaign();
	
	CampaignParser handler( _campaign );
	QFile file( filename );
	QXmlInputSource source( file );
	QXmlSimpleReader reader;
	reader.setContentHandler( &handler );
	reader.setErrorHandler( &handler );
	bool ok = reader.parse( source );
	file.close();
	if( ! ok ) {
		delete _campaign;
		_campaign = 0;
		logEE( "Parse Error (%s) : %s", filename.latin1(), handler.errorProtocol().latin1() );
		ret = false;
	}
	
	return ret;
}

bool Engine::loadGame( const QString & filename, bool silent )
{
	for( uint i = 0; i < _players.count(); i++ )
	{
		_currentPlayer = _players.at( i );
		_currentPlayer->cleanData();
	}

	reinit();

	ScenarioParser handler( (GameData*)this );
	QFile file( filename );
	QXmlInputSource source( file );
	QXmlSimpleReader reader;
	reader.setContentHandler( &handler );
	reader.setErrorHandler( &handler );
	bool ok = reader.parse( source );
	file.close();
	if ( !ok ) {
		logEE( "Parse Error (%s) : %s", filename.latin1(), handler.errorProtocol().latin1() );
		return false;
	}

	if( _players.count() != (uint) _nbPlayer ) {
		if(!silent){
			QMessageBox::critical( 0, tr( "Error with scenario" ), tr( "Wrong number of connected players" ) );
			for( uint i = 0; i < _players.count(); i++ ) {
				_currentPlayer = _players.at( i );
				logDD( "replayers %d ", _players.count());
				logDD( "relords %d ", _currentPlayer->numLord()  );
			}
		}
		_tmpPlay = _nbPlayer - _players.count();
		_state = NOT_PLAYING;
		GameData::reinit();
		return false;
	} else {
		return true;
	}
}

uint Engine::getAlivePlayersNumber()
{
	uint ret = 0;

	for( uint i = 0; i < _players.count(); i++ ) {
		if( _players.at( i )->isAlive() ) {
			ret++;
		}
	}

	return ret;
}

bool Engine::enoughPlayers()
{
	uint ret = 0;
	uint num = _players.count();

	ret = getAlivePlayersNumber();
	
	if( num == 1 && ret == 1 ) {
		 return true;
	}
	
	if( num > 1 && ret > 1 ) {
		return true;
	}

	return false; 
}

void Engine::updateMapVision( QPtrList<GenericCell> & removed, QPtrList<GenericCell> & added )
{
	for( uint i = 0; i < added.count(); i++ ) {
		GenericCell * cell = added.at( i );
		if( removed.findRef( cell ) == -1 ) {
			updateCellVision( cell );
		}
	}
}

void Engine::updateCellVision( GenericCell * cell )
{
	_server->sendCell( cell );
	if( cell->getLord() ) {
		_server->newLord( cell->getLord() );
	}
	if( cell->getCreature() ) {
		_server->newCreature( cell->getCreature() );
	}
	if( cell->getBase() ) {
		_server->newBase( cell->getBase() );
		if(cell->getBase()->getOwner()){
			_server->ownBase( cell->getBase(), _currentPlayer );
		}
	}
	if( cell->getBuilding() ) {
		_server->newBuilding( cell->getBuilding() );
		if(cell->getBuilding()->getOwner()){
			_server->ownBuilding( cell->getBuilding(), _currentPlayer );
		}
	}
	if( cell->getEvent() ) {
		_server->newEvent( cell->getEvent() );/*
		switch( cell->getEvent()->getType() ) {
		case GenericEvent::EventArtefact:
			_server->newArtefactMap( cell->getEvent()->getArtefact() );
			break;
		}*/
	}/// XXX: other cell attr to send...
}

Engine::CreatureAction Engine::computeCreatureAction( GenericMapCreature * creature, double ratio )
{
	CreatureAction ret = CreatureFight;

	switch( creature->getBehaviour() ) {
		case GenericMapCreature::Obedient:
			ret = CreatureJoin;
			break;
		case GenericMapCreature::Friendly:
			if( ratio < 0.75 ) {
				ret = CreatureFight;
			} else if( ratio < 1.25 ) {
				ret = CreatureMercenary;
			} else {
				ret = CreatureJoin;
			}
			break;
		case GenericMapCreature::Neutral:
			if( ratio < 1.0 ) {
				ret = CreatureFight;
			} else if( ratio < 1.5 ) {
				ret = CreatureFlee;
			} else if( ratio < 2.0 ) {
				ret = CreatureMercenary;
			} else {
				ret = CreatureJoin;
			}
			break;
		case GenericMapCreature::Aggressive:
			if( ratio < 1.5 ) {
				ret = CreatureFight;
			} else if( ratio < 3.0 ) {
				ret = CreatureFlee;
			} else {
				ret = CreatureMercenary;
			}
			break;
		case GenericMapCreature::Hostile:
			if( ratio < 2.5 ) {
				ret = CreatureFight;
			} else {
				ret = CreatureFlee;
			}
			break;
	}

	if( ( ! creature->isFleeing() ) && ( ret == CreatureFlee ) ) {
		ret = CreatureFight;
	}

	return ret;
}

void Engine::checkMainQuest()
{
	QuestData data;
	QuestCondition * failCondition, * successCondition;

	if( not _isFinished ) {
		data.setPlayer( _currentPlayer );
	
		Quest * main = _quests->getMainQuest();
	
		if( main ) {
			failCondition = main->getFailCondition();
			successCondition = main->getSuccessCondition();
	
			uint nbLord = _currentPlayer->numLord();
			for( uint i = 0; i < nbLord; i++ ) {
				data.setLord( _currentPlayer->getLord( i ) );
	
				if( failCondition ) {
					if( failCondition->check( &data ) ) {
						_currentPlayer->setAlive( false );
						_server->playerLose( _currentPlayer );
						if( ! enoughPlayers() ) {
							endGame();
						}
						return;
					}
				}
	
				if( successCondition ) {
					if( successCondition->check( &data ) ) {
						_server->playerWin( _currentPlayer );
						endGame();
						return;
					}
				}
			}
		}
	}
}

int Engine::getRandResource( int coeff)
{
	int ress,tmprand;
	
	tmprand =(int) ((200.0*rand())/(RAND_MAX+1.0));
	ress = (coeff*(100 + (100 - tmprand)))/100;
	//logDD("resource %d , coeff %d , rand %d", ress,coeff,tmprand);
	return ress;
}

void Engine::updatePlayerPrices( GenericPlayer * player)
{

	uint nbases;
	int cupr,tmpr;
	uint nbRes = DataTheme.resources.count();

	PriceMarket * realPrice = player->getPriceMarket();

	cupr = 10000;	

	for( uint i = 0; i < nbRes; i++ ) {
		nbases = player->numBase();
		for( uint j = 0 ; j < nbases ; j++){
			PriceMarket * price = DataTheme.bases.at( player->getBase( j )->getRace())->getPriceMarket();
			tmpr= price->getResourcePrice( i ) ;
			if( tmpr < cupr ) {
				cupr = tmpr;
			}
		}
		if(cupr==10000){
			cupr=1;
		}
		realPrice->setResourcePrice(i,cupr);
		cupr = 10000;
	}

}
