/*
 *  functions.c
 *
 *  Copyright 2009 Arnaud Soyez <weboide@codealpha.net>
 *
 *  This file is part of AudioPreview.
 *
 *  AudioPreview is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  AudioPreview is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with AudioPreview.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib/gi18n-lib.h>
#include <sys/ioctl.h>
#include "functions.h"
#include "ap_config.h"
#include "player.h"

GList* g_list_shuffle (GList *list)
{
  /*
   * Shuffle a GList (only data's) using the Fisher-Yates shuffle.
   * Complexity is O(n)      n = the length of the list.
   * 
   * <http://en.wikipedia.org/wiki/Fisher-Yates_shuffle>
   */
  
  guint len = g_list_length (list);
  
  // We start from the last element.
  GList *last_element = g_list_last (list);
  
  // TODO: Need fix? is 2147483647 streams a long enough playlist?
  if (len > 2147483647) // For g_random_int_range (gint32)
  {
    g_warning (_("Playlist is too long to be shuffled. Skipping shuffle."));
    return list;
  }
  
  g_debug ("%s: Shuffling %u elements.", __FUNCTION__, len);
  while (len > 1 && last_element!=NULL)
  {
    gint32 k = g_random_int_range (0, (gint32)len); // k is in [0, len-1]
    
    if (k != len-1)
    {
      // Swapping k and len-1 datas.
      
      GList *element_k = g_list_nth (list, k);
      gpointer tmp = element_k->data;
      element_k->data = last_element->data;
      last_element->data = tmp;
    }
    
    // We move to the left of the list.
    last_element = g_list_previous (last_element);
    len--;
  }
  
  g_debug ("%s: Shuffling done.", __FUNCTION__);
  return list;
}

/**
 * Check for user keyboard entry.
 * This function waits till the user hits a key,
 * then it will dispatch the key code. This is called by a thread
 * to avoid blocking the program.
 */
gpointer wait_for_key (GSourceFunc dispatch_func)
{
  if (set_tty_flags (TRUE))      /* Try to apply canonical mode to TTY */
    atexit (remove_tty_flags);   /* If applied, then deactivate canonical mode at exit */
  
  gint c;
  do
  {
    // Wait for the key.
    c = fgetc (stdin);
    
    // Dispatch the received key.
    if (dispatch_func != NULL)
      g_idle_add ((GSourceFunc) dispatch_func, g_memdup (&c, sizeof (gint)));
    
  } while (TRUE);
  
  return NULL;
}

/**
 * Activate/Deactivate the canonical mode of a TTY.
 */
gboolean set_tty_flags (gboolean activate)
{
  struct termios ttystate;

  if (activate)
  {
    if (isatty (STDIN_FILENO) && tcgetattr (STDIN_FILENO, &ttystate) != -1)
    {
      ttystate.c_lflag &= ~ICANON; // remove canonical mode
      ttystate.c_lflag &= ~ECHO; // Do not echo input
      ttystate.c_cc[VMIN] = 1; // minimum length to read before sending
      g_debug("Activating new TTY flags.");
      tcsetattr(STDIN_FILENO, TCSAFLUSH, &ttystate); // apply the changes
      return TRUE;
    }
    g_debug("Couldn't set TTY flags.");
    return FALSE;
  }
  else
  {
    g_debug("Cleaning TTY flags.");
    // put old tty flags back
    tcsetattr (STDIN_FILENO, TCSANOW, &ap_config.savedttystate);
    
    return TRUE;
  }
}

/**
 * Deactivate the canoncial mode (used with atexit())
 */
void remove_tty_flags ()
{
  set_tty_flags (0);
}

gboolean key_dispatcher (gint *car)
{
  //~ g_debug ("%s: Dispatching: '%c' (%d)", __FUNCTION__, car, car);
  
  if (car == NULL)
    return FALSE; // to get removed from the main loop
  
  switch (*car)
  {
    // PAUSE
    case ' ':
    {
      player_pause ();
      break;
    }
    
    // SKIP to NEXT stream
    case 'n':
    {
      player_prepare_for_next ();
      break;
    }
    
    // QUIT
    case 'q':
    {
      player_clear_pipeline ();
      done_playing_callback ();
      break;
    }
  }
  
  g_free (car);
  return FALSE; // to get removed from the main loop
}

/**
 * Starts the function wait_for_key in a thread.
 */
gboolean start_thread_wait_for_key ()
{
  GError  *error = NULL;

  g_debug("Starting thread wait for key");
  if (!g_thread_create ((GThreadFunc) wait_for_key, &key_dispatcher, FALSE, &error))
  {
    g_error (_("Thread failed: %s"), error->message);
  }

  return FALSE; // to get removed from the main loop
}

/**
 * Returns the number of columns of the terminal or 0 if fails.
 */
gint get_terminal_width ()
{
  int fd;
  fd = fileno (stderr);
  
  #ifdef TIOCGSIZE
      struct ttysize wsize;
  #elif defined(TIOCGWINSZ)
      struct winsize wsize;
  #else
      return 0;
  #endif
  
  #ifdef TIOCGSIZE
      if (ioctl (fd, TIOCGSIZE, &wsize))
          return 0;
      return wsize.ts_cols;
  #elif defined TIOCGWINSZ
      if (ioctl (fd, TIOCGWINSZ, &wsize))
          return 0;
      return wsize.ws_col;
  #endif
}

gboolean need_colors (gchar* type)
{
  if (strcmp (type, "no") == 0)
    return FALSE;
  if (strcmp (type, "yes") == 0)
    return TRUE;
  if (strcmp (type, "auto") == 0)
    return (isatty(STDOUT_FILENO) && getenv("TERM") && strcmp(getenv("TERM"), "dumb"));
}

/**
 * Updates terminal_cols and starts waiting for signal again.
 */
void terminal_size_changed (gint s)
{
  ap_config.terminal_cols = get_terminal_width();
  g_debug ("%s: Received SIGWINCH cols=%d",
           __FUNCTION__, ap_config.terminal_cols);
  //~ signal (SIGWINCH, terminal_size_changed);
}
