/*
 *  player.c
 *
 *  Copyright 2009 Arnaud Soyez <weboide@codealpha.net>
 *
 *  This file is part of AudioPreview.
 *
 *  AudioPreview is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  AudioPreview is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with AudioPreview.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gio/gio.h>
#include <glib/gi18n-lib.h>
#include "player.h"
#include "stream.h"

static gint64   dur;
static gboolean loop_stream_was_played = FALSE; ///< endless loop prevention flag

/**
 * Capture all the bus calls sent by gstreamer.
 */
gboolean bus_call (GstBus *bus, GstMessage *msg, gpointer user_data)
{
  switch (GST_MESSAGE_TYPE (msg))
  {
    case GST_MESSAGE_EOS:
    {
      stream_print_info (*currentstream);
      g_debug ("%s: End-of-Stream", __FUNCTION__);
      player_prepare_for_next ();
      break;
    }
    
    case GST_MESSAGE_ERROR:
    {
      GError *err;
      
      gst_message_parse_error (msg, &err, NULL);
      
      g_warning ("Player: %s. Skipping...", err->message);
      g_error_free (err);
      
      player_prepare_for_next ();
      
      break;
    }
    
    case GST_MESSAGE_WARNING:
    {
      GError *err;
      
      gst_message_parse_warning (msg, &err, NULL);
      
      g_warning ("Player: %s", err->message);
      g_error_free (err);
      
      break;
    }
    
    case GST_MESSAGE_STATE_CHANGED:
    {
      GstState oldstate, newstate, targetstate;
      gst_message_parse_state_changed (msg, &oldstate, &newstate, &targetstate);
      
      /* Sort all the bus messages: we need our playbin and only when the state
       * changes.
       */
      
      //~ if (newstate == GST_STATE_NULL)
        //~ g_debug ("NULL STATE !");
      
      if (msg->src != GST_OBJECT(pipeline) )
        break;
      
      g_debug ("%s: changed state: \t%s\t-->  %s\t(target: %s)", __FUNCTION__,
          gst_element_state_get_name (oldstate),
          gst_element_state_get_name (newstate),
          gst_element_state_get_name (targetstate));
      
      if (newstate == GST_STATE_READY)
      {
        /* This happens when:
         *  - when we need to play the next stream in queue.
         *    (NULL -> READY)
         * 
         * We ask to play the next stream in queue.
         */
        g_debug ("%s: playing next because GST_STATE_READY", __FUNCTION__);
        play_next ();
      }
      else if (oldstate == GST_STATE_READY && newstate == GST_STATE_PAUSED)
      {
        /* This happens when:
         *  - stream is getting ready to play (READY->PAUSED)
         * 
         * We need to seek/forward to the wanted position (ap_config.position).
         */
        player_seek ();
        stream_add_updater (*currentstream, 
                            (GSourceFunc) player_update_stream);
      }
      else if (oldstate != GST_STATE_PLAYING && newstate == GST_STATE_PLAYING)
      {
        /* This happens when:
         *  - stream has just started playing (PAUSED->PLAYING)
         *  - stream has resumed from user-requested-pause (PAUSED->PLAYING)
         *  - (maybe) when buffer was empty.
         */
          
        g_debug ("%s: Playing!", __FUNCTION__);
        
        if (!loop_stream_was_played)
          loop_stream_was_played = TRUE;
        
        stream_print_info (*currentstream);
      }
      else if (oldstate == GST_STATE_PLAYING && newstate == GST_STATE_PAUSED)
      {
        g_debug ("%s: Paused!", __FUNCTION__);
        
        stream_print_info (*currentstream);
      }
      
      break;
    }
    
    case GST_MESSAGE_TAG:
    {
      /* This happens when:
       *  - Title/Artists/... info is received/parsed from stream.
       */
      
      GstTagList *taglist;
      gchar *tmpchar;
      
      gst_message_parse_tag (msg, &taglist);
      
      if (gst_tag_list_get_string (taglist, GST_TAG_ARTIST, &tmpchar))
      {
        g_debug ("%s: Artist:\t%s", __FUNCTION__, tmpchar);
        stream_set_artist (*currentstream, tmpchar);
        g_free (tmpchar);
      }
      
      if (gst_tag_list_get_string (taglist, GST_TAG_TITLE, &tmpchar))
      {
        g_debug ("%s: Title:\t%s", __FUNCTION__, tmpchar);
        stream_set_title (*currentstream, tmpchar);
        g_free (tmpchar);
      }
      
      stream_print_info (*currentstream);
      gst_tag_list_free (taglist);
      break;
    }
    
    case GST_MESSAGE_BUFFERING:
    {
      /* This happens when:
       *  - stream is buffering...
       */
      gint perc;
      gst_message_parse_buffering (msg, &perc);
      stream_set_buffering_perc (*currentstream, perc);
      stream_print_info (*currentstream);
      
      break;
    }
    
    default:
      break;
  }
  
  return TRUE;
}

/**
 * Play the next stream in the playlist. (asynchronous)
 */
void play_next ()
{
  gboolean need_newline = FALSE;
  g_debug ("%s: Playing next...", __FUNCTION__);
  
  // if just started playing
  if (!ap_config.current)
  {
    ap_config.glist_current = ap_config.playlist;
  }
  else // else, we get the next stream
  {
    // we need a new line between streams
    need_newline = TRUE;
    
    // get the next stream from the list
    ap_config.glist_current = g_list_next (ap_config.glist_current);
    
    // if Loop && we are at the end && a stream was just played, we go back to the first one
    if (ap_config.loop && !ap_config.glist_current)
    {
      if (loop_stream_was_played)
      {
        g_debug ("Looping...");
        loop_stream_was_played = FALSE;
        ap_config.glist_current = ap_config.playlist;
      }
      else
      {
        g_warning ("Deactivating loop mode: useless endless loop detected.");
        ap_config.loop = FALSE;
      }
    }
  }
  
  // if we have a current stream
  if (ap_config.glist_current)
  {
    ap_config.current = ap_config.glist_current->data;
    
    // we print the new line now so that it's in between streams (to seperate them)
    if (!ap_config.debug && need_newline)
      g_print("\n");
    
    // Define the URI to play
    g_debug ("%s: URI: %s", __FUNCTION__, (*currentstream)->uri);
    g_object_set (G_OBJECT (pipeline), "uri", (*currentstream)->uri, NULL);
    
    // Show "Loading Filename..."
    g_info (_("Loading \"%s\"...\n"),
            (*currentstream)->filepath ?
              (*currentstream)->filepath : (*currentstream)->uri
           );
  
    // Starts playing
    player_begin ();
  }
  else // if no current stream, we clear the pipeline and call the callback
  {
    player_clear_pipeline ();
    done_playing_callback ();
  }
}

/**
 * Initialize the player.
 * This initializes the player before starting any play. This should be called
 * only once.
 */
void player_init (void *callback_when_done, void *bus_call_dispatcher)
{
  g_debug ("%s: Starting player...", __FUNCTION__);
  
  if (ap_config.use_playbin2)
    pipeline = gst_element_factory_make ("playbin2", "player");
  else
    pipeline = gst_element_factory_make ("playbin", "player");
  
  // AUDIO Disabled
  if (ap_config.no_audio)
  {
    g_debug ("%s: Adding fakesink to audio-sink property.", __FUNCTION__);
    audiofakesink = gst_element_factory_make ("fakesink", "audiofakesink");
    g_object_set (audiofakesink, "sync", TRUE, "silent", TRUE, NULL);
    g_object_set (pipeline, "audio-sink", audiofakesink, NULL);
  }
  
  // VIDEO Disabled
  if (ap_config.no_video)
  {
    g_debug ("%s: Adding fakesink to video-sink property.", __FUNCTION__);
    videofakesink = gst_element_factory_make ("fakesink", "videofakesink");
    g_object_set (videofakesink, "sync", TRUE, "silent", TRUE, NULL);
    g_object_set (pipeline, "video-sink", videofakesink, NULL);
  }
  
  // VOLUME
  if (ap_config.volume != 100)
  {
    g_debug ("%s: Setting volume to %d%%", __FUNCTION__, ap_config.volume);
    g_object_set (pipeline, "volume", ((gdouble)ap_config.volume)/100, NULL);
  }
  
  {
    GstBus *bus;
    
    bus = gst_pipeline_get_bus (GST_PIPELINE (pipeline));
    gst_bus_add_watch (bus, bus_call_dispatcher, NULL);
    gst_object_unref (bus);
  }
  
  done_playing_callback = callback_when_done;
  
  // currentstream is a pointer to pointer to stream
  currentstream = &ap_config.current;
  player_prepare_for_next ();
}

/**
 * Asks the player to prepare itself to play the next stream in queue.
 * This asks the player to prepare itself to play the next stream in queue. The
 * next stream will be played when the pipeline is in READY state.
 */
void player_prepare_for_next ()
{
  if (currentstream && *currentstream)
    stream_remove_updater (*currentstream); 
  
  g_debug ("%s: Getting pipeline READY...", __FUNCTION__);
  
  if (!ap_config.skip_null_state)
    gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_NULL);
  
  gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_READY);
}

/**
 * Same as player_prepare_for_next () except this returns FALSE (for g_idle).
 */
static gboolean player_prepare_for_next_query ()
{
  player_prepare_for_next ();
  return FALSE;
}

/**
 * Clear (free) the pipeline and associated objects.
 */
void player_clear_pipeline ()
{
  if (currentstream && *currentstream)
  {
    /* remove the updater function */
    stream_remove_updater (*currentstream);
    currentstream = NULL;
  }
  
  if (pipeline)
  {
    gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_NULL);
    gst_object_unref (GST_OBJECT (pipeline));
    pipeline = NULL;
  }
}

/**
 * Asks the player to be in PAUSED mode. (might be asynchronous)
 */
static void player_begin ()
{
  g_debug ("%s: PAUSED state: requesting...", __FUNCTION__);
  switch(gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_PAUSED))
  {
    case GST_STATE_CHANGE_FAILURE:
    {
      g_debug ("%s: received: GST_STATE_CHANGE_FAILURE", __FUNCTION__);
      g_warning (_("Unable to play stream (%s). Skipping..."), 
                  (*currentstream)->filepath ?
                   (*currentstream)->filepath : (*currentstream)->uri
                );
      player_prepare_for_next ();
      return;
      break;
    }
    
    case GST_STATE_CHANGE_SUCCESS:
    {
      g_debug ("%s: received: GST_STATE_CHANGE_SUCCESS", __FUNCTION__);
      break;
    }
    
    case GST_STATE_CHANGE_ASYNC:
    {
      g_debug ("%s: received: GST_STATE_CHANGE_ASYNC", __FUNCTION__);
      break;
    }
    
    case GST_STATE_CHANGE_NO_PREROLL:
    {
      g_debug ("%s: received: GST_STATE_CHANGE_NO_PREROLL", __FUNCTION__);
      break;
    }
    
    default:
    break;
  }
  g_debug ("%s: PAUSED state: waiting for state-changed bus call", __FUNCTION__);
}

/**
 * Seek to the necessary position (depending on ap_config). (synchronous)
 */
static void player_seek ()
{
  g_debug ("%s: Checking duration", __FUNCTION__);
  
  // Query the Stream duration
  if (player_update_stream_duration (*currentstream))
  {
    // -If- negative or null duration (possibly a continuous stream)
    // -OR- Play from the start.
    if (ap_config.position == START_PLAYING_START || (*currentstream)->duration <= 0)
    {
      if ((*currentstream)->duration <= 0)
        g_debug ("%s: No seek: duration is 0...", __FUNCTION__);
    }
    else // -else- Seek to requested position (duration is in nanosec)
    {
      // -If- Debug mode, then show total duration.
      if (ap_config.debug)
      {
        gint len_hour, len_min, len_sec;
        split_nanoseconds ((*currentstream)->duration, 
                            &len_hour, &len_min, &len_sec);
        
        g_debug ("%s: Duration: %u:%02u:%02u",
                 __FUNCTION__, len_hour, len_min, len_sec);
      }
      
      // Define default seek position.
      gint64 seekposition = 0;
      
      // -If- play from MIDDLE
      if (ap_config.position == START_PLAYING_MIDDLE)
      {
        // seek position = the middle of the duration
        seekposition = (*currentstream)->duration/2;
      }
      else if (ap_config.position == START_PLAYING_END) // -If- Play the end
      {
        gint64 configduration = ap_config.duration*GST_SECOND;
        // -If- wanted duration is shorter than stream duration, we can seek.
        if ((*currentstream)->duration > configduration)
        {
          seekposition = (*currentstream)->duration - configduration;
        }
        else // Otherwise we play the whole stream (don't change seekposition)
        {
          g_debug ("%s: wanted duration is longer than stream's duration",
                   __FUNCTION__);
        }
      }
      else if (ap_config.position == START_PLAYING_ANYWHERE) 
      { // -If- play from ANYWHERE
      
        gint streamsec = (*currentstream)->duration/GST_SECOND; // in sec
        
        // -If- wanted duration is shorter than stream duration, we can seek. 
        if (ap_config.duration < streamsec)
        {
          // Try to work around because g_random_int_range is only for guint32
          gint64 endposition;
          gint32 tmp;
          tmp = g_random_int_range (0, 1 + streamsec - ap_config.duration);
          seekposition = tmp * GST_SECOND;
          endposition = seekposition + ap_config.duration * GST_SECOND;
          tmp = g_random_int_range (0, 1 + 
                  (gint32) MIN((*currentstream)->duration - endposition,
                               GST_SECOND));
          seekposition += tmp;
        }
        else // Otherwise we play the whole stream (don't change seekposition)
          g_debug ("%s: wanted duration is longer than stream", __FUNCTION__);
      }
      
      // -If- we need to seek (seekpositon != 0)
      if (seekposition)
      {
        g_debug ("%s: Seeking...", __FUNCTION__);
        
        GstSeekFlags seekflags = GST_SEEK_FLAG_FLUSH;
        if (ap_config.fastseek)
        {
          g_debug ("%s: Fast seek.", __FUNCTION__);
          seekflags |= GST_SEEK_FLAG_KEY_UNIT;
        }
        else
          seekflags |= GST_SEEK_FLAG_ACCURATE;
        
        if (!gst_element_seek_simple (pipeline, GST_FORMAT_TIME, seekflags,
                                      seekposition))
          g_warning ("%s: Unable to seek stream", __FUNCTION__);
      }
    }
  }
  else
  {
    g_warning ("%s: Couldn't find stream duration...", __FUNCTION__);
  }
  
  // Request to start playing.
  player_start_playing();
}

/**
 * Asks the player to be in PLAYING state. (often asynchronous)
 */
static void player_start_playing ()
{
  g_debug ("%s: Setting pipeline playing...", __FUNCTION__);
  gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_PLAYING);
}

/**
 * Update the stream's duration with data from pipeline.
 */
gboolean player_update_stream_duration (Stream *stream)
{
  // TODO: It shouldn't be appliable to any stream (just currentstream)
  GstFormat format = GST_FORMAT_TIME;
  return gst_element_query_duration (pipeline, &format, &(stream->duration));
}

/**
 * Update the stream's position with data from pipeline.
 */
gboolean player_update_stream_position (Stream *stream)
{
  // TODO: It shouldn't be appliable to any stream (just currentstream)
  GstFormat format = GST_FORMAT_TIME;
  return gst_element_query_position (pipeline, &format, &(stream->position));
}

/**
 * Update the current stream's information and check if reached wanted duration.
 */
gboolean player_update_stream ()
{
  if (!currentstream || !(*currentstream))
    return FALSE;
  
  static GStaticMutex mutex = G_STATIC_MUTEX_INIT;
  gint64 elapsed;
  
  // LOCK
  g_static_mutex_lock (&mutex);
  
  player_update_stream_duration (*currentstream);
  player_update_stream_position (*currentstream);
  
  stream_print_info (*currentstream);
  
  if ((*currentstream)->startposition == -1)
    (*currentstream)->startposition = (*currentstream)->position;
  
  stream_get_elapsed_time(*currentstream, &elapsed);
  
  /* If we have reached Duration or Maxduration, we play the next stream */
  if (ap_config.duration >= 0 && elapsed >= (gint64)ap_config.duration*GST_SECOND
   || ap_config.maxduration >= 0 && elapsed >= (gint64)ap_config.maxduration*GST_SECOND)
    player_prepare_for_next();
  
  // UNLOCK
  g_static_mutex_unlock (&mutex);
  
  return TRUE;
}

/**
 * Asks the player to be in PAUSED state.
 */
gboolean player_pause ()
{
  GstState cur_state;
  GstStateChangeReturn ret;
  
  gst_element_get_state(GST_ELEMENT (pipeline), &cur_state, NULL,
                        GST_CLOCK_TIME_NONE);
  
  if (cur_state == GST_STATE_PLAYING)
    ret = gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_PAUSED);
  else if (cur_state == GST_STATE_PAUSED)
    ret = gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_PLAYING);
  else
    return FALSE;
  
  switch(ret)
  {
    case GST_STATE_CHANGE_FAILURE:
    {
      g_debug ("%s: received: GST_STATE_CHANGE_FAILURE", __FUNCTION__);
      if (cur_state == GST_STATE_PLAYING)
        g_warning (_("Unable to pause."));
      else
        g_warning (_("Unable to resume."));
      return FALSE;
    }
    
    case GST_STATE_CHANGE_SUCCESS:
    {
      g_debug ("%s: received: GST_STATE_CHANGE_SUCCESS", __FUNCTION__);
      return TRUE;
    }
    
    case GST_STATE_CHANGE_ASYNC:
    {
      g_debug ("%s: received: GST_STATE_CHANGE_ASYNC", __FUNCTION__);
      return TRUE;
    }
    
    case GST_STATE_CHANGE_NO_PREROLL:
    {
      g_debug ("%s: received: GST_STATE_CHANGE_NO_PREROLL", __FUNCTION__);
      return TRUE;
    }
    
    default:
    return FALSE;
  }
}
