#!/usr/bin/env python

# Copyright (C) Andrew Mitchell 2006

#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.

#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
############################################################################

## client side: enable pam_krb5

from AuthTool import pam, config, authtools, dnsconfig
import debconf, sys, os

config.methods['Kerberos'] = 'Kerberos V5 authentication'

class KerberosConfig(authtools.IAuthConfig):

    name = "Kerberos"
    pam_module = "pam_krb5.so"
    packages = ["libpam-heimdal","heimdal-clients"]
    requires = []
    display = False

    # Values to save/restore
    # Should match the values used in config
    persistent = ['default_realm','krb_tgt_server','krb_passwd_server','krb_admin_server']
    
    default_realm = ''
    krb_tgt_server = ''
    krb_passwd_server = ''
    krb_admin_server = ''
    discovery = False
    
    def __init__(self, authconfig):

        self.cfg = authconfig
        # Get the values from krb5-config for configuring krb5.conf
        self.db = authtools.Debconf()
        default_realm = self.db.get('krb5-config/default_realm')
        krb_servers = self.db.get('krb5-config/kerberos_servers')
        admin_server = self.db.get('krb5-config/admin_server')
        self.dns = dnsconfig.DNSConfig()
        # discover kerberos realm & servers
        self.default_realm  = self.dns.TXT('kerberos')['txt']
        self.krb_tgt_server = self.dns.SRV('kerberos')['host']
        self.krb_passwd_server = self.dns.SRV('kpasswd')['host']
        if self.krb_tgt_server != '':
            self.discovery = True

        authtools.IAuthConfig.__init__(self,authconfig)

    def config(self):
        """
        Return a list of configuration options in a standardised format
        {'id':['type','label',default','options'],...}
        """

        # Pull defaults from existing config
        # Some of these defaults are detected by DNS
        cfg = {
            'default_realm':['string', 'Default Realm',self.default_realm,''],
            'krb_tgt_server':['string', 'Kerberos TGT Server',self.krb_tgt_server,''],
            'krb_admin_server':['string', 'Admin Server',self.krb_admin_server,''],
            'krb_passwd_server':['string', 'Password Server',self.krb_passwd_server,'']
             }
            
        return cfg

    def enable(self):
        """Enable the config for this"""
        self.save()
        # setup /etc/krb5.conf (debconf, krb5-config.postinst)
        #try:
        self.db.set('krb5-config/default_realm', self.default_realm)
        self.db.set('krb5-config/kerberos_servers', self.krb_tgt_server)
        self.db.set('krb5-config/admin_server', self.krb_admin_server)
        if self.discovery:
            self.db.set('krb5-config/dns_for_default', 'true')

        # MUST let dpkg-reconfigure acquire debconf lock
        self.db.unlock()
        # This asks questions, which is hardly what we want
        # Add "-fnoninteractive
        # Running execl is a Bad Thing
        os.spawnl(os.P_WAIT, "/usr/sbin/dpkg-reconfigure", "dpkg-reconfigure", "-fnoninteractive", "krb5-config")
        self.db.reinit()
        #except:
        #    print "Debconf save failed"

        # enable pam_krb5
        pam_ = pam.PAMConfig()
        if not pam_.find('auth','pam_krb5.so', True):
            if pam_.find('auth','pam_unix.so', True):
                pam_.insert_after('auth', 'sufficient', 'pam_krb5.so', 'pam_unix.so')
                pam_.change_level('auth', 'sufficient', 'pam_unix.so')
            
            pam_.insert_after('password', 'sufficient', 'pam_krb5.so', 'pam_unix.so')
            pam_.insert_after('session', 'sufficient', 'pam_krb5.so', 'pam_unix.so')
            pam_.save()
        

    def preinst(self, mode):
        """Execute preinst script"""
        pass


    def gtk_config(self, container):
        """ """

        
