/*--------------------------------------------------------------------------*
 * AUTOPROFILE                                                              *
 *                                                                          *
 * A Gaim away message and profile manager that supports dynamic text       *
 *                                                                          *
 * AutoProfile is the legal property of its developers.  Please refer to    *
 * the COPYRIGHT file distributed with this source distribution.            *
 *                                                                          *
 * This program is free software; you can redistribute it and/or modify     *
 * it under the terms of the GNU General Public License as published by     *
 * the Free Software Foundation; either version 2 of the License, or        *
 * (at your option) any later version.                                      *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA *
 *--------------------------------------------------------------------------*/

#include "component.h"
#include "utility.h"
#include "util.h"
#include "gtkprefs.h"

/*---------- FORTUNE: Random quote grabber ----------*/
static GtkWidget *file_entry;

/* Read quote file, parse and spit out random quote */
static char *quote_generate ()
{
  GList *quotes, *quotes_start;
  char *ret;
  int num_quotes, index, max;
  time_t cur_time, old_time;
  char *time_string;

  max = gaim_prefs_get_int (
    "/plugins/gtk/autoprofile/components/quote/quote_size");
  index = gaim_prefs_get_int (
    "/plugins/gtk/autoprofile/components/quote/quote_current");

  quotes = read_fortune_file (
    gaim_prefs_get_string ("/plugins/gtk/autoprofile/components/quote/quote"), 
    max);

  /* Check for invalid file */
  num_quotes = g_list_length (quotes);
  
  if (num_quotes == 0) {
    auto_debug ("quote", "no valid quotes in file");
    return NULL; 
  }  

  /* Increment index if time has elapsed */
  old_time = gaim_str_to_time (gaim_prefs_get_string (
    "/plugins/gtk/autoprofile/components/quote/last_update"), TRUE);
  cur_time = time (NULL);

  if (difftime (cur_time, old_time) > 
    60.0 * 60.0 * (double) gaim_prefs_get_int (
      "/plugins/gtk/autoprofile/components/quote/update_rate")) {
    auto_debug ("quote", "time interval elapsed, moving to new quote");
    index++;
    gaim_prefs_set_int (
      "/plugins/gtk/autoprofile/components/quote/quote_current", index);
      
    time_string = (char *)malloc(1000);
    strftime (time_string, 1000, "%Y-%m-%dT%H:%M:%S+00:00", gmtime (&cur_time));
    gaim_prefs_set_string (
      "/plugins/gtk/autoprofile/components/quote/last_update", time_string);
    free (time_string);
  }

  /* Wrap around at end of file */
  if (index >= num_quotes) {
    index = 0; 
    gaim_prefs_set_int (
      "/plugins/gtk/autoprofile/components/quote/quote_current", 0);
  }

  /* Choose and output the quote */
  quotes_start = quotes;
  
  while (index != 0) {
    quotes = quotes->next;
    index--;
  }

  ret = strdup((char *) quotes->data); 
  free_string_list (quotes_start);

  return ret;
}

static void force_move (GtkButton *button, gpointer data)
{
  gaim_prefs_set_int (
    "/plugins/gtk/autoprofile/components/quote/quote_current",
    gaim_prefs_get_int (
      "/plugins/gtk/autoprofile/components/quote/quote_current") + 1);
}

static gboolean quote_update (GtkWidget *widget, GdkEventFocus *evt, 
                                gpointer data)
{
	gaim_prefs_set_string (
    "/plugins/gtk/autoprofile/components/quote/quote",
    gtk_entry_get_text (GTK_ENTRY (file_entry)));
  return FALSE;
}

static void quote_filename (GtkWidget *widget, gpointer user_data) {
  GtkWidget *file_selector = (GtkWidget *)user_data;
  const gchar *selected_filename;

  selected_filename = gtk_file_selection_get_filename (
    GTK_FILE_SELECTION (file_selector));

  gaim_prefs_set_string ("/plugins/gtk/autoprofile/components/quote/quote",
    selected_filename);
  gtk_entry_set_text (GTK_ENTRY (file_entry), gaim_prefs_get_string (
    "/plugins/gtk/autoprofile/components/quote/quote"));
}

/* Creates and pops up file selection dialog for quote file */
static void quote_file_selection (GtkWidget *widget, gpointer user_data) {
  GtkWidget *file_selector;
  const char *cur_file;

  /* Create the selector */
  file_selector = gtk_file_selection_new (
    "Select a file with quotes in fortune format");

  cur_file = gaim_prefs_get_string (
    "/plugins/gtk/autoprofile/components/quote/quote");
  if (strlen (cur_file) > 1) {
    gtk_file_selection_set_filename (
      GTK_FILE_SELECTION (file_selector), cur_file); 
  }
  
  g_signal_connect (GTK_OBJECT(
                      GTK_FILE_SELECTION(file_selector)->ok_button),
                    "clicked", G_CALLBACK (quote_filename),
                    (gpointer) file_selector);
   			   
  /* Destroy dialog box when the user clicks button. */
  g_signal_connect_swapped (GTK_OBJECT(
      GTK_FILE_SELECTION(file_selector)->ok_button), 
    "clicked", G_CALLBACK (gtk_widget_destroy), (gpointer) file_selector); 

  g_signal_connect_swapped (GTK_OBJECT (
      GTK_FILE_SELECTION (file_selector)->cancel_button), 
    "clicked", G_CALLBACK (gtk_widget_destroy), (gpointer) file_selector); 
   
  /* Display dialog */
  gtk_widget_show (file_selector);
}

/* Creates quote preferences menu */
static GtkWidget *quote_menu ()
{
  GtkWidget *ret = gtk_vbox_new (FALSE, 5);
  GtkWidget *hbox, *label, *button;

  label = gtk_label_new (_("Select quote database"));
  gtk_box_pack_start (GTK_BOX (ret), label, FALSE, FALSE, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0);

  hbox = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (ret), hbox, FALSE, FALSE, 0);
  /* Text entry to type in file name */
  file_entry = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (hbox), file_entry, FALSE, FALSE, 0);
  gtk_entry_set_text (GTK_ENTRY (file_entry), gaim_prefs_get_string (
    "/plugins/gtk/autoprofile/components/quote/quote"));
  g_signal_connect (G_OBJECT (file_entry), "focus-out-event",
                    G_CALLBACK (quote_update), NULL);
  /* Button to bring up file select dialog */
  button = gtk_button_new_with_label ("Browse ...");
  g_signal_connect (G_OBJECT (button), "clicked",
                    G_CALLBACK (quote_file_selection), NULL);

  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);

  gaim_gtk_prefs_labeled_spin_button (
    ret, "Max quote length",
    "/plugins/gtk/autoprofile/components/quote/quote_size",
    1, 1000, NULL);
  gaim_gtk_prefs_labeled_spin_button (
    ret, "Hours between updates",
    "/plugins/gtk/autoprofile/components/quote/update_rate",
    1, 1000, NULL);

  button = gtk_button_new_with_label ("Force move to next quote");
  g_signal_connect (G_OBJECT (button), "clicked",
                    G_CALLBACK (force_move), NULL);
  gtk_box_pack_start (GTK_BOX (ret), button, FALSE, FALSE, 0);

  return ret;
}

static void quote_init () {
  time_t the_time;
  char *time_string;
  
  time_string = (char *)malloc(1000);
  the_time = time(NULL);
  strftime (time_string, 1000, "%Y-%m-%dT%H:%M:%S+00:00", gmtime (&the_time));

  gaim_prefs_add_none ("/plugins/gtk/autoprofile/components/quote");
  gaim_prefs_add_string ("/plugins/gtk/autoprofile/components/quote/quote","");
  gaim_prefs_add_int (
    "/plugins/gtk/autoprofile/components/quote/quote_size", 1000);
  gaim_prefs_add_int (
    "/plugins/gtk/autoprofile/components/quote/update_rate", 24);
  gaim_prefs_add_int (
    "/plugins/gtk/autoprofile/components/quote/quote_current", 0);
  gaim_prefs_add_string (
    "/plugins/gtk/autoprofile/components/quote/last_update",
    time_string);
  free (time_string);
}

struct component quote =
{
  N_("Daily Quote"),
  N_("Daily quote select in order from text file with fortune format"),
  'q',
  &quote_generate,
  &quote_init,
  NULL,
  NULL,
  &quote_menu
};

