/*--------------------------------------------------------------------------*
 * AUTOPROFILE                                                              *
 *                                                                          *
 * A Gaim away message and profile manager that supports dynamic text       *
 *                                                                          *
 * AutoProfile is the legal property of its developers.  Please refer to    *
 * the COPYRIGHT file distributed with this source distribution.            *
 *                                                                          *
 * This program is free software; you can redistribute it and/or modify     *
 * it under the terms of the GNU General Public License as published by     *
 * the Free Software Foundation; either version 2 of the License, or        *
 * (at your option) any later version.                                      *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA *
 *--------------------------------------------------------------------------*/

#include "autoprofile.h"

#include "gtkdialogs.h"
#include "gtkimhtml.h"
#include "sound.h"

static time_t response_timeout;

/* Message queuing code yanked from regular Gaim */
struct queued_msg {
	char name[80];
  char *sender;
	char alias[80];
	char *message;
	time_t tm;
	GaimAccount *account;
	GaimMessageFlags flags;
};

/* Variable definitions */

/* List of messages */
static GtkListStore *message_list;
static GtkWidget *message_list_view;

/* The general window + text of message */
static GtkWidget *dialog = NULL;
static GtkWidget *msg_window = NULL;
static GtkWidget *awaytext = NULL;

/* Progress bar */
static GtkWidget *progress_box = NULL;
static GtkWidget *progress = NULL;
static int progress_minutes = 0;
static int progress_seconds = 0;
static gpointer fraction_timeout = NULL;
static gpointer second_timeout = NULL;

/* Queue variables */
static GtkWidget *awayqueuebox = NULL;
static GtkWidget *awayqueue = NULL;
static GtkWidget *awayqueuesw = NULL;
static GtkListStore *awayqueuestore = NULL;
GSList *message_queue = NULL;

/* Misc */
static gboolean was_queueing = FALSE;
static gboolean disconnected = TRUE;
/* Function definitions */
static int ap_find_queue_row_by_name(char *name)
{
  gchar *temp;
  gint i = 0;
  gboolean valid;
  GtkTreeIter iter;

  valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(awayqueuestore), &iter);
  while (valid) {
    gtk_tree_model_get (GTK_TREE_MODEL(awayqueuestore), &iter, 0, &temp, -1);
    if (!strcmp(name, temp))
      return i;
    g_free(temp);
		
    i++;
    valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(awayqueuestore), &iter);
  }

  return -1;
}

static int ap_find_queue_total_by_name(char *name)
{
  GSList *templist;
  struct queued_msg *qm;
  int i = 0;

  templist = message_queue;

  while (templist) {
    qm = (struct queued_msg *)templist->data;
    if ((qm->flags & GAIM_MESSAGE_RECV) && !strcmp(name, qm->name))
      i++;

    templist = templist->next;
  }

  return i;
}

static void queue_send_autoresponse (GaimAccount *account, char *recepient,
                                     char *msg)
{
  struct queued_msg *qm;
  const char *sender, *alias;
  GaimConnection *gc;
  GaimBuddy *b;

  sender = gaim_account_get_username (account);
  gc = gaim_account_get_connection (account);

  /* Send the msg */
  sending_auto = TRUE;
  serv_send_im (gc, recepient, msg, GAIM_CONV_IM_AUTO_RESP);
  sending_auto = FALSE;

  /* Record it in the queue */
  qm = g_new0(struct queued_msg, 1);
  g_snprintf(qm->name, sizeof(qm->name), "%s", recepient);
  qm->sender = strdup (sender);
  qm->message = g_strdup(msg);
  qm->account = gc->account;
  qm->tm = time (NULL);
  qm->flags = GAIM_CONV_IM_AUTO_RESP;
  message_queue = g_slist_append (message_queue, qm);
  b = gaim_find_buddy(gc->account, qm->name);
  alias = b ? gaim_buddy_get_alias (b) : qm->name;
  if (strcmp (alias, qm->name) != 0)
    g_snprintf (qm->alias, sizeof(qm->alias), "(%s)", alias);

}

static gboolean queue_received (GaimAccount *account, char **sender,
                                char **message, int *flags) 
{
  GaimConversation *cnv;
  GaimConnection *gc;
  GaimMessageFlags msgflags;

  struct queued_msg *qm;
  GtkTreeIter iter;
  gchar path[10];
  const char *alias;
  char number[32];
  int qtotal;
  int away_num;
  GaimBuddy *b;

  GList *msg_node, *start_node;
  char *text_respond, *generated, *text;

  int row;

  if (message == NULL || *message == NULL || sender == NULL || *sender == NULL)
    return FALSE;

  /* Check the flags */
	msgflags = GAIM_MESSAGE_RECV;
	if (*flags & GAIM_CONV_IM_AUTO_RESP)
		msgflags |= GAIM_MESSAGE_AUTO_RESP;
  
	//if (*flags & GAIM_CONV_IM_IMAGES)
	//	msgflags |= GAIM_MESSAGE_IMAGES;

  /* See if the conversation exists already */
  gc = gaim_account_get_connection (account);
	cnv = gaim_find_conversation_with_account (*sender, gc->account);

  if (!cnv && was_queueing) {
    /* 
     * Alright, so we're going to queue it. Neat, eh? :)
     * So first we create something to store the message, and add
     * it to our queue. Then we update the away dialog to indicate
     * that we've queued something.
     */
    qm = g_new0(struct queued_msg, 1);
    g_snprintf(qm->name, sizeof(qm->name), "%s", *sender);
    qm->sender = NULL;
    qm->message = g_strdup(*message);
    qm->account = gc->account;
    qm->tm = time (NULL);
    qm->flags = msgflags;
    message_queue = g_slist_append (message_queue, qm);
    b = gaim_find_buddy(gc->account, qm->name);
    alias = b ? gaim_buddy_get_alias (b) : qm->name;
    if (strcmp (alias, qm->name) != 0)
      g_snprintf (qm->alias, sizeof(qm->alias), "(%s)", alias);

    row = ap_find_queue_row_by_name (qm->name);

    /* Set some auto-response vars */
    away_num = gaim_prefs_get_int (
      "/plugins/gtk/autoprofile/current_away");
    if (away_num < 0) {
      auto_debug ("auto-response", "no away msg has been set");
      generated = strdup (" ");
      away_num = 0;
    } else {
      generated = NULL;
    }

    /* Set some auto-response vars */
    msg_node = gaim_prefs_get_string_list (
      "/plugins/gtk/autoprofile/message_texts");
    start_node = msg_node;
    while (away_num-- != 0) {
      msg_node = msg_node->next;
    }
 
    if (!generated) {
      generated = autoprofile_generate ((char *)msg_node->data);
    }
    free_string_list (start_node);

    if (row >= 0) {
      qtotal = ap_find_queue_total_by_name (qm->name);
      g_snprintf (number, 32, ngettext("(%d message)",
        "(%d messages)", qtotal), qtotal);
      g_snprintf (path, 10, "%d", row);
      gtk_tree_model_get_iter_from_string (
        GTK_TREE_MODEL(awayqueuestore), &iter, path);
      gtk_list_store_set (awayqueuestore, &iter,
        2, number, -1);

      if (gaim_prefs_get_bool("/gaim/gtk/sound/enabled/im_recv"))
        gaim_sound_play_event(GAIM_SOUND_RECEIVE);

      /* Old message, autoresponse stuff */
      if (!original_autoresponse || !is_away || sending_auto) {
        free (generated);
        return TRUE;
      }

      if (difftime (time(NULL), response_timeout) >
        gaim_prefs_get_int ("/plugins/gtk/autoprofile/delay_respond")) {
        text = gaim_markup_strip_html (*message);
        if (match_start (text, (char *)gaim_prefs_get_string (
                         "/plugins/gtk/autoprofile/text_trigger")) == 1) {
          queue_send_autoresponse (account, qm->name, generated);
          response_timeout = time(NULL);
          auto_debug ("autorespond", "string matched, responding");
        }
      }
    } else {
      gtk_tree_model_get_iter_first(GTK_TREE_MODEL(awayqueuestore), 
        &iter);
      gtk_list_store_append(awayqueuestore, &iter);
      gtk_list_store_set(awayqueuestore, &iter,
                         0, qm->name,
                         1, qm->alias, 
                         2, _("(1 message)"),
                         -1);

      if (gaim_prefs_get_bool("/gaim/gtk/sound/enabled/first_im_recv"))
        gaim_sound_play_event(GAIM_SOUND_FIRST_RECEIVE);

      /* New conversation, autoresponse stuff */
      if (!original_autoresponse || !is_away || sending_auto) {
        free (generated);
        return TRUE;
      }
      queue_send_autoresponse (account, qm->name, generated);
      if (gaim_prefs_get_bool ("/plugins/gtk/autoprofile/use_trigger")) {
        text_respond = stylize (gaim_prefs_get_string (
          "/plugins/gtk/autoprofile/text_respond"), 1000);
        queue_send_autoresponse (account, qm->name, text_respond);
        free (text_respond);
      }

    }
    return TRUE;
  } else {
    regular_received (account, *sender, *message, 0);
    return FALSE;
  }
}

/* Taken from Gaim server.c */
static void ap_purge_away_queue (GSList **queue)
{
  GSList *q = *queue;
  struct queued_msg *qm;
  GaimConversation *cnv;
  GaimAccount *account;
  gboolean orig_while_away;

  orig_while_away = gaim_prefs_get_bool("/core/sound/while_away");
  gaim_prefs_set_bool("/core/sound/while_away", FALSE);

  while (q) {
    qm = q->data;
    account = NULL;

    if (g_list_index (gaim_accounts_get_all(), qm->account) >= 0)
      account = qm->account;

    cnv = gaim_find_conversation_with_account (qm->name, account);

    if (!cnv)
      cnv = gaim_conversation_new (GAIM_CONV_IM, account, qm->name);
    else
      gaim_conversation_set_account (cnv, account);

    gaim_conv_im_write(GAIM_CONV_IM(cnv), qm->sender, qm->message, 
      qm->flags, qm->tm);

    g_free(qm->message);
    if (qm->sender)
      g_free(qm->sender);
    g_free(qm);

    q->data = NULL;
    q = q->next;
  }

  g_slist_free(*queue);
  *queue = NULL;

  gaim_prefs_set_bool("/core/sound/while_away", orig_while_away);
}

/* Taken from Gaim server.c */
static void ap_dequeue_message (GtkTreeIter *iter)
{
  gchar *name;
  GSList *templist;
  GaimConversation *cnv;
  struct queued_msg *qm;
  gboolean orig_while_away;

  gtk_tree_model_get (GTK_TREE_MODEL(awayqueuestore), iter, 0, &name, -1);

  gaim_debug(GAIM_DEBUG_INFO, "away", "Unqueueing messages from %s.\n",
    name);

  templist = message_queue;
  orig_while_away = gaim_prefs_get_bool("/core/sound/while_away");
  gaim_prefs_set_bool("/core/sound/while_away", FALSE);

  while (templist) {
    qm = templist->data;
    if (templist->data) {
      if (!gaim_utf8_strcasecmp (qm->name, name)) {
        GaimAccount *account = NULL;

        if (g_list_index (gaim_accounts_get_all(), qm->account) >= 0)
          account = qm->account;

        cnv = gaim_find_conversation_with_account (name, account);

        if (!cnv)
          cnv = gaim_conversation_new (GAIM_CONV_IM, account, qm->name);
        else
          gaim_conversation_set_account (cnv, account);

        gaim_conv_im_write (GAIM_CONV_IM(cnv), qm->sender, qm->message,
          qm->flags, qm->tm);
        g_free (qm->message);
        if (qm->sender)
          g_free (qm->sender);
        g_free (qm);
        templist = message_queue = g_slist_remove (message_queue, qm);
      } else {
        templist = templist->next;
      }
    }
  }

  g_free (name);
  /* In GTK 2.2, _store_remove actually returns whether iter is valid or not
   * after the remove, but in GTK 2.0 it is a void function. */
  gtk_list_store_remove(awayqueuestore, iter);
  gaim_prefs_set_bool("/core/sound/while_away", orig_while_away);
}

/* Taken from Gaim server.c */
static gint ap_dequeue_cb (GtkWidget *treeview, GdkEventButton *event, 
                           gpointer data) {
  GtkTreeIter iter;
  GtkTreeSelection *select;

  if (!(event->type == GDK_2BUTTON_PRESS && event->button == 1)) {
	  /* Double clicking on the list will unqueue that user's messages. */
    return FALSE;
  }

  select = gtk_tree_view_get_selection (GTK_TREE_VIEW(treeview));
  if (gtk_tree_selection_get_selected (select, NULL, &iter))
    ap_dequeue_message(&iter);

  return FALSE;
}

/* Changes fraction progress of progress bar */
static gboolean update_fraction (gpointer data)
{
  double fraction_increment;
  double cur_fraction;
  double result;

  fraction_increment = 250 /
    (60 * 1000 * (double) gaim_prefs_get_int (
     "/plugins/gtk/autoprofile/delay_away"));
  cur_fraction = gtk_progress_bar_get_fraction (GTK_PROGRESS_BAR(progress));

  result = cur_fraction + fraction_increment;
  if (result >= 1)
    result = 1;

  gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(progress), result);

  return TRUE;
}

/* Changes text of progress bar */
static gboolean update_seconds (gpointer data)
{
  char *text = (char *)malloc(1024);

  if (progress_seconds == 0) {
    progress_minutes--;
    progress_seconds = 59;
  } else {
    progress_seconds--;
  }

  if (progress_minutes < 0) {
    progress_minutes = 0;
    progress_seconds = 0;
  }
 
  sprintf (text, "next update in %d:%02d", progress_minutes, 
           progress_seconds);
  gtk_progress_bar_set_text (GTK_PROGRESS_BAR(progress), text);

  return TRUE;
}

/* Replace displayed message with a new one */
static void gtk_replace_displayed_message (char *msg)
{
  /* Update the main window w/ the text */
  if (awaytext)
    gtk_widget_destroy (awaytext);

  awaytext = gtk_imhtml_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER(msg_window), awaytext);
  gaim_setup_imhtml (awaytext);
  gtk_widget_show (awaytext);
  gtk_imhtml_append_text (GTK_IMHTML(awaytext), msg, GTK_IMHTML_NO_TITLE |
    GTK_IMHTML_NO_COMMENTS | GTK_IMHTML_NO_SCROLL);
  gtk_imhtml_append_text (GTK_IMHTML(awaytext), "<BR>",
    GTK_IMHTML_NO_TITLE | GTK_IMHTML_NO_COMMENTS |
    GTK_IMHTML_NO_SCROLL);
}

/* Changes the text displayed as the current away message */
static void gtk_away_msgs_update (char *msg)
{
  GtkTreeIter iter;
  struct tm *cur_time;
  char *time_string;
  time_t *general_time;
  GtkTreeSelection *selection;

  /* reset the progress bar */
  if (progress) {
    gtk_container_remove (GTK_CONTAINER(progress_box), progress);
    g_source_remove (GPOINTER_TO_INT (second_timeout));
    g_source_remove (GPOINTER_TO_INT (fraction_timeout));
  }
 
  /* create new progress bar */
  progress = gtk_progress_bar_new ();
  gtk_box_pack_start (GTK_BOX(progress_box), progress, TRUE, TRUE, 0);
  gtk_widget_show (progress);

  progress_seconds = 1;
  progress_minutes = gaim_prefs_get_int (
    "/plugins/gtk/autoprofile/delay_away");
  gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(progress), 0);
  update_seconds (NULL);

  fraction_timeout = GINT_TO_POINTER (g_timeout_add (
    250, update_fraction, NULL));
  second_timeout = GINT_TO_POINTER (g_timeout_add (
    1000, update_seconds, NULL));

  /* Update the list view */
  general_time = (time_t *)malloc(sizeof(time_t));
  time (general_time);
  cur_time = localtime (general_time);
  free (general_time);

  time_string = (char *)malloc (1024);
  *time_string = '\0';

  strftime (time_string, 1023, "%I:%M %p", cur_time);
  
  gtk_list_store_prepend (message_list, &iter);
  gtk_list_store_set (message_list, &iter,
                      0, time_string,
                      1, msg,
                      -1);
  selection = gtk_tree_view_get_selection (
    GTK_TREE_VIEW (message_list_view));
  gtk_tree_selection_select_iter (selection, &iter);

  /* Cap at 10 messages listed */
  if (gtk_tree_model_iter_nth_child 
        (GTK_TREE_MODEL(message_list), &iter, NULL, 10)) {
    gtk_list_store_remove (message_list, &iter);
  }

  /* Update the displayed message */
  gtk_replace_displayed_message (msg);
  response_timeout = time (NULL);
}

/* Disable everything, come back */
static void come_back (GtkWidget *w, GtkWidget *x)
{
  GtkWidget *tmp;
  
  if (dialog) {
    if (!disconnected) {
      if (was_queueing) {
        gaim_signal_disconnect (gaim_conversations_get_handle (),
                                "receiving-im-msg", plugin_handle,
                                GAIM_CALLBACK (queue_received));
      } else {
        gaim_signal_disconnect (gaim_conversations_get_handle (),
                                "received-im-msg", plugin_handle,
                                GAIM_CALLBACK (regular_received));
      }
      disconnected = TRUE;
    }

    tmp = dialog;
    ap_purge_away_queue (&message_queue);
    gtk_widget_destroy (tmp);

    dialog = NULL;
    msg_window = NULL;
    awaytext = NULL;
   
    if (w != tmp) {
      return;
    }
  }

  message_list = NULL;
  message_list_view = NULL;

  message_queue = NULL;

  awayqueue = NULL;
  awayqueuesw = NULL;
  awayqueuebox = NULL;
  if (awayqueuestore)
    g_object_unref (G_OBJECT(awayqueuestore));
  awayqueuestore = NULL; 

  progress = NULL;
  progress_box = NULL;
  g_source_remove (GPOINTER_TO_INT (second_timeout));
  g_source_remove (GPOINTER_TO_INT (fraction_timeout));

  /* Status_back may call come_back, so check */
  if (is_away) {
    status_back ();
  }
}            

static gboolean display_diff_msg (GtkTreeSelection *select, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *string;
 
  /* Figure out which component is selected */
  if (gtk_tree_selection_get_selected (select, &model, &iter)) {
    gtk_tree_model_get (model, &iter, 1, &string, -1);
    gtk_replace_displayed_message (string);
  }

  return FALSE;
}

/* Global procedures */
void gtk_away_msgs_create (char *msg)
{
  GtkWidget *button, *label;
  GtkWidget *hbox, *vbox;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  GtkTreeSelection *selection;
  GtkWidget *sw;
 
  /* Make sure dialog doesn't exist already */
  if (dialog) {
    gtk_away_msgs_update (msg);
    return;
  }

  /* Connect signal */
  if (gaim_prefs_get_bool ("/gaim/gtk/away/queue_messages")) {
    was_queueing = TRUE;
    gaim_signal_connect (gaim_conversations_get_handle (),
                         "receiving-im-msg", plugin_handle,
                         GAIM_CALLBACK (queue_received), NULL);
  } else {
    was_queueing = FALSE;
    gaim_signal_connect (gaim_conversations_get_handle (),
                         "received-im-msg", plugin_handle,
                         GAIM_CALLBACK (regular_received), NULL);
  }

  disconnected = FALSE;

  /* Create main display window */
  GAIM_DIALOG(dialog);
  gtk_window_set_title (GTK_WINDOW(dialog), _("AutoProfile Away!"));
  gtk_widget_realize (dialog);

  vbox = gtk_vbox_new (FALSE, 5);
  gtk_container_add (GTK_CONTAINER (dialog), vbox);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
  gtk_widget_show (vbox);

  /* Set up progress bar container */
  progress_box = gtk_hbox_new (TRUE, 0);
  gtk_box_pack_start (GTK_BOX(vbox), progress_box, FALSE, FALSE, 0);
  gtk_widget_show (progress_box);

  /* Set up box that contains info on the away messages */
  hbox = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
  gtk_widget_show (hbox);

  /* Set up list of past away messages */
  message_list = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  message_list_view = gtk_tree_view_new_with_model (
    GTK_TREE_MODEL (message_list));
  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (
    NULL, renderer, "text", 0, NULL);
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (message_list_view), FALSE);
  gtk_tree_view_append_column (GTK_TREE_VIEW (message_list_view), column);
  sw = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(sw), 
    GTK_POLICY_NEVER, GTK_POLICY_NEVER);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(sw), GTK_SHADOW_IN);
  gtk_container_add (GTK_CONTAINER (sw), message_list_view);
  gtk_box_pack_start (GTK_BOX(hbox), GTK_WIDGET(sw), FALSE, FALSE, 0);
  gtk_widget_show (message_list_view);
  gtk_widget_show (sw);

  selection = gtk_tree_view_get_selection (
    GTK_TREE_VIEW (message_list_view));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (display_diff_msg), NULL);
  gtk_widget_set_size_request (message_list_view, 65, 205);

  /* Set up the window to display away message in */
  msg_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(msg_window), 
    GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(msg_window), 
    GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX(hbox), msg_window, TRUE, TRUE, 0);
  gtk_widget_show (msg_window);

  /* Display the text */
  gtk_away_msgs_update (msg);

  /* Message queue stuff */
  awayqueuebox = gtk_vbox_new (FALSE, 2);
  gtk_box_pack_start (GTK_BOX(vbox), awayqueuebox, FALSE, FALSE, 0);
  
  label = gtk_label_new (NULL);
  gtk_label_set_markup (GTK_LABEL(label), _("<b>Queued messages</b>"));
  gtk_box_pack_start (GTK_BOX(awayqueuebox), label, FALSE, FALSE, 0);
  gtk_widget_set_size_request (awayqueuebox, 450, 80);
 
  awayqueuesw = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(awayqueuesw),
    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(awayqueuesw),
    GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX(awayqueuebox), awayqueuesw, TRUE, TRUE, 0);

  awayqueuestore = gtk_list_store_new (
    3, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
  awayqueue = gtk_tree_view_new_with_model (GTK_TREE_MODEL(awayqueuestore));
  renderer = gtk_cell_renderer_text_new ();

  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW(awayqueue), FALSE);
  column = gtk_tree_view_column_new_with_attributes (NULL, renderer,
    "text", 0, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW(awayqueue), column);
  column = gtk_tree_view_column_new_with_attributes (NULL, renderer,
    "text", 1, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW(awayqueue), column);
  column = gtk_tree_view_column_new_with_attributes (NULL, renderer, 
    "text", 2, NULL); 
  gtk_tree_view_append_column(GTK_TREE_VIEW(awayqueue), column); 


  gtk_container_add(GTK_CONTAINER(awayqueuesw), awayqueue);

  g_signal_connect(G_OBJECT(awayqueue), "button_press_event", 
    G_CALLBACK(ap_dequeue_cb), NULL);

  if (was_queueing) {
    gtk_widget_show_all (awayqueuebox);
  }
  
  button = gaim_pixbuf_button_from_stock(_("I'm Back!"), GTK_STOCK_JUMP_TO, 
      GAIM_BUTTON_HORIZONTAL);
  gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, 0);
  g_signal_connect (G_OBJECT(button), "clicked", 
    G_CALLBACK(come_back), dialog);
  gtk_window_set_focus (GTK_WINDOW(dialog), button);
  gtk_widget_show (button);
 
  /* Finish up */
  g_signal_connect (G_OBJECT(dialog), "destroy",
    G_CALLBACK(come_back), dialog);
  gtk_widget_show (dialog);
}

/* Wrapper function */
void gtk_away_msgs_destroy ()
{
  come_back (dialog, dialog);
}



