/*--------------------------------------------------------------------------*
 * AUTOPROFILE                                                              *
 *                                                                          *
 * A Gaim away message and profile manager that supports dynamic text       *
 *                                                                          *
 * AutoProfile is the legal property of its developers.  Please refer to    *
 * the COPYRIGHT file distributed with this source distribution.            *
 *                                                                          *
 * This program is free software; you can redistribute it and/or modify     *
 * it under the terms of the GNU General Public License as published by     *
 * the Free Software Foundation; either version 2 of the License, or        *
 * (at your option) any later version.                                      *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA *
 *--------------------------------------------------------------------------*/

#include "autoprofile.h"

/* Reads from fortune-style file and returns GList of each quote */
GList *read_fortune_file (const char *filename, int max)
{
  int state;
  char cur_char, *cur_string, *cur_string_start;
  GList *quotes = NULL;

  FILE *fp = fopen (filename, "r");

  if (fp == NULL)
    return NULL;

  /* Modeling the parser as a finite state machine (kind of) */
  state = 0;
  cur_string = (char *)malloc (sizeof (char[1024]));
  cur_string_start = cur_string;

  while (TRUE) {
    cur_char = getc (fp);          
    
    if (cur_char == EOF) { 
      /* End of the file, stop */
      free (cur_string_start);
      break; 
    }

    if (state == 1)  {  // Looking for fortune end
      if (cur_char == '%') {  // Found it
        // Create/assign values
        quotes = g_list_append (quotes, cur_string_start);
        *cur_string = '\0';
        cur_string = (char *)malloc (sizeof (char[1024]));
        cur_string_start = cur_string;
        state = 2; // Ignore the % sign
      } else {
        state = 0;
        if (cur_string - cur_string_start < max) {
          *cur_string = '\n';
          cur_string++;
        }
      } 
    }
    
    if (state == 2) { // Ignore chars right after end of quote
      if (cur_char != '\n' && cur_char != '%')
        state = 0;
    }

    if (state == 0) { // General state, read a char
      if (cur_char == '\n') {
        state = 1; 
      } else {
        // Truncate extra long quotes
        if (cur_string - cur_string_start < max) {
          *cur_string = cur_char;
          cur_string++;
        }
      }
    }
  }

  fclose (fp);
  return quotes;
}

/* Returns 1 if a pattern is found at the start of a string */
int match_start (const char *text, const char *pattern)
{
  while (*pattern) {
    if (!*text || *pattern++ != *text++) 
      return 0;
  }
  return 1;
}

/* Free's a GList as well as the internal contents */
void free_string_list (GList *list)
{
  GList *node = list;

  while (node) {
    free (node->data);
    node = node->next;
  }

  g_list_free (list);
}

/* Check if string is in GList */
gboolean string_list_find (GList *lst, const char *data)
{
  while (lst) {
    if (!strcmp (data, (char *) lst->data)) {
      return TRUE;
    }
    lst = lst->next;
  }

  return FALSE;
}

/* Prints out debug messages with repetitive formatting completed */
void auto_debug (const char *category, const char *message) 
{
  char *result;

  if (message == NULL) 
    message = "NULL";
  
  result = (char *)malloc (strlen(category)+strlen(message)+4);
  *result = '\0';
  
  /* Add in subcategory, message, and newline */
  strcat (result, category);
  strcat (result, ": ");
  strcat (result, message);
  strcat (result, "\n");

  /* Call regular gaim_debug */
  gaim_debug (GAIM_DEBUG_INFO, "autoprofile", result); 
  free (result);
}

