/*
 avicat
 Copyright (C) 2002 Oliver Kurth <oku@masqmail.cx>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "avifile.h"
#include "aviplay.h"
#include "videoencoder.h"
#include "avm_creators.h"
#include "avm_except.h"
#include "version.h"

#include <string.h>
#include <stdio.h>
#include <stdlib.h> // exit
#include <unistd.h>		// for getopt()
#include <limits.h>

static const char* g_pcProgramName = "avicat";

template <class First, class Second> struct pair
{
    First first;
    Second second;

    pair(){};
    pair(First f, Second s) : first(f), second(s) {}
};

typedef pair<double, double>  bounds_pair_t;

struct SInfo
{
    avm::IReadStream* in;
    avm::IWriteStream* out;
    avm::StreamInfo* info;
    avm::IAudioEncoder *aenc;
    avm::IVideoEncoder *venc;
    BITMAPINFOHEADER bh;
    unsigned idx;
    int map;
    bool opened;

    SInfo()
	: in(0), out(0), info(0), aenc(0), venc(0), idx(0),
	map(-1), opened(true) {}
};

struct FInfo
{
    avm::IReadFile* file;
    avm::vector<SInfo> vid;
    avm::vector<SInfo> aud;

    FInfo() : file(0) {}
};

uint_t findAttrInfo(const avm::CodecInfo& cinfo, const avm::string& name,
		    avm::CodecInfo::Direction direction = avm::CodecInfo::Encode)
{
    const avm::vector<avm::AttributeInfo>& list =
	(direction == avm::CodecInfo::Encode) ? cinfo.encoder_info : cinfo.decoder_info;

    for (unsigned i = 0; i < list.size(); i++)
	if (list[i].GetName() == name)
	    return i;
    return avm::vector<const avm::AttributeInfo>::invalid;
}

uint_t findCodecInfo(fourcc_t fourcc,
		     avm::CodecInfo::Direction direction = avm::CodecInfo::Encode)
{
    uint_t i;
    const avm::vector<avm::CodecInfo>& list = video_codecs;

    for(i = 0; i < list.size(); i++){
	if(list[i].direction & direction){
	    if(list[i].fourcc == fourcc)
		return i;
	}
    }
    return avm::vector<const avm::CodecInfo>::invalid;
}

#if 0
IAviWriteStream* createVidOutStream(const IAviReadStream* inStr,
				    IAviWriteFile* outFile)
{
    BITMAPINFOHEADER bh;
    inStr->GetVideoFormatInfo((uint8_t*)&bh, sizeof(bh));
    int ftm = (int) (1000000. * inStr->GetFrameTime());
    ftm = 41708;
    IAviWriteStream* outStr =
	outFile->AddStream(AviStream::Video, (uint8_t*)&bh, sizeof(bh),
			   bh.biCompression, ftm);
    return outStr;
}

IAviWriteStream*
createAudOutStream (const IAviReadStream* inStr, IAviWriteFile* outFile)
{

    WAVEFORMATEX wf;
    inStr->GetAudioFormatInfo((uint8_t*)&wf, 0);
    IAviWriteStream* outStr =
	outFile->AddStream(AviStream::Audio, (uint8_t*)&wf, 18,
			   wf.wFormatTag,
			   wf.nAvgBytesPerSec,
			   wf.nBlockAlign);
    return outStr;
}
#endif

class AviCutter
{
protected:
    static const size_t AUDIO_BUFFER_SIZE = 512000;
    enum stype { AUDIO, VIDEO };

    avm::IWriteFile* outFile;           // output

    avm::vector<bounds_pair_t> boundaries;
    const avm::vector<avm::string>& inFilenames;
    avm::vector<avm::IReadFile*> inFileNext;
    avm::vector<FInfo> finfo;

    avm::vector<SInfo> outVideo;
    avm::vector<SInfo> outAudio;

    //avm::IWriteStream* outVideo;
    //avm::vector<pair<WAVEFORMATEX*, avm::IWriteStream*> > outAudStreams;

    int64_t written_audio;
    int64_t written_frames;
    size_t buf_size;

    uint8_t *buf;
    uint8_t *aud_buf;
    double bpf;

    double streamtime;
    double preload;
    enum { JOIN, NOJOIN } m_Mode;

    void copyAudioFrames(unsigned, int, int, bool);
    void createVideoEncoder(fourcc_t);
    void AviCutter::writeVideoFrame(unsigned, avm::CImage *image);
    void AviCutter::copyVideoFrame();
    void parseBounds(char* bounds)
    {
	while (bounds)
	{
	    char* e = strchr(bounds, ':');
	    if (e)
		*e++ = 0;
	    char* p = strchr(bounds, ',');
	    if (p)
	    {
		*p++ = 0;
		bounds_pair_t pr;
		pr.first = atoi(bounds);
		pr.second = atoi(p);
		if (pr.second > pr.first)
		    boundaries.push_back(pr);
		else
		    fprintf(stderr, "second value must be greater than first: %s\n", bounds);
	    }
	    bounds = e;
	}
    }
    void parseMaps(char* maps)
    {
	while (maps)
	{
	    char* e = strchr(maps, ':');
	    if (e)
		*e++ = 0;
	    char* r = strchr(maps, '=');
	    if (r)
	    {
		*r++ = 0;
                int st = atoi(r);
		char* s = strchr(maps, ',');
		if (s)
		{
		    *s++ = 0;
		    unsigned int f = atoi(maps);
		    if (f < finfo.size())
		    {
			switch (*s++)
			{
			case 'V':
			    finfo[f].vid[st].map = atoi(s);
			    break;
			case 'A':
			    finfo[f].aud[st].map = atoi(s);
			    break;
			default:
			    fprintf(stderr, "not given video type for map: %s\n", maps);
			    break;
			}
		    }
		    else
			fprintf(stderr, "file index to large %d %s\n", f, maps);
		}
		else
		    fprintf(stderr, "not given A/V type for map: %s\n", maps);
	    }
	    else
		fprintf(stderr, "not given dest stream number: %s\n", maps);

            maps = e;
	}
    }
    SInfo* findMinSInfo(int type, int min)
    {
        SInfo* si = 0;
	for (unsigned i = 0; i < finfo.size(); i++)
	{
	    avm::vector<SInfo>* ti = 0;
	    if (type == AUDIO)
		ti = &finfo[i].aud;
	    else if (type == VIDEO)
                ti = &finfo[i].vid;

	    if (ti)
		for (unsigned j = 0; j < ti->size(); j++)
		    if ((!si || (*ti)[j].map < si->map)
			&& (*ti)[j].map > min)
                        si = &(*ti)[j];
	}
        return si;
    }

public:
    AviCutter(avm::IWriteFile* outF,
	      const avm::vector<avm::string>& files,
	      char* bounds, char* maps, char* apreloads
	     )
	: outFile(outF), inFilenames(files),
	written_audio(0), written_frames(0),
	buf_size(512000)
    {
	buf = new uint8_t[buf_size];
	aud_buf = new uint8_t[AUDIO_BUFFER_SIZE];

	parseBounds(bounds);
	if (!boundaries.size())
	{
	    bounds_pair_t all(0, INT_MAX-1);
	    boundaries.push_back(all);
	}

	// open all requested files
        // here we could make some check and eventually fail
	for (unsigned i = 0; i < inFilenames.size(); i++)
	    Open(inFilenames[i]);

	if (maps)
	    parseMaps(maps);
	else
	{
	    //for (unsigned i = 0; i < finfo.size(); i++)
	}

	if (maps)
	{
	    int i = -1;
            SInfo* si;
	    while ((si = findMinSInfo(VIDEO, i)))
	    {
		i = si->map;
		si->out = outFile->AddStream(si->in);
		outVideo.push_back(*si);
	    }
            i = -1;
	    while ((si = findMinSInfo(AUDIO, i)))
	    {
		i = si->map;
		si->out = outFile->AddStream(si->in);
                outVideo.push_back(*si);
	    }
	}
	else
	{

#if 0
	    for (unsigned i = 0; i < finfo.size(); i++)
	    {

            if
	    //if (mapVideo.size() || map
	    if (outVideo.size() <= i )
	    {
		if (maxis > buf_size)
		{
		    buf_size = maxis;
		    delete[] buf;
		    buf = new uint8_t[buf_size];
		}
		si.out = outFile->AddStream(si.in);
                si.in = 0;
                outVideo.push_back(si);
	    }
	    if (outAudio.size() <= i)
	    {
		//bpf = inVidStr->GetFrameTime() * wf->nAvgBytesPerSec;
		si.out = outFile->AddStream(si.in);
                si.in = 0;
		outAudio.push_back(si);
	    }
	    else
	    {
		/* check if format matches */
	    }
#endif
	}
    }

    ~AviCutter()
    {
	delete[] buf;
	delete[] aud_buf;
    }

    int Open(const avm::string& filename)
    {
	FInfo fi;
	fi.file = avm::CreateReadFile(filename.c_str());

	for (unsigned i = 0; i < fi.file->VideoStreamCount(); i++)
	{
	    SInfo si;
	    si.in = fi.file->GetStream(i, avm::IStream::Video);
	    si.info = si.in->GetStreamInfo();
	    uint_t fcc = si.info->GetFormat();

	    uint_t maxis = si.info->GetVideoWidth()
		* si.info->GetVideoHeight() * 4;
	    printf("F:%d Video Stream:%d  %dx%d %.4s\n", finfo.size(),
		   fi.vid.size(), si.info->GetVideoWidth(),
		   si.info->GetVideoHeight(), (const char*)&fcc);

	    fi.vid.push_back(si);
	}

	for (unsigned i = 0; i < fi.file->AudioStreamCount(); i++)
	{
	    SInfo si;

	    si.in = fi.file->GetStream(i, avm::IStream::Audio);
	    si.info = si.in->GetStreamInfo();
	    printf("F:%d Audio Stream:%d\n", finfo.size(), fi.aud.size());
	    fi.aud.push_back(si);
	}

	finfo.push_back(fi);
	return 0;
    }

    void Close(FInfo& fi)
    {
	for (unsigned i = 0; i < fi.vid.size(); i++)
	    delete fi.vid[i].info;

	for (unsigned i = 0; i < fi.aud.size(); i++)
	    delete fi.aud[i].info;

	fi.vid.clear();
	fi.aud.clear();
	delete fi.file;
	fi.file = 0;
    }

    void Copy();
};

/* life would be much easier if we could create the encoder and then
 set the attribute...
 */
void AviCutter::createVideoEncoder(fourcc_t fourcc)
{
#if 0
    if (vidEnc == NULL)
    {
	const avm::CodecInfo& info = video_codecs[findCodecInfo(fourcc)];

	avm::vector<avm::string>::iterator it;
	for(it = attributes.begin(); it != attributes.end(); it++){
	    char name[21], val[21];
	    sscanf(it->c_str(), "%20[^=]=%s", name, val);
	    printf("setting '%s' = '%s'\n", name, val);

	    uint_t index = findAttrInfo(info, name);
	    if(index != avm::vector<avm::AttributeInfo>::invalid){
		const avm::AttributeInfo& ainfo = info.encoder_info[index];
		switch(ainfo.kind){
		case avm::AttributeInfo::Integer:
		    avm::CodecSetAttr(info, name, atoi(val));
		    break;
		case avm::AttributeInfo::String:
		    avm::CodecSetAttr(info, name, val);
		    break;
		case avm::AttributeInfo::Select:
		    avm::CodecSetAttr(info, name, val);
		    break;
		}
	    }
	}
	avm::BitmapInfo bh1(bh.biWidth, bh.biHeight, 24);
	vidEnc = avm::CreateEncoderVideo(fourcc, bh1, NULL);
    }
#endif
}


void AviCutter::copyAudioFrames(unsigned idx, int frame, int max_frame, bool reseek)
{
#if 0
    for (unsigned i = 0; i < finfo[idx].aud.size(); i++)
    {
	avm::IReadStream* inAudStr = finfo[idx].aud.;

        // always according to video frame if there is video time!

	if (reseek)
	{
	    double time = inVidStr->GetTime();
	    inAudStr->SeekTime(time);
	}

	if (!inAudStr->Eof())
	{
	    int64_t excess = int64_t(bpf*(written_frames+1) - written_audio);
            int64_t to_read = int64_t(bpf*(max_frame - frame));

	    if (to_read > AUDIO_BUFFER_SIZE)
		to_read = AUDIO_BUFFER_SIZE;
	    to_read = to_read - (to_read%outAudStreams[i].first->nBlockAlign);

	    while ((excess>0) && (!inAudStr->Eof()))
	    {
		int flags = 0;
		uint_t bytes_read, samp_read;
		inAudStr->ReadDirect(aud_buf, (excess > to_read) ? excess : to_read,
				     to_read, samp_read, bytes_read, &flags);
		written_audio += bytes_read;
		excess -= bytes_read;
                flags |= IStream::KEYFRAME;
		//printf("ADDSAMP  %d\n", bytes_read);
		outAudStreams[i].second->AddChunk(aud_buf, bytes_read, flags);
	    }
	}
    }
#endif
}

void AviCutter::writeVideoFrame(unsigned i, avm::CImage *image)
{
    int is_kf;
    uint_t size;
    int hr = outVideo[i].venc->EncodeFrame(image, buf, &is_kf, &size);
    if (hr == 0)
	outVideo[i].out->AddChunk(buf, size, is_kf);
    else
	fprintf(stderr, "hr != 0\n");
    written_frames++;
}

void AviCutter::copyVideoFrame()
{
    for (unsigned i = 0; i < outVideo.size(); i++)
    {
	avm::IReadStream* inVidStr = 0;

	while (outVideo[i].idx < finfo[i].vid.size())
	{
	    inVidStr = finfo[i].vid[outVideo[i].idx].in;
	    if (!inVidStr->Eof())
		break;

	    // pick next stream
	    outVideo[i].idx++;
	    inVidStr = 0;
	}

	if (!inVidStr)
	    continue; // no more frames in this stream

	if (inVidStr->GetTime() < streamtime)
	{
	    outVideo[i].out->AddChunk(0, 0, 0);
	    printf("Adding empty frame\n");
	}
	else
	{
	    int flags = 0;
	    uint_t bytes_read, samp_read;
	    inVidStr->ReadDirect(buf, buf_size, 1, samp_read, bytes_read, &flags);
	    outVideo[i].out->AddChunk(buf, bytes_read, flags);
	}
    }
    written_frames++;
}

void AviCutter::Copy()
{
    unsigned sidx = 0;
    while (sidx < finfo.size())
    {
	double tmin = INT_MAX;
        SInfo* si = 0;
	for (unsigned i = 0; i < outVideo.size(); i++)
	    if (outVideo[i].in)
	    {
		double st = outVideo[i].in->GetTime();
		if (tmin < st)
		{
		    tmin = st;
                    si = &outVideo[i];
		}
	    }
	for (unsigned i = 0; i < outAudio.size(); i++)
	    if (outAudio[i].in)
	    {
		double st = outAudio[i].in->GetTime() - preload;
		if (tmin < st)
		{
		    tmin = st;
                    si = &outAudio[i];
		}
	    }

	avm::vector<bounds_pair_t>::iterator it_bound = boundaries.begin();
	//printf("INVIDEOF %d\n", inVidStr->Eof());
#if 0
	while (!inVidStr->Eof() && it_bound != boundaries.end())
	{
	    avm::IReadStream* inVidStr = finfo[i].vid[outVideo[i].idx].in;
	    //printf("SEEEK  %d  %d\n", it_bound->first, it_bound->second);
	    inVidStr->SeekToKeyFrame(it_bound->first);
	    uint_t next_kf_pos = inVidStr->GetNextKeyFrame(it_bound->first);
	    bool need_aud_resync = true;
	    bool have_encoder = false;

	    for(framepos_t frame = inVidStr->GetPos(); frame < it_bound->second; frame++)
	    {
		if (!inVidStr->Eof())
		{
		    if (frame < next_kf_pos)
		    {
			if (!have_encoder)
			{
			    if (outVideo[i].venc == NULL)
				createVideoEncoder(bh.biCompression);
			    if (outVideo[i].venc)
			    {
				inVidStr->SetDirection(1);
				inVidStr->StartStreaming();
				outVideo[i].venc->Start();
				have_encoder = true;
			    }
			}
			avm::CImage *image = inVidStr->GetFrame(true);

			if (frame >= it_bound->first)
			{
			    copyAudioFrames(frame, next_kf_pos, need_aud_resync);
			    need_aud_resync = false;

			    if (outVideo[i].venc)
			    {
				writeVideoFrame(image);
				printf("\rreencoded frame %d (%d)", frame, (int)written_frames); fflush(stdout);
			    }
			}
		    }
		    else
		    {
			copyAudioFrames(frame, it_bound->second, need_aud_resync);
			need_aud_resync = false;

			copyVideoFrame();
			printf("\rcopied frame %d (%d)", frame, (int)written_frames); fflush(stdout);
		    }
		    if ((frame+1 == next_kf_pos)
			|| (frame+1 == it_bound->second))
		    {
			if (have_encoder)
			{
			    inVidStr->StopStreaming();
			    outVideo[i].venc->Stop();
			}
		    }
		}
		else
		    break;
	    }

	    it_bound++;
	}
#endif
    }
}

int main (int argc, char* argv[])
try
{
    unsigned int i;
    int frames = 1;
    int debug = 0;
    int seven = 0;
    char* outFn = strdup("out.avi");
    char* bounds = 0;
    char* maps = 0;
    char* preload = 0;
    char* acodec = 0;
    char* vcodec = 0;
    char* oacodec = 0;
    char* ovcodec = 0;
    avm::vector<bounds_pair_t> boundaries;
    avm::vector<avm::string> inFiles;
    int seg_size = 0x7F000000;

    // Standard AVIlib sanity check:
    if (GetAvifileVersion() != AVIFILE_VERSION) {
	fprintf(stderr,
		"This binary was compiled for Avifile ver. %d, , but the library is ver. %d.\nAborting\n",
		AVIFILE_VERSION,
		GetAvifileVersion()
	       );
	return 0;
    }

    static const avm::Args::Option opts[] =
    {
	//{ avm::Args::Option::SUBOPTIONS, "s", "size", "set windows size", (void*) &sizesubopts },
	{ avm::Args::Option::DOUBLE, "p", "preload", "preload audio chunk 1=0.5s:2=10f", &preload },
	{ avm::Args::Option::BOOL, "f", "frames", "audio chunk", &frames },
	{ avm::Args::Option::CODEC, "ac", "audio-codec", "preferred audio decoder (use help for select)", &acodec },
	{ avm::Args::Option::CODEC, "vc", "video-codec", "preferred video decoder (use help for select)", &vcodec },
	//{ avm::Args::Option::CODEC, "oac", 0, "output audio codec (use help for select)", &oacodec },
	//{ avm::Args::Option::CODEC, "ovc", 0, "output video codec (use help for select)", &ovcodec },

	{ avm::Args::Option::STRING, "o", "output", "output filename", &outFn },
	{ avm::Args::Option::STRING, "b", "bounds", "boundaries   start1,end1:s2,e2:...", &bounds },
	{ avm::Args::Option::STRING, "m", "map", "stream mapping f=st_in,st_out  (1=1,0:2=1,1)", &maps },
	{ avm::Args::Option::BOOL, "d", "debug", "debug mode", &debug },
	{ avm::Args::Option::CODEC, "c", "codec", "use with help for more info" },
	{ avm::Args::Option::INT, "s", "size", "size of segment", &seg_size, 0, 2*1024*1024*1024 },
	{ avm::Args::Option::BOOL, "7", 0, "use 700MB size", &seven },
	{ avm::Args::Option::HELP },
	{ avm::Args::Option::NONE }
    };

    avm::Args(opts, &argc, argv,
	      " [options] [files]\n"
	      "  - Joins files\n"
              "  - With mapping it will add/remove streams from .avi files\n"
	      "Warnings: Do not use with VBR audio streams.\n"
	      "          Do not mix concatenating and cutting at once.\n"
	      "          Do just once operation at time!\n"
	      "  Options:\n"
	      "[-b start,end] ...] [-r #x] [-t x:y] file1 [file2] ...\n"
	      " -r x   remove track 'x' (1-based)"
	      " -t x:y use file2 channel 'y' as a track 'x' (1-based)\n\n",
	      g_pcProgramName);

    if (seven)
	seg_size = 700L*1024L*1024L; // 700MB for a CD

    //argc -= optind;
    //argv += optind;
    //if (maps) parseMaps(mapping, maps);
    if (argc > 0)
    {
	for (int arg = 0; arg < argc; arg++)
	    inFiles.push_back(argv[arg]);

	// Do the real work
	avm::IWriteFile* outFile = avm::CreateWriteFile(outFn, seg_size);
	AviCutter cutter(outFile, inFiles, bounds, maps, preload);
	cutter.Copy();
	delete outFile;
    }
}
catch(FatalError& error)
{
    error.Print();
}

/*

File1  File2
inV11  inV21
inA11  inA21
inA12  inA22

maxVtime
maxAtime

if (!veof())
{
for(each)
  copyAudioFrame()
for(each)
  copyVideoFrame()


}

 if (getVPos() < getAPos()

 atime < vtime


File2

/// function list
avicat --list

track
1: video w x h CODEC
2: audio xyz
3: audio xyz


/// function remove
avicat -r 1,2,3 -o avi filename

/// function set track
avicat -t 1 -o avi   filenamefrom  filename_track


*/
