#include <string.h>
#include <time.h>
#include <gtk/gtk.h>
#include "guiutils.h"
#include "cdialog.h"
#include "obj.h"
#include "win.h"
#include "winopcb.h"
#include "winlist.h"
#include "windnd.h"
#include "core.h"
#include "config.h"


static void WinDragDataReceivedNexus(
	win_struct *win, GtkWidget *w,
	GdkDragContext *dc, const guint info,
	GtkSelectionData *selection_data,
	const gint row, const gint column
);

void WinAnimIconDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);

void WinListDNDSetIcon(
	GtkCList *clist, gint row, gint column
);
void WinListDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data                                         
);
void WinListDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,                       
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data                                         
);
void WinListDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);

void WinEntryDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data                                         
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Win "drag_data_received" signal nexus.
 */
static void WinDragDataReceivedNexus(
	win_struct *win, GtkWidget *w,
	GdkDragContext *dc, const guint info,
	GtkSelectionData *selection_data,
	const gint row, const gint column
)
{
	GtkWidget *toplevel = win->toplevel;
	core_struct *core = win->core;

	WinSetBusy(win, TRUE);

	/* Animation Icon */
	if(w == win->anim_icon)
	{
	    GList *paths_list = NULL;
	    GtkWidget *toplevel = win->toplevel;

	    /* Is there a scan or update already in progress? */
	    if((win->scan_context->toid != 0) ||
	       (win->db_update_net_toid != 0)
	    )
	    {
#ifdef HAVE_EDV2
		EDVPlaySoundWarning(core->edv_ctx);
#endif
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Scan Failed",
"A scan or virus database update is already in progress.\n\
\n\
You may either wait for the current scan or update to\n\
complete itself or click on the stop button to stop the\n\
current scan or update before starting a new scan.",
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		WinSetBusy(win, FALSE);
		return;
	    }

	    /* Format the location string based on the target data
	     * type
	     *
	     * String
	     */
	    if((info == DND_INFO_TEXT_PLAIN) ||
	       (info == DND_INFO_TEXT_URI_LIST) ||
	       (info == DND_INFO_STRING)
	    )
	    {
		const gchar *path;
		GList *glist, *obj_list = ObjDDEBufferParsePath(
		    (const guint8 *)selection_data->data,
		    selection_data->length
		);
		obj_struct *obj;

		for(glist = obj_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    obj = OBJ(glist->data);
		    if(obj == NULL)
			continue;

		    path = obj->path;
		    if(STRISEMPTY(path))
			continue;

		    paths_list = g_list_append(
			paths_list,
			STRDUP(path)
		    );

		    ObjDelete(obj);
		}
		g_list_free(obj_list);
	    }
	    /* Scan Item */
	    else if(info == AVSCAN_DND_INFO_SCAN_ITEM)
	    {
		const gulong start_time = (gulong)time(NULL);
		gint i;
		gchar **strv;
		const gchar *location;
		GList *glist, *obj_list = ObjDDEBufferParse(
		    (const guint8 *)selection_data->data,
		    selection_data->length
		);
		obj_struct *obj;

		for(glist = obj_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    obj = OBJ(glist->data);
		    if(obj == NULL)
			continue;

		    location = obj->path;
		    if(STRISEMPTY(location))
			continue;

		    /* Update the last scanned time of all the
		     * scan items who's location matches this
		     * location
		     */
		    WinScanListTouchByLocation(
			win,
			location,
			start_time
		    );

		    /* Need to explode the location so we get
		     * each path
		     */
		    strv = g_strsplit(location, ",", -1);
		    if(strv == NULL)
			continue;

		    /* Append each path from the location to the
		     * paths list
		     */
		    for(i = 0; strv[i] != NULL; i++)
			paths_list = g_list_append(
			    paths_list,
			    STRDUP(strv[i])
			);

		    g_strfreev(strv);

		    ObjDelete(obj);
		}
		g_list_free(obj_list);
	    }
	    else
	    {
		/* Unsupported data */
		gchar	*target_name = gdk_atom_name(
		    selection_data->target
		),
			*type_name = gdk_atom_name(
		    selection_data->type
		),
			*msg = g_strdup_printf(
"Data \"%s\" of type \"%s\",\n\
is not supported by this application.",
		    target_name,
		    type_name
		);
		g_free(target_name);
		g_free(type_name);
#ifdef HAVE_EDV2
		EDVPlaySoundWarning(core->edv_ctx);
#endif
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Unsupported Data",
		    msg,
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(msg);
	    }
	    if(paths_list == NULL)
	    {
		WinSetBusy(win, FALSE);
		return;
	    }

	    /* Start the scan */
	    WinScanProcessStart(
		win,
		NULL,			/* Use the virus database path
					 * specified in the configuration */
		paths_list,
		WinGetCurrentOptions(win)
	    );

	    /* Switch to the results page */
	    WinSwitchPage(win, WIN_PAGE_NUM_RESULTS);

	    /* Delete the paths list */
	    g_list_foreach(paths_list, (GFunc)g_free, NULL);
	    g_list_free(paths_list);
	}
	/* Scan List? */
	else if(w == win->scan_clist)
	{
	    GList *obj_list;
	    GtkCList *clist = GTK_CLIST(w);

	    /* Get the list of objects based on the target data
	     * type
	     *
	     * String
	     */
	    if((info == DND_INFO_TEXT_PLAIN) ||
	       (info == DND_INFO_TEXT_URI_LIST) ||
	       (info == DND_INFO_STRING)
	    )
	    {
		obj_list = ObjDDEBufferParsePath(
		    (const guint8 *)selection_data->data,
		    selection_data->length
		);
	    }
	    /* Scan Item */
	    else if(info == AVSCAN_DND_INFO_SCAN_ITEM)
	    {
		obj_list = ObjDDEBufferParse(
		    (const guint8 *)selection_data->data,
		    selection_data->length
		);
	    }
	    else
	    {
		/* Unsupported data */
		gchar	*target_name = gdk_atom_name(
		    selection_data->target
		),
			*type_name = gdk_atom_name(
		    selection_data->type
		),
			*msg = g_strdup_printf(
"Data \"%s\" of type \"%s\",\n\
is not supported by this application.",
		    target_name,
		    type_name
		);
		g_free(target_name);
		g_free(type_name);
#ifdef HAVE_EDV2
		EDVPlaySoundWarning(core->edv_ctx);
#endif
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Unsupported Data",
		    msg,
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(msg);

		obj_list = NULL;
	    }
	    if(obj_list == NULL)
	    {
		WinSetBusy(win, FALSE);
		return;
	    }

	    /* Begin adding the objects to the scan list */
	    gtk_clist_freeze(clist);

	    /* Insert or append? */
	    if(row > -1)
	    {
		gint ins_row = row;
		GList *glist;
		obj_struct *obj;

		for(glist = obj_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    obj = OBJ(glist->data);
		    ins_row = WinScanListInsert(
			win, ins_row, ObjCopy(obj)
		    );
		    ObjDelete(obj);
		    if(ins_row < 0)
			break;
		    ins_row++;
		}
	    }
	    else
	    {
		gint new_row;
		GList *glist;
		obj_struct *obj;

		for(glist = obj_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{   
		    obj = OBJ(glist->data);
		    new_row = WinScanListAppend(win, ObjCopy(obj));
		    ObjDelete(obj);
		    if(new_row < 0)
			break;
		}
	    }
	    g_list_free(obj_list);

	    gtk_clist_thaw(clist);
	}
	/* Location Entry */
	else if(w == win->scan_location_entry)
	{
	    gchar *location;
	    GtkEntry *entry = GTK_ENTRY(w);

	    /* Format the location string based on the target data
	     * type
	     *
	     * String
	     */
	    if((info == DND_INFO_TEXT_PLAIN) ||
	       (info == DND_INFO_TEXT_URI_LIST) ||
	       (info == DND_INFO_STRING)
	    )
	    {
		gchar *s;
		GList *glist, *obj_list = ObjDDEBufferParsePath(
		    (const guint8 *)selection_data->data,
		    selection_data->length
		);
		obj_struct *obj;

		location = STRDUP("");
		for(glist = obj_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    obj = OBJ(glist->data);
		    if(obj == NULL)
			continue;

		    s = g_strconcat(
			location,
			obj->path,
			(g_list_next(glist) != NULL) ? "," : "",
			NULL
		    );
		    g_free(location);
		    location = s;

		    ObjDelete(obj);
		}
		g_list_free(obj_list);
	    }
	    /* Scan Item */
	    else if(info == AVSCAN_DND_INFO_SCAN_ITEM)
	    {
		gchar *s;
		GList *glist, *obj_list = ObjDDEBufferParse(
		    (const guint8 *)selection_data->data,
		    selection_data->length
		);
		obj_struct *obj;

		location = STRDUP("");
		for(glist = obj_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    obj = OBJ(glist->data);
		    if(obj == NULL)
			continue;

		    s = g_strconcat(
			location,
			obj->path,
			(g_list_next(glist) != NULL) ? "," : "",
			NULL
		    );
		    g_free(location);
		    location = s;

		    ObjDelete(obj);
		}
		g_list_free(obj_list);
	    }
	    else
	    {
		/* Unsupported data */
		gchar	*target_name = gdk_atom_name(
		    selection_data->target
		),
			*type_name = gdk_atom_name(
		    selection_data->type
		),
			*msg = g_strdup_printf(
"Data \"%s\" of type \"%s\",\n\
is not supported by this application.",
		    target_name,
		    type_name
		);
		g_free(target_name);
		g_free(type_name);
#ifdef HAVE_EDV2
		EDVPlaySoundWarning(core->edv_ctx);
#endif
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Unsupported Data",
		    msg,
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(msg);

		location = NULL;
	    }
	    if(location == NULL)
	    {
		WinSetBusy(win, FALSE);
		return;
	    }

	    /* Set the location entry */
	    gtk_entry_set_text(entry, location);
	    gtk_entry_set_position(entry, -1);

	    g_free(location);
	}
	/* Results List */
	else if(w == win->results_clist)
	{

	}

	WinUpdate(win);

	WinSetBusy(win, FALSE);
}


/*
 *	Animation Icon "drag_data_received" signal callback.
 */
void WinAnimIconDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	win_struct *win = WIN(data);
	if((widget == NULL) || (dc == NULL) || (win == NULL))
	    return;
	
	WinDragDataReceivedNexus(
	    win, widget, dc, info, selection_data,
	    0, 0
	);                                         
}


/*
 *	Sets the DND icon based on the specified cell on the Win's
 *	GtkCList.
 */
void WinListDNDSetIcon(
	GtkCList *clist, gint row, gint column   
)
{
	if(clist == NULL)
	    return;

	if((row >= 0) && (row < clist->rows))
	{
	    gint i;
	    gchar *text = NULL;
	    guint8 spacing = 0;
	    GdkPixmap *pixmap = NULL;
	    GdkBitmap *mask = NULL;

	    /* Search for the first cell that has a useable pixmap
	     * for the drag icon
	     */
	    for(i = 0; i < clist->columns; i++)
	    {
		switch(gtk_clist_get_cell_type(clist, row, i))
		{
		  case GTK_CELL_PIXMAP:
		    gtk_clist_get_pixmap(
			clist, row, i,
			&pixmap, &mask
		    );
		    break;
		  case GTK_CELL_PIXTEXT:
		    gtk_clist_get_pixtext(
			clist, row, i,
			&text, &spacing, &pixmap, &mask
		    );
		    break; 
		  case GTK_CELL_TEXT:
		  case GTK_CELL_WIDGET:
		  case GTK_CELL_EMPTY:
		    break;
		}  
		if(pixmap != NULL) 
		    break;
	    }

	    /* If we got a pixmap then set it as the drag icon */
	    if(pixmap != NULL)
	    {
		gint width, height;
		gdk_window_get_size(pixmap, &width, &height);
		GUIDNDSetDragIcon(
		    pixmap, mask,
		    width / 2, height / 2
		);
	    }     
	}
}


/*
 *	Win GtkCList "drag_data_get" signal callback.
 */
void WinListDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	gint buf_len = 0;
	guint8 *buf = NULL;
	GtkCList *clist;
	win_struct *win = WIN(data);
	if((widget == NULL) || (dc == NULL) || (win == NULL))
	    return;

	clist = GTK_IS_CLIST(widget) ? GTK_CLIST(widget) : NULL;
	if(clist == NULL)
	    return;


	/* Handle by widget */

	/* Scan List */
	if(widget == win->scan_clist)
	{
	    GList *glist;
	    obj_struct *obj;

	    for(glist = clist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		obj = OBJ(gtk_clist_get_row_data(clist, (gint)glist->data));
		if(obj == NULL)
		    continue;

	        buf = ObjDDEBufferAppend(
		    buf, &buf_len, obj
		);
	    }
	}
	/* Results List */
	else if(widget == win->results_clist)
	{
	    GList *glist;
	    obj_struct *obj;

	    for(glist = clist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		obj = OBJ(gtk_clist_get_row_data(clist, (gint)glist->data));
		if(obj == NULL)
		    continue;

		if(obj->path != NULL)
		    buf = ObjDDEBufferAppendPath(
			buf, &buf_len, obj
		    );
	    }
	}

	/* Got buffer to send out? */
	if(buf != NULL)
	{
	    gtk_selection_data_set( 
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,                      /* Bits Per Character */
		buf, buf_len
	    );
	    data_sent = TRUE;
	    g_free(buf);
	}

	/* If no data was sent out then send an error response */
	if(!data_sent)
	{
	    const gchar *s = "Error";
	    gtk_selection_data_set(
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,			/* Bits Per Character */
		s,			/* Data */
		STRLEN(s)		/* Length */
	    ); 
	    data_sent = TRUE;
	}
}

/*
 *	Win GtkCList "drag_data_received" signal callback.
 */
void WinListDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gint row, column;
	GtkCList *clist;
	win_struct *win = WIN(data);
	if((widget == NULL) || (dc == NULL) || (win == NULL))
	    return;

	clist = GTK_IS_CLIST(widget) ? GTK_CLIST(widget) : NULL;
	if(clist == NULL)
	    return;

	/* Find the row and column from the specified coordinates */
	if(!gtk_clist_get_selection_info(
	    clist,
	    x,    
	    y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
		clist->column_title_area.height +
		clist->column_title_area.y : 0),
	    &row, &column
	))               
	{
	    row = -1;
	    column = 0;
	}

	WinDragDataReceivedNexus(
	    win, widget, dc, info, selection_data,
	    row, column
	);                                         
}

/*
 *	Win GtkCList "drag_data_delete" signal callback.
 */
void WinListDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	gint row;
	GList *glist, *obj_list;
	GtkCList *clist;
	win_struct *win = WIN(data);
	if((widget == NULL) || (dc == NULL) || (win == NULL))
	    return;

	clist = GTK_IS_CLIST(widget) ? GTK_CLIST(widget) : NULL;
	if(clist == NULL)
	    return;

	if(clist->selection == NULL)
	    return;

	gtk_clist_freeze(clist);

	/* Copy each selected row's data to obj_list */
	obj_list = NULL;
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    row = (gint)glist->data;
	    obj_list = g_list_append(
		obj_list,
		gtk_clist_get_row_data(clist, row)
	    );
	}

	/* Delete each row who's has the data specified in the obj_list */
	for(glist = obj_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    row = gtk_clist_find_row_from_data(clist, glist->data);
	    if(row > -1)
		gtk_clist_remove(clist, row);
	}

	g_list_free(obj_list);

	gtk_clist_thaw(clist);
}


/*       
 *	Win GtkEntry "drag_data_received" signal callback.
 */
void WinEntryDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data                                         
)
{
	GtkEntry *entry;
	win_struct *win = WIN(data);
	if((widget == NULL) || (dc == NULL) || (win == NULL))
	    return;

	entry = GTK_IS_ENTRY(widget) ? GTK_ENTRY(widget) : NULL;
	if(entry == NULL)
	    return;

	WinDragDataReceivedNexus(
	    win, widget, dc, info, selection_data,
	    0, 0
	);
}
