#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/stat.h>
#ifdef _WIN32
# include <windows.h>
#else
# include <unistd.h>
#endif

#if defined(HAVE_CLAMAV)
# include <clamav.h>
# if !defined(USE_CLAMAV_88) && !defined(CL_DB_STDOPT)
#  error CL_DB_STDOPT was not defined, you are most likely using an older version of ClamAV, see comment about USE_CLAMAV_88 in the Makefile
# endif
#else
#warning No antivirus library defined, this module will not perform any virus scanning.
#endif

#include <glib.h>

#include "../include/disk.h"

#include "avscanop.h"
#include "config.h"


/* Utilities */
static void PRINT_ERR(const gchar *msg);
static void PRINT_SCANNING_FILE(
	const gchar *path,
	const gulong i, const gulong n
);
static void PRINT_SCANNED(
	const gulong files_scanned,
	const gulong data_scanned_blocks,
	const gulong total_data_scanned_blocks
);
static void PRINT_SCANNING_DIR(const gchar *path);
static void PRINT_CLEAN(const gchar *path);
static void PRINT_INFECTED(const gchar *path, const gchar *virus);
static void PRINT_PROBLEM(const gchar *path, const gchar *problem);

static gboolean AVScanIsExecutable(
	const gchar *path, const mode_t m
);
static gchar *AVScanCopyShortenString(const gchar *s, const gint m);


#if defined(HAVE_CLAMAV)
typedef struct {
	gchar		*database_path;		/* Database file or directory */
 	gchar		*database_dir;		/* Database directory */
#if defined(USE_CLAMAV_88)
	struct cl_node	*database;		/* Database handle */
#else	/* ClamAV 90.x or 91.x */
	struct cl_engine	*database;	/* Database handle */
#endif
	gint		database_entries;	/* Entries in the database */
	struct cl_stat	database_stat;		/* Database stats, for
						 * checking database changes */
	gulong		files_scanned,		/* Files scanned so far */
			size_scanned_blocks,
			infected_files,
			total_files;		/* Total files to be scanned */
} ClamAVData;
static ClamAVData *AVScanOPClamAVInit(
	const gchar *db_path,
	gint *status
);
static void AVScanOPClamAVShutdown(ClamAVData *d);
static gint AVScanOPClamAVScanFile(
	ClamAVData *d, const gchar *path,
	gint *stop_count,
	scanner_opts_struct *scanner_opts
);
static gint AVScanOPClamAVScanDir(
	ClamAVData *d, const gchar *path,
	const gboolean recursive,
	const gboolean executables_only,
	const gboolean ignore_links,
	gboolean *reload, gint *stop_count,
	scanner_opts_struct *scanner_opts
);
#endif	/* HAVE_CLAMAV */


static gulong AVScanOPCalculateTotalFilesInDirIterate(
	const gchar *path,
	const gboolean recursive,
	const gboolean executables_only,
	const gboolean ignore_links,
	gint *stop_count
);
static gulong AVScanOPCalculateTotalFiles(
	GList *paths_list,
	const gboolean recursive,
	const gboolean executables_only,
	const gboolean ignore_links,
	gint *stop_count
);

gchar *AVScanGetAVEngineNameVersionString(void);
gint AVScanOP(
	const gchar *db_path,
	GList *paths_list,
	const gboolean recursive,
	const gboolean executables_only,
	const gboolean ignore_links,
	gboolean *reload, gint *stop_count,
	scanner_opts_struct *scanner_opts
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)



/*
 *	Prints the specified path and message to stderr.
 *
 *	This is intended to indicate and have the monitoring process
 *	record the object specified by path as having a virus or
 *	other problem specified by msg.
 */
static void PRINT_ERR(const gchar *msg)
{
	g_printerr("%s\n", msg);
}

/*
 *	Prints the specified path in a message to stdout to indicate
 *	it is being scanned.
 *
 *	This is intended to indicate and have the monitoring process
 *	display this message on its status bar.
 */
static void PRINT_SCANNING_FILE(
	const gchar *path,
	const gulong i, const gulong n
)
{
	g_print("Scanning file: \"%s\" %ld %ld\n", path, i, n);
}

static void PRINT_SCANNED(
	const gulong files_scanned,
	const gulong data_scanned_blocks,
	const gulong total_data_scanned_blocks
)
{
	g_print("Scanned file: %ld %ld %ld\n",
	    files_scanned,
	    data_scanned_blocks, total_data_scanned_blocks
	);
}

static void PRINT_SCANNING_DIR(const gchar *path)
{
	g_print("Scanning directory: \"%s\"\n", path);
}

static void PRINT_CLEAN(const gchar *path)
{
	g_print("Clean: \"%s\"\n", path);
}

static void PRINT_INFECTED(const gchar *path, const gchar *virus)
{
	g_print("Infected: \"%s\" %s\n", path, virus);
}

static void PRINT_PROBLEM(const gchar *path, const gchar *problem)
{
	g_print("Problem: \"%s\" %s\n", path, problem);
}


/*
 *	Checks if the object is an executable.
 */
static gboolean AVScanIsExecutable(
	const gchar *path, const mode_t m
)
{
#if defined(_WIN32)
	const gchar *ext, *name;

	if(path == NULL)
	    return(FALSE);

	/* Get the name */
	name = g_basename(path);
	if(name == NULL)
	    name = path;

	/* Get the extension */
	ext = (const gchar *)strrchr((const char *)name, '.');
	if(ext == NULL)
	    return(FALSE);

	ext++;		/* Seek past the '.' deliminator */

	if(!g_strcasecmp(ext, "bat") ||
	   !g_strcasecmp(ext, "com") ||
	   !g_strcasecmp(ext, "exe") ||
	   !g_strcasecmp(ext, "bin") ||
	   !g_strcasecmp(ext, "dll") ||
	   !g_strcasecmp(ext, "lib") ||
	   !g_strcasecmp(ext, "so") ||
	   !g_strcasecmp(ext, "o") ||
	   !g_strcasecmp(ext, "a")
	)
	    return(TRUE);
	else
	    return(FALSE);
#else
	if((m & S_IXUSR) || (m & S_IXGRP) || (m & S_IXOTH))
	    return(TRUE);
	else
	    return(FALSE);
#endif
}


/*
 *	Returns a copy of the string that is no longer than max
 *	characters with appropriate shortened notation where 
 *	appropriate.
 */
static gchar *AVScanCopyShortenString(const gchar *s, const gint m)
{
	gint len;

	if(s == NULL)
	    return(NULL);

	len = STRLEN(s);
	if((len > m) && (m > 3))
	{
	    /* Need to shorten string */
	    const gint i = len - m + 3;

	    return(g_strdup_printf(
		"...%s", (const gchar *)(s + i)
	    ));
	}      
	else
	{   
	    return(STRDUP(s));
	}
}


#if defined(HAVE_CLAMAV)
/*
 *	Initializes the Clam AntiVirus.
 */
static ClamAVData *AVScanOPClamAVInit(
	const gchar *db_path,
	gint *status
)
{
	gint status2;
	gchar *s;
	struct stat stat_buf;
	ClamAVData *d = (ClamAVData *)g_malloc0(
	    sizeof(ClamAVData)
	);
	if(d == NULL)
	{
	    PRINT_ERR("Memory allocation error");
	    *status = 3;
	    return(NULL);
	}

	d->files_scanned = 0l;
	d->size_scanned_blocks = 0l;
	d->infected_files = 0l;
	d->total_files = 0l;

	/* If no virus database was specified then get the default
	 * virus database set by ClamAV
	 */
	if(db_path == NULL)
	    db_path = cl_retdbdir();
	if(db_path == NULL)
	{
	    PRINT_ERR("Virus database location was not specified");
	    g_free(d);
	    *status = 50;
	    return(NULL);
	}
	d->database_path = STRDUP(db_path);
	if(d->database_path == NULL)
	{
	    PRINT_ERR("Memory allocation error");
	    g_free(d);
	    *status = 3;
	    return(NULL);
	}

	/* Get the virus database's stats */
	if(stat((const char *)d->database_path, &stat_buf))
	{
	    const gint error_code = (gint)errno;
	    PRINT_PROBLEM(d->database_path, g_strerror(error_code));
	    g_free(d->database_path);
	    g_free(d);
	    *status = 50;
	    return(NULL);
	}

	/* Load the virus database */ 
	s = AVScanCopyShortenString(d->database_path, 50);
	g_print("Loading virus database \"%s\"...\n", s);
	g_free(s);

	d->database = NULL;
	d->database_entries = 0;

#ifdef S_ISDIR
	if(S_ISDIR(stat_buf.st_mode))
#else
	if(FALSE)
#endif
	{
	    d->database_dir = STRDUP(d->database_path);
	    if(d->database_dir == NULL)
	    {
		PRINT_ERR("Memory allocation error");
		g_free(d->database_path);
		g_free(d);
		*status = 3;
		return(NULL);
	    }

#if defined(USE_CLAMAV_88)
	    status2 = cl_loaddbdir(
		d->database_path,
		&d->database, &d->database_entries
	    );
#else	/* ClamAV 90.x or 91.x */
	    status2 = cl_load(
		d->database_path,
		&d->database, &d->database_entries,
		CL_DB_STDOPT
	    );
#endif
	}
	else
	{
	    d->database_dir = g_dirname(d->database_path);
	    if(d->database_dir == NULL)
	    {
		PRINT_ERR("Memory allocation error");
		g_free(d->database_path);
		g_free(d);
		*status = 3;
		return(NULL);
	    }

#if defined(USE_CLAMAV_88)
	    status2 = cl_loaddb(
		d->database_path,
		&d->database, &d->database_entries
	    );
#else	/* ClamAV 90.x or 91.x */
	    status2 = cl_load(
		d->database_path,
		&d->database, &d->database_entries,
		CL_DB_STDOPT
	    );
#endif
	}
	if((status2 != 0) || (d->database == NULL) || (d->database_entries <= 0))
	{
	    gchar *s = g_strconcat(
		"Virus database load error: ",
		cl_strerror(status2),
		NULL
	    );
	    PRINT_PROBLEM(d->database_path, s);
	    g_free(s);
	    if(d->database != NULL)
		cl_free(d->database);
	    g_free(d->database_path);
	    g_free(d->database_dir);
	    g_free(d);
	    *status = 50;
	    return(NULL);
	}

	/* Build the virus database */
	status2 = cl_build(d->database);
	if(status2 != 0)
	{
	    gchar *s = g_strconcat(
		"Virus database build error: ",
		cl_strerror(status2),
		NULL
	    );
	    PRINT_PROBLEM(d->database_path, s);
	    g_free(s);
	    if(d->database != NULL)
		cl_free(d->database);
	    g_free(d->database_path);
	    g_free(d->database_dir);
	    g_free(d);
	    *status = 50;
	    return(NULL);
	}

	/* Get the virus database directory stats */
	memset(&d->database_stat, 0x00, sizeof(d->database_stat));
	if(cl_statinidir(d->database_dir, &d->database_stat))
	{
	    PRINT_PROBLEM(
		d->database_dir,
		"Unable to get the virus database directory statistics"
	    );
	    if(d->database != NULL)
		cl_free(d->database);
	    g_free(d->database_path);
	    g_free(d->database_dir);
	    g_free(d);
	    *status = 50;
	    return(NULL);
	}

	return(d);
}

/*
 *	Shuts down the Clam AntiVirus.
 */
static void AVScanOPClamAVShutdown(ClamAVData *d)
{
	if(d == NULL)
	    return;

	/* Virus database */
	if(d->database != NULL)
	    cl_free(d->database);

	/* Virus database stat */
	cl_statfree(&d->database_stat);

	g_free(d->database_path);
	g_free(d->database_dir);

	g_free(d);
}

/*
 *	Use the Clam AntiVirus to scan the specified file.
 */
static gint AVScanOPClamAVScanFile(
	ClamAVData *d, const gchar *path,
	gint *stop_count,
	scanner_opts_struct *scanner_opts
)
{
	struct cl_limits limits;
	unsigned long int scanned_blocks = 0l;
	gint status, fd;
	guint options;
	const gchar *virus_name = NULL;

	/* Interrupted? */
	if(*stop_count > 0)
	    return(4);

	PRINT_SCANNING_FILE(
	    path,
	    d->files_scanned,
	    d->total_files
	);

	/* Open the file for reading */
	fd = open((const char *)path, O_RDONLY);
	if(fd < 0)
	{
	    /* Unable to open the file for reading, report error */
	    const gint error_code = (gint)errno;
	    PRINT_PROBLEM(path, g_strerror(error_code));

	    /* Unable to open an input file */
	    return(54);
	}

	/* Interrupted? */
	if(*stop_count > 0)
	{
	    /* Close the file but discard the return value because we
	     * are interrupting and thus do not care of buffered
	     * read()s or write()s
	     */
	    (void)close(fd);
	    return(4);
	}


	/* Set up archive limits */
	memset(&limits, 0x00, sizeof(struct cl_limits));

	/* Maximum files to scan within an archive */
	limits.maxfiles = (unsigned int)scanner_opts->max_files;

	/* Maximum archived file size in bytes */
	limits.maxfilesize = (unsigned long int)scanner_opts->max_file_size_bytes;

	/* Maximum recursion level */
	limits.maxreclevel = (unsigned int)scanner_opts->max_recursions;

	/* Maximum mail recursion level */
	limits.maxmailrec = (unsigned int)scanner_opts->max_mail_recursions;

	/* Maximum compression ratio */
	limits.maxratio = (unsigned int)scanner_opts->max_compression_ratio;

	/* Disable memory limit for bzip2 scanner */
	limits.archivememlim = 0;


	/* Set up scan options */
#if defined(CL_SCAN_STDOPT)
	options = CL_SCAN_STDOPT;
#elif defined(CL_SCAN_RAW) && defined(CL_SCAN_ARCHIVE) && defined(CL_SCAN_MAIL) && defined(CL_SCAN_HTML)
	options = CL_SCAN_RAW | CL_SCAN_ARCHIVE | CL_SCAN_MAIL |
		  CL_SCAN_HTML;
#elif defined(CL_ARCHIVE) && defined(CL_MAIL)
	options = CL_ARCHIVE | CL_MAIL;
#elif defined(CL_SCAN_RAW)
	options = CL_SCAN_RAW;
#else
#warning No supported CL_SCAN_* options are #defined, options will always be 0.
 	options = 0;
#endif

	/* Scan this file */
	status = (gint)cl_scandesc(
	    fd,
	    &virus_name, &scanned_blocks,
	    d->database,
	    &limits,
	    (unsigned int)options
	);

	/* Close the file */
	if(close(fd) != 0)
	{
	    /* Unable to close the file, report problem */
	    const gint error_code = (gint)errno;
	    PRINT_PROBLEM(path, g_strerror(error_code));

	    /* An error occured while reading an input file */
	    return(55);
	}

	/* Count this file as scanned and how much scanned */
	d->files_scanned++;
	d->size_scanned_blocks += (gulong)scanned_blocks;

#if defined(USE_CLAMAV_88)
	PRINT_SCANNED(
	    d->files_scanned,
# ifdef CL_COUNT_PRECISION
	    (gulong)(scanned_blocks *
		((gfloat)CL_COUNT_PRECISION / 1024.0f)
	    ),
	    (gulong)((gfloat)d->size_scanned_blocks *
		((gfloat)CL_COUNT_PRECISION / 1024.0f)
	    )
# else     
# warning CL_COUNT_PRECISION not defined, assuming old ClamAV 1:1 precision.
	    (gulong)scanned_blocks * 1l,
	    d->size_scanned_blocks * 1l
# endif	/* CL_COUNT_PRECISION */
	);
#else	/* ClamAV 90.x or 91.x */
	PRINT_SCANNED(
	    d->files_scanned,
	    (gulong)(scanned_blocks *
		((gfloat)CL_COUNT_PRECISION / 1024.0f)
	    ),
	    (gulong)((gfloat)d->size_scanned_blocks *
		((gfloat)CL_COUNT_PRECISION / 1024.0f)
	    )
	);
#endif

	/* Check the scan results */

	/* Got a virus? */
#if defined(CL_VIRUS)
	if(status == CL_VIRUS)
#else
	if(status == 1)
#endif
	{
	    /* Detected a virus, print the virus message */
	    PRINT_INFECTED(path, virus_name);

	    d->infected_files++;	/* Count this infected file */

	    return(1);
	}
	/* Clean? */
#if defined(CL_CLEAN)
	else if(status == CL_CLEAN)
#elif defined(CL_SUCCESS)
	else if(status == CL_SUCCESS)
#else
	else if(status == 0)
#endif
	{
	    /* Interrupted? */
	    if(*stop_count > 0)
		return(4);

	    /* Print the clean message only if reporting all objects */
	    if(scanner_opts->flags & SCANNER_FLAG_REPORT_ALL_OBJECTS)
		PRINT_CLEAN(path);

	    return(0);
	}
	/* Interrupted */
#if defined(CL_BREAK)
	else if(status == CL_BREAK)
#else
	else if(status == 2)
#endif
	{
	    return(4);
	}
	/* Other error */
	else
	{
	    gchar *error_msg = STRDUP(cl_strerror((int)status));
	    if(error_msg != NULL)
	    {
		*error_msg = (gchar)toupper((int)*error_msg);
		PRINT_PROBLEM(path, error_msg);
		g_free(error_msg);
	    }
	    switch(status)
	    {
#ifdef CL_EMAXREC
	      case CL_EMAXREC:
		status = 57;	/* Scan of object was incomplete because
				 * it would otherwise exceed a specified
				 * limit */
		break;
#endif
#ifdef CL_EMAXSIZE
	      case CL_EMAXSIZE:
		status = 57;	/* Scan of object was incomplete because
				 * it would otherwise exceed a specified
				 * limit */
		break;
#endif
#ifdef CL_EMAXFILES
	      case CL_EMAXFILES:
		status = 57;	/* Scan of object was incomplete because
				 * it would otherwise exceed a specified
				 * limit */
		break;
#endif
#ifdef CL_ERAR
	      case CL_ERAR:
		status = 59;	/* Scanner subsystem or plugin error */
		break;
#endif
#ifdef CL_EZIP
	      case CL_EZIP:
		status = 59;	/* Scanner subsystem or plugin error */
		break;
#endif
#ifdef CL_EMALFZIP
	      case CL_EMALFZIP:
		status = 51;	/* An input file has a corrupt format or
				 * a format not consistent with its
				 * extension */
		break;
#endif
#ifdef CL_EGZIP
	      case CL_EGZIP:
		status = 59;	/* Scanner subsystem or plugin error */
		break;
#endif
#ifdef CL_EBZIP
	      case CL_EBZIP:
		status = 59;	/* Scanner subsystem or plugin error */
		break;
#endif
#ifdef CL_EOLE2
	      case CL_EOLE2:
		status = 59;	/* Scanner subsystem or plugin error */
		break;
#endif
#ifdef CL_EMSCOMP
	      case CL_EMSCOMP:
		status = 59;	/* Scanner subsystem or plugin error */
		break;
#endif
#ifdef CL_EMSCAB
	      case CL_EMSCAB:
		status = 59;	/* Scanner subsystem or plugin error */
		break;
#endif
#ifdef CL_EACCES
	      case CL_EACCES:
		status = 55;	/* An error occured while reading an input
				 * file */
		break;
#endif
#ifdef CL_ENULLARG
	      case CL_ENULLARG:
		status = 2;	/* Invalid value */
		break;
#endif
#ifdef CL_ETMPFILE
	      case CL_ETMPFILE:
		status = 63;	/* Unable to create a tempory file or
				 * directory */
		break;
#endif
#ifdef CL_EFSYNC
	      case CL_EFSYNC:
		status = 3;	/* fsync() failed (systems error) */
		break;
#endif
#ifdef CL_EMEM
	      case CL_EMEM:
		status = 3;	/* Memory allocation error */
		break;
#endif
#ifdef CL_EOPEN
	      case CL_EOPEN:
		status = 54;	/* Unable to open an input file */
		break;
#endif
#ifdef CL_EMALFDB
	      case CL_EMALFDB:
		status = 50;	/* Unable to load or initialize the virus
                                 * database */
		break;
#endif
#ifdef CL_EPATSHORT
	      case CL_EPATSHORT:
		status = 2;	/* Pattern too short (invalid value) */
		break;
#endif
#ifdef CL_ETMPDIR
	      case CL_ETMPDIR:
		status = 63;	/* Unable to create a tempory file or
				 * directory */
		break;
#endif
#ifdef CL_ECVD
	      case CL_ECVD:
		status = 50;	/* Unable to load or initialize the virus
				 * database */
		break;
#endif
#ifdef CL_ECVDEXTR
	      case CL_ECVDEXTR:
		status = 50;	/* Unable to load or initialize the virus
				 * database */
		break;
#endif
#ifdef CL_EMD5
	      case CL_EMD5:
		status = 50;	/* Unable to load or initialize the virus
				 * database */
		break;
#endif
#ifdef CL_EDSIG
	      case CL_EDSIG:
		status = 50;	/* Unable to load or initialize the virus
				 * database */
		break;
#endif
#ifdef CL_EIO
	      case CL_EIO:
		status = 58;	/* I/O error */
		break;
#endif
#ifdef CL_EFORMAT
	      case CL_EFORMAT:
		status = 51;	/* An input file has a corrupt format or
				 * a format not consistent with its
				 * extension */
		break;
#endif
#ifdef CL_ESUPPORT
	      case CL_ESUPPORT:
		status = 60;	/* Unsupported file format */
		break;
#endif
#ifdef CL_ELOCKDB
	      case CL_ELOCKDB:
		status = 61;	/* Unable to lock the virus database
				 * directory */
		break;
#endif
	      default:
		status = 1;
		break;
	    }
	    return(status);
	}
}

/*
 *	Use Clam AntiVirus to scan the specified directory.
 */
static gint AVScanOPClamAVScanDir(
	ClamAVData *d, const gchar *path,
	const gboolean recursive,
	const gboolean executables_only,
	const gboolean ignore_links,
	gboolean *reload, gint *stop_count,
	scanner_opts_struct *scanner_opts
)
{
	mode_t m;
	gint status, status2;
	const gchar *s;
	gchar *child;
	struct dirent *dent;
	struct stat stat_buf;
	DIR *dp;

	/* Interrupted? */
	if(*stop_count > 0)
	    return(4);

	/* Open the directory */
	dp = opendir(path);
	if(dp == NULL)
	{
	    const gint error_code = (gint)errno;
	    PRINT_PROBLEM(path, g_strerror(error_code));
	    return(53);
	}

	/* Interrupted? */
	if(*stop_count > 0)
	{
	    closedir(dp);
	    return(4);
	}

	/* Scan directory */
	PRINT_SCANNING_DIR(path);
	status = 0;
	while(*stop_count == 0)
	{
	    dent = readdir(dp);
	    if(dent == NULL)
		break;

	    s = (const gchar *)dent->d_name;

	    /* Check for virus database change */
	    if(*reload || (cl_statchkdir(&d->database_stat) == 1))
	    {
		/* Reload the virus database */
		struct stat stat_buf;
		gchar *s = AVScanCopyShortenString(d->database_path, 50);
		g_print("Reloading virus database \"%s\"...\n", s);
		g_free(s);

		/* Reset the reload marker */
		if(*reload)
		    *reload = FALSE;

		/* Delete the existing virus database */
		if(d->database != NULL)
		{
		    cl_free(d->database);
		    d->database = NULL;
		    d->database_entries = 0;
		}

		/* Interrupted? */
		if(*stop_count > 0)
		    break;

		if(stat((const char *)d->database_path, &stat_buf))
		{
		    const gint error_code = (gint)errno;
		    PRINT_PROBLEM(d->database_path, g_strerror(error_code));
		    status = 50;
		    break;
		}

#ifdef S_ISDIR
		if(S_ISDIR(stat_buf.st_mode))
#else
		if(FALSE)
#endif
		{
#if defined(USE_CLAMAV_88)
		    status2 = cl_loaddbdir(
			d->database_path,
			&d->database, &d->database_entries
		    );
#else	/* ClamAV 90.x or 91.x */
		    status2 = cl_load(
			d->database_path,
			&d->database, &d->database_entries,
			CL_DB_STDOPT
		    );
#endif
		}
		else
		{
#if defined(USE_CLAMAV_88)
		    status2 = cl_loaddb(
			d->database_path,
			&d->database, &d->database_entries
		    );
#else	/* ClamAV 90.x or 91.x */
		    status2 = cl_load(
			d->database_path,
			&d->database, &d->database_entries,
			CL_DB_STDOPT
		    );
#endif
		}
		if((status2 != 0) || (d->database == NULL) ||
		   (d->database_entries <= 0)
		)
		{
		    gchar *s = g_strconcat(
			"Virus database reload error: ",
			cl_strerror(status2),
			NULL
		    );
		    PRINT_PROBLEM(d->database_path, s);
		    g_free(s);
		    if(d->database != NULL)
		    {
			cl_free(d->database);
			d->database = NULL;
			d->database_entries = 0;
		    }
		    status = 50;
		    break;
		}
		status2 = cl_build(d->database);
		if(status2 != 0)
		{
		    gchar *s = g_strconcat(
			"Virus database rebuild error: ",
			cl_strerror(status2),
			NULL
		    );
		    PRINT_PROBLEM(d->database_path, s);
		    g_free(s);
		    if(d->database != NULL)
		    {
			cl_free(d->database);
			d->database = NULL;
			d->database_entries = 0;
		    }
		    status = 50;
		    break;
		}
		/* Reget the virus database stats */
		cl_statfree(&d->database_stat);
		memset(&d->database_stat, 0x00, sizeof(d->database_stat));
		if(cl_statinidir(d->database_dir, &d->database_stat))
		{
		    PRINT_PROBLEM(
			d->database_dir,
			"Unable to get the virus database directory statistics"
		    );
		    if(d->database != NULL)
		    {
			cl_free(d->database);
			d->database = NULL;
			d->database_entries = 0;
		    }
		    status = 50;
		    break;
		}

		/* Interrupted? */
		if(*stop_count > 0)
		    break;
	    }

	    /* Skip the current and parent directory notations */
	    if(!strcmp((const char *)s, ".") ||
	       !strcmp((const char *)s, "..")
	    )
		continue;

	    /* Get the full path to the child object */
	    child = g_strconcat(
		path,
		G_DIR_SEPARATOR_S,
		s,
		NULL
	    );
	    if(child == NULL)
	    {
		if(status == 0)
		    status = 3;
		continue;
	    }

	    /* Get this object's local stats */
	    if(lstat((const char *)child, &stat_buf))
	    {
		const gint error_code = (gint)errno;
		PRINT_PROBLEM(child, g_strerror(error_code));
		g_free(child);
		if(status == 0)
		    status = 56;
		continue;
	    }

	    m = stat_buf.st_mode;

	    /* Link? */
#ifdef S_ISLNK
	    if(S_ISLNK(m))
#else
	    if(FALSE)
#endif
	    {
		gchar *dest;

		/* Ignore links? */
		if(ignore_links)
		{
		    g_free(child);
		    continue;
		}

		/* Get the destination value and check if it is a
		 * reference to the current or parent directory
		 */
		dest = (gchar *)GetAllocLinkDest((const char *)child);
		if(dest != NULL)
		{
		    if(!strcmp((const char *)dest, ".") ||
		       !strcmp((const char *)dest, "..")
		    )
		    {
			/* Skip this to prevent infinite recursion */
			g_free(child);
			g_free(dest);
			continue;
		    }
		    g_free(dest);
		}
	    }

	    /* Get this object's destination stats */
	    if(stat((const char *)child, &stat_buf))
	    {
		const gint error_code = (gint)errno;
		PRINT_PROBLEM(child, g_strerror(error_code));
		g_free(child);
		if(status == 0)
		    status = 56;
		continue;
	    }

	    m = stat_buf.st_mode;

	    /* Directory */
#ifdef S_ISDIR
	    if(S_ISDIR(m))
#else
	    if(FALSE)
#endif
	    {
		if(recursive)
		{
		    status2 = AVScanOPClamAVScanDir(
			d, child,
			recursive, executables_only, ignore_links,
			reload, stop_count,
			scanner_opts
		    );
		    if((status2 != 0) && (status == 0))
			status = status2;
		}
	    }
	    /* Regular File */
#ifdef S_ISREG
	    else if(S_ISREG(m))
#else
	    else if(TRUE)
#endif
	    {
		if(executables_only ?
		    AVScanIsExecutable(child, m) : TRUE
		)
		{
		    status2 = AVScanOPClamAVScanFile(
			d, child,
			stop_count,
			scanner_opts
		    );
		    if((status2 != 0) && (status == 0))
			status = status2;
		}
	    }

	    g_free(child);
	}

	/* Close the directory */
	closedir(dp);

	/* User aborted? */
	if(*stop_count > 0)
	{
	    if(status == 0)
		status = 4;
	}

	return(status);
}
#endif	/* HAVE_CLAMAV */


/*
 *	Calculates the total files to be scanned in the directory
 *	specified by path.
 *
 *	If recursive is TRUE then files to be scanned in subdirectories
 *	of the specified directory will be counted.
 *
 *	If executables_only is TRUE then only files that are set
 *	executable will be counted.
 */
static gulong AVScanOPCalculateTotalFilesInDirIterate(
	const gchar *path,
	const gboolean recursive,
	const gboolean executables_only,
	const gboolean ignore_links,
	gint *stop_count
)
{
	mode_t m;
	gulong subtotal = 0l;
	gchar *child;
	const gchar *s;
	struct dirent *dent;
	struct stat stat_buf;
	DIR *dp = opendir(path);
	if(dp == NULL)
	    return(subtotal);

	while(*stop_count == 0)
	{
	    dent = readdir(dp);
	    if(dent == NULL)
		break;

	    s = (const gchar *)dent->d_name;

	    /* Skip the current and parent directory notations */
	    if(!strcmp((const char *)s, ".") ||
	       !strcmp((const char *)s, "..")
	    )
		continue;

	    /* Get the full path to the child object */
	    child = g_strconcat(
		path,
		G_DIR_SEPARATOR_S,
		s,
		NULL
	    );
	    if(child == NULL)
		continue;

	    /* Get the object's local stats */
	    if(lstat((const char *)child, &stat_buf))
	    {
		/* Unable to get the local stats, do not count this */
		g_free(child);
		continue;
	    }

	    m = stat_buf.st_mode;

#ifdef S_ISLNK
	    /* Link? */
	    if(S_ISLNK(m))
#else
	    if(FALSE)
#endif
	    {
		gchar *dest;

		/* Ignore links? */
		if(ignore_links)
		{
		    /* Do not count this */
		    g_free(child);
		    continue;
		}

		/* Get the destination value and check if it is a
		 * reference to the current or parent directory
		 */
		dest = (gchar *)GetAllocLinkDest((const char *)child);
		if(dest != NULL)
		{
		    if(!strcmp((const char *)dest, ".") ||
		       !strcmp((const char *)dest, "..")
		    )
		    {
			/* Skip this to prevent infinite recursion */
			g_free(child);
			g_free(dest);
			continue;
		    }
		    g_free(dest);
		}
	    }

	    /* Get this object's destination stats */
	    if(stat((const char *)child, &stat_buf))
	    {
		/* Unable to get the destination stats, do not count this */
		g_free(child);
		continue;
	    }

	    m = stat_buf.st_mode;

	    /* Directory */
#ifdef S_ISDIR
	    if(S_ISDIR(m))
#else
	    if(FALSE)
#endif
	    {
		if(recursive)
		    subtotal += AVScanOPCalculateTotalFilesInDirIterate(
			child,
			recursive, executables_only, ignore_links,
			stop_count
		    );
	    }
	    /* Regular File */
#ifdef S_ISREG
	    else if(S_ISREG(m))
#else
	    else if(TRUE)
#endif
	    {
		if(executables_only ?
		    AVScanIsExecutable(child, m) : TRUE
		)
		    subtotal++;
	    }

	    g_free(child);
	} 

	closedir(dp);

	return(subtotal);
}

/*
 *	Calculates the total number of files to be scanned from the
 *	specified list of paths.
 *
 *	If a file is not accessable to be scanned in the specified
 *	path then it will be excluded from the returned total.
 */
static gulong AVScanOPCalculateTotalFiles(    
	GList *paths_list,
	const gboolean recursive,
	const gboolean executables_only,
	const gboolean ignore_links,
	gint *stop_count
)
{
	gulong total = 0l;
	mode_t m;
	struct stat stat_buf;
	const gchar *path;
	GList *glist;

	if(paths_list == NULL)
	    return(total);

	/* Iterate through each object in the paths list */
	for(glist = paths_list; glist != NULL; glist = g_list_next(glist))
	{
	    /* Interrupted */
	    if(*stop_count > 0)
		break;

	    path = (const gchar *)glist->data;
	    if(STRISEMPTY(path))
		continue;

	    /* Get this object's local stats */
	    if(lstat((const char *)path, &stat_buf))
	    {
		/* Unable to get local stats, do not count this */
		continue;
	    }

	    m = stat_buf.st_mode;

	    /* Link */
#ifdef S_ISLNK
	    if(S_ISLNK(m))
#else
	    if(FALSE)
#endif
	    {
		/* Links ignored? */
		if(ignore_links)
		{
		    /* Do not count this */
		    continue;
		}

		/* Get this link's destination stats */
		if(stat((const char *)path, &stat_buf))
		{
		    /* Unable to get destination stats, do not count this */
		    continue;
		}

		m = stat_buf.st_mode;

		/* Is this link's destination a directory? */
#ifdef S_ISDIR
		if(S_ISDIR(m))
#else
		if(FALSE)
#endif
		{
		    total += AVScanOPCalculateTotalFilesInDirIterate(
			path,
			recursive, executables_only, ignore_links,
			stop_count
		    );
		}
		else
		{
		    total++;
		}
	    }
	    /* Directory */
#ifdef S_ISDIR
	    else if(S_ISDIR(m))
#else
	    else if(FALSE)
#endif
	    {
		total += AVScanOPCalculateTotalFilesInDirIterate(
		    path,
		    recursive, executables_only, ignore_links,
		    stop_count
		);
	    }
	    /* Regular File */
#ifdef S_ISREG
	    else if(S_ISREG(m))
#else
	    else if(TRUE)
#endif
	    {
		total++;
	    }
	}

	return(total);
}


/*
 *	Returns a dynamically allocated string describing the name and
 *	the version of the Antivirus Engine.
 */
gchar *AVScanGetAVEngineNameVersionString(void)
{
#if defined(HAVE_CLAMAV)
	return(g_strdup_printf(
	    "%s Version %s",
	    "Clam AntiVirus",
	    cl_retver()
	));
#else
	return(NULL);
#endif
}


/*
 *	AntiVirus Scan Operation
 *
 *	This is the front end to performing scans.
 *
 *	The stop_count is a pointer to a gint which should be
 *	initialized to 0 prior to this call. This value will be checked
 *	throughout this module's functions and if (*stop_count > 0) then
 *	it will skip all scanning related calls.
 *
 *	Returns:
 *
 *	0	Success.
 *	1	General error or virus found.
 *	2	Invalid value.
 *	3	Systems error or memory allocation error.
 *	4	User aborted (otherwise no other error).
 *	5	User responded with "no" to a query (otherwise no
 *		other error).
 *	6	An identical operation is already in progress.
 *	7	Reserved.
 *	8	Segmentation fault.
 *
 *	50	Unable to load or initialize the virus database.
 *	51	An input file has a corrupt format or a format not
 *		consistent with its extension.
 *	52	Unsupported object type.
 *	53	Unable to open an input directory.
 *	54	Unable to open an input file.
 *	55	An error occured while reading an input file.
 *	56	Unable to stat() or lstat() an input file.
 *	57	Scan of object is incomplete because it would
 *		otherwise exceed a specified limit.
 *	58	Reserved.
 *	59	Scanner subsystem or plugin unable to scan an input
 *		file.
 *
 *	60	Unsupported file format.
 *	61	Unable to lock the virus database directory.
 *
 *	99	No virus scanner support compiled.
 */
gint AVScanOP(
	const gchar *db_path,
	GList *paths_list,
	const gboolean recursive,
	const gboolean executables_only,
	const gboolean ignore_links,
	gboolean *reload, gint *stop_count,
	scanner_opts_struct *scanner_opts
)
{
	gint status = 0;
#if defined(HAVE_CLAMAV)
	ClamAVData *d;
#endif

	if((paths_list == NULL) || (scanner_opts == NULL))
	{
	    status = 2;
	    return(status);
	}

	/* Interrupted? */
	if(*stop_count > 0)
	{
	    if(status == 0)
		status = 4;
	    return(status);
	}

#if defined(HAVE_CLAMAV)
	/* Initialize ClamAV */
	d = AVScanOPClamAVInit(db_path, &status);
	if(d != NULL)
	{
	    gint status2;
	    const gchar *path;
	    mode_t m;
	    GList *glist;
	    struct stat stat_buf;

	    /* Calculate the total files to be scanned */
	    d->total_files = AVScanOPCalculateTotalFiles(
		paths_list,
		recursive, executables_only, ignore_links,
		stop_count
	    );

	    /* Iterate through each object in the paths list */
	    for(glist = paths_list; glist != NULL; glist = g_list_next(glist))
	    {
		/* Interrupted */
		if(*stop_count > 0)
		    break;

		path = (const gchar *)glist->data;
		if(STRISEMPTY(path))
		    continue;

		/* Get this object's local stats */
		if(lstat((const char *)path, &stat_buf))
		{
		    const gint error_code = (gint)errno;
		    PRINT_PROBLEM(path, g_strerror(error_code));
		    if(status == 0)
			status = 56;
		    continue;
		}

		m = stat_buf.st_mode;

		/* Link */
#ifdef S_ISLNK
		if(S_ISLNK(m))
#else
		if(FALSE)
#endif
		{
		    if(ignore_links)
		    {
			/* Print a warning about this because a link
			 * was explicitly specified and ignore_links
			 * was also specified
			 */
			PRINT_PROBLEM(path, "Link ignored");
			continue;
		    }

		    /* Get this link's destination stats */
		    if(stat((const char *)path, &stat_buf))
		    {
			const gint error_code = (gint)errno;
			PRINT_PROBLEM(path, g_strerror(error_code));
			if(status == 0)
			    status = 56;
			continue;
		    }

		    m = stat_buf.st_mode;

		    /* Is this link's destination a directory? */
#ifdef S_ISDIR
		    if(S_ISDIR(m))
#else
		    if(FALSE)
#endif
		    {
			status2 = AVScanOPClamAVScanDir(
			    d, path,
			    recursive, executables_only, ignore_links,
			    reload, stop_count,
			    scanner_opts
			);
		    }
		    else
		    {
			status2 = AVScanOPClamAVScanFile(
			    d, path,
			    stop_count,
			    scanner_opts
			);
		    }
		}
		/* Directory */
#ifdef S_ISDIR
		else if(S_ISDIR(m))
#else
		else if(FALSE)
#endif
		    status2 = AVScanOPClamAVScanDir(
			d, path,
			recursive, executables_only, ignore_links,
			reload, stop_count,
			scanner_opts
		    );
		/* Regular File */
#ifdef S_ISREG
		else if(S_ISREG(m))
#else
		else if(TRUE)
#endif
		{
		    status2 = AVScanOPClamAVScanFile(
			d, path,
			stop_count,
			scanner_opts
		    );
		}
		else
		{
		    /* Unsupported object type or nothing to scan */
		    PRINT_PROBLEM(path, "Unsupported object type");
		    status2 = 52;
		}

		if((status2 != 0) && (status == 0))
		    status = status2;
	    }

	    AVScanOPClamAVShutdown(d);
	}
	else
	{
	    /* Unable to allocate the virus scanner library data */
	    if(status == 0)
		status = 1;
	}
#else
	PRINT_ERR(
	    "No AntiVirus support enabled at compile time"
	);
	status = 1;
#endif

	/* User aborted? */
	if(*stop_count > 0)
	{
	    if(status == 0)
		status = 4;
	}

	return(status);
}
