(***********************************************************************)
(*                                                                     *)
(*                             Active-DVI                              *)
(*                                                                     *)
(*                   Projet Cristal, INRIA Rocquencourt                *)
(*                                                                     *)
(*  Copyright 2002 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  All rights reserved.  This file is distributed    *)
(*  under the terms of the GNU Lesser General Public License.          *)
(*                                                                     *)
(*  Jun Furuse, Didier Rmy and Pierre Weis.                           *)
(*  Contributions by Roberto Di Cosmo, Didier Le Botlan,               *)
(*  Xavier Leroy, and Alan Schmitt.                                    *)
(*                                                                     *)
(*  Based on Mldvi by Alexandre Miquel.                                *)
(***********************************************************************)

open Misc;;

(* number of steps before checking for user interruptions *)
let checkpoint_frequency = 10;;

(*** Some utilities for specials ***)

let split_string s start = 
  Misc.split_string s (function ' ' -> true | _ -> false) start;;

(* "hello world" is one world *)
let rec split_string_quoted s start =
  let len = String.length s
  and i = ref start in
  (* find a space *)
  while !i < len && s.[!i] = ' ' do incr i; done; 
  if !i >= len then [] else begin
    let i0 = !i in
    while !i < len && s.[!i] <> ' ' do
      if s.[!i] = '"' then begin
        incr i;
        while !i < len && s.[!i] <> '"' do incr i done;
        if s.[!i] <> '"' then failwith "parse error (split_string_quoted)";
        incr i
      end else incr i
    done;
    let i1 = !i in
    String.sub s i0 (i1 - i0) :: split_string_quoted s i1
  end;;
(* '"' *)

(* "\"hello world\"" -> "hello world" *)
let unquote v =
  let s = if v.[0] = '"' then 1 else 0 in
  let l =
    if v.[String.length v - 1] = '"' then String.length v - 1 - s
    else String.length v - s in
  String.sub v s l;;

let split_record s =
  let tokens = split_string_quoted s 0 in
  List.map (fun token ->
    try
      let i = String.index token '=' in
      String.sub token 0 i,
      String.sub token (i + 1) (String.length token - i - 1)
    with
    | _ -> token, "") tokens
;;

module Dev = Grdev;;
module Symbol = Dev.Symbol;;
module DFont = Devfont.Make(Dev);;

let base_dpi = 600;;

(*** Cooked fonts ***)

exception Pause;;
exception Wait of float;;

type cooked_font = {
    name : string;
    ratio : float;
    mtable : (int * int) Table.t;
    mutable gtables : (int * Dev.glyph Table.t) list
  };;

let dummy_mtable = Table.make (fun _ -> raise Not_found);;
let dummy_gtable = Table.make (fun _ -> raise Not_found);;
let dummy_font =
  { name = "--nofont--"; ratio = 1.0; mtable = dummy_mtable; gtables = [] };;

let cook_font fdef dvi_res =
  let name = fdef.Dvi.name
  and sf = fdef.Dvi.scale_factor
  and ds = fdef.Dvi.design_size in
  let ratio = float sf /. float ds in
  let mtable =
    try DFont.find_metrics name (dvi_res *. ratio)
    with Not_found -> dummy_mtable in
  { name = name;
    ratio = ratio;
    mtable = mtable;
    gtables = [] };;

let get_gtable cfont sdpi =
  try List.assoc sdpi cfont.gtables
  with Not_found ->
    let dpi = ldexp (float sdpi) (-16) in
    let table =
      try DFont.find_glyphs cfont.name (dpi *. cfont.ratio)
      with Not_found -> dummy_gtable in
    cfont.gtables <- (sdpi, table) :: cfont.gtables;
    table;;

(*** Cooked DVI's ***)

type cooked_dvi = {
    base_dvi : Dvi.t;
    dvi_res : float;
    font_table : cooked_font Table.t
  };;

let cook_dvi dvi =
  let dvi_res = 72.27 in
  let build n =
    cook_font (List.assoc n dvi.Dvi.font_map) dvi_res in
  { base_dvi = dvi;
    dvi_res = dvi_res;
    font_table = Table.make build };;

(*** The rendering state ***)

type reg_set = {
    reg_h : int;
    reg_v : int;
    reg_w : int;
    reg_x : int;
    reg_y : int;
    reg_z : int
  };;
      
type state = {
    cdvi : cooked_dvi;
    sdpi : int;
    conv : float;
    x_origin : int;
    y_origin : int;
      (* Current font attributes *)
    mutable cur_font : cooked_font;
    mutable cur_mtable : (int * int) Table.t;
    mutable cur_gtable : Dev.glyph Table.t;
      (* Registers *)
    mutable h : int;
    mutable v : int;
    mutable w : int;
    mutable x : int;
    mutable y : int;
    mutable z : int;
      (* Register stack *)
    mutable stack : reg_set list;
      (* Color & Color stack *)
    mutable color : Dvicolor.color;
    mutable color_stack : Dvicolor.color list;
      (* Other attributes *)
    mutable alpha : Drawimage.alpha;
    mutable alpha_stack : Drawimage.alpha list;
    mutable blend : Drawimage.blend;
    mutable blend_stack : Drawimage.blend list;
    mutable epstransparent : bool;
    mutable epstransparent_stack : bool list;
    mutable direction : Transitions.direction option;
    mutable transition : Transitions.t;
    mutable transition_stack : Transitions.t list;
      (* TPIC specials state *)
    mutable tpic_pensize : float;
    mutable tpic_path : (float * float) list;
    mutable tpic_shading : float;
      (* PS specials page state *)
    mutable status : Dvi.known_status;
    mutable headers : (bool * string) list;
    mutable html : (Dev.H.tag * int) option;
    mutable draw_html : (int * int * Dev.glyph) list;
    mutable checkpoint : int;
  };;

type proc_unit = {
    escaped_register : reg_set;
    escaped_stack : reg_set list;
    escaped_cur_font : cooked_font;
    escaped_cur_mtable : (int * int) Table.t;
    escaped_cur_gtable : Dev.glyph Table.t;
    mutable escaped_commands : Dvi.command list
  };;

let procs = Hashtbl.create 107;;
type recording = { tag : string; unit : proc_unit}
let current_recording_proc = ref [];;

let visible = ref true;;
let is_recording () = !current_recording_proc <> [];;

(*** Rendering primitives ***)

let last_height = ref 0;;
let clear_symbols() = last_height := 2;;

let add_char st x y code glyph =
  let g : Symbol.g =
    { Symbol.fontname = st.cur_font.name;
      Symbol.fontratio = st.cur_font.ratio;
      Symbol.glyph = glyph
    } in
  last_height := (Dev.get_glyph glyph).Glyph.voffset;
  let s : Symbol.symbol = Symbol.Glyph g in
  Symbol.add st.color x y code s;;

let add_line st (line, file) =
  let x = st.x_origin + int_of_float (st.conv *. float st.h)
  and y = st.y_origin + int_of_float (st.conv *. float st.v) in
  Symbol.add st.color x y 0 (Symbol.Line (line, file));;

let add_blank nn st width =
  let x = st.x_origin + int_of_float (st.conv *. float st.h)
  and y = st.y_origin + int_of_float (st.conv *. float st.v)
  and w = int_of_float (st.conv *. float width) in
  Symbol.add st.color x y nn (Symbol.Space (w, !last_height));;

let add_rule st x y w h =
  Symbol.add st.color x y 0 (Symbol.Rule (w, h));;

let get_register_set st =
  { reg_h = st.h; reg_v = st.v;
    reg_w = st.w; reg_x = st.x;
    reg_y = st.y; reg_z = st.z };;

let set_register_set st rset =
  st.h <- rset.reg_h;
  st.v <- rset.reg_v;
  st.w <- rset.reg_w;
  st.x <- rset.reg_x;
  st.y <- rset.reg_y;
  st.z <- rset.reg_z;;

let push st =
  st.stack <- (get_register_set st) :: st.stack;;

let pop st =
  match st.stack with
  | [] -> ()
  | rset :: rest ->
      set_register_set st rset;
      st.stack <- rest;;

let color_push st col =
  st.color_stack <- st.color :: st.color_stack;
  st.color <- col;
  if !visible then Dev.set_color col;;

let color_pop st =
  match st.color_stack with
  | [] -> ()
  | col :: rest ->
      st.color <- col;
      if !visible then Dev.set_color col;
      st.color_stack <- rest;;

let alpha_push st v =
  st.alpha_stack <- st.alpha :: st.alpha_stack;
  st.alpha <- v;
  if !visible then Dev.set_alpha v;;

let alpha_pop st =
  match st.alpha_stack with
  | [] -> ()
  | v :: rest ->
      st.alpha <- v;
      if !visible then Dev.set_alpha v;
      st.alpha_stack <- rest;;

let blend_push st v =
  st.blend_stack <- st.blend :: st.blend_stack;
  st.blend <- v;
  if !visible then Dev.set_blend v;;

let blend_pop st =
  match st.blend_stack with
  | [] -> ()
  | v :: rest ->
      st.blend <- v;
      if !visible then Dev.set_blend v;
      st.blend_stack <- rest;;

let epstransparent_push st v =
  st.epstransparent_stack <- st.epstransparent :: st.epstransparent_stack;
  st.epstransparent <- v;
  if !visible then Dev.set_epstransparent v;;

let epstransparent_pop st =
  match st.epstransparent_stack with
  | [] -> ()
  | v :: rest ->
      st.epstransparent <- v;
      if !visible then Dev.set_epstransparent v;
      st.epstransparent_stack <- rest;;

let transition_push st v =
  (* st.transition_stack <- st.transition :: st.transition_stack; *)
  st.transition <- v;
  if !visible then Dev.set_transition v;;

(*
let transition_pop st =
  match st.transition_stack with
  | [] -> ()
  | v :: rest ->
      st.transition <- v;
      if !visible then Dev.set_transition v; 
     st.transition_stack <- rest;;
*)

let fnt st n =
  let (mtable, gtable, cfont) =
    try
      let cfont = Table.get st.cdvi.font_table n in
      (cfont.mtable, get_gtable cfont st.sdpi, cfont)
    with Not_found -> (dummy_mtable, dummy_gtable, dummy_font) in
  st.cur_mtable <- mtable;
  st.cur_gtable <- gtable;
  st.cur_font <- cfont;;

let put st code =
  try
    let x = st.x_origin + int_of_float (st.conv *. float st.h)
    and y = st.y_origin + int_of_float (st.conv *. float st.v)
    and glyph = Table.get st.cur_gtable code in
    if !visible then
      begin
        begin match st.html with
        | Some _ -> st.draw_html <- (x, y, glyph) :: st.draw_html
        | None -> ()
        end;
        Dev.draw_glyph (glyph : Dev.glyph) x y;
	add_char st x y code glyph
      end
  with _ -> ();;

let set st code =
  put st code;
  try
    let (dx, dy) = Table.get st.cur_mtable code in
    st.h <- st.h + dx;
    st.v <- st.v + dy
  with _ -> ();;

let put_rule st a b =
  let x = st.x_origin + int_of_float (st.conv *. float st.h)
  and y = st.y_origin + int_of_float (st.conv *. float st.v)
  and w = int_of_float (ceil (st.conv *. float b))
  and h = int_of_float (ceil (st.conv *. float a)) in
  add_rule st x (y-h) w h;
  if !visible then Dev.fill_rect x (y - h) w h;;

let set_rule st a b =
  put_rule st a b;
  st.h <- st.h + b;;

(*** Specials ***)

let ill_formed_special s =
  Misc.warning (Printf.sprintf "Ill formed special <<%s>>" s);;


let line_special st s =
  match split_string s 0 with
  | key :: line :: rest ->
      begin try
        let l = int_of_string line in
        let f = match rest with file :: _ -> Some file | _ -> None in
        add_line st (l, f)
      with Failure _ -> ill_formed_special s
      end
  | _ -> ill_formed_special s;;

let color_special st s =
  match split_string s 0 with
  | "color" :: "push" :: args ->
      color_push st (Dvicolor.parse_color_args args)
  | "color" :: "pop" :: [] ->
      color_pop st
  | _ -> ();;

let parse_float s =
 try float_of_string s
 with _ -> failwith ("advi: cannot read a floating number in \"" ^ s ^ "\"");;

let option_parse_float s r =
  try Some(parse_float (List.assoc s r))
  with _ -> None;;

let alpha_special st s =
  match split_string s 0 with
  | ["advi:"; "alpha"; "push"; arg] ->
      alpha_push st (parse_float arg)
  | ["advi:"; "alpha"; "pop"] ->
      alpha_pop st
  | _ -> ();;

let parse_blend s =
  match String.lowercase s with
  | "none" -> Drawimage.Normal
  | "normal" -> Drawimage.Normal
  | "multiply" -> Drawimage.Multiply
  | "screen" -> Drawimage.Screen
  | "overlay" -> Drawimage.Overlay
  | "dodge" -> Drawimage.ColorDodge
  | "burn" -> Drawimage.ColorBurn
  | "darken" -> Drawimage.Darken
  | "lighten" -> Drawimage.Lighten
  | "difference" -> Drawimage.Difference
  | "exclusion" -> Drawimage.Exclusion
  | _ ->
      Misc.warning ("blend: invalid blend mode " ^ s);
      Drawimage.Normal;;

let blend_special st s =
  match split_string s 0 with
  | ["advi:"; "blend"; "push"; arg] ->
      blend_push st (parse_blend arg)
  | "advi:" :: "blend" :: "pop" :: [] ->
      blend_pop st
  | _ -> ();;

let parse_epstransparent s =
  match String.lowercase s with
  | "true" -> true
  | "false" -> false
  | _ -> raise (Failure "epstransparent: invalid mode");;

let epstransparent_special st s =
  match split_string s 0 with
  | ["advi:"; "epstransparent"; "push"; arg] ->
      epstransparent_push st (parse_epstransparent arg)
  | "advi:" :: "epstransparent" :: "pop" :: [] ->
      epstransparent_pop st
  | _ -> ();;

let get_records s =
  List.map (fun (k, v) -> String.lowercase k, v) (split_record s);;

let psfile_special st s =
  let records = get_records s in
  let file =
    try unquote (List.assoc "psfile" records)
    with Not_found -> raise (Failure "psfile: invalid special") in
  (* prerr_endline ("PSFILE=" ^ file); *)
  (* bbox *)
  let llx, lly, urx, ury =
    try
      let llx = int_or_float_of_string (List.assoc "llx" records)
      and lly = int_or_float_of_string (List.assoc "lly" records)
      and urx = int_or_float_of_string (List.assoc "urx" records)
      and ury = int_or_float_of_string (List.assoc "ury" records) in
      (* prerr_endline
         ("BBOX=" ^ Printf.sprintf "%d %d %d %d" llx lly urx ury); *)
      llx, lly, urx, ury
    with
    | _ -> raise (Failure "psfile: no bbox") in
  let width, height = (* return Big Points *)
    let w = try int_of_string (List.assoc "rwi" records) with _ -> 0
    and h = try int_of_string (List.assoc "rhi" records) with _ -> 0 in
    match w, h with
    | 0, 0 -> float (urx - llx), float (ury - lly)
    | 0, _ ->
       let h = float h /. 10.0 in
       let w = float (urx - llx) *. (h /. float (ury - lly)) in
       w, h
    | _, 0 ->
       let w = float w /. 10.0 in
       let h = float (ury - lly) *. (w /. float (urx - llx)) in
       w, h
    | _, _ -> float w /. 10.0, float h /. 10.0 in
  let dpi = ldexp (float st.sdpi) (-16) in
  let width_pixel = truncate (width /. 72.0 *. dpi) in
  let height_pixel = truncate (height /. 72.0 *. dpi) in
  (* prerr_endline (Printf.sprintf "%dx%d pixel" width_pixel height_pixel);*)
  file, (llx, lly, urx, ury), (width_pixel, height_pixel);;

let int_of_signal = function
  | "SIGABRT" | "sigabrt" -> Sys.sigabrt (* -1 *)
  | "SIGALRM" | "sigalrm" -> Sys.sigalrm (* -2 *)
  | "SIGFPE" | "sigfpe" -> Sys.sigfpe (* -3 *)
  | "SIGHUP" | "sighup" -> Sys.sighup (* -4 *)
  | "SIGILL" | "sigill" -> Sys.sigill (* -5 *)
  | "SIGINT" | "sigint" -> Sys.sigint (* -6 *)
  | "SIGKILL" | "sigkill" -> Sys.sigkill (* -7 *)
  | "SIGPIPE" | "sigpipe" -> Sys.sigpipe (* -8 *)
  | "SIGQUIT" | "sigquit" -> Sys.sigquit (* -9 *)
  | "SIGSEGV" | "sigsegv" -> Sys.sigsegv (* -10 *)
  | "SIGTERM" | "sigterm" -> Sys.sigterm (* -11 *)
  | "SIGUSR1" | "sigusr1" -> Sys.sigusr1 (* -12 *)
  | "SIGUSR2" | "sigusr2" -> Sys.sigusr2 (* -13 *)
  | "SIGCHLD" | "sigchld" -> Sys.sigchld (* -14 *)
  | "SIGCONT" | "sigcont" -> Sys.sigcont (* -15 *)
  | "SIGSTOP" | "sigstop" -> Sys.sigstop (* -16 *)
  | "SIGTSTP" | "sigtstp" -> Sys.sigtstp (* -17 *)
  | "SIGTTIN" | "sigttin" -> Sys.sigttin (* -18 *)
  | "SIGTTOU" | "sigttou" -> Sys.sigttou (* -19 *)
  | "SIGVTALRM" | "sigvtalrm" -> Sys.sigvtalrm (* -20 *)
  | "SIGPROF" | "sigprof" -> Sys.sigprof (* -21 *)
  | "" -> Sys.sigquit
  | s -> int_of_string s;;

let kill_embed_special st s =
  (* advi: kill name=? signal=? *)
  let records = get_records s in
  let app_name =
    try unquote (List.assoc "name" records)
    with Not_found -> raise (Failure ("No command to kill in " ^ s)) in
  let sign = List.assoc "signal" records in
  (* prerr_endline (Printf.sprintf "Signal is ``%s''" sign); *)
  let sig_val =
    try int_of_signal (unquote (List.assoc "signal" records))
    with
    | Not_found -> raise (Failure ("No signal to kill command in " ^ s))
    | Failure _ -> raise (Failure ("Cannot understand signal in " ^ s))  in
  Embed.kill_embedded_app sig_val app_name;;

let app_mode_of_string = function
  | "sticky" -> Embed.Sticky
  | "persistent" -> Embed.Persistent
  | "ephemeral" -> Embed.Ephemeral
  | s -> raise (Failure ("Unknown embedding mode " ^ s));;


let embed_special st s =
  (* advi: embed mode=? width=? height=? command="command string" *)
  let records = get_records s in
  let app_mode =
    try app_mode_of_string (List.assoc "mode" records)
    with Not_found ->
      raise (Failure ("embed: no embedding mode in special " ^ s)) in
  let app_name =
    try unquote (List.assoc "name" records) with Not_found -> "" in
  let command =
    try unquote (List.assoc "command" records)
    with Not_found ->
        raise (Failure ("embed: no command to embed in special " ^ s)) in
  (* prerr_endline ("embed command=" ^ command); *)
  let get_dim dim records =
    match Dimension.normalize
            (Dimension.dimen_of_string (List.assoc dim records)) with
    | Dimension.In d -> d
    | _ -> assert false in

  let width_pixel, height_pixel =
    let w, h =
      try
        let width = get_dim "width" records in
        let height = get_dim "height" records in
        width, height
      with
      | _ -> raise (Failure ("embed: no size in special " ^ s)) in
    let dpi = ldexp (float st.sdpi) (-16) in
    let width_pixel = truncate (w *. dpi) in
    let height_pixel = truncate (h *. dpi) in
   (* prerr_endline (Printf.sprintf "%d x %d pixel" width_pixel height_pixel);*)
    width_pixel, height_pixel in
  let x = st.x_origin + int_of_float (st.conv *. float st.h)
  and y = st.y_origin + int_of_float (st.conv *. float st.v) in
  if !visible then
    Grdev.embed_app command app_mode app_name width_pixel height_pixel x y;;

(* When scanning the page, we gather information on the embedded commands *)
let scan_embed_special st s = 
  let records = get_records s in
  let command =
    try unquote (List.assoc "command" records)
    with Not_found ->
        raise (Failure ("embed: no command to embed in special " ^ s)) in
  Launch.add_whiterun_command command ;;

let parse_transition dir mode record =
  let default_dir =
    match dir with Some d -> d | None -> Transitions.DirNone in
  let parse_genpath record =
    try
      List.assoc "genpath" record
    with _ ->
      warning "special: trans push: genpath function not found";
      "spiral"
  in
  let parse_pathelem s =
      (option_parse_float (s ^ "x") record,
       option_parse_float (s ^ "y") record,
       None,
       None) (* to complete with parsed scale and rotation *)
  in
  let parse_steps =
    try
      let stepsstr = List.assoc "steps" record in
      try Some (int_of_string stepsstr)
      with _ ->
        warning
          ("special: trans push: steps parse failed: \"" ^ stepsstr ^ "\"");
        None
    with Not_found -> None
  in
  let parse_direction key default =
    try
      match String.lowercase (List.assoc key record) with
      | "left" -> Transitions.DirLeft
      | "right" -> Transitions.DirRight
      | "top" | "up" -> Transitions.DirTop
      | "bottom" | "down" -> Transitions.DirBottom
      | "topleft" | "upleft" -> Transitions.DirTopLeft
      | "topright" | "upright" -> Transitions.DirTopRight
      | "bottomleft" | "downleft" -> Transitions.DirBottomLeft
      | "bottomright" | "downright" -> Transitions.DirBottomRight
      | "center" -> Transitions.DirCenter
      | s ->
         warning ("special: trans push: direction parse failed: \""^ s ^"\"");
         raise Exit
    with _ -> default (* Transitions.DirNone *)
  in
  match String.lowercase mode with
  | "slide" ->
      Transitions.TransSlide (parse_steps, parse_direction "from" default_dir)
  | "wipe" ->
      Transitions.TransWipe (parse_steps, parse_direction "from" default_dir)
  | "block" ->
      Transitions.TransBlock
        (parse_steps, parse_direction "from" Transitions.DirNone)
  | "path" ->
      Transitions.TransPath
        (parse_steps,
         parse_genpath record,
         parse_pathelem "start",
         parse_pathelem "stop")
  | "none" ->
      Transitions.TransNone
  | _ ->
     warning ("special: trans push: mode parse failed \"" ^ mode ^ "\"");
     Transitions.TransNone;;

let transition_special st s =
  match split_string s 0 with
  | "advi:" :: "trans" :: mode :: args ->
      let record = split_record (String.concat " " args) in
      let trans = parse_transition st.direction mode record in
      transition_push st trans
  | _ -> ();;

let transbox_save_special st s =
  match split_string s 0 with
  | "advi:" :: "transbox" :: "save" :: args ->
      let dpi = ldexp (float st.sdpi) (-16) in
      let record = split_record (String.concat " " args) in 
      let width = Dimension.dimen_of_string (List.assoc "width" record) in
      let height = Dimension.dimen_of_string (List.assoc "height" record) in
      let depth = Dimension.dimen_of_string (List.assoc "depth" record) in
      let pixels_of_dimen dim =
	match Dimension.normalize dim with
	| Dimension.Px x -> x
	| Dimension.In x -> truncate (x *. dpi)
        | _ -> assert false
      in
      let width_pixel = pixels_of_dimen width
      and height_pixel = pixels_of_dimen height
      and depth_pixel = pixels_of_dimen depth
      in
      let x = st.x_origin + int_of_float (st.conv *. float st.h)
      and y = st.y_origin + int_of_float (st.conv *. float st.v) + depth_pixel 
      in
      Dev.transbox_save x y width_pixel (height_pixel + depth_pixel)
  | _ -> raise (Failure "advi: transbox save special failed");;

let transbox_go_special st s =
  match split_string s 0 with
  | "advi:" :: "transbox" :: "go" :: mode :: args ->
      let record = split_record (String.concat " " args) in 
      let trans = parse_transition None mode record in
      Dev.transbox_go trans
  | _ -> raise (Failure "advi: transbox go special failed");;

exception Ignore
let edit_special st s =
  try
  match split_string s 0 with
  | "advi:" :: "edit" :: args ->
      let record = split_record (String.concat " " args) in
      let field x =
        try List.assoc x record
        with Not_found -> 
          warning (Printf.sprintf "Field %s missing in special %s" x s);
          raise Ignore in
      let dpi = ldexp (float st.sdpi) (-16) in
      let pixels dim =
	match Dimension.normalize (Dimension.dimen_of_string dim) with
	| Dimension.Px x -> float x
	| Dimension.In x -> x *. dpi
        | _ -> assert false in
      let prop = function
        | "X" -> Dev.E.X | "Y" -> Dev.E.Y | "XY" -> Dev.E.XY | "Z" -> Dev.E.Z
        | _ -> ill_formed_special s; Dev.E.Z in
      let unit = pixels (field "unit") in
      let float_field x =
        let fx = field x in
        try float_of_string fx
        with _  ->
          warning
            (Printf.sprintf "Field %s=%s not a float in special %s" x fx s);
            raise Ignore in
      let float_to_pixel f = truncate (f *. unit) in
      let  r = {
        Dev.x = float_field "x"; Dev.y = float_field "y";
        Dev.w = float_field "w"; Dev.h = float_field "h";
      } in
      let rect = {
        Dev.x = st.x_origin + int_of_float (st.conv *. float st.h) 
          + float_to_pixel r.Dev.x;
        Dev.y = st.y_origin + int_of_float (st.conv *. float st.v)
          - float_to_pixel r.Dev.y;
        Dev.w = float_to_pixel r.Dev.w; 
        Dev.h = float_to_pixel r.Dev.h; 
      } in
      let info =
        { Dev.E.comm = field "comm";
          Dev.E.name = field "name";
          Dev.E.line = field "line";
          Dev.E.file = field "file";
          Dev.E.unit = unit;
          Dev.E.origin = r; 
          Dev.E.move =
            (try prop (List.assoc "move" record) with Not_found -> Dev.E.Z);
          Dev.E.resize =
            (try prop (List.assoc "resize" record) with Not_found -> Dev.E.Z);
        } in
      Dev.E.add rect info

  | _ ->
      warning (Printf.sprintf "Incorrect advi Special `%s' ignored" s)
  with Ignore -> ()
;;


let forward_eval_command = ref (fun _ _ -> ());;

let playing = ref 0;;

(* Setting the forward function Grdev.get_playing. *)
let play = Grdev.get_playing in
play := (fun () -> !playing);;

let visible_stack = ref [];;

let proc_clean () =
  current_recording_proc := [];
  playing := 0;
  visible_stack := [];
  visible := true;
  Hashtbl.clear procs;;

let proc_special st s =
  let records = get_records s in
  try
    let v = List.assoc "record" records in
    match v with
    | "start" ->
        let procname =
          try unquote (List.assoc "proc" records)
          with Not_found -> raise (Failure "proc: invalid special") in
        visible_stack := !visible :: !visible_stack;
        visible := List.mem_assoc "play" records;
        if !playing = 0 then
          begin
            let recording =
              { tag = procname;
                unit = 
                { escaped_register = get_register_set st;
                  escaped_stack = st.stack;
                  escaped_cur_mtable = st.cur_mtable;
                  escaped_cur_gtable = st.cur_gtable;
                  escaped_cur_font = st.cur_font;
                  escaped_commands = [] }
              } in
            current_recording_proc := recording :: !current_recording_proc;
          end;
    | "end" ->
        if !playing = 0 then
          begin match !current_recording_proc with
          | [] ->
              Misc.warning
                (Printf.sprintf "'xxx %s' not recording" s)
          | recording :: rest ->
              let procname = recording.tag in
              current_recording_proc := rest;
              let u = recording.unit in
              Hashtbl.add procs procname u;
              match u.escaped_commands with
              | h :: rest -> u.escaped_commands <- List.rev rest
              | [] -> assert false
          end;
        begin match !visible_stack with
        | h :: rest ->
            visible := h; visible_stack := rest;
        | [] ->
            (* Ill-formed DVI not recording error should have ben reported
               right above *) 
            (); 
        end;
    | _ -> ill_formed_special s
  with
  | Not_found ->
      let procname =
        try unquote (List.assoc "proc" records)
        with Not_found -> raise (Failure "proc: invalid special") in
      try
        ignore (List.assoc "play" records);
        if not (is_recording ()) then
          begin
            let us = Hashtbl.find_all procs procname in
            let escaped_cur_font = st.cur_font
            and escaped_cur_mtable = st.cur_mtable
            and escaped_cur_gtable = st.cur_gtable in
            let escaped_stack = push st; st.stack in
            incr playing;
            List.iter
              (fun u ->
                set_register_set st u.escaped_register;
                st.stack <- u.escaped_stack;
                st.cur_mtable <- u.escaped_cur_mtable;
                st.cur_gtable <- u.escaped_cur_gtable;
                st.cur_font <- u.escaped_cur_font;
                List.iter (fun com -> !forward_eval_command st com)
                  u.escaped_commands
              ) us;
            decr playing;
            st.stack <- escaped_stack; pop st;
            st.cur_mtable <- escaped_cur_mtable;
            st.cur_gtable <- escaped_cur_gtable;
            st.cur_font <- escaped_cur_font;
          end
      with
      | Not_found ->
          Misc.warning
            (Printf.sprintf "xxx '%s': %s not recorded" s procname);;

let wait_special st s =
  let records = get_records s in
  let second =
    try parse_float (List.assoc "sec" records) 
    with
    | Not_found | Failure _ -> raise (Failure "wait: invalid special") in
  (* Wait is treated like Pause, as an exception *)
  if !visible then raise (Wait second);
  st.checkpoint <- 0;;

(* Background object configuration. RDC *)
let inherit_background_info =
  Options.flag false
    "-inherit-background"
    "\tBackground options are inherited from previous page";;

let setup_bkgd status =
  (* propagate bkgd preferences to graphics device *)
  Dev.blit_bkgd_data status.Dvi.bkgd_prefs Dev.bkgd_data;
  (* store the default/inherited prefs into Dev *)
  Dev.set_bg_options status.Dvi.bkgd_local_prefs;
  (* apply local modifications                  *)
  Dev.blit_bkgd_data Dev.bkgd_data status.Dvi.bkgd_prefs
  (* recover modified preferences               *);;


let ratios_alist = [
  ("auto", Drawimage.ScaleAuto);
  ("center", Drawimage.ScaleCenter);
  ("top", Drawimage.ScaleTop);
  ("bottom", Drawimage.ScaleBottom);
  ("left", Drawimage.ScaleLeft);
  ("right", Drawimage.ScaleRight);
  ("topleft", Drawimage.ScaleTopLeft);
  ("bottomright", Drawimage.ScaleBottomRight);
  ("topright", Drawimage.ScaleTopRight);
  ("bottomleft", Drawimage.ScaleBottomLeft);
] ;;

let bkgd_alist = [
  ("color", fun s -> fun st ->
     let c = Dvicolor.parse_color_args (split_string (unquote s) 0)
     in [Dev.BgColor c]);
  ("image", fun s -> fun st ->
     [Dev.BgImg s]);
  ("reset", fun s -> fun st ->
     Dev.blit_bkgd_data (Dev.default_bkgd_data ()) st.Dvi.bkgd_prefs;
     []);
  ("inherit", fun s -> fun st ->
     inherit_background_info := true;
     []);
  ("alpha", fun s -> fun st ->
     let a = parse_float (unquote s) in
     [Dev.BgAlpha a]);
  ("blend", fun s -> fun st ->
     let b = parse_blend (unquote s) in
     [Dev.BgBlend b]);
  ("fit", fun s -> fun st ->
     let f =
       try List.assoc (unquote s) ratios_alist with _ -> Drawimage.ScaleAuto in
     [Dev.BgRatio f]);
];;

let filter_alist alist falist =
  let aux k alist okalist =
    try (k, List.assoc k alist) :: okalist
    with Not_found -> okalist in
  List.fold_left (fun l -> fun k -> aux k alist l) []
                 (List.map (fun (k, v) -> k) falist);;

(* When scanning the page, we just fill the info structure for backgrounds *)
let scan_bkgd_special st s =
  let records = get_records s in
  st.Dvi.bkgd_local_prefs <-
    List.flatten
      (List.map (fun (k, v) -> (List.assoc k bkgd_alist) v st)
                (filter_alist records bkgd_alist)) @
    st.Dvi.bkgd_local_prefs;;

(* When not scanning, we ignore the background information *)
let bkgd_special st s = ();;

(* Support for TPIC specials. *)

let milli_inch_to_sp = Units.from_to Units.IN Units.SP 1e-3;;

let tpic_milli_inches s = parse_float s *. milli_inch_to_sp;;

let tpic_pen st =
  int_of_float (st.conv *. st.tpic_pensize +. 0.5)
let tpic_x st x =
  st.x_origin + int_of_float (st.conv *. (float st.h +. x));;
let tpic_y st y =
  st.y_origin + int_of_float (st.conv *. (float st.v +. y));;

let tpic_flush_path st cntr =
  let path = Array.of_list (List.rev st.tpic_path) in
  (* Convert points in path to pixel coordinates *)
  let pixpath =
    Array.map (fun (x, y) -> (tpic_x st x, tpic_y st y)) path in
  (* If shading requested and path is closed, fill *)
  if st.tpic_shading > 0.0 &&
     Array.length path >= 2 &&
     path.(0) = path.(Array.length path - 1)
  then
    if !visible then Dev.fill_path pixpath ~shade:st.tpic_shading;
  (* If requested, draw outline of path *)
  if cntr then
    if !visible then Dev.draw_path pixpath ~pensize:(tpic_pen st);
  (* Reset path *)
  st.tpic_path <- [];
  st.tpic_shading <- 0.0;;

let dist (x0,y0) (x1,y1) = abs(x0 - x1) + abs(y0 - y1);;

let tpic_spline_path st =
  (* Code shamelessly stolen from xdvi *)
  let path =
    Array.of_list
      (List.map (fun (x, y) -> (tpic_x st x, tpic_y st y))
                (List.rev st.tpic_path)) in
  let p =
    Array.concat [[|path.(0)|]; path; [|path.(Array.length path - 1)|]] in
  let r = ref [] in
  for i = 0 to Array.length p - 3 do
    let steps = (dist p.(i) p.(i+1) + dist p.(i+1) p.(i+2)) / 4 in
    let (x2, y2) = p.(i+2)
    and (x1, y1) = p.(i+1)
    and (x0, y0) = p.(i) in
    for j = 0 to steps - 1 do
      let w = (j * 1000 + 500) / steps in
      let t1 = w * w / 20 in
      let w = w - 500 in
      let t2 = (750000 - w * w) / 10 in
      let w = w - 500 in
      let t3 = w * w / 20 in
      let xp = (t1 * x2 + t2 * x1 + t3 * x0 + 50000) / 100000
      and yp = (t1 * y2 + t2 * y1 + t3 * y0 + 50000) / 100000 in
      r := (xp, yp) :: !r
    done
  done;
  if !visible then
      Dev.draw_path (Array.of_list (List.rev !r)) ~pensize:(tpic_pen st);
  st.tpic_path <- [];
  st.tpic_shading <- 0.0;;

let rad_to_deg = 45.0 /. atan 1.0;;

let tpic_arc st x y rx ry s e cntr =
  let x = tpic_x st x
  and y = tpic_y st y
  and rx = int_of_float (st.conv *. rx)
  and ry = int_of_float (st.conv *. ry)
  and s = int_of_float (s *. rad_to_deg)
  and e = int_of_float (e *. rad_to_deg) in
  (* If shading requested, fill the arc *)
  if st.tpic_shading > 0.0 then
    if !visible then
      Dev.fill_arc ~x ~y ~rx ~ry ~start:s ~stop:e ~shade:st.tpic_shading;
  (* If requested, draw outline of arc *)
  if cntr then
    if !visible then
      Dev.draw_arc ~x ~y ~rx ~ry ~start:s ~stop:e ~pensize:(tpic_pen st);
  (* Reset shading *)
  st.tpic_shading <- 0.0;;

let tpic_specials st s =
  match split_string s 0 with
  | "pn" :: size :: _ ->
      st.tpic_pensize <- tpic_milli_inches size
  | "pa" :: x :: y :: _ ->
      st.tpic_path <-
        (tpic_milli_inches x, tpic_milli_inches y) :: st.tpic_path
  | "fp" :: _ ->
      tpic_flush_path st true
  | "ip" :: _ ->
      tpic_flush_path st false
  | "da" :: _ -> (* TODO: dashed lines *)
      tpic_flush_path st true
  | "dt" :: _ -> (* TODO: dotted lines *)
      tpic_flush_path st true
  | "sp" :: _ -> (* TODO: dashed/dotted splines *)
      tpic_spline_path st
  | "ar" :: x :: y :: rx :: ry :: s :: e :: _ ->
      tpic_arc st (tpic_milli_inches x) (tpic_milli_inches y)
               (tpic_milli_inches rx) (tpic_milli_inches ry)
               (parse_float s) (parse_float e)
               true
  | "ia" :: x :: y :: rx :: ry :: s :: e :: _ ->
      tpic_arc st (tpic_milli_inches x) (tpic_milli_inches y)
               (tpic_milli_inches rx) (tpic_milli_inches ry)
               (parse_float s) (parse_float e)
               true
  | "sh" :: s :: _ ->
      st.tpic_shading <- parse_float s
  | "wh" :: _ ->
      st.tpic_shading <- 0.0
  | "bk" :: _ ->
      st.tpic_shading <- 1.0
  | _ ->
      ();;
(* End of TPIC hacks *)

let moveto_special st b s =
  if !Options.dops then
    let x, y = Dev.current_pos () in
    if b then begin
       st.h <- int_of_float (float (x - st.x_origin) /. st.conv);
       st.v <- int_of_float (float (y - st.y_origin) /. st.conv);
    end else begin
       st.h <- st.h + int_of_float (float x /. st.conv);
       st.v <- st.v + int_of_float (float y /. st.conv);
    end;;

let ps_special st s =
  if !Options.dops && st.status.Dvi.hasps then
     let x = int_of_float (st.conv *. float st.h) in
     let y = int_of_float (st.conv *. float st.v) in
     if !visible then
       begin try
         Dev.exec_ps s x y
       with Dev.GS ->
         st.status.Dvi.hasps <- false
       end;;

(* header are not "rendered", only stored during scan *)
let header_special st s = ();;

(* For html specials *)

(* Should check that a pause is not in the middle of some html code *)
let open_html st html tag tag_string =
  let name = String.sub html 9 (String.length html - 11) in
  let x = st.x_origin + int_of_float (st.conv *. float st.h)
  and y = st.y_origin + int_of_float (st.conv *. float st.v) in
  begin match st.html with
  | Some (t, k) ->
      st.html <- Some (t, succ k)
  | None ->
      st.html <- Some (tag name, 0)
  end;;

let close_html st =
  match st.html with
  | Some (tag, k) when k > 0 ->
      st.html <- Some (tag, k-1)
  | Some (tag, 0) ->
      Dev.H.add {Dev.H.tag =tag; Dev.H.draw = List.rev st.draw_html};
      st.html <- None;
      st.draw_html <- []
  | Some (_, k) -> assert false
  | None -> warning ("Closing html tag that is not open");;

let html_special st html =
  if has_prefix "<A name=\"" html || has_prefix "<a name=\"" html then
    open_html st html (fun x -> Dev.H.Name x) "Name" else
  if  has_prefix "<A href=\"" html || has_prefix "<a href=\"" html then
    open_html st html (fun x -> Dev.H.Href x) "Href" else
  if  has_prefix "<A advi=\"" html || has_prefix "<a advi=\"" html then
    let advi x =
      let play () = proc_special st ("advi: proc=" ^ x ^ " play") in
      Dev.H.Advi
        {Dev.H.link = x;
         Dev.H.action = play;
         Dev.H.mode = Dev.H.Over;
         Dev.H.color = None;
         Dev.H.area = None} in 
      open_html st html advi "Advi" else
  if  has_prefix "<A hdvi=\"" html || has_prefix "<a hdvi=\"" html then
    let advi x =
      let play () = proc_special st ("advi: proc=" ^ x ^ " play") in
      Dev.H.Advi
        {Dev.H.link = x;
         Dev.H.action = play;
         Dev.H.mode = Dev.H.Click_down;
         Dev.H.color = None;
         Dev.H.area = None} in 
      open_html st html advi "Advi" else
  if has_prefix "</A>" html || has_prefix "</a>" html then close_html st
  else warning ("Unknown html suffix" ^ html);;

let scan_special_html (headers, xrefs) page s =
  let name = String.sub s 14 (String.length s - 16) in
  Hashtbl.add xrefs name page;;

(* This function is iterated on the current DVI page BEFORE
   rendering it, to gather the information contained in some
   "asynchronous" specials (typically, PS headers, background
   commands, html references) *)
let scan_special status (headers, xrefs) pagenum s =
  if Launch.whiterun () &&
     has_prefix "advi: embed " s then scan_embed_special status s
  (* Embedded Postscript, better be first for speed when scanning *)
  else if has_prefix "\" " s || has_prefix "ps: " s then
    (if !Options.dops then status.Dvi.hasps <- true)
  else if has_prefix "!" s then
    (if !Options.dops then
      headers := (true, get_suffix "!" s) :: !headers)
  (* Embedded Postscript, better be first for speed when scanning *)
  else if has_prefix "header=" s then
    (if !Options.dops then
      headers := (false, get_suffix "header=" s) :: !headers)
  else if has_prefix "advi: setbg " s then scan_bkgd_special status s
  else if has_prefix "html:<A name=\"" s || has_prefix "html:<a name=\"" s then
    scan_special_html (headers, xrefs) pagenum s;;

(* Scan a page calling function scan_special when seeing a special and
   the function otherwise for other DVI stuff. *)
let scan_special_page otherwise cdvi globals pagenum =
Misc.debug_stop "Scanning specials";
   let page = cdvi.base_dvi.Dvi.pages.(pagenum) in
   match page.Dvi.status with
   | Dvi.Unknown ->
       let status =
         {Dvi.hasps = false;
          Dvi.bkgd_local_prefs = [];
          Dvi.bkgd_prefs =
            (if !inherit_background_info
             then Dev.copy_of_bkgd_data ()
             else Dev.default_bkgd_data ())} in
       let eval = function
         | Dvi.C_xxx s -> scan_special status globals pagenum s
         | x -> otherwise x in
       Dvi.page_iter eval cdvi.base_dvi.Dvi.pages.(pagenum);
       page.Dvi.status <- Dvi.Known status;
       status
   | Dvi.Known stored_status -> stored_status;;

let special st s =
  if has_prefix "\" " s || has_prefix "ps: " s || has_prefix "! " s then
    ps_special st s else
  if has_prefix "advi: moveto" s then moveto_special st true s else
  if has_prefix "advi: rmoveto" s then moveto_special st false s else

  (* Other specials *)
  if has_prefix "color " s then color_special st s else
  if has_prefix "html:" s then html_special st (get_suffix "html:" s) else
  if has_prefix "PSfile=" s || has_prefix "psfile=" s then
    begin
      try
  	let file, bbox, size = psfile_special st s in
  	let x = st.x_origin + int_of_float (st.conv *. float st.h)
  	and y = st.y_origin + int_of_float (st.conv *. float st.v) in
  	if !visible then 
  	  let draw = 
  	    if has_prefix "`" file then
  	     Dev.draw_img (zap_to_char ' ' file)
  	       Drawimage.ScaleAuto false 1.0 st.blend (Some bbox)
  	    else Dev.draw_ps file bbox in
  	  draw size x y
      with
      |	Failure s -> Misc.warning s
      |	e -> Misc.warning (Printexc.to_string e)
  end else
  if has_prefix "advi: " s then begin
    if has_prefix "advi: edit" s then edit_special st s else
    if has_prefix "advi: alpha" s then alpha_special st s else
    if has_prefix "advi: blend" s then blend_special st s else
    if has_prefix "advi: epstransparent" s then epstransparent_special st s else
    if has_prefix "advi: pause" s then raise Pause else
    if has_prefix "advi: proc" s then proc_special st s else
    if has_prefix "advi: wait " s then wait_special st s else
    if has_prefix "advi: embed " s then
      (if !visible then embed_special st s) else
    if has_prefix "advi: trans " s then transition_special st s else
    if has_prefix "advi: transbox save " s then transbox_save_special st s else
    if has_prefix "advi: transbox go " s then transbox_go_special st s else
    if has_prefix "advi: kill " s then
      (if !visible then kill_embed_special st s) else
    if has_prefix "advi: setbg " s then bkgd_special st s else
    if has_prefix "advi:" s then Misc.warning ("unknown special: "^ s)
   end else
  if has_prefix "line: " s then line_special st s else
  if has_prefix "pn " s || has_prefix "pa " s || s = "fp" || s = "ip" ||
     has_prefix "da " s || has_prefix "dt " s || s = "sp" ||
     has_prefix "sp " s || has_prefix "ar " s || has_prefix "ia " s ||
     has_prefix "sh " s || s = "wh" || s = "bk" then tpic_specials st s;;

(*** Page rendering ***)
let eval_dvi_command st = function
  | Dvi.C_set code -> set st code
  | Dvi.C_put code -> put st code
  | Dvi.C_set_rule(a, b) -> set_rule st a b
  | Dvi.C_put_rule(a, b) -> put_rule st a b
  | Dvi.C_push -> push st
  | Dvi.C_pop -> pop st
  | Dvi.C_right k -> add_blank 1 st k; st.h <- st.h + k
  | Dvi.C_w0 ->  add_blank 2 st st.w; st.h <- st.h + st.w
  | Dvi.C_w k -> st.w <- k; add_blank 3 st st.w; st.h <- st.h + st.w
  | Dvi.C_x0 ->  add_blank 4 st st.x; st.h <- st.h + st.x
  | Dvi.C_x k -> st.x <- k; add_blank 5 st st.x; st.h <- st.h + st.x
  | Dvi.C_down k -> st.v <- st.v + k
  | Dvi.C_y0 -> st.v <- st.v + st.y
  | Dvi.C_y k -> st.y <- k; st.v <- st.v + st.y
  | Dvi.C_z0 -> st.v <- st.v + st.z
  | Dvi.C_z k -> st.z <- k; st.v <- st.v + st.z
  | Dvi.C_fnt n -> fnt st n
  | Dvi.C_xxx s -> special st s
  | _ -> ();;

(* Unused ???
let scan_command st = function
  | Dvi.C_xxx s -> special st s
  | _ -> ();;*)

let eval_command st c =
  let record r =
    let u = r.unit in
    match c with
    (* The advi: proc specials are not recorded *)
    (* | Dvi.C_xxx s when has_prefix "advi: proc" s -> () *)
    | _ -> u.escaped_commands <- c :: u.escaped_commands in
  List.iter record !current_recording_proc;
  eval_dvi_command st c;;

forward_eval_command := eval_command;;

let newpage h st magdpi x y =
  try Dev.newpage h st.sdpi magdpi x y
  with Dev.GS -> st.status.Dvi.hasps <- false;;

let find_prologues l =
  let l = List.rev l in
  let h = List.map snd (List.filter (function b, _ -> not b) l)  in
  let h' = Search.true_file_names [] h in
  try
    let table = List.combine h h' in
    List.map
      (function b, s as p -> if b then p else b, List.assoc s table) l
  with
    Invalid_argument _  -> 
      Misc.warning
        "Cannot find postscript prologue. Continuing without Postscript"; 
      Options.dops := false;
      []
  | Not_found -> assert false
;;

(* function to be removed in the future, or replaced by
   the proper iteration of render_step *)
let render_page cdvi num dpi xorig yorig =
  failwith "Render_page is deprecated.";;

let render_step cdvi num ?trans dpi xorig yorig =
  proc_clean ();
  if num < 0 || num >= Array.length cdvi.base_dvi.Dvi.pages then
    invalid_arg "Driver.render_step";
  let mag = float cdvi.base_dvi.Dvi.preamble.Dvi.pre_mag /. 1000.0
  and page = cdvi.base_dvi.Dvi.pages.(num) in
  let otherwise _ = () in
  let status =
    let headers = ref []
    and xrefs = cdvi.base_dvi.Dvi.xrefs in
    let s = scan_special_page otherwise cdvi (headers, xrefs) num in
    if !headers <> [] then
      Dev.add_headers (find_prologues !headers);
    s in
  if not !Options.dops then status.Dvi.hasps <- false;
  let st =
    { cdvi = cdvi;
      sdpi = int_of_float (mag *. ldexp dpi 16);
      conv = mag *. dpi /. cdvi.dvi_res /. 65536.0;
      x_origin = xorig; y_origin = yorig;
      cur_mtable = dummy_mtable;
      cur_gtable = dummy_gtable;
      cur_font = dummy_font;
      h = 0; v = 0; w = 0; x = 0; y = 0; z = 0;
      stack = []; color = Grdev.fgcolor (); color_stack = [];
      alpha = 1.0; alpha_stack = [];
      blend = Drawimage.Normal; blend_stack = [];
      epstransparent = true; epstransparent_stack = [];
      direction = trans;
      transition = Transitions.TransNone;
      transition_stack = [];
      tpic_pensize = 0.0; tpic_path = []; tpic_shading = 0.0;
      status = status;
      headers = [];
      html = None;
      draw_html = [];
      checkpoint = 0;
    } in
  if st.status.Dvi.hasps then newpage [] st  (mag *. dpi) xorig yorig;
  setup_bkgd st.status; (* apply the background preferences in Dev *)
  Dev.clear_dev ();     (* and redraw the background               *)
  Dev.set_color st.color;
  Dev.set_transition st.transition;
  st.checkpoint <- 0;
  let check () =
    begin try Dev.continue () with
    (* try with exn -> raise exn ?? What does that mean ? *)
    | Dev.Stop as exn -> raise exn
    end;
    st.checkpoint <- checkpoint_frequency in
  let eval st x =
    st.checkpoint <- st.checkpoint - 1;
    let b = eval_command st x in
    if st.checkpoint < 0 then check ();
    b in
  Dvi.page_step (eval st) page;;

let unfreeze_font cdvi n =
  try
    let cfont = Table.get cdvi.font_table n in
    ignore (Table.get cfont.mtable (Char.code 'A'))
  with _ -> ();;

let unfreeze_fonts cdvi =
  List.iter (fun (n, _) -> unfreeze_font cdvi n)
    cdvi.base_dvi.Dvi.font_map;;

let scan_special_pages cdvi lastpage =
  let headers = ref []
  and xrefs = cdvi.base_dvi.Dvi.xrefs in
  let otherwise _ = () in
  for n = 0 to min lastpage (Array.length cdvi.base_dvi.Dvi.pages) - 1 do
    ignore (scan_special_page otherwise cdvi (headers, xrefs) n);
  done;
  if !headers <> [] then
    Dev.add_headers (find_prologues !headers);;

let unfreeze_glyphs cdvi dpi =
  let mag = float cdvi.base_dvi.Dvi.preamble.Dvi.pre_mag /. 1000.0 in
  let sdpi = int_of_float (mag *. ldexp dpi 16)
  and mtable = ref dummy_mtable
  and gtable = ref dummy_gtable in
  let headers = ref []
  and xrefs = cdvi.base_dvi.Dvi.xrefs in
  let otherwise = function
    | Dvi.C_fnt n ->
        let (mt, gt) =
          try
            let cfont = Table.get cdvi.font_table n in
            (cfont.mtable, get_gtable cfont sdpi)
          with Not_found -> (dummy_mtable, dummy_gtable) in
        mtable := mt;
        gtable := gt
    | Dvi.C_set code ->
        begin try ignore (Table.get !mtable code) with _ -> () end;
        begin try ignore (Table.get !gtable code) with _ -> () end
    | _ -> () in

  let headers = ref []
  and xrefs = cdvi.base_dvi.Dvi.xrefs in
  let globals = headers, xrefs in
  for n = 0 to Array.length cdvi.base_dvi.Dvi.pages - 1 do
    mtable := dummy_mtable;
    gtable := dummy_gtable;
    ignore (scan_special_page otherwise cdvi globals n);
  done;
  Dev.add_headers (find_prologues !headers);;

