/*****************************************************************************
 *
 * xfloat_t.c: Test external float representation.
 *
 * Copyright (c) 1990-2007 Aldor Software Organization Ltd (Aldor.org).
 *
 ****************************************************************************/

#if !defined(TEST_XFLOAT) && !defined(TEST_ALL)

void testXFloat(void) { }

#else

# include "axlgen.h"

#ifdef XFLOAT_DEBUG
# define xfloatDEBUG(x)	x
#else
# define xfloatDEBUG(x)
#endif

/******************************************************************************
 *
 * :: Main test function for portable floating point representation.
 *
 *****************************************************************************/

extern void	testSFloatRoundTrip		(void);
extern void	testDFloatRoundTrip		(void);
extern void	testDFloatExamples		(void);
extern void	testSFExhaustive		(void);
extern void	testDFExhaustive		(void);
extern void	testDFloatToSFloat		(void);
extern void	testFloatInteractiveConversion	(void);

void
testXFloat(void)
{

	xfloatDumpInfo(osStdout);
	
	testSFloatRoundTrip();
	testDFloatRoundTrip();

	testDFloatExamples();

	testSFExhaustive();
	testDFExhaustive();
/*
	testDFloatToSFloat();
*/
	if (osGetEnv("XFLOAT_INTERACTIVE"))
		testFloatInteractiveConversion();

}


/******************************************************************************
 *
 * :: Macros for defining floating-point test cases.
 *
 *****************************************************************************/

#ifdef CC_non_ieee_floats
#  define	floateq(a,b)	((a) == (b))
#else
#  define	floateq(a,b)	ubEqual(a, b)
#endif

#define ubEqual(a,b)	_ubEqual(sizeof(a),(UByte *)&(a),(UByte *)&(b))
#define ubPrint(a)	_ubPrint(sizeof(a),(UByte *)&(a))
#define ubPrintSF(a)	_ubPrintSF((UByte *)&(a))
#define ubPrintDF(a)	_ubPrintDF((UByte *)&(a))

Bool
_ubEqual(int n, UByte *ub1, UByte *ub2)
{
	int	i;
	
	for (i = 0; i < n; i++) if (ub1[i] != ub2[i]) return false;
	return true;
}

void
_ubPrint(int n, UByte *ub)
{
	int	i;
	printf("0x");
	for (i = 0; i < n; i++)
		printf(i ? ".%02x" : "%02x", ub[i]);
}

void
_ubPrintSF(UByte *ub)
{
	UByte	data[sizeof(SFloat)];

	unsigned int i;

	for (i = 0; i < sizeof(SFloat); i++)
		data[i] = SF_UByte(ub, i);
	_ubPrint(sizeof(SFloat), data);
}

void
_ubPrintDF(UByte *ub)
{
	UByte	data[sizeof(DFloat)];

	unsigned int i;

	for (i = 0; i < sizeof(DFloat); i++)
		data[i] = DF_UByte(ub, i);
	_ubPrint(sizeof(DFloat), data);
}

/******************************************************************************
 *
 * :: Test round-trip: Native->Extended->Native
 *
 *****************************************************************************/

struct sfTestCase { Bool isTest; UByte data[sizeof(float)];  };
struct dfTestCase { Bool isTest; UByte data[sizeof(double)]; };

extern struct sfTestCase	sfTestCase[];
extern struct dfTestCase	dfTestCase[];

void
testSFloatRoundTrip(void)
{
	SFloat	test;
	SFloat	sbuf;
	XSFloat	xbuf;

	unsigned int i, j;

	printf("SFloat round trip:\n");

	for (i = 0; sfTestCase[i].isTest; i++) {
		for (j = 0; j < sizeof(float); j++)
			SF_UByte(&test, j) = sfTestCase[i].data[j];

		xsfFrNative(&xbuf, &test);
		xsfToNative(&xbuf, &sbuf);

	       if (!floateq(test, sbuf)) {
			printf("\n id(");
			ubPrintSF(test);
			printf(") -> ");
			ubPrint(xbuf);
			printf(" -> ");
			ubPrintSF(sbuf);
		}
		else {
			printf("\n id(");
			ubPrintSF(test);
			printf(")");
		}
	}
	printf("\n");
}

void
testDFloatRoundTrip(void)
{
	DFloat	test;
	DFloat	dbuf;
	XDFloat	xbuf;

  	xfloatDEBUG(int 	expon;)
  	xfloatDEBUG(Bool	sign;)
  	xfloatDEBUG(UByte	pb[sizeof(XDFloat)];)

	unsigned int i, j;

	printf("DFloat round trip:\n");
	for (i = 0; dfTestCase[i].isTest; i++) {
		for (j = 0; j < sizeof(double); j++)
			DF_UByte(&test, j) = dfTestCase[i].data[j];

		xdfFrNative(&xbuf, &test);
		xdfToNative(&xbuf, &dbuf);

		if (!ubEqual(test, dbuf)) {
			printf("\n id(");
			ubPrintDF(test);
			printf(") -> ");

			xfloatDEBUG({
			  dfDissemble(&test, &sign,&expon,pb,NULL);
			  printf("=%d%c",expon,sign?'-':'+');
			  for (j=0;j<sizeof(double);j++) printf("%02x.",pb[j]);
			  fflush(osStdout);
			});

			ubPrint(xbuf);
			printf(" -> ");

			xfloatDEBUG({
			  fflush(osStdout);
			  xdfDissemble(&xbuf, &sign,&expon,pb);
			  printf("=%d%c",expon,sign?'-':'+');
			  for (j=0;j<sizeof(double);j++) printf("%02x.",pb[j]);
			  fflush(osStdout);
			});

			ubPrintDF(dbuf);

			xfloatDEBUG({
			  fflush(osStdout);
			  dfDissemble(&dbuf, &sign,&expon,pb,NULL);
			  printf("=%d%c",expon,sign?'-':'+');
			  for (j=0;j<sizeof(double);j++) printf("%02x.",pb[j]);
			  fflush(osStdout);
			});

		}
		else {
			printf("\n id(");
			ubPrintDF(test);
			printf(")");
		}

		xfloatDEBUG(fflush(osStdout));
	}
	printf("\n");
}


void
testDFloatExamples(void)
{
#ifdef XFLOAT_DEBUG
#	define	NDV	100
	double	dv[NDV];
	int	i, j, n;
	UByte	pb[sizeof(double)];
	int	expon;
	Bool	sign;
	double	d;
	XDFloat	xdf;

	n = 0;

	dv[n++] = 1.0;
	dv[n++] = 2.0;
	dv[n++] = 4.0;
	dv[n++] = 8.0;
	dv[n++] =  .1234e40;
	dv[n++] =  .2234e40;
	dv[n++] = -.2234e40;
	dv[n++] = 0;
	dv[n++] =  1.234;
	dv[n++] = log(0.0);
	dv[n++] = asin(2.0);
	dv[n++] = 1e-310;
	dv[n++] = 1.2345e-313;

	for (j = 0; j < n; j++) {
		printf("%2d => %12e = ", j, dv[j]);

		dfDissemble(&dv[j], &sign, &expon, pb,NULL);
		for (i = 0; i < sizeof(double); i++) printf("%02x.", pb[i]);
		printf("  ");
		switch (dfClassify(&dv[j])) {
		case FLOAT_NORM:	printf("Norm "); break;
		case FLOAT_DENORM: 	printf("Deno "); break;
		case FLOAT_ZERO:	printf("Zero "); break;
		case FLOAT_NAN:		printf("NAN  "); break;
		case FLOAT_INF:		printf("INF  "); break;
		default:        	printf("???  "); break;
		}
		printf("(%d) ", bfFirst1(sizeof(double), pb));
		printf("Sgn= %d, ", sign);
		printf("Exp= %d, ", expon);
		printf("\n");

		dfAssemble(&d, sign, expon, pb);
		printf("   ...%12e = ", d);

		dfDissemble(&d, &sign, &expon, pb,NULL);
		fracNormalize(&expon, sizeof(double), pb);
		for (i = 0; i < sizeof(double); i++) printf("%02x.", pb[i]);
		printf("       (%d) ", bfFirst1(sizeof(double), pb));
		printf("Sgn= %d, ", sign);
		printf("Exp= %d, ", expon);
		printf("\n");

		fracDenormalize(&expon, dfExponMin, sizeof(double), pb,
				DF_LgLgBase, DF_HasNorm1);
		dfAssemble(&d, sign, expon, pb);
		printf("   ...%12e = ", d);

		xdfFrNative(&xdf, &d);
		xdfPrint(osStdout, &xdf);
		xdfToNative(&xdf, &d);
		printf("\n");
		printf("   -->%12e = ", d);

		dfDissemble(&d, &sign, &expon, pb,NULL);
		for (i = 0; i < sizeof(double); i++) printf("%02x.", pb[i]);
		printf("  ");
		switch (dfClassify(&d)) {
		case FLOAT_NORM:	printf("Norm "); break;
		case FLOAT_DENORM: 	printf("Deno "); break;
		case FLOAT_ZERO:	printf("Zero "); break;
		case FLOAT_NAN:		printf("NAN  "); break;
		case FLOAT_INF:		printf("INF  "); break;
		default:        	printf("???  "); break;
		}
		printf("(%d) ", bfFirst1(sizeof(double), pb));
		printf("Sgn= %d, ", sign);
		printf("Exp= %d, ", expon);
		printf("\n");
		printf("\n");

	}
#endif
}


/******************************************************************************
 *
 * :: Exhaustive tests
 *
 *****************************************************************************/

void
testOneSFloatDebug(SFloat *test)
{
	int	expon, j;
	Bool	sign;
	UByte	pb[sizeof(SFloat)];

	sfDissemble(test, &sign, &expon, pb,NULL);
	fprintf(osStdout, "=%d%c", expon, sign ? '-' : '+');
	for (j = 0; j < sizeof(SFloat); j += 1)
		fprintf(osStdout, "%02x.", pb[j]);
	fflush(osStdout);
}

void
testOneXSFloatDebug(XSFloat *test)
{
	int	expon, j;
	Bool	sign;
	UByte	pb[sizeof(XSFloat)];

	xsfDissemble(test, &sign, &expon, pb);
	fprintf(osStdout, "=%d%c", expon, sign ? '-' : '+');
	for (j = 0; j < sizeof(XSFloat); j += 1)
		fprintf(osStdout, "%02x.", pb[j]);
	fflush(osStdout);
}

void
testOneDFloatDebug(DFloat *test)
{
	int	expon, j;
	Bool	sign;
	UByte	pb[sizeof(DFloat)];

	dfDissemble(test, &sign, &expon, pb,NULL);
	fprintf(osStdout, "=%d%c", expon, sign ? '-' : '+');
	for (j = 0; j < sizeof(DFloat); j += 1)
		fprintf(osStdout, "%02x.", pb[j]);
	fflush(osStdout);
}

void
testOneXDFloatDebug(XDFloat *test)
{
	int	expon, j;
	Bool	sign;
	UByte	pb[sizeof(XDFloat)];

	xdfDissemble(test, &sign, &expon, pb);
	fprintf(osStdout, "=%d%c", expon, sign ? '-' : '+');
	for (j = 0; j < sizeof(XDFloat); j += 1)
		fprintf(osStdout, "%02x.", pb[j]);
	fflush(osStdout);
}

void
testOneSFloat(SFloat *test)
{
	SFloat	sbuf;
	XSFloat	xbuf;

	xsfFrNative(&xbuf, test);
	xsfToNative(&xbuf, &sbuf);

	if (!floateq(*test, sbuf)) {
		printf("\n id(");
		ubPrintSF(*test);
		printf(") -> ");

		xfloatDEBUG(testOneSFloatDebug(test));

		ubPrint(xbuf);
		printf(" -> ");

		xfloatDEBUG(testOneXSFloatDebug(&xbuf));

		ubPrintSF(sbuf);
		printf("(%e,%e)\n", *test, sbuf);

		xfloatDEBUG(testOneSFloatDebug(&sbuf));
	}
}

void
testOneDFloat(DFloat *test)
{
	DFloat	dbuf;
	XDFloat	xbuf;

	xdfFrNative(&xbuf, test);
	xdfToNative(&xbuf, &dbuf);

	if (!floateq(*test, dbuf)) {
		printf("\n id(");
		ubPrintDF(*test);
		printf(") -> ");

		xfloatDEBUG(testOneDFloatDebug(test));

		ubPrint(xbuf);
		printf(" -> ");

		xfloatDEBUG(testOneXDFloatDebug(&xbuf));

		ubPrintDF(dbuf);
		printf("(%e,%e)\n", *test, dbuf);

		xfloatDEBUG(testOneDFloatDebug(&dbuf));
	}
}

#define SFADD0(_s0,_s1,_s2,_s3) {		\
	SF_UByte(&test, int0) = _s0;		\
	SF_UByte(&test, 1) = _s1;		\
	SF_UByte(&test, 2) = _s2;		\
	SF_UByte(&test, 3) = _s3;		\
}

#define DFADD0(_d0,_d1,_d2,_d3,_d4,_d5,_d6,_d7) {	\
	DF_UByte(&test, int0) = _d0;		\
	DF_UByte(&test, 1) = _d1;		\
	DF_UByte(&test, 2) = _d2;		\
	DF_UByte(&test, 3) = _d3;		\
	DF_UByte(&test, 4) = _d4;		\
	DF_UByte(&test, 5) = _d5;		\
	DF_UByte(&test, 6) = _d6;		\
	DF_UByte(&test, 7) = _d7;		\
}


#define SFADD0(_s0,_s1,_s2,_s3) {		\
	SF_UByte(&test, int0) = _s0;		\
	SF_UByte(&test, 1) = _s1;		\
	SF_UByte(&test, 2) = _s2;		\
	SF_UByte(&test, 3) = _s3;		\
}

#define DFADD0(_d0,_d1,_d2,_d3,_d4,_d5,_d6,_d7) {	\
	DF_UByte(&test, int0) = _d0;		\
	DF_UByte(&test, 1) = _d1;		\
	DF_UByte(&test, 2) = _d2;		\
	DF_UByte(&test, 3) = _d3;		\
	DF_UByte(&test, 4) = _d4;		\
	DF_UByte(&test, 5) = _d5;		\
	DF_UByte(&test, 6) = _d6;		\
	DF_UByte(&test, 7) = _d7;		\
}

void
testDFExhaustive(void)
{
	UByte first, second;
	ULong x;
	DFloat test = 0.0;
	int y = -1;

	printf("Warning: Running exhaustive tests. ");
	printf("It may take several minutes...\n");
	
	for ( x = 0 ; x <= 0xffff ; x++) {
		first = x >> 8;
		second = x & 0xff;
		
		DFADD0(first , second, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00);
		testOneDFloat(&test);
		DFADD0(first , second, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00);
		testOneDFloat(&test);

		DFADD0(first , second, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01);
		testOneDFloat(&test);
		DFADD0(first , second, 0x88, 0x00, 0x00, 0x00, 0x00, 0x01);
		testOneDFloat(&test);

		DFADD0(first , second, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00);
		testOneDFloat(&test);
		DFADD0(first , second, 0x88, 0x00, 0x00, 0x00, 0x80, 0x00);
		testOneDFloat(&test);

		DFADD0(first , second, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08);
		testOneDFloat(&test);
		DFADD0(first , second, 0x88, 0x00, 0x00, 0x00, 0x40, 0x08);
		testOneDFloat(&test);

		if (y++ == 999) {
			printf("[dfloat: %lu/65535 done]\n", x);
			y = 0;
		}
	}
}


void
testSFExhaustive(void)
{
	UByte first, second;
	ULong x;
	SFloat test = 0.0;
	int y = -1;

	printf("Warning: Running exhaustive tests.");
	printf(" It may take several minutes...\n");
	
	for ( x = 0 ; x <= 0xffff ; x++) {
		first = x >> 8;
		second = x & 0xff;
		
		SFADD0(first , second, 0x00, 0x00);
		testOneSFloat(&test);
		SFADD0(first , second, 0x88, 0x00);
		testOneSFloat(&test);

		SFADD0(first , second, 0x00, 0x01);
		testOneSFloat(&test);
		SFADD0(first , second, 0x88, 0x01);
		testOneSFloat(&test);

		SFADD0(first , second, 0x80, 0x00);
		testOneSFloat(&test);

		SFADD0(first , second, 0x40, 0x08);
		testOneSFloat(&test);

		if (y++ == 999) {
			printf("[sfloat: %lu/65535 done]\n", x);
			y = 0;
		}
	}
}

/******************************************************************************
 *
 * :: Double->Extended->Single
 *
 *****************************************************************************/

void
testOneDFloatToSFloat(DFloat df)
{
	SFloat	sbuf;
	XDFloat xdbuf;
	Bool failed;

	int	sexpon, dexpon;
	Bool	ssign, dsign;
	UByte	spb[sizeof(XSFloat)];
	UByte	dpb[sizeof(XDFloat)];

	xdfFrNative(&xdbuf, &df);
	xsfToNative((XSFloat *)&xdbuf, &sbuf);

	dfDissemble(&df, &dsign, &dexpon, dpb,NULL);
	sfDissemble(&sbuf, &ssign, &sexpon, spb,NULL);

	failed = false;

	if (dexpon != sexpon) {
		if (! ((sexpon == sfExponMin() ||
		        (sexpon - dexpon >= CHAR_BIT * sizeof(SFloat))) ||
		       (sexpon == sfExponNAN() && dexpon == dfExponNAN())))
		{
			printf("FAILED: different exponent. (");
			failed = true;
		}
	}
	else if (ssign != dsign) {
		printf("FAILED: different sign. (");
		failed = true;
	}
	else if (sexpon != sfExponMin() &&
		 sexpon - dexpon < CHAR_BIT * sizeof(SFloat) &&
		 !_ubEqual(2, (UByte *)spb, (UByte *)dpb))
	{
		printf("FAILED: different fraction (first 2 bytes) (");
		_ubPrint(sizeof(spb), spb);
		printf(" <> ");
		_ubPrint(sizeof(dpb), dpb);
		printf(")\n");
	}
	else if ((float) df != sbuf) {
		printf("DIFFERENT cast: %f %f (", df, sbuf);
		failed = true;
	}

	if (failed) {
		ubPrintSF(sbuf);
		printf(" <> ");
		ubPrintDF(df);
		printf(")\n");
	}
}

void
testDFloatToSFloat(void)
{
	UByte first, second;
	ULong x;
	DFloat test = 0.0;
	int y = -1;

	printf("Dfloat to SFloat trip:\n");

	testOneDFloatToSFloat(0.21231231231231243243);
	testOneDFloatToSFloat(0.000000012303123000123);
	testOneDFloatToSFloat(312311231.32424342342342);
	testOneDFloatToSFloat(2343242342.23432434343242);
	testOneDFloatToSFloat((DFloat) 1);
	testOneDFloatToSFloat(1.00);
	testOneDFloatToSFloat(10.300);
	testOneDFloatToSFloat(232.22200000000001);
	testOneDFloatToSFloat(0.00000000000000001);
	testOneDFloatToSFloat(0.2311239895435453543);
	testOneDFloatToSFloat(40.232);
	testOneDFloatToSFloat(232132.12313213);
	testOneDFloatToSFloat(0.3123232432345454);
	testOneDFloatToSFloat(0.4444444444444444444444444444444444444444);
	testOneDFloatToSFloat(3.4444444444444444444444444444444444444444);
	testOneDFloatToSFloat((DFloat) 0);

	printf("Warning: Running exhaustive tests. ");
	printf("It may take several minutes...\n");
	
	for ( x = 19000 ; x <= 50000 ; x++) {
		first = x >> 8;
		second = x & 0xff;
		
		DFADD0(first , second, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00);
		testOneDFloatToSFloat(test);
		DFADD0(first , second, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00);
		testOneDFloatToSFloat(test);

		DFADD0(first , second, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01);
		testOneDFloatToSFloat(test);
		DFADD0(first , second, 0x88, 0x00, 0x00, 0x00, 0x00, 0x01);
		testOneDFloatToSFloat(test);

		DFADD0(first , second, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00);
		testOneDFloatToSFloat(test);
		DFADD0(first , second, 0x88, 0x00, 0x00, 0x00, 0x80, 0x00);
		testOneDFloatToSFloat(test);

		DFADD0(first , second, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08);
		testOneDFloatToSFloat(test);
		DFADD0(first , second, 0x88, 0x00, 0x00, 0x00, 0x40, 0x08);
		testOneDFloatToSFloat(test);

		if (y++ == 999) {
			printf("[dfloat to sfloat: %lu/65535 done]\n", x);
			y = 0;
		}
	}


}

/******************************************************************************
 *
 * :: Interactive tests
 *
 *****************************************************************************/

/* Takes 4 bytes (the float) and outputs the corresponding xsfloat */
void
testDirectSFloatToXSFloat(void)
{
	SFloat test;
	XSFloat xbuf;
/*	unsigned b0, b1, b2, b3; */

	UByte pb[sizeof(SFloat)];
	int expon;
	Bool sign;
/*
	printf("Enter a 4-bytes float:\n");
	scanf("%x",&b9);
	scanf("%x",&b1);
	scanf("%x",&b2);
	scanf("%x",&b3);

	SFADD0((UByte) b9 ,(UByte) b1, (UByte) b2, (UByte) b3);
*/

/*	printf("Enter exp: ");
	scanf("%d", &expon);
	pb[0] = pb[1] = pb[2] = pb[3] = 0;
	pb[0] = 0x80;
	sfAssemble(&test, 0, expon, pb);
*/
	printf("Enter a float:");
	scanf("%f", &test);

	sfDissemble(&test, &sign, &expon, pb,NULL);
	printf("[%d|%d]\n", sign, expon);

	xsfFrNative(&xbuf, &test);
	printf("\n -->  %e\n --> id(", test);
	ubPrintBits(test);
	printf(")\n");

	printf(" --> id(");
	ubPrintSF(test);
	printf(")\n");

	printf(" --> id(");
	ubPrintBits(xbuf);
	printf(")\n");

	printf(" --> id(");
	ubPrint(xbuf);
	printf(")\n");
}

/* Takes 6 bytes (the xsfloat) and output the corrisponding float */
void
testDirectXSFloatToSFloat(void)
{
	SFloat test;
	XSFloat xbuf;
	UByte * p = (UByte *) &xbuf;
	unsigned u0;
	int i;

	UByte pb[sizeof(SFloat)];
	int expon;
	Bool sign;

	printf("Enter a 6-bytes xsfloat:\n");

	for (i = 0; i < 6; i++) {
		scanf("%x",&u0);
		*p++ = (UByte) u0;
	}

	xsfToNative(&xbuf, &test);

	sfDissemble(&test, &sign, &expon, pb,NULL);
	printf("[%d|%d]\n", sign, expon);

	printf("\n --> id(");
	ubPrintBits(xbuf);
	printf(")");

	printf("\n --> id(");
	ubPrint(xbuf);
	printf(")");

	printf("\n --> id(");
	ubPrintBits(test);
	printf(") %e\n", test);

	printf(" --> id(");
	ubPrintSF(test);
	printf(")\n");

}

void
testFloatInteractiveConversion(void)
{
	char c = '1';
	while (c != '0') {
		printf("Chose direct test:\n");
		printf("\t1: native -> extended;");
		printf("\t2: extended -> native;");
		printf("\t0: exit > ");

		scanf("%c", &c);

		if (c == '1')
			testDirectSFloatToXSFloat();
		else if (c == '2')
			testDirectXSFloatToSFloat();
	
	}
}


/******************************************************************************
 *
 * :: IEEE 754 Data
 *
 *****************************************************************************/

#ifndef CC_non_ieee_floats

/*
 * IEEE 754 Single precision floats.
 */
struct sfTestCase sfTestCase[] = {
	/* Positive */
	{1,{0x00,0x00,0x00,0x00}}, /* zero */
	{1,{0x00,0x00,0x00,0x01}}, /* epsilon */
	{1,{0x00,0x7f,0xff,0xff}}, /* small - epsilon */
	{1,{0x00,0x80,0x00,0x00}}, /* small */
	{1,{0x00,0x80,0x00,0x01}}, /* small + epsilon */
	{1,{0x00,0xff,0xff,0xff}}, /* 2*small - epsilon */
	{1,{0x01,0x00,0x00,0x00}}, /* 2*small */
	{1,{0x01,0x00,0x00,0x01}}, /* 2*small + epsilon */
	{1,{0x3f,0x7f,0xff,0xff}}, /* 1 - epsilon */
	{1,{0x3f,0x80,0x00,0x00}}, /* 1 */
	{1,{0x3f,0x80,0x00,0x01}}, /* 1 + epsilon */
	{1,{0x3f,0xff,0xff,0xff}}, /* 2 - epsilon */
	{1,{0x40,0x00,0x00,0x00}}, /* 2 */
	{1,{0x40,0x00,0x00,0x01}}, /* 2 + epsilon */
	{1,{0x40,0x7f,0xff,0xff}}, /* 4 - epsilon */
	{1,{0x7e,0xff,0xff,0xff}}, /* large - epsilon */
	{1,{0x7f,0x00,0x00,0x00}}, /* large */
	{1,{0x7f,0x00,0x00,0x01}}, /* large + epsilon */
	{1,{0x7f,0x7f,0xff,0xff}}, /* 2*large - epsilon */
	{1,{0x7f,0x80,0x00,0x00}}, /* infinity */
	{1,{0x7f,0x80,0x00,0x01}}, /* NaN */
	{1,{0x7f,0xff,0xff,0xff}}, /* NaN */

	/* Negative */
	{1,{0x80,0x00,0x00,0x00}}, /* zero */
	{1,{0x80,0x00,0x00,0x01}}, /* epsilon */
	{1,{0x80,0x7f,0xff,0xff}}, /* small - epsilon */
	{1,{0x80,0x80,0x00,0x00}}, /* small */
	{1,{0x80,0x80,0x00,0x01}}, /* small + epsilon */
	{1,{0x80,0xff,0xff,0xff}}, /* 2*small - epsilon */
	{1,{0x81,0x00,0x00,0x00}}, /* 2*small */
	{1,{0x81,0x00,0x00,0x01}}, /* 2*small + epsilon */
	{1,{0xbf,0x7f,0xff,0xff}}, /* 1 - epsilon */
	{1,{0xbf,0x80,0x00,0x00}}, /* 1 */
	{1,{0xbf,0x80,0x00,0x01}}, /* 1 + epsilon */
	{1,{0xbf,0xff,0xff,0xff}}, /* 2 - epsilon */
	{1,{0xc0,0x00,0x00,0x00}}, /* 2 */
	{1,{0xc0,0x00,0x00,0x01}}, /* 2 + epsilon */
	{1,{0xc0,0x7f,0xff,0xff}}, /* 4 - epsilon */
	{1,{0xfe,0xff,0xff,0xff}}, /* large - epsilon */
	{1,{0xff,0x00,0x00,0x00}}, /* large */
	{1,{0xff,0x00,0x00,0x01}}, /* large + epsilon */
	{1,{0xff,0x7f,0xff,0xff}}, /* 2*large - epsilon */
	{1,{0xff,0x80,0x00,0x00}}, /* infinity */
	{1,{0xff,0x80,0x00,0x01}}, /* NaN */
	{1,{0xff,0xff,0xff,0xff}}, /* NaN */

	/* Normalized numbers */
	{1,{0x40,0x40,0x00,0x00}},
	{1,{0x40,0x20,0x00,0x00}},
	{1,{0x40,0x10,0x00,0x00}},
	{1,{0x40,0x08,0x00,0x00}},
	{1,{0x40,0x04,0x00,0x00}},
	{1,{0x40,0x02,0x00,0x00}},
	{1,{0x40,0x01,0x00,0x00}},
	{1,{0x40,0x00,0x80,0x00}},
	{1,{0x40,0x00,0x40,0x00}},
	{1,{0x40,0x00,0x20,0x00}},
	{1,{0x40,0x00,0x10,0x00}},
	{1,{0x40,0x00,0x08,0x00}},
	{1,{0x40,0x00,0x04,0x00}},
	{1,{0x40,0x00,0x02,0x00}},
	{1,{0x40,0x00,0x01,0x00}},
	{1,{0x40,0x00,0x00,0x80}},
	{1,{0x40,0x00,0x00,0x40}},
	{1,{0x40,0x00,0x00,0x20}},

	{1,{0x40,0x00,0x00,0x10}},
	{1,{0x40,0x00,0x00,0x08}},
	{1,{0x40,0x00,0x00,0x04}},
	{1,{0x40,0x00,0x00,0x02}},
	{1,{0x40,0x00,0x00,0x01}},

	/* Denormalized numbers */
	{1,{0x00,0x40,0x00,0x00}},
	{1,{0x00,0x20,0x00,0x00}},
	{1,{0x00,0x10,0x00,0x00}},
	{1,{0x00,0x08,0x00,0x00}},
	{1,{0x00,0x04,0x00,0x00}},
	{1,{0x00,0x02,0x00,0x00}},
	{1,{0x00,0x01,0x00,0x00}},
	{1,{0x00,0x00,0x80,0x00}},
	{1,{0x00,0x00,0x40,0x00}},
	{1,{0x00,0x00,0x20,0x00}},
	{1,{0x00,0x00,0x10,0x00}},
	{1,{0x00,0x00,0x08,0x00}},
	{1,{0x00,0x00,0x04,0x00}},
	{1,{0x00,0x00,0x02,0x00}},
	{1,{0x00,0x00,0x01,0x00}},
	{1,{0x00,0x00,0x00,0x80}},
	{1,{0x00,0x00,0x00,0x40}},
	{1,{0x00,0x00,0x00,0x20}},
	{1,{0x00,0x00,0x00,0x10}},
	{1,{0x00,0x00,0x00,0x08}},
	{1,{0x00,0x00,0x00,0x04}},
	{1,{0x00,0x00,0x00,0x02}},
	{1,{0x00,0x00,0x00,0x01}},

	/* Exponent ranges */
	{1,{0x00,0x00,0x00,0x01}},
	{1,{0x00,0x80,0x00,0x01}},
	{1,{0x01,0x00,0x00,0x01}},
	{1,{0x01,0x80,0x00,0x01}},
	{1,{0x02,0x00,0x00,0x01}},
	{1,{0x3e,0x00,0x00,0x01}},
	{1,{0x3e,0x80,0x00,0x01}},
	{1,{0x3f,0x00,0x00,0x01}},
	{1,{0x3f,0x80,0x00,0x01}},
	{1,{0x40,0x00,0x00,0x01}},
	{1,{0x40,0x80,0x00,0x01}},
	{1,{0x41,0x00,0x00,0x01}},
	{1,{0x41,0x80,0x00,0x01}},
	{1,{0x7e,0x00,0x00,0x01}},
	{1,{0x7e,0x80,0x00,0x01}},
	{1,{0x7f,0x00,0x00,0x01}},
	{1,{0x7f,0x80,0x00,0x01}},
	{0}
};


/*
 * IEEE 754 Double precision floats.
 */
struct dfTestCase dfTestCase[] = {
	/* Positive */
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}}, /* zero */
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01}}, /* epsilon */
	{1,{0x00,0x0f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* small - epsilon */
	{1,{0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00}}, /* small */
	{1,{0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x01}}, /* small + epsilon */
	{1,{0x00,0x1f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 2*small - epsilon */
	{1,{0x00,0x20,0x00,0x00,0x00,0x00,0x00,0x00}}, /* 2*small */
	{1,{0x00,0x20,0x00,0x00,0x00,0x00,0x00,0x01}}, /* 2*small + epsilon */
	{1,{0x3f,0xef,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 1 - epsilon */
	{1,{0x3f,0xf0,0x00,0x00,0x00,0x00,0x00,0x00}}, /* 1 */
	{1,{0x3f,0xf0,0x00,0x00,0x00,0x00,0x00,0x01}}, /* 1 + epsilon */
	{1,{0x3f,0xff,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 2 - epsilon */
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00}}, /* 2 */
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x01}}, /* 2 + epsilon */
	{1,{0x40,0x0f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 4 - epsilon */
	{1,{0x7f,0xdf,0xff,0xff,0xff,0xff,0xff,0xff}}, /* large - epsilon */
	{1,{0x7f,0xe0,0x00,0x00,0x00,0x00,0x00,0x00}}, /* large */
	{1,{0x7f,0xe0,0x00,0x00,0x00,0x00,0x00,0x01}}, /* large + epsilon */
	{1,{0x7f,0xef,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 2*large - epsilon */
	{1,{0x7f,0xf0,0x00,0x00,0x00,0x00,0x00,0x00}}, /* infinity */
	{1,{0x7f,0xf0,0x00,0x00,0x00,0x00,0x00,0x01}}, /* NaN */
	{1,{0x7f,0xff,0xff,0xff,0xff,0xff,0xff,0xff}}, /* NaN */

	/* Negative */
	{1,{0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00}}, /* zero */
	{1,{0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x01}}, /* epsilon */
	{1,{0x80,0x0f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* small - epsilon */
	{1,{0x80,0x10,0x00,0x00,0x00,0x00,0x00,0x00}}, /* small */
	{1,{0x80,0x10,0x00,0x00,0x00,0x00,0x00,0x01}}, /* small + epsilon */
	{1,{0x80,0x1f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 2*small - epsilon */
	{1,{0x80,0x20,0x00,0x00,0x00,0x00,0x00,0x00}}, /* 2*small */
	{1,{0x80,0x20,0x00,0x00,0x00,0x00,0x00,0x01}}, /* 2*small + epsilon */
	{1,{0xbf,0xef,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 1 - epsilon */
	{1,{0xbf,0xf0,0x00,0x00,0x00,0x00,0x00,0x00}}, /* 1 */
	{1,{0xbf,0xf0,0x00,0x00,0x00,0x00,0x00,0x01}}, /* 1 + epsilon */
	{1,{0xbf,0xff,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 2 - epsilon */
	{1,{0xc0,0x00,0x00,0x00,0x00,0x00,0x00,0x00}}, /* 2 */
	{1,{0xc0,0x00,0x00,0x00,0x00,0x00,0x00,0x01}}, /* 2 + epsilon */
	{1,{0xc0,0x0f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 4 - epsilon */
	{1,{0xff,0xdf,0xff,0xff,0xff,0xff,0xff,0xff}}, /* large - epsilon */
	{1,{0xff,0xe0,0x00,0x00,0x00,0x00,0x00,0x00}}, /* large */
	{1,{0xff,0xe0,0x00,0x00,0x00,0x00,0x00,0x01}}, /* large + epsilon */
	{1,{0xff,0xef,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 2*large - epsilon */
	{1,{0xff,0xf0,0x00,0x00,0x00,0x00,0x00,0x00}}, /* infinity */
	{1,{0xff,0xf0,0x00,0x00,0x00,0x00,0x00,0x01}}, /* NaN */
	{1,{0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff}}, /* NaN */

	/* Normalized numbers */
	{1,{0x40,0x08,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x04,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x02,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x01,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x80,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x40,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x20,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x10,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x08,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x04,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x02,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x01,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x80,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x40,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x20,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x10,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x08,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x04,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x02,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x01,0x00,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x80,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x40,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x20,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x10,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x08,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x04,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x02,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x01,0x00,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x80,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x40,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x20,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x10,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x08,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x04,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x02,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x01,0x00,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x80,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x40,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x20,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x10,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x08,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x04,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x02,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x01,0x00}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x80}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x40}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x20}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x10}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x08}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x04}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x02}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x01}},

	/* Denormalized numbers */
	{1,{0x00,0x08,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x40,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x20,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x08,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x40,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x40,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x20,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x08,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x40,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x08,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x04,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x02,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x40,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x40}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01}},

	/* Exponent ranges */
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x00,0x20,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x00,0x30,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x3f,0xc0,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x3f,0xd0,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x3f,0xe0,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x3f,0xf0,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x40,0x10,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x40,0x20,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x40,0x30,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x7f,0xc0,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x7f,0xd0,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x7f,0xe0,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x7f,0xf0,0x00,0x00,0x00,0x00,0x00,0x01}},
	{0}
};

#else  /* CC_non_ieee_floats */


/******************************************************************************
 *
 * :: IBM 370 Data
 *
 *****************************************************************************/

#ifdef HW_370

/*
 * IBM 370 Single precision floats
 */
struct sfTestCase sfTestCase[] = {
	/* Positive */
	{1,{0x00,0x00,0x00,0x00}}, /* zero */
	{1,{0x00,0x00,0x00,0x01}}, /* epsilon */
	{1,{0x00,0xff,0xff,0xff}}, /* small - epsilon */
	{1,{0x01,0x10,0x00,0x00}}, /* small */
	{1,{0x01,0x10,0x00,0x01}}, /* small + epsilon */
	{1,{0x01,0x1f,0xff,0xff}}, /* 2*small - epsilon */
	{1,{0x01,0x20,0x00,0x00}}, /* 2*small */
	{1,{0x01,0x20,0x00,0x01}}, /* 2*small + epsilon */
	{1,{0x40,0xff,0xff,0xff}}, /* 1 - epsilon */
	{1,{0x41,0x10,0x00,0x00}}, /* 1 */
	{1,{0x41,0x10,0x00,0x01}}, /* 1 + epsilon */
	{1,{0x41,0x1f,0xff,0xff}}, /* 2 - epsilon */
	{1,{0x41,0x20,0x00,0x00}}, /* 2 */
	{1,{0x41,0x20,0x00,0x01}}, /* 2 + epsilon */
	{1,{0x41,0x3f,0xff,0xff}}, /* 4 - epsilon */
	{1,{0x7f,0x3f,0xff,0xff}}, /* large - epsilon */
	{1,{0x7f,0x40,0x00,0x00}}, /* large */
	{1,{0x7f,0x40,0x00,0x01}}, /* large + epsilon */
	{1,{0x7f,0x7f,0xff,0xff}}, /* 2*large - epsilon */
	{1,{0x7f,0x80,0x00,0x00}}, /* 2*large */
	{1,{0x7f,0x80,0x00,0x01}}, /* 2*large + epsilon */
	{1,{0x7f,0xff,0xff,0xff}}, /* 4*large - epsilon */

	/* Negative */
	/* Only Ieee floats have signed zeros. */
	/* SFADD(0x80,0x00,0x00,0x00}}, */
	{1,{0x80,0x00,0x00,0x01}}, /* epsilon */
	{1,{0x80,0xff,0xff,0xff}}, /* small - epsilon */
	{1,{0x81,0x10,0x00,0x00}}, /* small */
	{1,{0x81,0x10,0x00,0x01}}, /* small + epsilon */
	{1,{0x81,0x1f,0xff,0xff}}, /* 2*small - epsilon */
	{1,{0x81,0x20,0x00,0x00}}, /* 2*small */
	{1,{0x81,0x20,0x00,0x01}}, /* 2*small + epsilon */
	{1,{0xc0,0xff,0xff,0xff}}, /* 1 - epsilon */
	{1,{0xc1,0x10,0x00,0x00}}, /* 1 */
	{1,{0xc1,0x10,0x00,0x01}}, /* 1 + epsilon */
	{1,{0xc1,0x1f,0xff,0xff}}, /* 2 - epsilon */
	{1,{0xc1,0x20,0x00,0x00}}, /* 2 */
	{1,{0xc1,0x20,0x00,0x01}}, /* 2 + epsilon */
	{1,{0xc1,0x3f,0xff,0xff}}, /* 4 - epsilon */
	{1,{0xff,0x3f,0xff,0xff}}, /* large - epsilon */
	{1,{0xff,0x40,0x00,0x00}}, /* large */
	{1,{0xff,0x40,0x00,0x01}}, /* large + epsilon */
	{1,{0xff,0x7f,0xff,0xff}}, /* 2*large - epsilon */
	{1,{0xff,0x80,0x00,0x00}}, /* 2*large */
	{1,{0xff,0x80,0x00,0x01}}, /* 2*large + epsilon */
	{1,{0xff,0xff,0xff,0xff}}, /* 4*large - epsilon */

	/* Normalized numbers */
	{1,{0x41,0x80,0x00,0x00}},
	{1,{0x41,0x40,0x00,0x00}},
	{1,{0x41,0x20,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00}},
	{1,{0x41,0x18,0x00,0x00}},
	{1,{0x41,0x14,0x00,0x00}},
	{1,{0x41,0x12,0x00,0x00}},
	{1,{0x41,0x11,0x00,0x00}},
	{1,{0x41,0x10,0x80,0x00}},
	{1,{0x41,0x10,0x40,0x00}},
	{1,{0x41,0x10,0x20,0x00}},
	{1,{0x41,0x10,0x10,0x00}},
	{1,{0x41,0x10,0x08,0x00}},
	{1,{0x41,0x10,0x04,0x00}},
	{1,{0x41,0x10,0x02,0x00}},
	{1,{0x41,0x10,0x01,0x00}},
	{1,{0x41,0x10,0x00,0x80}},
	{1,{0x41,0x10,0x00,0x40}},
	{1,{0x41,0x10,0x00,0x20}},
	{1,{0x41,0x10,0x00,0x10}},
	{1,{0x41,0x10,0x00,0x08}},
	{1,{0x41,0x10,0x00,0x04}},
	{1,{0x41,0x10,0x00,0x02}},
	{1,{0x41,0x10,0x00,0x01}},

	/* Denormalized numbers */
	{1,{0x00,0x80,0x00,0x00}},
	{1,{0x00,0x40,0x00,0x00}},
	{1,{0x00,0x20,0x00,0x00}},
	{1,{0x00,0x10,0x00,0x00}},
	{1,{0x00,0x08,0x00,0x00}},
	{1,{0x00,0x04,0x00,0x00}},
	{1,{0x00,0x02,0x00,0x00}},
	{1,{0x00,0x01,0x00,0x00}},
	{1,{0x00,0x00,0x80,0x00}},
	{1,{0x00,0x00,0x40,0x00}},
	{1,{0x00,0x00,0x20,0x00}},
	{1,{0x00,0x00,0x10,0x00}},
	{1,{0x00,0x00,0x08,0x00}},
	{1,{0x00,0x00,0x04,0x00}},
	{1,{0x00,0x00,0x02,0x00}},
	{1,{0x00,0x00,0x01,0x00}},
	{1,{0x00,0x00,0x00,0x80}},
	{1,{0x00,0x00,0x00,0x40}},
	{1,{0x00,0x00,0x00,0x20}},
	{1,{0x00,0x00,0x00,0x10}},
	{1,{0x00,0x00,0x00,0x08}},
	{1,{0x00,0x00,0x00,0x04}},
	{1,{0x00,0x00,0x00,0x02}},
	{1,{0x00,0x00,0x00,0x01}},

	/* Exponent ranges */
	{1,{0x00,0x10,0x00,0x01}},
	{1,{0x00,0x20,0x00,0x01}},
	{1,{0x00,0x40,0x00,0x01}},
	{1,{0x00,0x80,0x00,0x01}},
	{1,{0x01,0x10,0x00,0x01}},
	{1,{0x01,0x20,0x00,0x01}},
	{1,{0x40,0x20,0x00,0x01}},
	{1,{0x40,0x40,0x00,0x01}},
	{1,{0x40,0x80,0x00,0x01}},
	{1,{0x41,0x10,0x00,0x01}},
	{1,{0x41,0x20,0x00,0x01}},
	{1,{0x41,0x40,0x00,0x01}},
	{1,{0x41,0x80,0x00,0x01}},
	{1,{0x42,0x10,0x00,0x01}},
	{1,{0xfe,0x20,0x00,0x01}},
	{1,{0xfe,0x40,0x00,0x01}},
	{1,{0xfe,0x80,0x00,0x01}},
	{1,{0xff,0x10,0x00,0x01}},
	{1,{0xff,0x20,0x00,0x01}},
	{1,{0xff,0x40,0x00,0x01}},
	{1,{0xff,0x80,0x00,0x01}},
	{0}
};


/*
 * IBM 370 Double precision floats
 */
struct dfTestCase dfTestCase[] = {
	/* Positive */
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}}, /* zero */
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01}}, /* epsilon */
	{1,{0x00,0xff,0xff,0xff,0xff,0xff,0xff,0xff}}, /* small - epsilon */
	{1,{0x01,0x10,0x00,0x00,0x00,0x00,0x00,0x00}}, /* small */
	{1,{0x01,0x10,0x00,0x00,0x00,0x00,0x00,0x01}}, /* small + epsilon */
	{1,{0x01,0x1f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 2*small - epsilon */
	{1,{0x01,0x20,0x00,0x00,0x00,0x00,0x00,0x00}}, /* 2*small */
	{1,{0x01,0x20,0x00,0x00,0x00,0x00,0x00,0x01}}, /* 2*small + epsilon */
	{1,{0x40,0xff,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 1 - epsilon */
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x00,0x00}}, /* 1 */
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x00,0x01}}, /* 1 + epsilon */
	{1,{0x41,0x1f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 2 - epsilon */
	{1,{0x41,0x20,0x00,0x00,0x00,0x00,0x00,0x00}}, /* 2 */
	{1,{0x41,0x20,0x00,0x00,0x00,0x00,0x00,0x01}}, /* 2 + epsilon */
	{1,{0x41,0x3f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 4 - epsilon */
	{1,{0x7f,0x3f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* large - epsilon */
	{1,{0x7f,0x40,0x00,0x00,0x00,0x00,0x00,0x00}}, /* large */
	{1,{0x7f,0x40,0x00,0x00,0x00,0x00,0x00,0x01}}, /* large + epsilon */
	{1,{0x7f,0x7f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 2*large - epsilon */
	{1,{0x7f,0x80,0x00,0x00,0x00,0x00,0x00,0x00}}, /* 2*large */
	{1,{0x7f,0x80,0x00,0x00,0x00,0x00,0x00,0x01}}, /* 2*large + epsilon */
	{1,{0x7f,0xff,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 4*large - epsilon */

	/* Negative */
	/* Only Ieee floats have signed zeros. */
	/* DFADD(0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00}}, */
	{1,{0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x01}}, /* epsilon */
	{1,{0x80,0xff,0xff,0xff,0xff,0xff,0xff,0xff}}, /* small - epsilon */
	{1,{0x81,0x10,0x00,0x00,0x00,0x00,0x00,0x00}}, /* small */
	{1,{0x81,0x10,0x00,0x00,0x00,0x00,0x00,0x01}}, /* small + epsilon */
	{1,{0x81,0x1f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 2*small - epsilon */
	{1,{0x81,0x20,0x00,0x00,0x00,0x00,0x00,0x00}}, /* 2*small */
	{1,{0x81,0x20,0x00,0x00,0x00,0x00,0x00,0x01}}, /* 2*small + epsilon */
	{1,{0xc0,0xff,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 1 - epsilon */
	{1,{0xc1,0x10,0x00,0x00,0x00,0x00,0x00,0x00}}, /* 1 */
	{1,{0xc1,0x10,0x00,0x00,0x00,0x00,0x00,0x01}}, /* 1 + epsilon */
	{1,{0xc1,0x1f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 2 - epsilon */
	{1,{0xc1,0x20,0x00,0x00,0x00,0x00,0x00,0x00}}, /* 2 */
	{1,{0xc1,0x20,0x00,0x00,0x00,0x00,0x00,0x01}}, /* 2 + epsilon */
	{1,{0xc1,0x3f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 4 - epsilon */
	{1,{0xff,0x3f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* large - epsilon */
	{1,{0xff,0x40,0x00,0x00,0x00,0x00,0x00,0x00}}, /* large */
	{1,{0xff,0x40,0x00,0x00,0x00,0x00,0x00,0x01}}, /* large + epsilon */
	{1,{0xff,0x7f,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 2*large - epsilon */
	{1,{0xff,0x80,0x00,0x00,0x00,0x00,0x00,0x00}}, /* 2*large */
	{1,{0xff,0x80,0x00,0x00,0x00,0x00,0x00,0x01}}, /* 2*large + epsilon */
	{1,{0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff}}, /* 4*large - epsilon */

	/* Normalized numbers */
	{1,{0x41,0x80,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x40,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x20,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x18,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x14,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x12,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x11,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x80,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x40,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x20,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x10,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x08,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x04,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x02,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x01,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x80,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x40,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x20,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x10,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x08,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x04,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x02,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x01,0x00,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x80,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x40,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x20,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x10,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x08,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x04,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x02,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x01,0x00,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x80,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x40,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x20,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x10,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x08,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x04,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x02,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x01,0x00,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x80,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x40,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x20,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x10,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x08,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x04,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x02,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x01,0x00}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x00,0x80}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x00,0x40}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x00,0x20}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x00,0x10}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x00,0x08}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x00,0x04}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x00,0x02}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x00,0x01}},

	/* Denormalized numbers */
	{1,{0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x20,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x08,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x40,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x20,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x08,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x40,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x40,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x20,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x08,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x40,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x08,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x04,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x02,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x40,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x40}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02}},
	{1,{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01}},

	/* Exponent ranges */
	{1,{0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x00,0x20,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x01,0x10,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x01,0x20,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x40,0x20,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x40,0x40,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x40,0x80,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x41,0x10,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x41,0x20,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x41,0x40,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x41,0x80,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0x42,0x10,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0xfe,0x20,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0xfe,0x40,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0xfe,0x80,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0xff,0x10,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0xff,0x20,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0xff,0x40,0x00,0x00,0x00,0x00,0x00,0x01}},
	{1,{0xff,0x80,0x00,0x00,0x00,0x00,0x00,0x01}},
	{0}
};
#endif /* HW_370 */

#endif /* CC_non_ieee_floats */

#endif /* TEST_XFLOAT || TEST_ALL */
