# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::tree::changelog;
use base 'AXP::Command::tree';

use Arch::Util qw(standardize_date parse_creator_email);

sub infoline {
	"generate a standard ChangeLog file from tree logs"
}

sub optusage {
	"[options] [archive/version]"
}

sub options {
	(
		$_[0]->tree_options,
		output    => { sh => 'o', type => "=s", arg => 'FILE', desc => "write to FILE rather than stdout" },
		tagged    => { sh => 't', desc => "tag the output file" },
		show_time => { desc => "include time and timezone in dates" },
		show_dirs => { desc => "include changes to the directories" },
		show_self => { desc => "include changes to ChangeLog file" },
		show_ids  => { desc => "include explicit file ids" },
		no_files  => { sh => 'n', desc => "exlude file lists from changes" },
	)
}

sub helptext {
	q{
		Generate a ChangeLog for the given version from the tree
		patch logs.  If DIR is not given, the current directory is
		assumed for the arch tree.

		Unlike "tla changelog", the output is closer to the de-facto
		standard format and arguably more readable.  Also, the output
		file is not tagged by default.
	}
}

sub execute {
	my $self = shift;
	my %opt = %{$self->{options}};

	my $tree = $self->tree(hide_ids => !$opt{show_ids});
	my $version = shift @ARGV || $tree->get_version;
	die "No tree version, exiting\n" unless $version;

	my @logs = reverse $tree->get_logs($version);
	die "No patch logs for version $version, exiting\n" unless @logs;

	require IO::File;  # no 'use', speed up generation stuff
	my $file = $opt{output} || "-";
	my $out = new IO::File "> $file" or die "Can't write to $file: $!";

	print $out
		"# Do not edit - automatically generated by axp changelog\n",
		$opt{tagged}?
			"# arch-tag: automatic-ChangeLog--$version":
			"# for version $version",
		"\n\n";

	foreach my $log (@logs) {
		my $headers = $log->get_headers;
		my $revision = $headers->{revision};
		$revision =~ s/^.*--//;
		my $date = standardize_date($headers->{standard_date});
		$date =~ s/ .*// unless $opt{show_time};
		my ($creator, $email) = parse_creator_email($headers->{creator});
		print $out "$date  $creator  <$email>  $revision\n\n";

		unless ($opt{no_files}) {
			my @modified = @{$headers->{modified_files} || []};
			my @added    = @{$headers->{new_files}      || []};
			my @removed  = @{$headers->{removed_files}  || []};
			my @renamed  = @{$headers->{renamed_files}  || []};
			if ($opt{show_dirs}) {
				push @added, map { "$_/" }
					@{$headers->{new_directories} || []};
				push @removed,	map { "$_/" }
					@{$headers->{removed_directories} || []};
				push @renamed,	map { [ "$_->[0]/", "$_->[1]/" ] }
					@{$headers->{renamed_directories} || []};
			}			
			@renamed = map { "$_->[0] -> $_->[1]" } @renamed;
			@modified = grep { !/^changelog$/i } @modified
				unless $opt{show_self};

			my $output = "";
			$output .= $self->format_changed_files("*", @modified) if @modified;
			$output .= $self->format_changed_files("+", @added)    if @added;
			$output .= $self->format_changed_files("-", @removed)  if @removed;
			$output .= $self->format_changed_files("*", @renamed)  if @renamed;
			print $out $output, "\n";
		}

		my $summary = $headers->{summary};
		$summary =~ s/^\s*(.*?)\s*$/$1/s;
		$summary ||= "(no summary)";
		my $body = $headers->{body};
		$body =~ s/^\s*(.*?)\s*$/$1/s;

		my $message = "$summary\n";
		$message .= "\n$body\n" if $body ne "";
		$message =~ s/(^|\n)([^\n])/$1\t$2/g;
		print $out $message, "\n";
	}
}

sub format_changed_files ($$@) {
	my $self = shift;

	my $prefix = shift;
	$prefix = "\t$prefix ";
	my $output = join('', map { "$prefix$_:\n" } @_);
	return $output;
}

1;
