# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::tree::history;
use base 'AXP::Command::tree';

use Arch::Tree;

sub infoline {
	"show descriptions of ancestry revisions from logs"
}

sub optusage {
	"[options] [filepath]"
}

sub options {
	(
		$_[0]->tree_options,
		reverse  => { sh => 'r', desc => "sort from oldest to newest" },
		summary  => { sh => 's', desc => "show summary of each revision" },
		creator  => { sh => 'c', desc => "show creator of each revision" },
		date     => { sh => 'D', desc => "show date of each revision" },
		desc     => { desc => "equivalent to -Dcs" },
		full     => { sh => 'f', desc => "show full revision names" },
		filename => { sh => 'n', desc => "show file name if applicable" },
		no_email => { sh => 'e', desc => "don't show email of creator" },
		one_version => { sh => 'o', desc => "don't follow tags from other versions" },
	)
}   
 
sub helptext {
	q{
		Without filepath parameter, show all ancestry revision names
		and optionally their summary, date and creator.  The format is
		the same as in 'tla logs' and 'tla revisions'.

		With filepath parameter, only show the revisions that modified
		the given filepath.  In this case, additional field (filename)
		may be shown too and the file renames are reported properly.
	}
}

sub execute {
	my $self = shift;
	my %opt = %{$self->{options}};

	$opt{date} = $opt{creator} = $opt{summary} = 1 if $opt{desc};

	my $tree = $self->tree;
	my $filepath = shift(@ARGV);
	warn "Post file-name parameters (@ARGV) are ignored\n" if @ARGV;

	my $revision_descs = $tree->get_ancestry_revision_descs($filepath, $opt{one_version});
	@$revision_descs = reverse @$revision_descs if $opt{reverse};

	foreach my $revision_desc (@$revision_descs) {
		my $creator_line = "";
		$creator_line .= $revision_desc->{date} if $opt{date};
		if ($opt{creator}) {
			$creator_line .= "      " if $opt{date};
			$creator_line .= $revision_desc->{creator};
			$creator_line .= " <" . $revision_desc->{email} . ">" unless $opt{no_email};
		}

		print "$revision_desc->{version}--" if $opt{full};
		print $revision_desc->{name}, "\n";
		print "    ", $revision_desc->{filepath}, "\n" if $filepath && $opt{filename};
		print "    $creator_line\n" if $creator_line;
		print "    ", $revision_desc->{summary}, "\n" if $opt{summary};
	}
}

1;
