#! /bin/sh
#
# Configure apache for Axyl
# P Waite
set -e

# Name of this script
THIS=`basename $0`

# Gotta have the power..
if [ $(id -u) != 0 ]
then
  echo "You must be root to run this install script."
  exit 1
fi

# Must have all parms for this..
if [ $# -lt 5 ]
then
  echo "usage: $THIS vhost servername admin_email documentroot axylroot"
  exit 1
fi

# Get common funcs and config vars etc..
if [ -f ./axyl-common.sh ]
then
  . ./axyl-common.sh
else
  echo "ERROR: axyl-common.sh not found. You must execute this"
  echo "script from within the Axyl 'install' directory."
  exit 1
fi

# Get parms..
VHOST=$1
VHOST=`echo $VHOST | sed -e "s;ASTERISK;*;g"`
VSERVERNAME=$2
ADMIN_EMAIL=$3
DOCROOT=$4
AXYL_HOME=$5
AXYL_LOGS=$6
if [ ! -d $AXYL_HOME ]
then
  echo "error: the Axyl root directory '$AXYL_HOME' does not exist."
  exit 1
fi

# Functions
function fix_phpini () {
  TITLE=$1
  AXYL_HOME=$2
  PHP_INI=$3
  PHP_INI_OLD=$4
  TSTAMP=`date +"%Y%m%d%H%M%S"`
  PATH_PATT="^include_path[\s]*=[\s]*.*${AXYL_HOME}/lib"
  if [ -f $PHP_INI ]
  then
    echo ""
    echo "Configuring Php for ${TITLE}.."
    # php ini file
    perl -pi -e "s;^[\s]*expose_php[\s]*.*$;expose_php = Off;i" ${PHP_INI}
    perl -pi -e "s;^[\s]*max_execution_time[\s]*.*$;max_execution_time = 0;i" ${PHP_INI}
    perl -pi -e "s;^[\s]*register_globals[\s]*.*$;register_globals = On;i" ${PHP_INI}
    perl -pi -e "s;^[\s]*magic_quotes_gpc[\s]*.*$;magic_quotes_gpc = Off;i" ${PHP_INI}
    CHKAX=`perl -n -e "m;${PATH_PATT};i && print;" ${PHP_INI}`
    if [ "$CHKAX" = "" ]
    then
      echo "IMPORTANT:"
      echo "For Axyl Php scripts to work properly for $TITLE we need to make sure"
      echo "the 'include_path' is set in ${PHP_INI}. The path can be"
      echo "re-written for you now, or if you already have a path you want to preserve,"
      echo "you can do it yourself after this installation."
      echo -n "Re-write this include path now? (y/n) [n] :"
      read ANS
      if [ "$ANS" = "y" -o "$ANS" = "Y" ]
      then
        cp -a ${PHP_INI} ${PHP_INI_OLD}
        A=`perl -n -e "m;^include_path[\s]*=;i && print;" ${PHP_INI}`
        if [ "$A" = "" ]
        then
          # Append include path to file..
          echo "" >> ${PHP_INI}
          echo "; Include path appended by Axyl installation.." >> ${PHP_INI}
          echo "include_path = \".:${AXYL_HOME}/lib\"" >> ${PHP_INI}
          echo "include path was appended to ${PHP_INI}"
        else
          # Replace include path
          perl -pi -e "s;^[\s]*include_path[\s]*.*$;include_path = \".:${AXYL_HOME}/lib\";i" ${PHP_INI}
          echo "include path was re-written in ${PHP_INI}"
        fi
        echo "NOTE: the original file was saved as:"
        echo "  ${PHP_INI_OLD}"
      else
        echo ""
        echo "Ok, no changes were made. You will need to make sure the following path"
        echo "is inserted into the 'include_path' in ${PHP_INI}: "
        echo "  '.:${AXYL_HOME}/lib'"
      fi
    fi
  fi
}

# Files..
CONF=${APACHE_CONFDIR}/httpd.conf
CONF_OLD=${CONF}.axyl-old
if [ -e $CONF ]
then
  # Always make a copy of existing httpd.conf..
  cp -a $CONF $CONF_OLD

  # New format or old?
  if [ -e ${APACHE_CONFDIR}/conf.d ]
  then
    new_conf=1
    CONF_NEW=${APACHE_CONFDIR}/conf.d/${VSERVERNAME}.conf
    [ -f $CONF_NEW ] &&  echo "WARNING: re-defining existing Axyl apache configuration"
    echo "# Axyl virtual host: ${VSERVERNAME}" > $CONF_NEW
  else
    new_conf=0
    if grep -q "^ServerName ${VSERVERNAME}" $CONF
    then
      echo ""
      echo "ERROR: apache was already configured for ${VSERVERNAME}."
      echo "Edit and remove the Axyl virtual host from ${CONF}"
      echo "and then re-run this script."
      exit 0
    fi
    CONF_NEW=${APACHE_CONFDIR}/httpd.conf.axyl-new
    cp -a $CONF $CONF_NEW
    echo "" >> $CONF_NEW
    echo "# Axyl virtual host: '${VSERVERNAME}'" >> $CONF_NEW
    echo "# Created: " `date`
  fi

  # Insert NameVirtualHost directive..
  if ! grep -q -i -e "^[\s]*NameVirtualHost.*$VHOST" $CONF
  then
    echo "NameVirtualHost ${VHOST}" >> $CONF_NEW
  fi

  # Logging directory. We use an Axyl location, since various distros
  # have an assortment of locations in use and it is impossible to
  # be sure of finding the correct area otherwise..
  if [ -z $AXYL_LOGS ]
  then
    AXYL_LOGS="/var/log/axyl"
  fi
  APACHE_LOGS=${AXYL_LOGS}/apache
  mkaxyldir $APACHE_LOGS

  # Insert the virtual host definition..
  echo "<VirtualHost ${VHOST}>" >> $CONF_NEW
  echo "  ServerAdmin ${ADMIN_EMAIL}" >> $CONF_NEW
  echo "  DocumentRoot ${DOCROOT}" >> $CONF_NEW
  echo "  DirectoryIndex index.php index.html" >> $CONF_NEW
  echo "  ServerName ${VSERVERNAME}" >> $CONF_NEW
  echo "  ErrorLog ${APACHE_LOGS}/${VSERVERNAME}-error.log" >> $CONF_NEW
  echo "  CustomLog ${APACHE_LOGS}/${VSERVERNAME}-access.log common" >> $CONF_NEW
  echo "  php_value include_path \".:${AXYL_HOME}/lib:${DOCROOT}\"" >> $CONF_NEW
  echo "  <Directory ${DOCROOT}/>" >> $CONF_NEW
  echo "    Order Deny,Allow" >> $CONF_NEW
  echo "    Allow from All" >> $CONF_NEW
  echo "    Options FollowSymLinks" >> $CONF_NEW
  echo "    AllowOverride" >> $CONF_NEW
  echo "  </Directory>" >> $CONF_NEW
  echo "  Options FollowSymLinks" >> $CONF_NEW
  echo "</VirtualHost>" >> $CONF_NEW

  # Tidy up..
  if [ $new_conf -eq 0 ]
  then
    mv $CONF_NEW $CONF
  fi

  # Make sure Php4 module is enabled
  APACHE_MODCONF=/usr/sbin/modules-config
  if [ -x ${APACHE_MODCONF} ]
  then
    APACHEID=`basename ${APACHE_CONFDIR}`
    EXISTS=`${APACHE_MODCONF} ${APACHEID} query mod_php4`
    [ "$EXISTS" = "" ] && ${APACHE_MODCONF} ${APACHEID} enable mod_php4
  else
    perl -pi -e "s/# *LoadModule php4_module/LoadModule php4_module/gi" $CONF
  fi
else
  echo "ERROR: apache does not seem to be installed."
  exit 1
fi

# Make sure Apache is configured for PHP..
TSTAMP=`date +"%Y%m%d%H%M%S"`
ETCPHP=/etc/php4
# Php ini file for Apache processes..
PHP_INI_APACHE=${ETCPHP}/apache/php.ini
PHP_INI_APACHE_OLD=${PHP_INI_APACHE}.axyl-saved.${TSTAMP}
# Php ini file for CGI processes..
PHP_INI_CGI=${ETCPHP}/cgi/php.ini
PHP_INI_CGI_OLD=${PHP_INI_CGI}.axyl-saved.${TSTAMP}
# Php ini file for command-line scripts..
PHP_INI_CLI=${ETCPHP}/cli/php.ini
PHP_INI_CLI_OLD=${PHP_INI_CLI}.axyl-saved.${TSTAMP}
# Php global ini file, used in some distros..
PHP_INI_ETC=/etc/php.ini
PHP_INI_ETC_OLD=${PHP_INI_ETC}.axyl-saved.${TSTAMP}

# Apache is done via virtual hosts, and the global php.ini is probably
# best left alone. Uncomment as appropriate for your system..
#fix_phpini "Apache" $AXYL_HOME $PHP_INI_APACHE $PHP_INI_APACHE_OLD
#fix_phpini "All" $AXYL_HOME $PHP_INI_ETC $PHP_INI_ETC_OLD

fix_phpini "CGI" $AXYL_HOME $PHP_INI_CGI $PHP_INI_CGI_OLD
fix_phpini "CLI" $AXYL_HOME $PHP_INI_CLI $PHP_INI_CLI_OLD

# Restart apache..
echo ""
if [ -x /usr/sbin/apache ]
then
  /etc/init.d/apache restart
fi
if [ -x /usr/sbin/apache-ssl ]
then
  /etc/init.d/apache-ssl restart
fi

# END