<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    htmlarea-defs.php                                       */
/* Author:      Paul Waite                                              */
/* Description: Definitions for using HTMLArea wysiwyg editor.          */
/*                                                                      */
/* ******************************************************************** */
/** @package form */

/** Form elements */
include_once("form-defs.php");

// NOTES: How to implement HTMLAreas in your forms:
//
//  // Include htmlarea classes at the top of the page..
//  include_once("htmlarea-defs.php");
//  // Define which plugins to load..
//  htmlarea_plugins("ContextMenu,CSS");
//  ...
//  // Build your form element..
//  $Fld = new form_wysiwygfield("my_content", "Article", $my_content);
//  // Register plugins this editor will use..
//  $Fld->register_plugins("all");
//  // Render the field..
//  $myfield = $Fld->render();
//  ...
//  ... more htmlareas as required ...

// ----------------------------------------------------------------------
/**
* The htmlarea class. This is just an aggregator of functionality to
* initialise and properly render all the bits required to make one
* or more HTMLArea widgets work in the $RESPONSE. NB: this class is
* not required to be used directly, being automatically instantiated
* in this module (see below). It is also automatically rendered
* when the $RESPONSE is delivered back to the user agent.
* @see webpage::generate()
* @access private
* @package form
*/
class htmlarea extends HTMLObject {
  // Public
  // Private
  /** Plugins activated. Initially none.
      @access private */
  var $loaded_plugins = array();
  /** Array of all possible allowed plugins
      @access private */
  var $all_plugins = array(
        "ContextMenu",
        "CSS",
        "TableOperations",
        "ListType",
        "CharacterMap",
        "HtmlTidy",
        "EnterParagraphs",
        "FullPage"
        );
  /** Array of default styles we will use for the CSS plugin. This
      is the default set of Axyl styles.
      @access private */
  var $default_css_styles = array(
        "&mdash; styles &mdash;:",
        "Title: axtitle",
        "Heading: axhdg",
        "Sub-heading: axsubhdg",
        "Light backgnd: axbglite",
        "Dark backgnd: axbgdark",
        "Highlight: axhl",
        "Form label: axfmlbl",
        "Number field: axnumbox",
        "Text field: axtxtbox",
        "Memo field: axmemo",
        "Menu: menu",
        "Sub-menu: submenu"
        );
  /** Whether we have initialised or not
      @access private */
  var $init_done = false;
  /** Collected script content
      @access private */
  var $script = "";
  /** Whether we have been activated. Activation occurs
      when something calls the 'add_script()' method
      @access private */
  var $activated = false;
  /** Whether we have already been rendered
      @access private */
  var $rendered = false;
  // .....................................................................
  /** Constructor for an HTMLArea manager */
  function htmlarea() {
  } // htmlarea
  // .....................................................................
  /**
  * Set the plugins for ALL the Wysiwyg editors. This represents the list
  * of plugins which should be loaded. These can then be used by any
  * HTMLArea form object to register against.
  * NOTE: This can be called multiple times, but will only ever ADD
  *       extra (new) plugins to what was already there.
  * @param string $pluginlist Comma-delimited list of plugins to permit
  */
  function load_plugins($pluginlist="all") {
    debug_trace($this);
    if ($pluginlist == "all") {
      $this->loaded_plugins = $this->all_plugins;
    }
    else {
      $plugins = explode(",", $pluginlist);
      foreach ($plugins as $plugin) {
        if (in_array($plugin, $this->all_plugins)
        && !in_array($plugin, $this->loaded_plugins)) {
          $this->loaded_plugins[] = $plugin;
        }
      }
    }
    debug_trace();
  } // load_plugins
  // .....................................................................
  /**
  * Internal method to initialise the HTMLArea environment. This sets up
  * the URL and i18n settings, and also loads the nominated plugins.
  * Plugins must be set up before this is called. This methid is an
  * internal one which is usually called late, just prior to render.
  * @access private
  */
  function initialise() {
    debug_trace($this);
    global $RESPONSE, $LIBDIR;
    // One-time-only javascript for all HTMLAreas on page..
    if (!$this->init_done) {
      $RESPONSE->head->add_named_script(
          "_editor_url = \"$LIBDIR/js/htmlarea/\";\n"
        . "_editor_lang = \"en\";\n",
        "htmlarea settings"
        );
      $RESPONSE->set_onload("HTMLArea.init();");
      $RESPONSE->add_scriptsrc("$LIBDIR/js/htmlarea/htmlarea.js");

      foreach ($this->loaded_plugins as $plugin) {
        $RESPONSE->add_named_script("HTMLArea.loadPlugin(\"$plugin\");\n", "htmlarea");
      }
      $this->init_done = true;
    }
    debug_trace();
  } // initialise
  // .....................................................................
  /**
  * Script collector for htmlarea-specific javascript.
  * @access private
  */
  function add_script($script) {
    debug_trace($this);
    $this->script .= $script;
    $this->activated = true;
    debug_trace();
  } // add_script
  // .....................................................................
  /**
  * Render the htmlarea javascript as HTML. Note, this does not render
  * the editor widget(s) since these will be rendered as individual
  * form elements separately wherever they are intended to go. This
  * method renders the environment for those widgets to operate in
  * the $RESPONSE, and returns nothing to the caller.
  */
  function html() {
    debug_trace($this);
    global $RESPONSE;
    if (!$this->rendered && $this->script != "") {
      $this->initialise();
      $RESPONSE->add_named_script("HTMLArea.onload = function () {\n", "htmlarea");
      $RESPONSE->add_named_script($this->script, "htmlarea");
      $RESPONSE->add_named_script("}\n", "htmlarea");
      $this->rendered = true;
    }
    debug_trace();
  } // html
} // htmlarea class

// ----------------------------------------------------------------------
/**
* A container class for configuration settings for the HTMLArea
* internal configuration structure. We provide a selection of all the
* possible HTMLArea parameters, which represents a subset. This also
* renders the Javascript definition for the config @see render()
* @package form
*/
class htmlarea_config {
  /** Width of HTMLarea element (default 'auto') */
  var $width;
  /** Height of HTMLarea element (default 'auto') */
  var $height;
  /** enable creation of a status bar? (default true) */
  var $statusBar;
  /** intercept ^V and use the HTMLArea paste command
      If false, then passes ^V through to browser editor widget
      (default false) */
  var $htmlareaPaste;
  /** maximum size of the undo queue (default 20) */
  var $undoSteps;
  /** the next parameter specifies whether the toolbar should
      be included in the size or not (default true) */
  var $sizeIncludesToolbar;
  /** if true then HTMLArea will retrieve the full HTML, starting
      with the HTML tag (default false) */
  var $fullPage;
  /** style included in the iframe document (default nullstring) */
  var $pageStyle;
  /** set to true if you want Word code to be cleaned upon Paste
      (default true)*/
  var $killWordOnPaste;
  /** HTML tags to disallow (these have to be a regexp, or null
      if this functionality is not desired) (default null) */
  var $htmlRemoveTags;
  /** The toolbar. This takes the form of an array of arrays of toolbar
      buttons. Each sub-array is a new line of buttons.
      (defaulted to all standard buttons) */
  var $toolbar;
  /** The list of font families to allow user to choose from
      (defaulted to standard list of font families) */
  var $fontnames;
  /** The list of font sizes to allow user to choose from
      (defaulted to standard list of font sizes) */
  var $fontsizes;
  /** The list of block format styles to allow user to choose from
      (defaulted to standard list of block formats) */
  var $blockformats;
  // Private
  /** A basic toolbar definition. Use for entering text in given
      font, with basic bold/italic/underline styling.
      @access private */
  var $toolbar_basic = array(
        array(
          "fontname", "space",
          "fontsize", "space",
          "formatblock", "space",
          "bold", "italic", "underline", "separator",
          "undo", "redo", "space", "removeformat"
          ),
        array("space")
        );
  /** A medium toolbar definition, adds colour, links, raw HTML editing,
      lists, and horizontal rule, etc.
      @access private */
  var $toolbar_medium = array(
        array(
          "fontname", "space",
          "fontsize", "space",
          "formatblock", "space",
          "bold", "italic", "underline", "strikethrough", "separator",
          "copy", "cut", "paste", "space", "undo", "redo", "space", "removeformat"
          ),
        array(
          "justifyleft", "justifycenter", "justifyright", "justifyfull", "separator",
          "orderedlist", "unorderedlist", "outdent", "indent", "separator",
          "forecolor", "hilitecolor", "separator",
          "inserthorizontalrule", "createlink", "htmlmode", "separator",
          "popupeditor", "separator", "showhelp", "about"
          ),

        );
  /** A fully-populated toolbar definition for power users
      @access private */
  var $toolbar_full = array(
        array(
          "fontname", "space",
          "fontsize", "space",
          "formatblock", "space",
          "bold", "italic", "underline", "strikethrough", "separator",
          "subscript", "superscript", "separator",
          "copy", "cut", "paste"
          ),
        array(
          "justifyleft", "justifycenter", "justifyright", "justifyfull", "separator",
          "lefttoright", "righttoleft", "separator",
          "orderedlist", "unorderedlist", "outdent", "indent"
          ),
        array(
           "undo", "redo", "space", "removeformat", "forecolor", "hilitecolor", "separator",
          "inserthorizontalrule", "createlink", "insertimage", "inserttable", "htmlmode", "separator",
          "popupeditor", "killword", "separator", "showhelp", "about"
          )
        );

  // .....................................................................
  /** Constructor for new HTMLArea configuration object  */
  function htmlarea_config() {
  } // HTMLArea_config
  // .....................................................................
  /**
  * Render this config as a string of Javascript which could be put into
  * a webapage and used to set the configuration of a real HTMLArea
  * object.
  */
  function render($configvar) {
    $s  = "";
    $s .= "var $configvar = new HTMLArea.Config();\n";
    if (isset($this->width) && $this->width != "") {
      $s .= "$configvar.width = '$this->width';\n";
    }
    if (isset($this->height) && $this->height != "") {
      $s .= "$configvar.height = '$this->height';\n";
    }
    if (isset($this->statusBar) && $this->statusBar === false) {
      $s .= "$configvar.statusBar=" . ($this->statusBar ? "true" : "false") . ";\n";
    }
    if (isset($this->htmlareaPaste) && $this->htmlareaPaste === true) {
      $s .= "$configvar.htmlareaPaste=" . ($this->htmlareaPaste ? "true" : "false") . ";\n";
    }
    if (isset($this->undoSteps)) {
      $s .= "$configvar.undoSteps=" . $this->undoSteps . ";\n";
    }
    if (isset($this->sizeIncludesToolbar) && $this->sizeIncludesToolbar === false) {
      $s .= "$configvar.sizeIncludesToolbar=" . ($this->sizeIncludesToolbar ? "true" : "false") . ";\n";
    }
    if (isset($this->pageStyle) && $this->pageStyle != "") {
      $s .= "$configvar.pageStyle = '$this->pageStyle';\n";
    }
    if (isset($this->killWordOnPaste) && $this->killWordOnPaste === false) {
      $s .= "$configvar.killWordOnPaste=" . ($this->killWordOnPaste ? "true" : "false") . ";\n";
    }
    if (isset($this->htmlRemoveTags) && $this->htmlRemoveTags != "") {
      $s .= "$configvar.htmlRemoveTags = '$this->htmlRemoveTags';\n";
    }
    if (isset($this->toolbar)) {
      $toolbar = $this->toolbar;
      if (is_string($toolbar)) {
        switch ($toolbar) {
          case "basic":
            $toolbar = $this->toolbar_basic;
            break;
          case "medium":
            $toolbar = $this->toolbar_medium;
            break;
          default:
            $toolbar = $this->toolbar_full;
        } // switch
      }
      $toolbar_lines = array();
      foreach ($toolbar as $toolbar_line) {
        $toolbar_lines[] = "[\"" . implode("\",\"", $toolbar_line) . "\"]";
      }
      $s .= "$configvar.toolbar = [" . implode(",\n", $toolbar_lines) . "];\n";
    }
    if (isset($this->fontnames)) {
      $fonts = array();
      foreach ($this->fontnames as $fontlabel => $fontvalue) {
        $fonts[] = "\"$fontlabel\": \"$fontvalue\"";
      }
      $s .= "$configvar.fontname = {" . implode(",", $fonts) . "};\n";
    }
    if (isset($this->fontsizes)) {
      $sizes = array();
      foreach ($this->fontsizes as $sizlabel => $sizvalue) {
        $sizes[] = "\"$sizlabel\": \"$sizvalue\"";
      }
      $s .= "$configvar.fontsize = {" . implode(",", $sizes) . "};\n";
    }
    if (isset($this->blockformats)) {
      $formats = array();
      foreach ($this->blockformats as $fmtlabel => $fmtvalue) {
        $formats[] = "\"$fmtlabel\": \"$fmtvalue\"";
      }
      $s .= "$configvar.formatblock = {" . implode(",", $formats) . "};\n";
    }
    // Return the Javascript code..
    return $s;
  } // render
} // htmlarea_config class
// ----------------------------------------------------------------------
/** The global control object which has to be used
* to initialise the plugins, and render the environment
*/
global $HTMLAREA;
$HTMLAREA = new htmlarea();

// ----------------------------------------------------------------------
// FUNCTIONS
/**
* Function to set plugins to be loaded. When you are using one of
* these HTMLArea editors on a webpage, call this to define which of
* the allowed plugins you want loaded. The possibilities currently
* are as follows:
*    ContextMenu
*    CSS
*    TableOperations
*    ListType
*    CharacterMap
*    HtmlTidy
*    EnterParagraphs
*    FullPage
* @param string $plugins A comma-delimited list of plugins
*/
function htmlarea_plugins($plugins) {
  global $HTMLAREA;
  if (is_object($HTMLAREA) && method_exists($HTMLAREA, "load_plugins")) {
    $HTMLAREA->load_plugins($plugins);
  }
} // htmlarea_plugins

// ----------------------------------------------------------------------
/**
* Wysiwyg Field class
* A field which renders a textarea form element as a Wysiwyg editor.
* This is based on the package 'HTMLarea', by Mihai Bazon.
* This class leverages the standard memofield. In fact we render a standard
* memofield, and the main diference is only the setting of a classid, and
* rendering the javascript for HTMLarea to initialisei itself.
* @package form
*/
class form_wysiwygfield extends form_memofield {
  /** Unique ID of this form element
      @access private */
  var $wysiwyg_id = "wysiwyg";
  /** Plugins we want to register for this field
      @access private */
  var $plugins = array();
  /** CSS Styles we want to display for this field
      @access private */
  var $css_styles;
  /** The configuration override settings to use. This is
      an object of htmlarea_config class.
      @access private */
  var $config;
  // .....................................................................
  /** Constructor - create a wysiwyg form element..
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css      CSS class or style to apply to the button
  * @param integer $width    Width of the memo field in characters
  * @param integer $rows     Height of the memo field in lines
  */
  function form_wysiwygfield($name="", $label="", $value="", $editable=EDITABLE, $css="", $width=STD_WIDTH, $rows=5) {
    global $HTMLAREA;
    $this->form_memofield($name, $label, $value, $editable, $css, $width, $rows);
    $this->wysiwyg_id = $name;
    $this->setid($this->wysiwyg_id);
    // Default CSS styles..
    $this->css_styles = $HTMLAREA->default_css_styles;
    // Empty config object..
    $this->config = new htmlarea_config();
  } // form_wysiwygfield
  // .....................................................................
  /**
  * Set the styles to provide for this Wysiwyg editor. Styles are displayed
  * when you register the "CSS" plugin. This method either replaces (the
  * default action) or appends the given list (or array) of styles to the
  * editor. Pass the styles list either as a comma-delimited string, or
  * an array. Each style should be in the format 'label : classname' as
  * shown in this example:  'Light background : axbglite'
  * (You should omit the quotes, which are just to delimit the example)
  * @param mixed $new_styles Comma-delimited list or array of style definitions
  * @param string $mode If "append" add styles to existing, else replace
  */
  function set_styles($new_styles, $mode="replace") {
    global $HTMLAREA;
    if (!is_array($new_styles)) {
      $new_styles = explode(",", $new_styles);
    }
    if ($mode == "replace") {
      $this->css_styles = $new_styles;
    }
    else {
      foreach ($new_styles as $style) {
        $this->css_styles[] = $style;
      }
    }
  } // set_styles
  // .....................................................................
  /** Set the HTMLArea page style. This is the overall style setting(s)
  * which are applied to the HTMLArea widget and so here is where you
  * determine such basic things as the font settings of the content which
  * is displayed/edited etc.
  * @param string $css_style The style setting for the editor 'page'
  */
  function set_pagestyle($css_style) {
    $this->config->pageStyle = $css_style;
  } // set_pagestyle
  // .....................................................................
  /** Set a new toolbar definition. Toolbars are arrays of arrays of lists
  * of HTMLArea widgets. Each array is a line of these widgets. For an
  * example see the default setting for the toolbar in this class. You
  * can EITHER give a full toolbar definition here as an array of arrays
  * of toolbar widgets, OR give a preset toolbar ID string. Presets that
  * we have defined so far are:
  *   "basic"  - Just enough to enter text, bold/italic etc.
  *   "medium" - As above but with colour, links, raw HTML etc.
  *   "full"   - Everything available
  * @param mixed $new_toolbat Array of toolbar lines, or preset toolbar id
  */
  function set_toolbar($new_toolbar) {
    $this->config->toolbar = $new_toolbar;
  } // set_toolbar
  // .....................................................................
  /** Set a new list of selectable font families. This array should be an
  * associative one in the form:
  *     "Arial" =>  "arial,helvetica,sans-serif",
  *     "Courier New" => "courier new,courier,monospace",
  *     ...
  * @param array $new_fontnames Array of font family names to use
  */
  function set_fontnames($new_fontnames) {
    if (is_array($new_fontnames)) {
      $this->config->fontnames = $new_fontnames;
    }
  } // set_fontnames
  // .....................................................................
  /** Set a new list of selectable font sizes. The sizes array should be
  * an associative one in the form:
  *     "1 (8 pt)"  => "1",
  *     "2 (10 pt)" => "2",
  *     ...
  * @param array $new_blocksizes Array of font sizes to use
  */
  function set_fontsizes($new_fontsizes) {
    if (is_array($new_fontsizes)) {
      $this->config->fontsizes = $new_fontsizes;
    }
  } // set_fontsizes
  // .....................................................................
  /** Set a new list of selectable block formats. The array should be an
  * associative one in the form:
  *     "Heading 1" => "h1",
  *     "Heading 2" => "h2",
  *     ...
  * @param array $new_blockformats Array of block formats to use
  */
  function set_blockformats($new_blockformats) {
    if (is_array($new_blockformats)) {
      $this->config->blockformats = $new_blockformats;
    }
  } // set_blockformats
  // .....................................................................
  /** Set the HTMLArea sizing. The default is 'auto' for both width
  * and height, which gives you an editing area filling the are that
  * the original memofield widget would have filled.
  * @param string $width Width of the HTMLArea editor
  * @param string $height Height of the HTMLArea editor
  * @param boolean $inctoolbar Whether sizing includes the toolbar
  */
  function set_metrics($width="", $height="", $inctoolbar=true) {
    if ($width != "") {
      $this->config->width = $width;
    }
    if ($height != "") {
      $this->config->height = $height;
    }
    $this->config->sizeIncludesToolbar = ($inctoolbar === true);
  } // set_metrics
  // .....................................................................
  /** Set the regular expression to use for removing dis-allowed HTML
  * tags from the editor content. This allows you to specify a regex
  * matching those tags you don't want to ever appear in the content.
  * @param string $regex Regular expression matching disallowed tags
  */
  function set_remove_regex($regex="") {
    $this->config->htmlRemoveTags = $regex;
  } // set_remove_regex
  // .....................................................................
  /** Set the HTMLArea undo steps maximum. The default number is 20. Use
  * this method to change that.
  * @param integer $steps Number of undo steps to record.
  */
  function set_undo_steps($steps=20) {
    $this->config->undoSteps = $steps;
  } // set_undo_steps
  // .....................................................................
  /** Set the HTMLArea statusbar display mode, true or false.
  * @param boolean $mode Whether to display the statusbar or not
  */
  function set_statusbar($mode=true) {
    $this->config->statusBar = $mode;
  } // set_statusbar
  // .....................................................................
  /** Set the HTMLArea Paste mode. If set to true, then the HTMLArea
  * widget will intercept ^V and use the HTMLArea paste command. If
  * false, then it passes ^V through to browser editor widget
  * @param boolean $mode Whether to intercept ^V or not
  */
  function set_paste_intercept($mode=true) {
    $this->config->htmlareaPaste = $mode;
  } // set_paste_intercept
  // .....................................................................
  /**
  * Set the plugins to register for this Wysiwyg editor. This is a subset
  * of the allowed plugins which were given in the original call to the
  * @see htmlarea_plugins() function. If you want to just register all
  * of the possibilities, then just leave the argument list empty, or
  * explicitly set it to "all" (the default).
  * @param string $pluginlist Comma-delimited list of plugins to register
  */
  function register_plugins($pluginlist="all") {
    global $HTMLAREA;
    if ($pluginlist == "all") {
      $this->plugins = $HTMLAREA->loaded_plugins;
    }
    else {
      $plugins = explode(",", $pluginlist);
      $this->plugins = array();
      foreach ($plugins as $plugin) {
        if (in_array($plugin, $HTMLAREA->loaded_plugins)) {
          $this->plugins[] = $plugin;
        }
      }
    }
  } // register_plugins
  // .....................................................................
  /**
  * Register with HTMLArea for this editor to work. We produce the required
  * javascript. This is an internal method usually executed just prior to
  * rendering.
  * @access private
  */
  function register() {
    global $HTMLAREA, $RESPONSE, $LIBDIR;
    if (isset($RESPONSE) && file_exists($RESPONSE->site_docroot."$LIBDIR/js/htmlarea/htmlarea.js")) {
      $wysiwyged = $this->wysiwyg_id . "_ed";
      $wysiwygconfig = $this->wysiwyg_id . "_config";

      $ssheet = new stylesheet($RESPONSE->site_docroot . $RESPONSE->head->stylesheet);
      if ($ssheet->opened) {
        //$bodystyle = $ssheet->style("body");
        //$parastyle = $ssheet->style("p");
        //$this->set_pagestyle("body {" . $bodystyle . "} p {" . $parastyle . "}");
        $this->set_pagestyle("@import url( " . $RESPONSE->site_url . $RESPONSE->head->stylesheet . " );");
      }

      // HTMLArea editor widget for this field..
      $RESPONSE->add_named_script("var $wysiwyged=null;\n", "htmlarea objects");
      $HTMLAREA->add_script(
          $this->config->render($wysiwygconfig)
        . "$wysiwyged=new HTMLArea(\"$this->wysiwyg_id\",$wysiwygconfig);\n"
        );

      // Register plugins with this field..
      foreach ($this->plugins as $plugin) {
        switch ($plugin) {
          case "CSS":
            if (count($this->css_styles > 0)) {
              $HTMLAREA->add_script(
                  "$wysiwyged.registerPlugin(CSS, {\n"
                . "combos : [\n"
                . " { label: \"\",\n"
                . "  options: {\n"
                );
              $css_styles = array();
              foreach ($this->css_styles as $style) {
                $style = preg_replace("/[\"',]/", "", $style);
                $stylebits = explode(":", $style);
                $css_styles[] = "  \"" . trim($stylebits[0]) . "\" : \"" . trim($stylebits[1]) . "\"";
                $style_list = implode(",\n", $css_styles);
                if (!strstr($css_styles[0], "&mdash; styles &mdash;")) {
                  $style_list = "  \"&mdash; styles &mdash;\" : \"\",\n" . $style_list;
                }
              } // foreach
              $HTMLAREA->add_script($style_list);
              $HTMLAREA->add_script(
                  "  }\n"
                . " }\n"
                . "]\n"
                . "});\n"
                );
            }
            break;
          default:
            $HTMLAREA->add_script("$wysiwyged.registerPlugin($plugin);\n");
        } // switch
      }
      // Generate section for field..
      $HTMLAREA->add_script("$wysiwyged.generate();\n");
    } // if not already done
  } // register
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The wysiwyg HTMLArea editor field as HTML.
  */
  function html($name="") {
    if ($name != "") {
      $this->name = $name;
      $this->wysiwyg_id = $name;
      $this->setid($this->wysiwyg_id);
    }
    $this->register();
    return form_memofield::html();
  } // html
} // class form_wysiwygfield

// ----------------------------------------------------------------------
?>