<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    layout-defs.php                                         */
/* Author:      Paul Waite                                              */
/* Description: Definitions for content layout management in webpages.  */
/*                                                                      */
/* ******************************************************************** */
/** @package cm */

/** Block management classes */
include_once("block-defs.php");

// ----------------------------------------------------------------------
/** New layout to be created */
define("NEW_LAYOUT", -1);

// ......................................................................
// Anything which is editing and uses layouts might need to save data.
$update_delim = "^~";
$RESPONSE->body->add_script(
    "function sto(fld, fm) {\n"
  . " var datFld = eval('document.forms.' + fm + '._update_data');\n"
  . " if (datFld != null) {\n"
  . "  fname = fld.name;\n"
  . "  if (datFld.value != '') datFld.value += '$update_delim';\n"
  . "  datFld.value += fname + '|' + escape(fld.value);\n"
  . " }\n"
  . "}\n"
  );

// ......................................................................
/**
* Find content layout tags in a string.
* These are of the format: <!--LAYOUTID="my layout id"--> and there
* may be any number (including zero) of these in the string.
* Returns a string which is the passed-in string with the tags
* replaced by the rendered layout content.
* @param string The string to search for layout definitions
* @return string The string with all layout content rendered into it
*/
function render_layouts($s) {
  $mytimer = new microtimer();
  $rs = $s;
  if (stristr($rs, "LAYOUTID=")) {
    preg_match_all("/<!--LAYOUTID=\"(.+)\"-->/i", $rs, $matches);
    for ($i=0; $i< count($matches[0]); $i++) {
      $layouttag = $matches[0][$i];
      $layoutid  = $matches[1][$i];
      debugbr("layout: rendering $layoutid", DBG_DEBUG);
      $mylayout = new named_layout($layoutid);
      $rs = str_replace($layouttag, $mylayout->render(), $rs);
    }
  }
  debug_trace();
  return $rs;
} // render_layouts

// ......................................................................
/**
* Layout
* A layout can be simply viewed as a table definition. The table cells
* can contain blocks, and so this entity is provided to allow control
* of how blocks of content can be arranged on a webpage.
* @package cm
*/
class layout extends RenderableObject {
  // Public
  /** The unique ID of this layout */
  var $layoutid = 0;
  /** The name of the current layout */
  var $layout_name = "";
  /** The language of the layout (0 = default) */
  var $language = 0;
  /** The language encoding code */
  var $lang_encoding = "";
  /** The language text direction */
  var $lang_direction = "";
  /** True if we should display last modified date */
  var $show_last_modified = false;
  /** The format string for last modified datetime */
  var $format_last_modified = NICE_DATE;
  /** The prefix string for last modified datetime */
  var $prefix_last_modified = "";
  /** The index category of the current layout - used with Lucene indexing */
  var $index_category = "";
  /** Table width specification */
  var $table_width = "";
  /** Table autojustify flag */
  var $table_autojustify = false;
  /** Table rowstriping mode flag */
  var $table_rowstripes = false;
  /** Whether the layout exists in database or not */
  var $exists = false;

  // Private
  /** The layout table itself
      @access private */
  var $layout_table;
  /** The name of the layout form
      @access private */
  var $layoutfm = "layoutform";
  /** The layout blocks, keyed on 'row|col'
      @access private */
  var $layout_blocks = array();
  /** Supplemental layout table style
      @access private */
  var $layout_style = "";
  /** Total rows in layout
      @access private */
  var $tot_rows = 0;
  /** Total columns in layout
      @access private */
  var $tot_cols = 0;
  /** Total empty/undefined cells
      @access private */
  var $tot_empty = 0;
  /** Total cells containing a content block
      @access private */
  var $tot_block = 0;
  /** Total plain content cells
      @access private */
  var $tot_plain = 0;
  /** Total wysiwyg content cells
      @access private */
  var $tot_wysiwyg = 0;
  /** Total editable plain content cells
      @access private */
  var $tot_editable = 0;
  /** Total viewable plain content cells
      @access private */
  var $tot_viewable = 0;
  /** Array of layout blocks in edit mode
      @access private */
  var $edit_blocks = array();
  /** Array of layout cells with edit permission
      @access private */
  var $editable_cells = array();
  /** Array of layout cells with view permission
      @access private */
  var $viewable_cells = array();
  /** Last modified date/time string.
      Most recently modified block in layout.
      @access private */
  var $last_modified = "";
  /** Table style to apply for plain cells table
      @access private */
  var $table_style = "";
  /** Message to display (optional)
      @access private */
  var $message = "";
  /** Local layouteditor object, only instantiated if the
      layout requires editing services.
      @access private */
  var $layouteditor;
  // ....................................................................
  /**
  * Constructor
  * Create a new layout object. To create a new layout then just leave
  * leave the argument list empty.
  * @param string $id The unique name/identity of the layout.
  */
  function layout($id=NEW_LAYOUT) {
    // Creating new layout..
    if ($id == NEW_LAYOUT) {
      $id = get_next_sequencevalue("seq_layout_id", "ax_layout", "layout_id");
    }

    // Save block ID..
    $this->layoutid = $id;

    // Define a unique form name..
    $this->layoutfm = "layoutfm_$id";

    // Read it from database
    $this->get($id);
  } // layout
  // ....................................................................
  /**
  * Provide a layouteditor. This is used to instantiate a layouteditor
  * object for when we need to change this layout somewhow. We only
  * need one, so we check if it's already been done first.
  */
  function activate_editing() {
    if (!isset($this->layouteditor)) {
      global $RESPONSE, $LIBDIR;
      include_once("layout-editor-defs.php");
      $this->layouteditor = new layouteditor($this);
    }
  } // activate_editing
  // ....................................................................
  /**
  * Get the layout.
  * Retrieves the specified layout from database.
  * @param string $name The name/identity of the layout to get
  */
  function get($id) {
    global $RESPONSE;
    debug_trace($this);
    $this->exists = false;

    // Try and find it..
    if ($RESPONSE->multilang) {
      $q  = "SELECT * FROM ax_layout, ax_language";
      $q .= " WHERE ax_layout.layout_id=$id";
      $q .= "   AND ax_language.lang_id=ax_layout.lang_id";
    }
    else {
      $q  = "SELECT * FROM ax_layout";
      $q .= " WHERE ax_layout.layout_id=$id";
    }
    $Lq = dbrecordset($q);
    if ($Lq->hasdata) {
      $this->layoutid = $id;
      $this->layout_name = $Lq->field("layout_name");
      if ($RESPONSE->multilang) {
        $this->language = $Lq->field("lang_id");
        $this->lang_encoding = $Lq->field("char_encoding");
        $this->lang_direction = $Lq->field("direction");
      }
      $this->layout_style = $Lq->field("layout_style");
      $this->index_category = $Lq->field("index_category");
      $sertable = $Lq->field("layout_table");
      if ($sertable != "") {
        $this->layout_table = unserialize($sertable);
      }
      $this->format_last_modified = $Lq->field("format_last_modified");
      if ($this->format_last_modified == "") {
        $this->format_last_modified = NICE_DATE;
      }
      $this->prefix_last_modified = $Lq->field("prefix_last_modified");
      $this->show_last_modified = ($Lq->field("show_last_modified") == "t");
      if ($this->show_last_modified) {
        $lmts = 0;
        $q  = "SELECT MAX(last_modified) AS lastmod FROM ax_block";
        $q .= " WHERE layout_id=$id";
        $Lm = dbrecordset($q);
        if ($Lm->hasdata) {
          $lmdt = $Lm->field("lastmod");
          if ($lmdt != "") {
            $lmts = datetime_to_timestamp($lmdt);
          }
        }
        // Table modification time..
        if (isset($this->layout_table) && isset($this->layout_table->last_modified)) {
          $ts = $this->layout_table->last_modified;
          if ($ts > $lmts) $lmts = $ts;
        }
        // Save modstamp string..
        if ($lmts > 0) {
          $this->last_modified = timestamp_to_displaydate($this->format_last_modified, $lmts);
          if ($this->prefix_last_modified != "") {
            $this->last_modified = $this->prefix_last_modified . " " . $this->last_modified;
          }
        }
      }
      $this->exists = true;
    }
    // Ensure we have a table for layout..
    if (!isset($this->layout_table) || $this->layout_table->cellcount() == 0) {
      $this->layout_table = new matrix(1, 1, "&nbsp;");
      $this->layout_table->tablename = "layout:$this->layout_name";
      $this->layout_table->setwidth("100%");
    }

    // Get the layout cell information..
    $this->tot_empty = 0;
    $this->tot_block = 0;
    $this->tot_plain = 0;
    $this->tot_wysiwyg = 0;
    $this->tot_editable = 0;
    $this->tot_viewable = 0;
    $this->editable_cells = array();
    $this->viewable_cells = array();
    $this->tot_rows = $this->layout_table->rowcount();
    $this->tot_cols = $this->layout_table->cellcount();
    if ($this->exists) {
      for ($row=0; $row < $this->tot_rows; $row++) {
        for ($col=0; $col < $this->tot_cols; $col++) {
          if ($this->layout_table->cell_exists($row, $col)) {
            $cell = $this->layout_table->get_cell($row, $col);
            $this->layout_blocks["$row|$col"] = $cell->blockid;
            if (!isset($cell->celltype)) {
              $this->tot_empty += 1;
            }
            else {
              switch ($cell->celltype) {
                case BLOCK_CONTENT: $this->tot_block   += 1; break;
                case PLAIN_CELL: $this->tot_plain   += 1; break;
                case WYSIWYG_EDITOR: $this->tot_wysiwyg += 1; break;
              }
              if (isset($cell->access) && isset($RESPONSE)) {
                if ($RESPONSE->ismemberof_group_in("Editor,Author") ||
                   ($RESPONSE->ismemberof_group("Entry") && $this->is_pendingver())
                ) {
                  if ($cell->access->anypermitted($RESPONSE->group_names_list(), PERM_UPDATE)) {
                    $this->tot_editable += 1;
                    $this->editable_cells["$row|$col"] = true;
                  }
                }
                if ($cell->access->anypermitted($RESPONSE->group_names_list(), PERM_READ)) {
                  $this->tot_viewable += 1;
                  $this->viewable_cells["$row|$col"] = true;
                }
              }
            }
          }
          else {
            $this->tot_empty += 1;
          }
        }
      }
    }
    debug_trace();
    // Return true if at least the block exists..
    return $this->exists;
  } // get
  // ....................................................................
  /**
  * Save the layout.
  * Save this layout to the database. Create a new one if it
  * doesn't already exist.
  */
  function put() {
    debug_trace($this);
    // Timestamp the change in the table object itself..
    $this->layout_table->last_modified = time();

    // Deal with brand new layout..
    if ($this->exists) {
      $Lq = new dbupdate("ax_layout");
      $Lq->where("layout_id=" . $this->layoutid);
    }
    else {
      $Lq = new dbinsert("ax_layout");
      $Lq->set("layout_id", $this->layoutid);
    }
    $Lq->set("layout_name",          $this->layout_name);
    $Lq->set("lang_id",              $this->language);
    $Lq->set("layout_style",         $this->layout_style);
    $Lq->set("index_category",       $this->index_category);
    $Lq->set("layout_table",         serialize($this->layout_table));
    $Lq->set("show_last_modified",   $this->show_last_modified);
    $Lq->set("format_last_modified", $this->format_last_modified);
    $Lq->set("prefix_last_modified", $this->prefix_last_modified);
    $this->exists = $Lq->execute();
    debug_trace();
  } // put
  // ....................................................................
  /**
  * Replicate the hosted layout as a new layout. Creates a brand new
  * layout in the database, with same data as this one. The end result
  * is that this current object becomes the new layout, and a duplicate
  * set of layout records exist in the database. The layout ID of this
  * new layout is, of course, updated to being a brand new one.
  * NOTES: The layout name is normally left null, which keeps the layout
  * in the same 'family' of layout versions. You can force the layout
  * name to be different, and this will create a new 'layout_set'
  * record of that name for you, if required.
  * @param string $layoutname New layout name. If null, keeps same name.
  */
  function replicate($layoutname="") {
    $this->activate_editing();
    $this->layouteditor->replicate($layoutname);
  } // replicate
  // ....................................................................
  /**
  * Delete the hosted layout from the database. Afterwards, the current object
  * still exists as it was before this method was executed, but the
  * $this->layout->exists flag will have been reset to false.
  */
  function delete() {
    $this->activate_editing();
    $this->layouteditor->delete();
  } // delete
  // ....................................................................
  /**
  * Paste the given layout into this layout, replacing the complete
  * definition as it currently stands, with the new one. To do this
  * we delete the current layout from the database, get() the new
  * layout from the database, and then replicate it, morphing this
  * object into the brand new layout. All layout and associated block
  * ID's are changed, and are brand new.
  */
  function paste_layout($layoutid) {
    $layoutname = $this->layout_name;
    $this->delete();
    $this->get($layoutid);
    $this->replicate($layoutname);
  } // paste_layout
  // ....................................................................
  /**
  * Index all blocks in this layout.
  * If Lucene indexing is enabled, then we call the indexer for all of
  * the blocks which are in the hosted layout, using the webpage path and title as
  * provided in the call to this method.
  * @param string $path The relative path to the webpage the hosted layout is in
  * @param string $title The title of the webpage the hosted layout is in
  */
  function index($path, $title) {
    global $RESPONSE;
    $lcq = dbrecordset("SELECT block_id FROM ax_block WHERE layout_id=" . $this->layoutid);
    if ($lcq->hasdata) {
      // Include metadata if there is any..
      $metadata = false;
      if ($RESPONSE->metadata_mode == METADATA_ENABLED) {
        include_once("metadata-defs.php");
        $metadata = new layout_metadata_elements($this->layoutid, "", true);
      }
      do {
        $blockid = $lcq->field("block_id");
        $b = new block($blockid);
        $b->index($path, $title, $this->index_category, $metadata);
      } while ($lcq->get_next());
    }
  } // index
  // ....................................................................
  /**
  * Un-Index all blocks in this layout. After calling this method all
  * the bloacks in the layout will have been removed from the Lucene
  * index.
  */
  function unindex() {
    $lcq = dbrecordset("SELECT block_id FROM ax_block WHERE layout_id=" . $this->layoutid);
    if ($lcq->hasdata) {
      do {
        $blockid = $lcq->field("block_id");
        $b = new block($blockid);
        $b->unindex();
      } while ($lcq->get_next());
    }
  } // unindex
  // ....................................................................
  /**
  * Return true if the current user is permitted to edit layout details.
  * We allow editing only for versions VERSION_PENDING and VERSION_LIVE
  * and the latter only for Editors.
  * @return boolean True if editing is permitted by current user.
  */
  function user_can_edit($required_version=VERSION_UNDEFINED) {
    global $RESPONSE;
    $perm = false;
    if ($required_version == VERSION_UNDEFINED ||
        $required_version == $this->version) {
      // Pending version
      if ($this->is_pendingver()) {
        if ($RESPONSE->ismemberof_group_in("Editor,Author")) {
          $perm = true;
        }
      }
      // Live version
      elseif ($this->version == VERSION_LIVE) {
        if ($RESPONSE->ismemberof_group_in("Editor")) {
          $perm = true;
        }
      }
    }
    return $perm;
  } // user_can_edit
  // ....................................................................
  /**
  * Return PENDING version status.
  * @return boolean True if current versin is PENDING, or only one version.
  */
  function is_pendingver() {
    return ($this->version == VERSION_PENDING || $this->version_count == 1);
  }
  // ....................................................................
  /**
  * Render the layout editing suite.
  * @return string The HTML for the editing suite form etc.
  * @access private
  */
  function editform() {
    $this->activate_editing();
    return $this->layouteditor->editform();
  } // editform
  // ....................................................................
  /**
  * Render the layout content.
  * @return string The HTML
  * @access private
  */
  function layoutcontent() {
    debug_trace($this);
    global $LIBDIR;
    global $RESPONSE;

    $pwidth = "150px";

    // Create a simple container table to view layout..
    $Tvw = new table($this->layout_name);
    $Tvw->tr();
    $Tvw->td();

    // Make our layout table and populate it with our blocks..
    $Tlay = $this->layout_table;

    // Apply supplemental style if defined..
    if ($this->layout_style != "") {
      $Tlay->setstyle($this->layout_style);
    }

    // Ensure any blank cells return "&nbsp;"..
    $Tlay->setnbsp();

    if ($this->tot_plain > 0) {
      $edbox = new form_textfield();
      $edbox->set_onchange("sto(this,'$this->layoutfm')");
      $group_names_list = $RESPONSE->group_names_list();
      $profile = $Tlay->get_width_profile();
      $tblwdth = $Tlay->width;
    }
    if ($this->tot_editable > 0) {
      $RESPONSE->set_style("input {background-color:#F9F7ED;}");
    }
    for ($r = 0; $r < $this->tot_rows; $r++) {
      for ($c = 0; $c < $this->tot_cols; $c++) {
        if (isset($this->layout_blocks["$r|$c"])) {
          $blockid = $this->layout_blocks["$r|$c"];
          if ($blockid != 0) {
            // Render as content block cell..
            $block = new block($blockid);
            if ($block->mode == "editing") {
              $this->edit_blocks[] = $block->blockid;
            }
            $block->set_layout_info(
                  $this->version,
                  $this->version_count,
                  $this->language,
                  $this->lang_encoding,
                  $this->lang_direction
                  );
            $cell = $Tlay->get_cell($r, $c);
            $cell->setcontent($block->render());
            $cell->setalignment($block->justify, $block->valign);
            // Apply supplemental style if defined..
            if ($this->layout_style != "") {
              $cell->setstyle($this->layout_style);
            }
            $Tlay->set_cell($r, $c, $cell);
          }
          else {
            if ($this->tot_editable > 0 && $this->mode != "previewing") {
              // Render as editable textfield, if update is permitted..
              $cell = $Tlay->get_cell($r, $c);
              if (isset($this->editable_cells["$r|$c"])) {
                // Try to ascertain a pixel width..
                $edbox->clearstyle();
                if (isset($profile[$c])) {
                  $pwdth = $profile[$c];
                  if (strstr($pwdth, "%")) {
                    if ($tblwdth != "" && !strstr($tbldwdth, "%")) {
                      $w = (int) ($tblwdth * $pwdth / 100);
                      $edbox->setstyle("width:" . $w . "px;");
                    }
                  }
                  else {
                    if ($pwdth != "") {
                      $edbox->setstyle("width:" . $pwdth . "px;");
                    }
                  }
                }
                $edbox->setvalue($cell->content->content);
                $cell->content->content = $edbox->render($cell->cellid);
                // Apply supplemental style if defined..
                if ($this->layout_style != "") {
                  $cell->setstyle($this->layout_style);
                }
                $Tlay->set_cell($r, $c, $cell);
              }
            }
            // Overriding read permission blanking here. If it isn't
            // a viewable cell then they draw a blank..
            if (!isset($this->viewable_cells["$r|$c"])) {
              $cell = $Tlay->get_cell($r, $c);
              $cell->content->content = "";
              // Apply supplemental style if defined..
              if ($this->layout_style != "") {
                $cell->setstyle($this->layout_style);
              }
              $Tlay->set_cell($r, $c, $cell);
            }
          }
        }
      }
    }

    // Add in editing tools if authorised..
    debugbr("layoutcontent: mode is $this->mode", DBG_DEBUG);
    if ($this->mode != "previewing") {
      $layedit = ($RESPONSE->ismemberof_group_in("Editor,Author,Entry"));
      $Tvw->td_content("<form name=\"$this->layoutfm\" method=\"post\">\n");
      $formcontent = "";

      if ($layedit) {
        $Tvw->setstyle("border-width:1px;border-style:dotted;border-color:#0000ff;");
        $Tlay->setborder(1);

        // Version selector
        $verCombo = new form_combofield("layout_version");
        $verCombo->setclass("axcombo");
        $verCombo->setstyle("width:$pwidth;font-size:85%;");
        $verCombo->additem(0, "Pending");
        $verCombo->additem(1, "Live");
        $verCombo->additem(2, "Previous");
        $verCombo->setvalue($this->version);
        $verCombo->set_onchange("document.forms.$this->layoutfm.submit()");

        // Tools for the toolbar
        $toolbar = array();
        $toolbar[] = $verCombo;

        // Tools only for edit-enabled users..
        if ($this->user_can_edit()) {
          if ($RESPONSE->metadata_mode == METADATA_ENABLED) {
            $btn = new image_button("_meta", "", "", "", "$LIBDIR/img/_meta.gif", 42, 15, "Edit metadata");
            $url = "/axyl-metadata-editor.php?layout_id=" . urlencode($this->layoutid);
            $btn->set_onclick("meta_edit('$url')");
            $RESPONSE->body->add_popup_script(
                  "meta_edit",
                  600, 650, 50, 50,
                  "toolbar=no,status=no,scrollbars=yes,resizable=yes"
                  );
            $toolbar[] = $btn;
          }
          $toolbar[] = new image_button(
                    "_edit",
                    "", "", "",
                    "$LIBDIR/img/_edit.gif",
                    42, 15,
                    "Edit layout"
                    );
          // Paste layout from clipboard button..
          $layconf = new configuration("layout", "clipboard");
          $copy_layoutid = $layconf->value("copy_layoutid");
          if ($copy_layoutid != "" && $copy_layoutid != $this->layoutid) {
            $bpaste = new image_button(
                    "_paste",
                    "", "", "",
                    "$LIBDIR/img/_paste.gif",
                    57, 15,
                    "Paste layout"
                    );
            $bpaste->set_confirm_text("Overwrite this layout with the one on the clipboard?");
            $toolbar[] = $bpaste;
          }
          // Copy layout to clipboard button..
          $toolbar[] = new image_button(
                    "_copy",
                    "", "", "",
                    "$LIBDIR/img/_copy.gif",
                    42, 15,
                    "Copy layout to clipboard"
                    );
        } // can edit

        // Toolbar table
        $Tbar = new table("toolbar");
        $Tbar->tr("axtitle");
        if ($this->user_can_edit()) {
          $Tbar->th("[$this->layout_name]", "axtitle");
        }
        else {
          $Tbar->th("&nbsp;", "axtitle");
        }
        $tools = "";
        foreach ($toolbar as $tool) {
          $tools .= $tool->render();
        }
        // Render tools, if any..
        $Tbar->th($tools, "axtitle");
        $Tbar->th_css("text-align:right");
        $Tvw->td_content( $Tbar->render(), "axtitle" );

        // Hidden form fields..
        $update_data = new form_hiddenfield("_update_data", "");
        $update_flag = new form_hiddenfield("_update_flag", "false");

        // Put hidden fields into the toolbar form..
        $form_content .=
            $update_data->render()
          . $update_flag->render();

      } // if layedit

      $layfm = new form_hiddenfield("edit_layoutform", $this->layoutfm);
      $mode  = new form_hiddenfield("layoutmode", $this->mode);
      $elid  = new form_hiddenfield("edit_layoutid", $this->layoutid);
      $export_flag = new form_hiddenfield("_export_flag", "false");
      $form_content .=
          $mode->render()
        . $layfm->render()
        . $elid->render()
        . $export_flag->render();

      // Put hidden fields into the toolbar form..
      $Tvw->td_content( $form_content );
      $Tvw->td_content("</form>\n");

    } // not previewing

    // Put layout into viewer table..
    $Tvw->td_content( $Tlay->render() );
    $Tvw->td_alignment("", "top");

    if ($this->show_last_modified && $this->last_modified != "") {
      $Tvw->tr();
      $Tvw->td($this->last_modified, "axyl_lastmod");
    }

    $bottoolbar = array();
    // If editable content, give them a save button,
    // and remember the layout ID and the mode..
    if ($this->mode != "previewing"
      && $this->tot_editable > 0
      && count($this->edit_blocks) == 0
    ) {
      $bupdate = new form_imagebutton("_update", "", "", "$LIBDIR/img/_save.gif", "Save changes", 57, 15);
      $clkstr  = "document.forms.$this->layoutfm._update_flag.value='true';";
      $clkstr .= "document.forms.$this->layoutfm.submit();";
      $bupdate->set_onclick($clkstr);
      $bottoolbar[] = $bupdate->render();
    }
    if ($this->mode != "previewing"
      && $this->tot_plain > 0
      && $this->tot_blocks == 0
    ) {
      $bexport = new form_imagebutton("_export", "", "", "$LIBDIR/img/_export.gif", "Export in CSV format", 57, 15);
      $clkstr  = "document.forms.$this->layoutfm._export_flag.value='true';";
      $clkstr .= "document.forms.$this->layoutfm.submit();";
      $bexport->set_onclick($clkstr);
      $bottoolbar[] = $bexport->render();
    }
    if (count($bottoolbar > 0)) {
      $tools = implode("&nbsp;", $bottoolbar);
      $Tvw->tr();
      $Tvw->td( $tools );
    }

    // Return the html..
    $s = $Tvw->render();
    debug_trace();
    return $s;
  } // layoutcontent
  // ....................................................................
  /**
  * Render the block content according to the mode of operation
  * we are in. Possible modes: 'viewing', 'editing', 'saving'.
  * @return string The HTML
  */
  function html() {
    debug_trace($this);
    global $LIBDIR;
    global $RESPONSE;
    global $edit_layoutid;

    $s = "";
    if ($this->message != "") {
      $s .= "<center><span class=error>$this->message</span></center>";
    }

    switch($this->mode) {
      case "editing":
        // Make sure edit request is meant for us..
        if (!isset($edit_layoutid) || $edit_layoutid != $this->layoutid) {
          return "";
        }
        // Deal with first layout edit. In this case it won't yet
        // exist in the database, so we create it here..
        if (!$this->exists) {
          $this->put();
        }
        $this->mode = "saving";
        $s .= $this->editform();
        break;

      default:
        if ($this->mode != "viewing" && $this->mode != "previewing") {
          $this->mode = "viewing";
        }
        // Insert any layout metadata into the page..
        if ($RESPONSE->metadata_mode == METADATA_ENABLED) {
          include_once("metadata-defs.php");
          $laymeta = new layout_metadata_elements($this->layoutid, "", true);
          $laymeta->insert_metatags($RESPONSE);
        }

        // Render this layout now..
        $s .= $this->layoutcontent();

    } // switch
    return $s;
  } // html
  // ....................................................................
  /**
  * Assign the cell IDs to the layout. We iterate across all cells
  * assigning the unique ID. This is used when the table shape changes.
  * @access private
  */
  function assign_cellids() {
    for ($row=0; $row < $this->tot_rows; $row++) {
      for ($col=0; $col < $this->tot_cols; $col++) {
        if ($this->layout_table->cell_exists($row, $col)) {
          $cell = $this->layout_table->get_cell($row, $col);
          $cell->setcellid("_tcell|$row|$col|tbody");
          $this->layout_table->set_cell($row, $col, $cell);
        }
      }
    }
  } // assign_cellids
  // ....................................................................
  /**
  * Vacate the given layout cell. Removes any defined Plain cell
  * or Content Block cell from this cell position. Note: this will
  * also delete any defined blocks/blocklets and lose all the content
  * in these records.
  * @param integer $row The row number of the layout cell
  * @param integer $col The column number of the layout cell
  * @access private
  */
  function cell_vacate($row, $col) {
    if ($this->layout_table->cell_exists($row, $col)) {
      $cell = $this->layout_table->get_cell($row, $col);
      if (isset($cell->celltype)) unset($cell->celltype);
      if (isset($cell->blockid)) {
        if ($cell->blockid != "") {
          $b = new block($cell->blockid);
          $b->delete();
        }
        if (isset($cell->celltype)) unset($cell->celltype);
        if (isset($cell->blockid))  unset($cell->blockid);
        if (isset($cell->access))   unset($cell->access);
      }
      $this->layout_table->set_cell($row, $col, $cell);
    }
  } // cell_vacate
  // ....................................................................
  /**
  * Merge the settings of the two cells. We assume that cell1 is
  * going to be the surviving cell, having "absorbed" cell2.
  * Rules: If cell1 is defined already, then we just vacate cell2
  * and leave cell1 as-is. If cell1 is empty, and cell2 is
  * defined, then we copy cell2 to cell1 and unset cell2 without
  * removing any blocks.
  * @param integer $row1 The row number of the absorbing cell
  * @param integer $col1 The column number of the absorbing cell
  * @param integer $row2 The row number of the absorbed cell
  * @param integer $col2 The column number of the absorbed cell
  * @access private
  */
  function cell_merge($row1, $col1, $row2, $col2) {
    if ($this->layout_table->cell_exists($row1, $col1) &&
        $this->layout_table->cell_exists($row2, $col2)) {
      $cell1 = $this->layout_table->get_cell($row1, $col1);
      $cell2 = $this->layout_table->get_cell($row2, $col2);
      if (isset($cell1->celltype)) {
        $this->cell_vacate($row2, $col2);
      }
      else {
        if (isset($cell2->celltype)) {
          $cell1->celltype = $cell2->celltype;
          unset($cell2->celltype);
          if (isset($cell2->blockid)) {
            $cell1->blockid = $cell2->blockid;
            unset($cell2->blockid);
          }
          // Put absorbed cell back in place..
          $this->layout_table->set_cell($row2, $col2, $cell2);
        }
      }
      // Put resultant cell back in place..
      $this->layout_table->set_cell($row1, $col1, $cell1);
    }
  } //cell_merge
  // ....................................................................
  /**
  * Process a block edit form POST.
  * Assume that the fields have been submitted in a form as named
  * in the config, and grab the POSTed values. This method is executed
  * from the constructor usually, before anything is read in from
  * the database. We get first shot to change data here.
  * @param text $id The identity of the set to update from POST
  * @access private
  */
  function POSTprocess() {
    debug_trace($this);
    global $HTTP_POST_VARS, $RESPONSE, $LIBDIR;
    global $edit_layoutform, $edit_layoutid, $layoutmode;

    // Only interested in our own postings..
    if (isset($edit_layoutform) && $edit_layoutform == $this->layoutfm) {
      if (isset($layoutmode) && isset($edit_layoutid)) {
        $this->get($edit_layoutid);
        if ($this->layoutid == $edit_layoutid) {
          // Posted buttons, and hidden fields..
          global $_done_x, $_edit_x, $_update_x, $_copy_x, $_paste_x;
          global $_update_flag, $_update_data, $_export_flag, $update_delim;
          global $layoutmode;

          $this->mode = $layoutmode;
          debugbr("layoutid $this->layoutid mode is $this->mode", DBG_DEBUG);
          switch ($this->mode) {
            case "viewing":
              // Clicked Edit button
              if (isset($_edit_x)) {
                $this->mode = "editing";
              }
              elseif (isset($_copy_x)) {
                $this->get($edit_layoutid);
                $layconf = new configuration("layout", "clipboard");
                if (!$layconf->field_exists("copy_layoutid")) {
                  $layconf->field_insert("copy_layoutid", "text");
                }
                $layconf->set_value("copy_layoutid", $this->layoutid);
                $layconf->put();
              }
              elseif (isset($_paste_x) && isset($edit_layoutid)) {
                $layconf = new configuration("layout", "clipboard");
                $copy_layoutid = $layconf->value("copy_layoutid");
                if ($copy_layoutid != "" && $edit_layoutid != "") {
                  $this->get($edit_layoutid);
                  debugbr("pasting layout $copy_layoutid", DBG_DEBUG);
                  $this->paste_layout($copy_layoutid);
                  $layconf->set_value("copy_layoutid", "");
                  $layconf->put();
                }
              }
              // Updating plain cell content..
              elseif (isset($_update_flag) && $_update_flag == "true") {
                global $_update_data;
                // Get layout table, so we can alter it..
                $this->get($edit_layoutid);
                $updated = false;
                if (isset($_update_data) && $_update_data != "") {
                  $updates = explode($update_delim, $_update_data);
                  foreach ($updates as $update) {
                    $bits = explode("|", $update);
                    if ($bits[0] == "_tcell") {
                      $row     = $bits[1];
                      $col     = $bits[2];
                      $tgroup  = $bits[3];
                      $postval = rawurldecode($bits[4]);
                      debugbr("POSTED cell: $row,$col = '$postval'", DBG_DEBUG);
                      $this->layout_table->poke_cell($row, $col, $postval, "", "", $tgroup);
                      $updated = true;
                    }
                  }
                }
                if ($updated) {
                  $this->put();
                }
              }
              elseif (isset($_export_flag) && $_export_flag == "true") {
                // Get layout table, so we can export it..
                $this->get($edit_layoutid);
                $RESPONSE->discard();
                header("Content-Type: text/csv");
                header("Content-Disposition: attachment; filename=$this->layout_name" . ".csv");
                echo $this->layout_table->csv();
                exit;
              }
              break;

            case "saving":
              // Get layout table, so we can alter it..
              $this->get($edit_layoutid);

              global $_new_x, $_save_x, $_cancel_x, $_done_x;
              global $layout_action, $_perm_set_x, $_perm_unset_x;

              // Let me out of here..
              if (isset($_done_x) || isset($_cancel_x)) {
                // Drop through to viewing..
                $this->mode = "viewing";
              }
              // Blow away whole layout, and start again..
              elseif (isset($_new_x)) {
                start_transaction();
                dbcommand("DELETE FROM ax_block WHERE layout_id=$this->layoutid");
                $Lup = new dbupdate("ax_layout");
                $Lup->set("layout_table", NULLVALUE);
                $Lup->where("layout_id=$this->layoutid");
                $Lup->execute();
                commit();
                if (isset($this->layout_table))  unset($this->layout_table);
                if (isset($this->layout_blocks)) unset($this->layout_blocks);
                $this->mode = "editing";
              }
              // Save button clicked..
              elseif (isset($_save_x)) {
                global $index_category, $layout_rows, $layout_cols, $layout_padding;
                global $background_colour, $width_profile, $layout_newcell;
                global $show_last_modified, $format_last_modified;
                global $prefix_last_modified, $table_style, $table_width;
                global $table_autojustify, $table_rowstripes, $layout_style;
                global $language;

                // Updated layout parameters..
                if ($this->index_category != $index_category) {
                  $this->index_category = $index_category;
                  $this->index();
                }
                $this->language = $language;
                $this->show_last_modified = isset($show_last_modified);
                $this->format_last_modified = $format_last_modified;
                $this->prefix_last_modified = $prefix_last_modified;
                $this->layout_style = $layout_style;

                // Table size changed flag..
                $sizechanged = false;

                // Set layout table parameters..
                $this->layout_table->setcss("");
                $this->layout_table->setclass($table_style);
                $this->layout_table->setwidth($table_width);
                if (isset($table_rowstripes)) {
                  $this->layout_table->rowstripes("axyl_rowstripe_lite,axyl_rowstripe_dark");
                }
                else {
                  $this->layout_table->rowstripes("");
                }
                $this->layout_table->autojustify(isset($table_autojustify));
                $this->layout_table->setpadding($layout_padding);
                $this->layout_table->setbgcolor($background_colour);
                if (isset($width_profile)) {
                  $this->layout_table->set_width_profile($width_profile);
                }
                // Adjust row changes..
                $Trows = $this->layout_table->rowcount();
                if ($layout_rows != $Trows) {
                  $sizechanged = true;
                  if ($layout_rows < $Trows) {
                    $last = $Trows - 1;
                    for ($i = 0; $i < $Trows - $layout_rows; $i++) {
                      $this->layout_table->delete_row($last);
                      $last -= 1;
                    }
                  }
                  else {
                    for ($i = 0; $i < $layout_rows - $Trows; $i++) {
                      $this->layout_table->append_row( new tablecell("&nbsp;") );
                    }
                  }
                }
                // Adjust for column changes..
                $Tcols = $this->layout_table->cellcount();
                if ($layout_cols != $Tcols) {
                  $sizechanged = true;
                  if ($layout_cols < $Tcols) {
                    $last = $Tcols - 1;
                    for ($i = 0; $i < $Tcols - $layout_cols; $i++) {
                      $this->layout_table->delete_cols($last);
                      $last -= 1;
                    }
                  }
                  else {
                    $this->layout_table->append_cols($layout_cols - $Tcols, new tablecell("&nbsp;"));
                  }
                }
                // Look for new layout cells..
                if (isset($layout_newcell)) {
                  foreach ($layout_newcell as $request) {
                    $bits = explode("|", $request);
                    $celltype = $bits[0];
                    $row = $bits[1];
                    $col = $bits[2];
                    // Get table cell to change it..
                    if ($this->layout_table->cell_exists($row, $col)) {
                      $cell = $this->layout_table->get_cell($row, $col);
                      // Block cells get new block, plain cells don't..
                      switch ($celltype) {
                        // Content Block or Wysiwyg cell..
                        case BLOCK_CONTENT:
                        case WYSIWYG_EDITOR:
                          $b = new block();
                          $b->layoutid = $this->layoutid;
                          $b->block_type = $celltype;
                          $b->put();
                          $cell->celltype = $celltype;
                          $cell->blockid = $b->blockid;
                          break;
                        // Plain cell..
                        case PLAIN_CELL:
                          $cell->celltype = PLAIN_CELL;
                          $cell->blockid = "";
                          // Set default cell permissions..
                          $cell->permit("Editor", PERM_ALL);
                          $cell->permit("Author", PERM_READ|PERM_UPDATE);
                          break;
                      } // switch
                      // Return cell to table..
                      $this->layout_table->set_cell($row, $col, $cell);
                    }
                  } // foreach
                } // new layout cells

                // Save the layout, stay in edit mode..
                $this->assign_cellids();
                $this->put();
                $this->mode = "editing";
              }
              // Permissions setting activity..
              elseif (isset($_perm_set_x) || isset($_perm_unset_x)) {
                global $layout_cellsel, $perm_groups, $perm_perms;
                if (isset($layout_cellsel) && isset($perm_groups) && isset($perm_perms)) {
                  foreach ($layout_cellsel as $cellsel) {
                    $bits = explode("|", $cellsel);
                    $row = $bits[0];
                    $col = $bits[1];
                    if ($this->layout_table->cell_exists($row, $col)) {
                      $setgroups = implode(",", $perm_groups);
                      $setperms = 0;
                      foreach ($perm_perms as $perm) {
                        $setperms |= $perm;
                      }
                      if (isset($_perm_set_x)) {
                        $this->layout_table->permit_cell($row, $col, $setgroups, $setperms);
                      }
                      else {
                        $this->layout_table->unpermit_cell($row, $col, $setgroups, $setperms);
                      }
                    }
                  }
                  $this->put();
                }
                $this->mode = "editing";
              }
              // Other layout management action..
              elseif (isset($layout_action) && $layout_action != "") {
                $req = explode("|", $layout_action);
                $activity = $req[0];
                switch ($activity) {
                  // Merging of rows or cols
                  case "merge":
                    $object = $req[1];
                    $row = $req[2]; $col = $req[3];
                    switch ($object) {
                      case "col":
                        $this->cell_merge($row, $col, $row, $col + 1);
                        $this->layout_table->merge_cols($row, $col, 2);
                        break;
                      case "row":
                        $this->cell_merge($row, $col, $row + 1, $col);
                        $this->layout_table->merge_rows($row, $col, 2);
                        break;
                      case "allcols":
                        $this->layout_table->merge_cols($row, $col, $this->tot_cols);
                        break;
                    } // switch
                    break;
                  // Splitting of merged rows or cols
                  case "split":
                    $object = $req[1];
                    $row = $req[2]; $col = $req[3];
                    switch ($object) {
                      case "col":
                        $this->layout_table->split_cols($row, $col);
                        break;
                      case "row":
                        $this->layout_table->split_rows($row, $col);
                        break;
                    } // switch
                    break;
                  // Delete content block or plain cell..
                  case "deletecell":
                    $row = $req[1];
                    $col = $req[2];
                    $this->cell_vacate($row, $col);
                    $this->layout_blocks = array();
                    break;
                  // Inserting a row
                  case "insrow":
                    $row = $req[1]; $col = $req[2];
                    $this->layout_table->insert_row($row);
                    $this->layout_blocks = array();
                    break;
                  // Inserting a column
                  case "inscol":
                    $row = $req[1]; $col = $req[2];
                    $this->layout_table->insert_cols($col);
                    $this->layout_blocks = array();
                    break;
                  case "delrow":
                    $row = $req[1]; $col = $req[2];
                    $this->layout_table->delete_row($row);
                    $this->layout_blocks = array();
                    break;
                  // Inserting a column
                  case "delcol":
                    $row = $req[1]; $col = $req[2];
                    $this->layout_table->delete_cols($col);
                    $this->layout_blocks = array();
                    break;
                } // switch

                // Save..
                $this->assign_cellids();
                $this->put();
                // Stay in editing..
                $this->mode = "editing";
              }
              break;
          } // switch
        } // layoutid is this one
      } // got $layoutmode and $layoutid
    } // layout form is us

    // If the user preference says so, then set the mode to
    // content preview mode..
    $prefs = new configuration("preferences", $RESPONSE->userid);
    if ($prefs->field_exists("Content Preview Mode")) {
      if ($prefs->value("Content Preview Mode") === true) {
        $this->mode = "previewing";
      }
    }
    debugbr("mode now set to: $this->mode", DBG_DEBUG);
    debug_trace();
  } // POSTprocess
} // layout class

// ----------------------------------------------------------------------
/**
* Named Layout. A named layout is just another way of grabbing a layout,
* but by name, rather than by ID. A given "name" can have multiple
* versions in existence, if it has been published, and these will all
* have unique ID's, so this class is concerned with sorting out which
* version of a named layout is required, and acquiring the correct
* layout ID.
* @package cm
*/
class named_layout extends layout {
  // Public
  /** The version of the layout we have */
  var $version = VERSION_UNDEFINED;
  /** Total versions of this layout in database */
  var $version_count = 0;

  // Private
  /** Flag to indicate POST told us to publish */
  var $posted_publish = false;
  /** Flag to indicate POST told us to revert */
  var $posted_revert = false;
  // ....................................................................
  /**
  * Constructor
  * Create a new named_layout object. A named layout is a layout which
  * is identified by its name, and by the version. Versions are numbered
  * from zero (0), which represents the most recent. The higher the
  * version number, the further back in time you go. We define three
  * special version numbers:
  *    VERSION_PENDING  (0) The layout waiting to be made live
  *    VERSION_LIVE     (1) The currently live layout
  *    VERSION_PREVIOUS (2) The version previously live
  * Accordingly, these versions are the most recent three in the set of
  * all versions of the layout.
  * @param string $name Layout name
  * @param integer $version Version number, (zero=most recent)
  */
  function named_layout($name="", $version=VERSION_UNDEFINED) {
    global $RESPONSE;

    // Initialise version..
    if ($version != VERSION_UNDEFINED) {
      $this->version = $version;
    }

    // Process any form submissions...
    $this->POSTprocess();

    // Fallback if required version still not defined..
    if ($this->version == VERSION_UNDEFINED) {
      if ($this->mode != "previewing" && $RESPONSE->ismemberof_group_in("Editor,Author,Entry")) {
        $this->version = VERSION_PENDING;
      }
      else {
        $this->version = VERSION_LIVE;
      }
    }

    // Process POSTs, and determine the layout ID..
    if ($name != "") {

      // How many versions do we have..
      $q  = "SELECT COUNT(*) AS vercnt FROM ax_layout";
      $q .= " WHERE layout_name='". addslashes($name) . "'";
      $Lcnt = dbrecordset($q);
      if ($Lcnt->hasdata) {
        $this->version_count = $Lcnt->field("vercnt");
      }
      debugbr("layout: version history of $this->version_count", DBG_DEBUG);

      // Now, if there are no versions of this layout then we are
      // accessing it for the very first time, so we create it,.
      if ($this->version_count == 0) {
        // Create new set if required..
        $checkSet = dbrecordset("SELECT * FROM ax_layout_set WHERE layout_name='". addslashes($name) . "'");
        if ($checkSet->rowcount == 0) {
          $LSin = new dbinsert("ax_layout_set");
          $LSin->set("layout_name", $name);
          $LSin->execute();
        }
        // Create new layout..
        $newlay = new layout();
        $newlay->layout_name = $name;
        $newlay->put();
        $this->version_count = 1;
      }

      // Grab all relevant versions of the layout..
      $LQ = new dbselect("ax_layout");
      $LQ->where("layout_name='". addslashes($name) . "'");
      $LQ->orderby("layout_id DESC");
      $LQ->execute();
      if ($LQ->hasdata) {
        $row = false;
        // NON-PENDING..
        if ($this->version > 0) {
          if ($LQ->rowexists($this->version)) {
            debugbr("layout: going for version $this->version", DBG_DEBUG);
            $row = $LQ->get_row($this->version);
          }
          else {
            if ($this->mode != "previewing"
              && $RESPONSE->ismemberof_group_in("Editor,Author,Entry")) {
              $this->message = "Selected layout does not exist. Falling back to Pending.";
            }
          }
        }
        // PENDING..
        if ($row === false) {
          // Show pending, if no live..
          $this->version = VERSION_PENDING;
          debugbr("layout: falling back to version $this->version", DBG_DEBUG);
          if ($LQ->rowexists($this->version)) {
            $row = $LQ->get_row($this->version);
          }
        }
        if ($row !== false) {
          $this->layout( $LQ->field("layout_id") );
        }
      }

      // Do parent POST processing..
      layout::POSTprocess();

      // Refresh local picture..
      $this->get( $this->layoutid );

      // Now do any of the actions, such as publishing
      // which may have been specified by POST..
      if ($this->posted_publish) {
        $this->publish();
      }
      if ($this->posted_revert) {
        $this->unpublish();
      }
    }
  } // named_layout
  // ....................................................................
  /**
  * Index the named layout. We only do this if the layout version
  * is LIVE or there is only a single version in existence.
  */
  function index() {
    // Find page we belong to..
    $q  = "SELECT * FROM ax_layout_set ls, ax_sitepage pg";
    $q .= " WHERE ls.layout_name='" . addslashes($this->layout_name) . "'";
    $q .= "   AND pg.page_id=ls.page_id";
    $sitepage = dbrecordset($q);
    if ($sitepage->hasdata) {
      $path  = $sitepage->field("page_path");
      $title = $sitepage->field("page_title");
      layout::index($path, $title);
    }
  } // index
  // ....................................................................
  /**
  * Un-Index the named layout. We only do this if the layout version
  * is LIVE or there is only a single version in existence.
  */
  function unindex() {
    if ($this->version_count == 1 || $this->version == VERSION_LIVE) {
      layout::unindex();
    }
  } // unindex
  // ....................................................................
  /**
  * Publish a pending named layout. All we do in fact, is to replicate
  * the current pending version of the layout (this one) into a new
  * version. That automatically makes this layout the current LIVE one,
  * and the newly created version becomes the new PENDING one.
  */
  function publish() {
    if ($this->version == VERSION_PENDING) {
      debugbr("publishing this layout as LIVE", DBG_DEBUG);
      $this->replicate();
      $this->index();
    }
  } // publish
  // ....................................................................
  /**
  * Un-Publish a live named layout. This simply deletes the current
  * pending version of the layout. That makes the current LIVE version
  * the new pending version.
  */
  function unpublish() {
    if ($this->version == VERSION_LIVE) {
      debugbr("unpublishing this layout, reverting previous layout to LIVE", DBG_DEBUG);
      $pending = new named_layout($this->layout_name, VERSION_PENDING);
      if ($pending->exists) {
        debugbr("deleting layout ID $pending->layoutid", DBG_DEBUG);
        $pending->unindex();
        $pending->delete();
        $this->version = VERSION_PENDING;
        $this->index();
      }
    }
  } // unpublish
  // ....................................................................
  /**
  * Return HTML for this named layout.
  */
  function html() {
    if ($this->exists) {
      return layout::html();
    }
    else {
      return "No such layout exists.";
    }
  } // html
  // ....................................................................
  /**
  * Process a block edit form POST.
  * access private
  */
  function POSTprocess() {
    debug_trace($this);
    global $layout_version, $edit_layoutid;
    global $_publish_x, $_revert_x, $_copy_x, $_paste_x;
    if (isset($layout_version)) {
      if ($this->version == VERSION_UNDEFINED) {
        debugbr("setting layout version to $layout_version", DBG_DEBUG);
        $this->version = $layout_version;
      }
    }
    // Buttons..
    if (isset($_publish_x)) {
      $this->posted_publish = true;
    }
    elseif (isset($_revert_x)) {
      $this->posted_revert = true;
    }

    // Do parent POST processing..
    //layout::POSTprocess();

    debug_trace($this);
  }
} // named_layout class
// ----------------------------------------------------------------------
?>