<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    xmenu-defs.php                                          */
/* Author:      Paul Waite                                              */
/* Description: Definitions for a Javascript-based multi-level menu.    */
/*              The system creates a semi-static config file in a       */
/*              dynamic manner and the javascript content is generated  */
/*              from that. The static file (XmenuConfig.js) is only     */
/*              rebuilt when either the stylesheet or the menu data in  */
/*              the database is changed. The stylesheet should contain  */
/*              .menu and .submenu classes as per the default Axyl      */
/*              installation.                                           */
/*                                                                      */
/* ******************************************************************** */
/** @package menu */

/** Menu classes */
include_once("menu-defs.php");

// ----------------------------------------------------------------------
/**
* The HV Menu Javascript variables file. This file is
* dynamically generated by this class.
* @access private
*/
define("MENU_VARS_FILE", "XmenuConfig.js");
/**
* XMenu class
* A Generic Javascript Menu. This class provides a multi-level menu
* implemented in cross-browser Javascript. It should be compatible with
* Netscape 4, Netscape 6, and IE 4+.
* @package menu
*/
class Xmenu extends RenderableObject {
  // Public
  /** Menu name eg: 'main' */
  var $menu_name = "";
  /** Menu language */
  var $language = 0;
  /** Name of stylesheet file to get menu styles from */
  var $stylesheet = "";

  // Private
  /** The menu instance
      @access private */
  var $menu;
  /** Unique database menu ID
      @access private */
  var $menu_id = 0;
  /** Path to menu variables file - theme aware
      @access private */
  var $xmenu_configfile_path = "";
  /** Whether this menu exists in the database
      @access private */
  var $exists = false;
  /** Home dir for Xmenu scripts
      @access private */
  var $xmenu_js = "";
  /** Home dir for Xmenu images
      @access private */
  var $xmenu_img = "";
  /** WWW Xmenu config URL - theme aware
      @access private */
  var $xmenu_configfile_url = "";
  /** Menu type from stylesheet: vertical, horizontal
      or collapsing.
      @access private */
  var $menu_type = "vertical";
  /** Menu image object: vertical arrow
      @access private */
  var $img_vert_arrow;
  /** Menu image object: horizontal arrow
      @access private */
  var $img_horiz_arrow;
  /** Menu image object: spacer
      @access private */
  var $img_spacer;
  /** Whether to force rebuild or not
      @access private */
  var $force_rebuild = false;
  /** Whether menu structure was rebuilt or not
      @access private */
  var $menu_structure_changed = false;
  /** Whether images were changed or not
      @access private */
  var $imgs_changed = false;
  // ....................................................................
  /**
  * Constructor
  * Create the Xmenu object.
  * @param string  $menu_name  Menu name
  * @param object  $webpage    Webpage object that this menu is being created for
  * @param string  $stylsheet  Name of stylesheet file to reference for menu styles
  * @param integer $lang       Optional language variant of this menu (zero = default)
  */
  function Xmenu($menu_name="main", $webpage=false, $stylesheet="", $lang=-1) {
    global $RESPONSE, $LIBDIR;
    // Set menu name..
    $this->menu_name = $menu_name;
    // Set the menu language..
    if ($lang != -1) {
      $this->language = $lang;
    }
    elseif ($webpage !== false && $webpage->multilang && isset($webpage->languages[0])) {
      $this->language = $webpage->languages[0];
    }
    elseif (isset($RESPONSE) && $RESPONSE->multilang && isset($RESPONSE->languages[0])) {
      $this->language = $RESPONSE->languages[0];
    }
    // Check if we can use a menu stored in the session..
    // Manage selected menuoption..
    if (is_object($webpage)) {
      $this->session_id = $webpage->session_id;
      if ($webpage->session_record["menu_status"] != "") {
        debugbr("unserializing menu..");
        $menu = unserialize($webpage->session_record["menu_status"]);
        if (is_object($menu) && $menu->valid
           && ($menu->menu_name == $this->menu_name && $menu->language == $this->language)) {
          debugbr("installing pre-existing menu..");
          $this->menu = $menu;
          if ($this->menu->get_if_modified()) {
            debugbr("rebuilt modified menu..");
            $this->menu_structure_changed = true;
            $this->save_to_session();
          }
        }
      }
    }
    // Create it if it doesn't exist, re-create if required..
    if (!isset($this->menu)
    || (isset($cachecontrol) && $cachecontrol == "refresh")) {
      $this->menu = new menu_instance($this->menu_name, $this->language);
      $this->save_to_session();
      debugbr("xmenu: new menu generated", DBG_DEBUG);
    }
    // If valid, continue processing..
    if ($this->menu->valid) {
      $this->menu_id = $this->menu->menu_id;
      $this->exists = true;
    }

    if ($webpage) {
      $this->stylesheet = $webpage->site_docroot . $webpage->head->stylesheet;
      global $CMDIR;
      // Paths etc..
      $this->xmenu_configfile_path = $webpage->site_docroot . "$CMDIR/" . $webpage->theme . "_" . MENU_VARS_FILE;
      $this->xmenu_js = "$LIBDIR/js/xmenu/lib/js";
      $this->xmenu_img = "$LIBDIR/js/xmenu/img";

      // Menu images..
      $this->img_horiz_arrow = new img("$this->xmenu_img/arrow_horiz.gif", "", "", 3, 5);
      $this->img_vert_arrow  = new img("$this->xmenu_img/arrow_vert.gif", "", "", 5, 3);
      $this->img_spacer      = new img("$this->xmenu_img/spacer.gif", "", "", 1, 1);
      // If images are set in session menu, use those..
      if (isset($this->menu->img_horiz_arrow)) {
        $this->img_horiz_arrow = $this->menu->img_horiz_arrow;
      }
      if (isset($this->menu->img_vert_arrow)) {
        $this->img_vert_arrow = $this->menu->img_vert_arrow;
      }
      if (isset($this->menu->img_spacer)) {
        $this->img_spacer = $this->menu->img_spacer;
      }

      // Menu configuration file out on disk..
      $this->xmenu_configfile_url = "$CMDIR/" . $webpage->theme . "_" . MENU_VARS_FILE;

      // Insert menu javascript calls into webpage..
      $this->display_in_webpage($webpage);
    }
    // Over-ridden stylesheet to use..
    if ($stylesheet != "") {
      $this->stylesheet = $webpage->site_docroot . "/" . $stylesheet;
    }
  } // Xmenu
  // ....................................................................
  /**
  * Over-rides the standard icon images used to indicate submenus and
  * spaces. The values passed should be image objects as instances
  * of the 'img' class (@see img()).
  * @param object $img_horiz_arrow   New image for 'horizontal arrow' icon
  * @param object $img_vert_arrow    New image for 'vertical arrow' icon
  * @param object $img_spacer        New image for 'spacer' icon
  */
  function set_menu_images($img_horiz_arrow=false, $img_vert_arrow=false, $img_spacer=false) {
    $imgs_changed = false;
    if ($img_horiz_arrow !== false && $this->img_horiz_arrow->src != $img_horiz_arrow->src) {
      $this->img_horiz_arrow = $img_horiz_arrow;
      $imgs_changed = true;
    }
    if ($img_vert_arrow !== false && $this->img_vert_arrow->src != $img_vert_arrow->src) {
      $this->img_vert_arrow = $img_vert_arrow;
      $imgs_changed = true;
    }
    if ($img_spacer !== false && $this->img_spacer->src != $img_spacer->src) {
      $this->img_spacer = $img_spacer;
      $imgs_changed = true;
    }
    if ($imgs_changed) {
      $this->imgs_changed = true;
    }
  } // set_menu_images
  // ....................................................................
  /**
  * Set the standard images up. This method just re-asserts that the
  * Xmenu should use the standard Xmenu images for arrows, spacer etc.
  */
  function set_default_images() {
    $this->set_menu_images(
            new img("$this->xmenu_img/arrow_horiz.gif", "", "", 3, 5),
            new img("$this->xmenu_img/arrow_vert.gif", "", "", 5, 3),
            new img("$this->xmenu_img/spacer.gif", "", "", 1, 1)
            );
  } // set_default_images
  // ....................................................................
  /**
  * This method saves the menu_instance to the user session for next
  * time. This gives the menu persistence.
  * @access private
  */
  function save_to_session() {
    $this->menu->img_horiz_arrow = $this->img_horiz_arrow;
    $this->menu->img_vert_arrow = $this->img_vert_arrow;
    $this->menu->img_spacer = $this->img_spacer;
    $up = new dbupdate("ax_wwwsession");
    $up->set("menu_status", serialize($this->menu));
    $up->where("session_id='$this->session_id'");
    $up->execute();
  } // save_to_session
  // ....................................................................
  /**
  * Requires build
  * Check if the database records containing the menu
  * have been modified since the last modification time
  * of the menu vars file. Returns true if so.
  * @return bool True if menu requires build (config has been changed)
  * @access private
  */
  function requires_build() {
    // First check existence and file timestamp of the
    // menu setup file that is used by the system..
    if (!file_exists($this->xmenu_configfile_path)) {
      debugbr("xmenu rebuild: config file absent", DBG_DEBUG);
      return true;
    }
    elseif ($this->menu_structure_changed) {
      debugbr("xmenu rebuild: menu structure changed", DBG_DEBUG);
      return true;
    }
    elseif ($this->imgs_changed) {
      debugbr("xmenu rebuild: images changed", DBG_DEBUG);
      return true;
    }
    elseif ($this->force_rebuild) {
      debugbr("xmenu rebuild: forced rebuild", DBG_DEBUG);
      return true;
    }
    clearstatcache();
    $mvarmod_ts = filemtime($this->xmenu_configfile_path);
    $mvarmod_dt = timestamp_to_datetime($mvarmod_ts);
    // If stylesheet file is provided, then check against
    // that also, since this influences things like fonts
    // and colours etc..
    if ($this->stylesheet != "") {
      $stylemod_ts = filemtime($this->stylesheet);
      if ($stylemod_ts > $mvarmod_ts) {
        debugbr("xmenu rebuild: stylesheet changed", DBG_DEBUG);
        return true;
      }
    }
    // If we got here, no need to build..
    return false;
  } // requires build
  // ....................................................................
  /**
  * Build menu config
  * Build the static menu config file which is used to set menu
  * look and feel, driven by the given stylesheet. This is a physical
  * file which gets (re-)built and saved to disk.
  * @access private
  */
  function build() {
    global $RESPONSE;
    global $LIBDIR;
    // Get fonts and colours etc. If this stylesheet is undefined or
    // doesn't exist on disk, then we should still be able to generate
    // the vars below, but will get the defaults in each case..
    $ss = new stylesheet($this->stylesheet);
    // Read in all the style settings..
    $this->menu_type      = defaulted($ss->style("menu",              "menu-type"),                "vertical");
    $FontLowColor         = defaulted($ss->style("menu",              "color"),                    "#efefef");
    $LowBgColor           = defaulted($ss->style("menu",              "background-color"),         "black");
    $FontHighColor        = defaulted($ss->style("menu_highlight",    "color"),                    "white");
    $HighBgColor          = defaulted($ss->style("menu_highlight",    "background-color"),         "grey");
    $BorderColor          = defaulted($ss->style("menu",              "border-color"),             "black");
    $MenuTextAlign        = defaulted($ss->style("menu",              "text-align"),               "left");
    $MenuCentered         = defaulted($ss->style("menu",              "menu-align"),               "left");
    $MenuVerticalCentered = defaulted($ss->style("menu",              "menu-vertical-align"),      "top");
    $FontFamily           = defaulted($ss->style("menu",              "font-family"),              "arial");
    $ClickedColor         = defaulted($ss->style("menu",              "clicked-color"),            "black");
    $ClickedBgColor       = defaulted($ss->style("menu",              "clicked-background-color"), "white");
    $FontSubLowColor      = defaulted($ss->style("submenu",           "color"),                    "#efefef");
    $LowSubBgColor        = defaulted($ss->style("submenu",           "background-color"),         "black");
    $FontSubHighColor     = defaulted($ss->style("submenu_highlight", "color"),                    "white");
    $HighSubBgColor       = defaulted($ss->style("submenu_highlight", "background-color"),         "grey");
    $BorderSubColor       = defaulted($ss->style("submenu",           "border-color"),             "black");
    $ItemTextAlign        = defaulted($ss->style("submenu",           "border-color"),             "black");
    $SubFontFamily        = defaulted($ss->style("submenu",           "font-family"),              "arial");

    // Elements with superfluous bits to be removed..
    $val                  = defaulted($ss->style("menu",    "margin-top"),              "0");
    $StartTop             = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "margin-left"),             "0");
    $StartLeft            = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "padding-left"),            "0");
    $LeftPadding          = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "padding-top"),             "0");
    $TopPadding           = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "padding-right"),           "0");
    $RightPadding         = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "padding-bottom"),          "0");
    $BotPadding           = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "font-size"),               "8");
    $FontSize             = str_replace("pt", "", $val);
    $val                  = defaulted($ss->style("menu",    "menu-disappear-delay"),    "1");
    $DisappearDelay       = str_replace("ms", "", $val);
    // Sanity check for milliseconds mode..
    if ($DisappearDelay >= 100) {
      $DisappearDelay = ceil($DisappearDelay / 1000);
    }
    $val                  = defaulted($ss->style("menu",    "border-width"),            "1");
    $BorderWidth          = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "item-spacing"),            "0");
    $BorderBtwnElmnts     = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "menu-x-offset"),           "0");
    $MenuXOffset          = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "menu-y-offset"),           "0");
    $MenuYOffset          = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("submenu", "parent-overlap"),          "0");
    $ParentOverlap        = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("submenu", "parent-vertical-overlap"), "0");
    $ParentVertOverlap    = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("submenu", "padding-left"),            "0");
    $SubLeftPadding       = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("submenu", "padding-top"),             "0");
    $SubTopPadding        = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("submenu", "padding-right"),           "0");
    $SubRightPadding      = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("submenu", "padding-bottom"),          "0");
    $SubBotPadding        = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("submenu", "font-size"),               "8");
    $SubFontSize          = str_replace("pt", "", $val);

    // Integers, Logicals, et al..
    $val = defaulted($ss->style("menu", "font-weight"), "normal");
    $FontBold = ($val == "bold") ? "true" : "false";

    $val = defaulted($ss->style("submenu", "font-weight"), "normal");
    $SubFontBold = ($val == "bold") ? "true" : "false";

    $val = defaulted($ss->style("menu", "menu-unfold-onclick"), "no");
    $UnfoldOnClick = ($val == "yes") ? "true" : "false";

    $val = defaulted($ss->style("menu", "show-arrows"), "yes");
    $ShowArrows = ($val == "yes") ? true : false;

    // This can take values: 'url' or 'description'. It determines
    // what is shown in the browser status bar..
    $StatusContent = defaulted($ss->style("menu", "status-content"), "url");

    $val = defaulted($ss->style("menu", "clicked-node-highlighted"), "no");
    $ClickedNodeHighlighted = ($val == "yes") ? "true" : "false";

    $val = defaulted($ss->style("menu", "relative-positioned"), "no");
    $this->relative_positioned = ($val == "yes");

    $val = defaulted($ss->style("menu", "menu-sticky-state"), "no");
    $StickyState = ($val == "yes") ? "true" : "false";

    $val = defaulted($ss->style("menu", "menu-below"), "yes");
    $MenuBelow = ($val == "yes") ? "true" : "false";

    $val = defaulted($ss->style("menu", "child-left-align"), "no");
    $ChildLeftAlign = ($val == "yes") ? "true" : "false";

    $val = defaulted($ss->style("menu", "menu-fade-effect"), "no");
    $FadeEffect = ($val == "yes");
    if ($FadeEffect) {
      $FadeStart = defaulted($ss->style("menu", "menu-fade-start"), "0");
      $FadeStop  = defaulted($ss->style("menu", "menu-fade-stop"),  "100");
      $FadeSteps = defaulted($ss->style("menu", "menu-fade-steps"), "10");
      $FadeDelay = defaulted($ss->style("menu", "menu-fade-delay"), "100");
    }

    // Write the variables file out..
    // Complete example of a menu configuration for a
    /*
    var pastaVertical = [
      // globals -----------------
      [ Xmenu.prototype.VERTICAL,
        1, // delay in sec. before closing menu
        false, // onclick / onmouseover
        true, // horizontal & vertical menu: menu appears below/right of the root-node
        false, // horizontal menu: each hierarchy starts on same X
        false, // keep expansion state
        true, // highlight clicked nodes
        [ 20, 1 ], // XlayerParent width, height
        "img/spacer.gif" // spacer url
      ],
      // Menu level style 0..
      [
        [ "#ff4400", "#ffaaaa" ], // onclick-menu: color of clicked node
        // xlayer style:
        [
          // xOffset, yOffset, width, height,
          0, 0, 100, 35,
          // fading:
          // [start_val, stop_val, steps, delay(ms)],
          [0, 100, 10, 100], // or null,
          // onmouseout:
          [
            // bgcolor, fgcolor, align,
            "#ff5500", "white", "center",
            // topPad, rightPad, bottomPad, leftPad,
            10, 15, 0, 0,
            // bold, fontFace, fontSize,
            true, "Arial, Helvetica, sans serif", 1,
            // always_show_img, img_src, img_width, img_height, img_padding,
            false, "img/arrow_horiz.gif", 3, 5, 10
          ],
          // onmouseover:
          [
            // bgcolor, fgcolor, align,
            "#ff7c00", "#ffbbaa", "center",
            // topPad, rightPad, bottomPad, leftPad,
            0, 0, 0, 10,
            // bold, fontFace, fontSize,
            true, "Arial, Helvetica, sans serif", 1,
            // img_src, img_width, img_height, img_padding,
            "img/spacer.gif", 1, 1, 0
          ]
        ]
      ]
      // Menu level style 1..
      [ ] .. etc.
    ]
    */

    $newMV = new outputfile($this->xmenu_configfile_path);
    if ($newMV->opened) {
      $newMV->writeln("var menu_$this->menu_id = [");
      $newMV->writeln("// globals");
      $newMV->writeln("[ Xmenu.prototype." . strtoupper($this->menu_type) . ", // menu type");
      $newMV->writeln(" $DisappearDelay, // disappear delay (S)");
      $newMV->writeln(" $UnfoldOnClick, // unfolds onclick");
      $newMV->writeln(" $MenuBelow, // menu appears below root-node if true, else to the right");
      $newMV->writeln(" $ChildLeftAlign, // horizontal menus: each child left aligned with parent");
      $newMV->writeln(" $StickyState, // keep expansion state");
      $newMV->writeln(" $ClickedNodeHighlighted, // onclick menu: highlight clicked nodes");
      $newMV->writeln(" [20,1], // XlayerParent width,height");
      $newMV->writeln(" \"" . $this->img_spacer->src . "\" // spacer image url");
      $newMV->writeln("],");

      $styles = array();

      $s = "\"$ClickedColor\",\"$ClickedBgColor\"";
      $styles[] = "[$s]";

      // Start Level 0 styling..
      $s  = "";
      $s .= "$MenuXOffset,$MenuYOffset,"
          . $this->menu->level_widths[0] . ","
          . $this->menu->level_heights[0] . ",";
      // Fade effect
      $s .= ($FadeEffect) ? "[$FadeStart,$FadeStop,$FadeSteps,$FadeDelay]," : "null,";
      // On-mouseout
      $s .= "[";
      $s .= "\"$LowBgColor\",\"$FontLowColor\",\"$MenuTextAlign\",";
      $s .= "$TopPadding,$RightPadding,$BotPadding,$LeftPadding,";
      $s .= "$FontBold,\"$FontFamily\",$FontSize,";
      $s .= "false,";
      if ($ShowArrows) {
        if ($this->menu_type == "horizontal") {
          $s .= "\"" . $this->img_vert_arrow->src . "\",";
          $s .= $this->img_vert_arrow->width . "," . $this->img_vert_arrow->height . ",";
        }
        else {
          $s .= "\"" . $this->img_horiz_arrow->src . "\",";
          $s .= $this->img_horiz_arrow->width . "," . $this->img_horiz_arrow->height . ",";
        }
        $s .= "10],";
      }
      else {
        $s .= "null,0,0,0],";
      }
      // On-mouseover
      $s .= "[";
      $s .= "\"$HighBgColor\",\"$FontHighColor\",\"$MenuTextAlign\",";
      $s .= "$TopPadding,$RightPadding,$BotPadding,$LeftPadding,";
      $s .= "$FontBold,\"$FontFamily\",$FontSize,";
      $s .= "\"" . $this->img_spacer->src . "\",";
      $s .= $this->img_spacer->width . "," . $this->img_spacer->height . ",";
      $s .= "0],";
      // End Level 0 styling..
      $styles[] = "// level 0\n[$s]";

      // Level 1..N styling all the same, currently..
      for ($level = 1; $level <= $this->menu->level_depth; $level++) {
        // Start Level N styling..
        $s  = "";
        $ParentOverlap = -($ParentOverlap);
        $ParentVertOverlap = -($ParentVertOverlap);
        $s .= "$ParentOverlap,$ParentVertOverlap,"
            . $this->menu->level_widths[$level] . ","
            . $this->menu->level_heights[$level] . ",";
        // Fade effect
        $s .= ($FadeEffect) ? "[$FadeStart,$FadeStop,$FadeSteps,$FadeDelay]," : "null,";
        // On-mouseout
        $s .= "[";
        $s .= "\"$LowSubBgColor\",\"$FontSubLowColor\",\"$ItemTextAlign\",";
        $s .= "$TopPadding,$RightPadding,$BotPadding,$LeftPadding,";
        $s .= "$FontBold,\"$FontFamily\",$FontSize,";
        $s .= "false,";
        if ($ShowArrows) {
          if ($this->menu_type == "horizontal") {
            $s .= "\"" . $this->img_vert_arrow->src . "\",";
            $s .= $this->img_vert_arrow->width . "," . $this->img_vert_arrow->height . ",";
          }
          else {
            $s .= "\"" . $this->img_horiz_arrow->src . "\",";
            $s .= $this->img_horiz_arrow->width . "," . $this->img_horiz_arrow->height . ",";
          }
          $s .= "10],";
        }
        else {
          $s .= "null,0,0,0],";
        }
        // On-mouseover
        $s .= "[";
        $s .= "\"$HighBgColor\",\"$FontHighColor\",\"$MenuTextAlign\",";
        $s .= "$SubTopPadding,$SubRightPadding,$SubBotPadding,$SubLeftPadding,";
        $s .= "$SubFontBold,\"$SubFontFamily\",$SubFontSize,";
        $s .= "\"" . $this->img_spacer->src . "\",";
        $s .= $this->img_spacer->width . "," . $this->img_spacer->height . ",";
        $s .= "0],";
        // End Level N styling..
        $styles[] = "// level $level\n[$s]";
      } // for
      // End styles

      $newMV->writeln("// styles");
      $newMV->writeln("[" . implode(",\n", $styles) . "],");

      // Menu structure
      $newMV->writeln("// content");
      $newMV->writeln($this->menu_structure());

      $newMV->writeln("];");

      // Finish up..
      $newMV->closefile();
    } // newMV file opened
  } // build
  // ....................................................................
  /**
  * Menu structure
  * This function returns a set of Javascript array definitions which
  * should be put into the page which will be hosting the menu. Note
  * that this will be built for each user, since each user has a
  * unique auth_code which might be a part of the menu..
  * @return string Javascript var definitions for menu structure
  * @access private
  */
  function menu_structure() {
    global $RESPONSE;
    // MENU ITEM DETAIL..
    if ($this->menu->menuop_count() > 0) {
      // Store each row..
      $menuopts = array();
      foreach ($this->menu->menu_top as $item_no => $mopid) {
        $prefix = "$item_no";
        $this->Xmenu_entry($menuopts, $prefix, $mopid);
      } // foreach
    }
    return "[" . implode(",\n", $menuopts) . "]";
  } // menu_structure
  // ....................................................................
  /**
  * Recursively produce menu definition entries. These array
  * definitions define the menu structure in terms of the actual
  * menus and their sub-menus. This routine produces a single
  * menu-option definition, but will recursively call all child
  * (sub-menu) definitions.
  @param reference $menuopts Reference to an array of menuoptions
  @param string $mopid Menu option ID
  @access private
  */
  function Xmenu_entry(&$menuopts, $prefix, $mopid) {
    // Store current menu option..
    $mop = $this->menu->menuop($mopid);
    if ($mop !== false) {
      $details    = $mop->details();
      $menu_level = $mop->menu_level;
      $label      = $mop->label;
      if ($label == MENU_ITEM_SPACER) {
        $label = "&nbsp;";
      }
      if ($label == MENU_ITEM_SEPARATOR) {
        switch ($this->menu_type) {
          case "vertical":
          case "collapsing":
            $label = "<hr width='" . $this->menu->level_widths[$menu_level] . "' noshade>";
            break;
        } // switch
      }
      $action     = $mop->action;
      $expanded   = $mop->expanded;
      $auth_code  = $mop->auth_code;
      $heading    = $mop->is_submenuheading(); // Whether it is a heading for sub-menu
      $pseudo     = $mop->is_pseudo();        // Whether it is a pseudo menu-item
      $menuoption = !$heading;

      // Current menu option..
      if ($action != "" && !$pseudo) {
        if ($authcode == "t") {
          $action = href_addparm($action, "auth_code", $RESPONSE->get_auth_code());
        }
        $action = "new WinTarget(\"$action\")";
      }
      else {
        $action = "null";
      }
      // Make menu entry..
      $menuopts[] = "[\"$label\",$action,$menu_level]";

      // Process children..
      if (isset($mop->children) && $mop->children != "") {
        $childoptions = explode("|", $mop->children);
        $childcount = count($childoptions);
        $pfxcount = 1;
        foreach ($childoptions as $childmopid) {
          $childprefix = $prefix . "|" . $pfxcount;
          $this->Xmenu_entry($menuopts, $childprefix, $childmopid);
          $pfxcount += 1;
        }
      }
    }
  } // Xmenu_entry
  // ....................................................................
  /**
  * Display in webpage
  * Inserts the javascript necessary to embed the menu into a given webpage.
  * NB: Normally the webpage passed in here is $RESPONSE.
  * @param object $webpage Webpage object that this menu is being created for
  */
  function display_in_webpage($webpage) {
    global $LIBDIR;
    $webpage->head->add_scriptsrc("$this->xmenu_js/Browser.js");
    $webpage->head->add_scriptsrc("$this->xmenu_js/XlayerParent.js");
    $webpage->head->add_scriptsrc("$this->xmenu_js/Xlayer.js");
    $webpage->head->add_scriptsrc("$this->xmenu_js/Xmenu.js");
    $webpage->head->add_scriptsrc("$this->xmenu_js/Xmenus.js");
    $webpage->head->add_scriptsrc("$this->xmenu_js/XmenuConfig.js");
    $webpage->head->add_scriptsrc($this->xmenu_configfile_url);
    $webpage->head->add_scriptsrc("$this->xmenu_js/Debug.js");
    $webpage->head->add_script(
            "var menus=new Xmenus('','');\n"
          . "function initXmenu() {\n"
          . " menus.create();\n"
          . " debug.flushBuffer();\n"
          . "}\n"
          );
    $webpage->set_onload("initXmenu()");
    //$webpage->body->set_parms(" onResize=\"window.location.reload()\"");
  } // display_in_webpage
  // ....................................................................
  /**
  * This renders the field as HTML.
  * Inserts the HTML DIV tag which the HVmenu will use to position
  * itself to. The name of the DIV is taken from the unique menu ID,
  * and corresponds to the TargetLoc variable defined above..
  * @return string The menu anchor point (DIV) as HTML.
  */
  function html() {
    // Always build menu variables file if required..
    if ($this->requires_build()) {
      // Build static menu configuration file..
      $this->build();
      // Save menu object to session..
      $this->save_to_session();
    }
    $js  = "<script language=\"javascript\">";
    $js .= "menus.add( menu_$this->menu_id );";
    $js .= "</script>";
    return $js;
  } // html
} // Xmenu class

// ----------------------------------------------------------------------
?>