<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    query-defs.php                                          */
/* Author:      Paul Waite                                              */
/* Description: Definitions for making queries on the database          */
/*                                                                      */
/* ******************************************************************** */
/** @package database */

/** Date-time functions */
include_once("datetime-defs.php");

// ----------------------------------------------------------------------
// TRANSACTION Class
/** Transaction response to failure - rollback */
define("ROLLBACK_ON_FAIL",     true);
/** Transaction response to failure - do nothing */
define("NO_ROLLBACK_ON_FAIL",  false);

// ----------------------------------------------------------------------
// NULL query term
/** This value indicates a NULL field value for queries */
define("NULLVALUE", "NULL!FIELD!VALUE");

// ----------------------------------------------------------------------
/**
* Transaction class
* Encapsulates the query transaction.
* @package database
* @access private
*/
class transaction {
  /** Transaction mode */
  var $mode = ROLLBACK_ON_FAIL;
  /** True if there is an open transaction */
  var $open = false;
  /** True if the transaction failed */
  var $failed = false;
  /** If failed, message describing failure */
  var $failed_msg = "";
  // ....................................................................
  /**
  * Constructor
  *
  * Create a transaction. Sets basic transaction attributes.
  * This transaction object is used, via global functions, for
  * every transaction using these query routines.
  * @param bool $mode Transaction rollback mode (true = rollback on failure)
  */
  function transaction($mode=ROLLBACK_ON_FAIL) {
    $this->mode = $mode;
  }
  // ....................................................................
  /**
  * Begin transaction
  * Start a transaction.
  * @param bool $mode Transaction rollback mode (true = rollback on failure)
  */
  function begin($mode=ROLLBACK_ON_FAIL) {
    global $RESPONSE;
    if ($RESPONSE->db_backed && !$this->open) {
      $RESPONSE->datasource->begin_transaction();
      $this->open = true;
      $this->mode = $mode;
      $this->failed = false;
      $this->failed_msg = "";
      if (debugging()) {
        debugbr("TOK: Start transaction.", DBG_SQL);
      }
    }
  }
  // ....................................................................
  /**
  * Commit transaction
  * Try to commit the open transaction. If an error
  * occurs then try to roll back if flagged to do so.
  * @return boolean True if the transaction succeeded.
  */
  function commit() {
    global $RESPONSE;
    $tok = false;
    if ($RESPONSE->db_backed && $this->open) {
      // Try to commit the transaction now..
      if (!$this->failed) {
        if ($RESPONSE->datasource->commit()) {
          if (debugging()) {
            debugbr("TOK: transaction committed.", DBG_SQL);
          }
          $tok = true;
        }
        else {
          $this->failed = true;
        }
      }
      // Deal with failed transactions..
      if ($this->failed) {
        $errmsg = "transaction failed";
        error_log("TFAIL: " . APP_NAME . ": $errmsg", 0);
        if (debugging()) {
          debugbr("TFAIL: $errmsg", DBG_SQL);
        }
        if ($this->mode == ROLLBACK_ON_FAIL) {
          $this->rollback();
        }
      }
      // Always end up with it closed..
      $this->open = false;
    }
    return $tok;
  }
  // ....................................................................
  /**
  * Roll back transaction
  * Tell the database to roll back the transaction..
  */
  function rollback() {
    global $RESPONSE;
    if ($RESPONSE->db_backed && $this->open) {
      $RESPONSE->datasource->rollback();
      $errmsg = "transaction rolled back";
      error_log("TFAIL: " . APP_NAME . " $errmsg", 0);
      if (debugging()) {
        debugbr("TFAIL: $errmsg", DBG_SQL);
      }
      $this->open = false;
    }
  }
} // transaction class

// Instantiate a generic transaction to use..
$global_tran = new transaction();

// ----------------------------------------------------------------------
/**
* List of things class
* Encapsulates lists of items. A general-purpose class for containing
* lists of things. A utility class to hold lists of things like field
* lists, lists of tablenbames, orderby lists, etc.
* @package database
* @access private
*/
class listofthings {
  /** The list of things being held */
  var $things;
  /** Total things we have */
  var $total = 0;
  // ....................................................................
  /**
  * Constructor
  * Create a new listofthings object.
  */
  function listofthings() {
    $this->clear();
  }
  // ....................................................................
  /**
  * Add thing
  * @param string $thing The identifier of the thing we are adding
  * @param mixed  $val   The value of the thing we are adding
  */
  function add($thing, $val="") {
    $this->things[$thing] = $val;
    $this->total += 1;
  }
  // ....................................................................
  /**
  * Clear things
  * Clears all things from the list.
  */
  function clear() {
    if (isset($this->things)) unset($this->things);
    $this->total = 0;
  }
  // ....................................................................
  /**
  * Return list
  * @param  string $delim The delimiter for the returned list
  * @return string The delimited list of names of things
  */
  function listed($delim=",") {
    $list = "";
    if (isset($this->things)) {
      reset($this->things);
      while (list($thing, $val) = each($this->things)) {
        $list .= "$thing~^";
      }
      $list = str_replace("~^", $delim, trim($list));
      if (substr($list, -1) == $delim) {
        $list = substr($list, 0, strlen($list) - 1);
      }
    }
    return $list;
  }
  // ....................................................................
  /**
  * Return values
  * @param  string $delim The delimiter for the returned list
  * @return string The delimited list of values of things
  */
  function values($delim=",") {
    $list = "";
    if (isset($this->things)) {
      reset($this->things);
      while (list($thing, $value) = each($this->things)) {
        if ($value === "") $value = "''";
        $list .= "$value~^";
      }
      $list = str_replace("~^", $delim, trim($list));
      if (substr($list, -1) == $delim) {
        $list = substr($list, 0, strlen($list) - 1);
      }
    }
    return $list;
  }
  // ....................................................................
  /**
  * Return equates
  * Returns the things we contain in key=value format, and all
  * as a string delimited by the given character.
  * @param  string $delim The delimiter for the returned list
  * @return string The delimited list of equated things
  */
  function equated($delim=",") {
    $list = "";
    if (isset($this->things)) {
      reset($this->things);
      while (list($thing, $value) = each($this->things)) {
        if ($value === "") $value = "''";
        $list .= "$thing=$value~^";
      }
      $list = str_replace("~^", $delim, trim($list));
      if (substr($list, -1) == $delim) {
        $list = substr($list, 0, strlen($list) - 1);
      }
    }
    return $list;
  }
} // listofthings class

// ----------------------------------------------------------------------
/**
* SQLquery class
* An SQL Statement Text Container.
* This class is the parent of the main dbquery class which directs the
* query to the database. It is mainly a container of SQL query text, in
* the variable 'sql', but also offers a few basic methods for building
* queries. For complex queries however, build your own in a string and
* then just set the 'sql' variable.
* @package database
*/
class sqlquery {
  /** Type of query 'SELECT', 'DELETE', 'INSERT' or 'UPDATE' */
  var $type = "";
  /** List of fields in the query */
  var $fields;
  /** List of tables in the query */
  var $tables;
  /** The query WHERE clause components */
  var $where;
  /** The GROUP BY clause */
  var $groupby;
  /** The ORDER BY clause */
  var $orderby;
  /** The LIMIT value */
  var $limit;
  /** The OFFSET value */
  var $offset;
  /** The formatted SQL query itself @see build() */
  var $sql = "";
  // ....................................................................
  /**
  * Constructor
  * Create a new SQL Query object.
  * @param string $sql The SQL statement in full
  */
  function sqlquery($sql="") {
    $this->clear();
    $this->sql = $sql;
  } // sqlquery
  // ....................................................................
  /**
  * Clear query - Wipe all of the current query definitions.
  */
  function clear() {
    if (isset($this->fields))  unset($this->fields);
    if (isset($this->tables))  unset($this->tables);
    if (isset($this->where))   unset($this->where);
    if (isset($this->groupby)) unset($this->groupby);
    if (isset($this->orderby)) unset($this->orderby);
    $this->fields  = new listofthings();
    $this->tables  = new listofthings();
    $this->where   = new listofthings();
    $this->groupby = new listofthings();
    $this->orderby = new listofthings();
    $this->sql = "";
    $this->limit = 0;
    $this->offset = 0;
  } // clear
  // ....................................................................
  /**
  * Utility function to help building list of things
  * @param listofthings  $list_of_things  listofthings to add to
  * @param mixed         $list            A simple array or a delimited list
  * @param string        $delim           Delimiter, "," default
  * @access private
  */
  function addlist(&$list_of_things, $list, $delim=",") {
    if (is_array($list)) {
      $items = $list;
    }
    else {
      $items = explode($delim, $list);
    }
    // Add to our existing list..
    foreach ($items as $item) {
      if ($item != "") $list_of_things->add($item);
    }
  } // addlist
  // ....................................................................
  /**
  * Define field list
  * Add a list of fields to return in query. This is a cumulative function
  * which may be called more than once to add fields. You can specify the
  * list of fields either as an array, or as a delimited list. If the latter,
  * then default delimiter is a comma, unless you specify your own.
  * Applicable to SELECT, DELETE and UPDATE.
  * @param string $field_spec The field list to add to the query
  * @param string $delim The delimter you want to separate fields with
  */
  function fieldlist($field_spec="*", $delim=",") {
    $this->addlist($this->fields, $field_spec, $delim);
  } // fieldlist
  // ....................................................................
  /**
  * Define table list
  * Add the table specification to our list. This is a cumulative function
  * which may be called more than once to add tables. You can specify the
  * list of tables either as an array, or as a delimited list. If the latter,
  * then default delimiter is a comma, unless you specify your own.
  * @param string $table_spec The table list to add to the query
  * @param string $delim The delimiter you want to separate tables with
  */
  function tables($table_spec, $delim=",") {
    $this->addlist($this->tables, $table_spec, $delim);
  } // tables
  // ....................................................................
  /**
  * Define table FROM list
  * A nicer synonym for "tables()" for SELECT
  * @param string $table_spec The table list to add to the query
  * @param string $delim The delimiter you want to separate tables with
  */
  function from($table_spec, $delim=",") {
    $this->tables($table_spec, $delim);
  } // from
  // ....................................................................
  /**
  * Define table INSERT INTO list
  * A nicer synonym for "tables()" for INSERT
  * @param string $table_spec The table list to add to the query
  * @param string $delim The delimiter you want to separate tables with
  */
  function into($table_spec, $delim=",") {
    $this->tables($table_spec, $delim);
  } // into
  // ....................................................................
  /**
  * Define group by field list
  * The fields can be an array, or a delimited list. If the latter, then default delimiter is a comma,
  * unless you specify your own.
  * @param string $field_spec The field list to add to the GROUP BY. Do not include words "GROUP BY".
  * @param string $delim The delimiter you want to separate the fields with
  */
  function groupby($field_spec="", $delim=",") {
    $this->addlist($this->groupby, $field_spec, $delim);
  } // groupby
  // ....................................................................
  /**
  * Define order field list
  * Defines the Sort order field list. The fields can be an array, or a
  * delimited list. If the latter, then default delimiter is a comma,
  * unless you specify your own.
  * @param string $field_spec The field list to add to the ORDER BY. Do not include words "ORDER BY".
  * @param string $delim The delimiter you want to separate the fields with
  */
  function orderby($field_spec="", $delim=",") {
    $this->addlist($this->orderby, $field_spec, $delim);
  } // orderby
  // ....................................................................
  /**
  * Define query LIMIT
  * @param integer $limit Numeric value for limit rows to return. Do not include the word "LIMIT".
  * @param integer $offset Numeric value for start row. Do not include the word "OFFSET".
  */
  function limit($limit) {
    $this->limit = $limit;
  } // limit
  // ....................................................................
  /**
  * Define query OFFSET
  * @param integer $offset Numeric value for start row. Do not include the word "OFFSET".
  */
  function offset($offset) {
    $this->offset = $offset;
  } // set
  // ....................................................................
  /**
  * Define field assignments
  * Defines the field assignment clauses for UPDATE and INSERT queries.
  * @param string $field The name of the field to assign a value to
  * @param mixed  $val   The value to assign to the field. Processed according to type.
  */
  function set($field, $val) {
    global $RESPONSE;
    // Numerics are done without quotes
    if (is_int($val) || is_float($val)) {
      $this->fields->add($field, $val);
    }
    // Boolean formats dependent on database type..
    elseif (is_bool($val)) {
      $val = $RESPONSE->datasource->db_value_from_bool($val);
      if (!is_int($val)) $val = "'$val'";
      $this->fields->add($field, $val);
    }
    // Everything else is a quoted, escaped string..
    else {
      $val = trim($val);
      $bits = explode("::", $val);
      $val = $bits[0];
      if ($RESPONSE->multilang && $RESPONSE->mbstring_avail) {
        if (mb_substr($val, 0, 1) == "'") $val = mb_substr($val, 1);
        if (mb_substr($val, -1) == "'") $val = mb_substr($val, 0, strlen($val) - 1);
      }
      else {
        if (substr($val, 0, 1) == "'") $val = substr($val, 1);
        if (substr($val, -1) == "'") $val = substr($val, 0, strlen($val) - 1);
      }
      $val = addslashes($val);
      $this->fields->add($field, "'$val'");
    }
  } // set
  // ....................................................................
  /**
  * Add WHERE clause component
  * This function allows you to add a WHERE clause component. An example might
  * be something like: "AND c.foo='myval'". Either call this once with the whole
  * WHERE cluase string (minus the word "WHERE"), or multiple times with
  * parts of the where clause as in the example above.
  * @param string $where_clause A WHERE clause component, without the "WHERE".
  */
  function where($where_clause) {
    if ($where_clause != "") {
      $this->where->add($where_clause);
    }
  } // where
  // ....................................................................
  /**
  * This is useful when you change some part of the query after it has been
  * executed once, and want it to rebuild the SQL anew before it gets
  * executed again.
  */
  function rebuild() {
    $this->sql = "";
    $this->build();
  } // rebuild
  // ....................................................................
  /**
  * Build the SQL query
  * This takes the various components which have been added to the object
  * and parses them to build the full SQL statement which will be sent
  * to the server. The result is stored in $this->sql.
  * NOTE: this method calls the appropriate database-specific SQL
  *       builder method.
  */
  function build() {
    global $RESPONSE;
    $this->sql = $RESPONSE->datasource->SQL($this);
    return $this->sql;
  } // build
} // sqlquery class

// ----------------------------------------------------------------------
/**
* DB Query class
* This class is the one which executes queries against the
* connected database.
* @package database
*/
class dbquery extends sqlquery {
  /** Number of rows returned after execute */
  var $rowcount = 0;
  /** Number of rows affected by query */
  var $affectedrowcount = 0;
  /** Current row in the query */
  var $rowno = 0;
  /** Current row resource ID */
  var $rid = false;
  /** True if query is valid, post execution */
  var $valid = false;
  /** True if data was returned, after execute */
  var $hasdata = false;
  /** Record last error/info message */
  var $last_errormsg = "";
  // ....................................................................
  /**
  * Constructor
  * Create a new DB Query object.
  * @param string $sql An SQL statement in full
  */
  function dbquery($sql="") {
    $this->sqlquery($sql);
    return $this;
  } // dbquery
  // ....................................................................
  /**
  * Exceute the query
  * If we have an SQL phrase, execute it now. We store
  * the result in this->valid, and also return it. If
  * a transaction is open, update the status.
  * @return bool True if query was executed successfully
  */
  function execute() {
    global $RESPONSE;

    // Head it off at the pass if the system is being run
    // standalone, or there is no defined datasource..
    if (!isset($RESPONSE->datasource) || !$RESPONSE->db_backed) {
      return false;
    }

    global $global_tran;
    $this->rid = false;
    if ($this->sql == "") {
      $this->build();
    }
    if ($this->sql != "") {
      // Execute the query using low-level DB module..
      $this->rid = $RESPONSE->datasource->query($this->sql);
      // Now examine the result..
      if ($this->rid != false) {
        if (preg_match("/(^select|^\(select)/i", $this->sql)) {
          $this->rowcount = $RESPONSE->datasource->numrows($this->rid);
        }
        $this->rowno = 0;
        $this->hasdata = ($this->rowcount > 0);
      }
      else {
        // Log the failed query..
        $db_err = $RESPONSE->datasource->errormessage();
        if ($db_err) $errstr .= " DBSERVER: $db_err";
        $this->last_errormsg = $errstr;
        if (debugging()) {
          debugbr($errstr, DBG_SQL);
        }
        // Set failed status for any open transaction..
        if ($global_tran->open) {
          $global_tran->failed = true;
          $global_tran->failed_msg = $errstr;
        }
      }
    }
    $this->valid = ($this->rid != false);
    return $this->valid;
  } // execute
  // ....................................................................
  /**
  * Set the SQL statement
  * @param string $sql An SQL statement in full
  */
  function set_sql($sql) {
    $this->tidyup();
    $this->sql = $sql;
    return $this;
  } // set_sql
  // ....................................................................
  /**
  * Free resources.
  * Not really necessary, but you might be that fastidious kind of person.
  */
  function tidyup() {
    global $RESPONSE;
    if ($this->rid) {
      $RESPONSE->datasource->freeresult($this->rid);
      $this->clear();
      $this->rowcount = 0;
      $this->affectedrowcount = 0;
      $this->rid = false;
      $this->valid = false;
      $this->hasdata = false;
    }
  } // tidyup
} // dbquery class

// ----------------------------------------------------------------------
/**
* DB Rows class
* Renders a query into data and allows access to the data either
* directly or via the usual get first,last,next,previous cursor
* navigation.
* This class returns data as "rows" which is to say a standard
* array of data. For the associative array version then please
* @see dbrecords
* NOTE: On creation, it executes the query and positions to the
* initial record (defaulted to the first).
* @package database
*/
class dbrows extends dbquery {
  /** An array containing the current DB row */
  var $current_row;
  // ....................................................................
  /**
  * Constructor
  * Create a new DB Rows object.
  * @param string $sql An SQL statement in full
  */
  function dbrows($sql="") {
    $this->dbquery($sql);
    if ($sql != "") {
      $this->execute();
    }
  } // dbrows
  // ....................................................................
  /**
  * Execute query
  * Execute this query. We override the parent method here
  * simply to ensure we are positioned at the first row.
  * @return bool True if query was executed successfully
  */
  function execute() {
    dbquery::execute();
    if ($this->valid) {
      $this->get_first();
    }
    return $this->valid;
  } // execute
  // ....................................................................
  /**
  * Set the SQL statement
  * In this case we re-execute the SQL automatically.
  * @param string $sql An SQL statement in full
  * @return bool True if query was executed successfully
  */
  function set_sql($sql) {
    $this->tidyup();
    $this->sql = $sql;
    return $this->execute();
  } // set_sql
  // ....................................................................
  /**
  * Get row raw
  * Return the given database row from the resultset. This method may
  * be over-ridden in subsequent child classes.
  * @param integer $rowno The row number to return
  * @return array True if row was available
  * @access private
  */
  function get_row_raw($rowno) {
    global $RESPONSE;
    if ($this->rid != false) {
      return $RESPONSE->datasource->fetch_row($this->rid, $rowno);
    }
    else return false;
  } // get_row_raw
  // ....................................................................
  /**
  * Get row
  * Return the given database row from the resultset. Uses the
  * get_row_raw() method applicable to this class.
  * @see get_row_raw()
  * @param integer $rowno The row number to return
  * @return mixed The row if it is available, else returns FALSE.
  */
  function get_row($rowno) {
    if ($this->valid && ($this->rowcount > 0)) {
      if ($rowno > ($this->rowcount - 1)) $rowno = $this->rowcount - 1;
      elseif ($rowno < 0) $rowno = 0;
      $this->current_row = $this->get_row_raw($rowno);
      if ($this->current_row !== false) {
        $this->rowno = $rowno;
        if (debugging()) {
          $errstr = "";
          for($i=0; $i < count($this->current_row); $i++) {
            if ($errstr != "") $errstr .= ", ";
            $errstr .= $this->current_row[$i];
          }
          //$errstr = var_dump($this->current_row);
          debugbr("QDATA: Row $rowno: $errstr", DBG_SQLDATA);
        }
      }
      else {
        debugbr("QDATA: Row $rowno: returned FALSE", DBG_SQLDATA);
      }
    }
    else {
      if (isset($this->current_row)) unset($this->current_row);
      $this->current_row = false;
    }
    return $this->current_row;
  } // get_row
  // ....................................................................
  /**
  * Returns true if the row number exists in the returned resultset.
  * The query has to be valid, and there have to be some rows in it.
  * @param integer $rowno Number of the row, zero (0) is first row
  * @return bool True if the row is present in the current resultset
  */
  function rowexists($rowno) {
    return (
         ($this->valid)
      && ($this->rowcount > 0)
      && ($rowno >= 0)
      && ($rowno <= ($this->rowcount - 1))
      );
  } // rowexists
  // ....................................................................
  /**
  * Refresh the query
  * Re-run the current SQL query. If successful the row will be stored
  * in $this->current_row.
  */
  function refresh() {
    $rowno = $this->rowno;
    $this->execute();
    $this->get_row($rowno);
  } // refresh
  // ....................................................................
  /**
  * Get current row
  * If current query is invalid, try to execute it first, then do a
  * get_first(). If query is then valid, return the current row.
  * @see get_first()
  * @return mixed The row if it is available, else returns FALSE.
  */
  function get_current() {
    if (!$this->valid) {
      $this->execute();
      $this->get_first();
    }
    return $this->current_row;
  } // get_current
  // ....................................................................
  /**
  * Get current row
  * If current query is invalid, try to execute it first, then do a
  * get_first(). If query is then valid, return the current row.
  * @see get_first()
  * @return mixed The row if it is available, else returns FALSE.
  */
  function get_first() {
    if (!$this->valid) $this->execute();
    return $this->get_row(0);
  } // get_first
  // ....................................................................
  /**
  * Get last row
  * If current query is invalid, try to execute it first, then get
  * the last row from the resultset.
  * @return mixed The row if it is available, else returns FALSE.
  */
  function get_last() {
    if (!$this->valid) $this->execute();
    return $this->get_row($this->rowcount - 1);
  } // get_last
  // ....................................................................
  /**
  * Get previous row
  * If current query is invalid, try to execute it first, then get
  * the previous row from the resultset.
  * @return mixed The row if it is available, else returns FALSE.
  */
  function get_previous() {
    if (!$this->valid) $this->execute();
    if ($this->rowno > 0) {
      return $this->get_row($this->rowno - 1);
    }
    else return false;
  } // get_previous
  // ....................................................................
  /**
  * Get next row
  * If current query is invalid, try to execute it first, then get
  * the next row from the resultset.
  * @return mixed The row if it is available, else returns FALSE.
  */
  function get_next() {
    if (!$this->valid) $this->execute();
    if ($this->rowno < ($this->rowcount - 1)) {
      return $this->get_row($this->rowno + 1);
    }
    else return false;
  } // get_next
  // ....................................................................
  /**
  * Return the EOF (end-of-file) indicator for this query. Returns
  * true if no more results can be returned with get_next(), ie. we
  * are at the end of the results set.
  * @return boolean True if we are at the end of the results set.
  */
  function eof() {
    if (!$this->valid) return true;
    else return ($this->rowno >= ($this->rowcount - 1));
  } // eof
} // dbrows class

// ----------------------------------------------------------------------
/**
* DB Records class
* Renders a query into data and allows access to the data either
* directly or via the usual get first,last,next,previous cursor
* navigation.
* This class returns data as an associative array and is thus
* the most useful of all the data access methods. It extends the
* dbrows class, and over-rides the get_row_raw method to retrieve
* data.
* @see dbrows.
* @package database
*/
class dbrecords extends dbrows {
  /**
  * Constructor
  * Create a new DB Records object.
  * @param string $sql An SQL statement in full
  */
  function dbrecords($sql="") {
    $this->dbrows($sql);
  } // dbrecords
  // ....................................................................
  /**
  * Get row raw
  * Return the given database row from the resultset. This over-rides
  * the parent method of the same name and returns an array.
  * @param integer $rowno The row number to return
  * @return array True if row was available
  * @access private
  */
  function get_row_raw($rowno) {
    global $RESPONSE;
    if ($this->rid) {
      return $RESPONSE->datasource->fetch_array($this->rid, $rowno);
    }
    else return false;
  } // get_row_raw
  // ....................................................................
  /**
  * Return whether the name field exists in the resultset, true or false.
  * @param string $fieldname The name of the field to check existence of
  * @return boolean True if the named field exists in the resultset.
  */
  function field_exists($fieldname) {
    global $RESPONSE;
    // Intercept any errant querying in standalone mode..
    if (!$RESPONSE->db_backed) return false;
    if ($this->rid) {
      return isset($this->current_row[$fieldname]);
    }
    else return false;
  } // field_exists
  // ....................................................................
  /**
  * Get field content
  * Return the field content from the current database array (row).
  * Does not provide ANY pre/post-processing.
  * @param string $fieldname The name of the field to return value of
  * @return mixed Value of the named field
  */
  function rawfield($fieldname) {
    global $RESPONSE;
    // Intercept any errant querying in standalone mode..
    if (!$RESPONSE->db_backed) return "";
    if ($this->rid) {
      $value = $this->current_row[$fieldname];
      return $value;
    }
    else return false;
  } // rawfield
  // ....................................................................
  /**
  * Get field content
  * Return the field content from the current database array (row).
  * If the value is a string, then stripslashes is done automatically.
  * @param string $fieldname The name of the field to return value of
  * @return mixed Value of the named field
  */
  function field($fieldname) {
    global $RESPONSE;
    // Intercept any errant querying in standalone mode..
    if (!$RESPONSE->db_backed) return "";
    if ($this->rid) {
      $value = $this->rawfield($fieldname);
      if (is_string($value)) {
        $value = stripslashes($value);
      }
      return $value;
    }
    else return false;
  } // field
  // ....................................................................
  /**
  * Database independent boolean handling. Returns TRUE if the named
  * field in the current row is boolean true according to the rules of the
  * underlying database, else returns FALSE.
  * @param string $fieldname The name of the field to return boolean value of
  * @return boolean True if field contains database-dependent true value
  */
  function istrue($fieldname) {
    global $RESPONSE;
    $value = $this->field($fieldname);
    return $RESPONSE->datasource->bool_from_db_value($value);
  } // istrue
} // dbrecords class

// ----------------------------------------------------------------------
// SPECIFIC DBQUERY TYPES..
// Wrappers which save you specifying some variables, when instantiating
// a new 'dbquery' object that's all..
/**
* DB Select class
* A special case of the dbrecords class.
* @package database
*/
class dbselect extends dbrecords {
  /**
  * Constructor
  * Create a new DB Select object. This is for selecting rows from
  * the database, and returning fields from those rows.
  * @param string $table Table(s) to run select on
  */
  function dbselect($table="") {
    $this->dbrecords();
    $this->type = "SELECT";
    if ($table != "") {
      $this->from($table);
    }
  } // dbselect
} // dbselect class
// ----------------------------------------------------------------------
/**
* DB Delete class
* A special case of the dbquery class. This is for deleting
* rows from the database.
* @package database
*/
class dbdelete extends dbquery {
  /**
  * Constructor
  * Create a new DB Delete object.
  * @param string $table Table to delete rows from.
  */
  function dbdelete($table="") {
    $this->dbquery();
    $this->type = "DELETE";
    if ($table != "") {
      $this->into($table);
    }
  } // dbdelete
} // dbdelete class
// ----------------------------------------------------------------------
/**
* DB tablemod class
* Parent class for classes which only modify a single table. This
* means either update or inserts. This class is provided so we can
* define a common method for sequence definition.
* @package database
* @abstract
*/
class dbtablemod extends dbquery {
  /**
  * Constructor
  * Create a new DB Insert object. This is for inserting
  * a record into the database.
  * @param string $table Table to modify, mandatory parameter.
  */
  function dbtablemod($table) {
    $this->dbquery();
    $this->tables($table);
  }
  // ....................................................................
  /**
  * Set the next sequence value for a column, using either a named
  * sequence or, if that is nullstring or the underlying DB does
  * not support sequences, other means. See the next_sequencevalue()
  * method in the underlying DB module db-xxxx.php.
  */
  function next_sequencevalue($sequencename, $column) {
    global $RESPONSE;
    $nextseq = $RESPONSE->datasource->next_sequencevalue(
               $sequencename,
               $this->tables->listed(),
               $column
               );
    $this->set($column, $nextseq);
  }
}
// ----------------------------------------------------------------------
/**
* DB Insert class
* A special case of the dbtablemod class.
* @package database
*/
class dbinsert extends dbtablemod {
  /**
  * Constructor
  * Create a new DB Insert object. This is for inserting
  * a record into the database.
  * @param string $table Table to insert into
  */
  function dbinsert($table) {
    $this->dbtablemod($table);
    $this->type = "INSERT";
  } // dbinsert
} // dbinsert class
// ----------------------------------------------------------------------
/**
* DB Update class
* A special case of the dbquery class. This is for updating data in
* particular rows in the database.
* @package database
*/
class dbupdate extends dbtablemod {
  /**
  * Constructor
  * Create a new DB Select object.
  * @param string $table Table to update
  */
  function dbupdate($table) {
    $this->dbtablemod($table);
    $this->type = "UPDATE";
  } // dbupdate
} // dbupdate class
// ----------------------------------------------------------------------
/**
* DB seq class
* A class which allows the management and use of sequences.
* @package database
*/
class dbseq extends dbquery {
  // Public
  // Private
  /** The name of the sequence
      @access private */
  var $sequencename = "";
  // ....................................................................
  /**
  * Create a new object to manage a sequence, optionally
  * specifying the sequence name..
  * @param string $sequencename Name of the sequence to manage
  */
  function dbseq($sequencename) {
    $this->sequencename = $sequencename;
    $this->dbquery();
  } // dbseq
  // ....................................................................
  /**
  * Get the next sequence value. We can optionally specify the table and
  * column associated with it. The requirement for these parameters is in fact
  * implementation-specific. If your underlying database does not support
  * named sequences, then you will probably have to nominate the table/column
  * so that the low-level DB access module can do a MAX() to obtain the next
  * value. If it does upport tham then you probably only need the sequence
  * name as specified in the constructor.
  * @param string $table Name of the table associated with this sequence
  * @param string $column Name of the column associated with this sequence
  * @return integer The value of the next integer in this sequence
  */
  function next_sequencevalue($table="", $column="") {
    global $RESPONSE;
    return $RESPONSE->datasource->next_sequencevalue($this->sequencename, $table, $column);
  } // next_sequencevalue
  // ....................................................................
  /**
  * Get the current sequence value.
  * @param string $table Name of the table associated with this sequence
  * @param string $column Name of the column associated with this sequence
  * @return integer The current sequence value
  */
  function current_sequencevalue($table="", $column="") {
    global $RESPONSE;
    return $RESPONSE->datasource->current_sequencevalue($this->sequencename, $table, $column);
  } // current_sequencevalue
  // ....................................................................
  /**
  * Set a sequence value.
  * @param integer $newval New integer value to set sequence to
  * @param string $table Name of the table associated with this sequence
  * @param string $column Name of the column associated with this sequence
  */
  function set_sequencevalue($newval, $table="", $column="") {
    global $RESPONSE;
    return $RESPONSE->datasource->set_sequencevalue($newval, $this->sequencename, $table, $column);
  } // set_sequencevalue
} // dbseq class

// ######################################################################
// Utility functions..
/**
* Execute a DB command
* A wrapper which caters for the 'command' type of SQL
* query where no results are reauired, such as for a
* DELETE or UPDATE, or INSERT etc. Returns true if all
* ok, otherwise returns false.
* @param string $sql An SQL statement in full
* @return bool True if dbcommand succeeded
*/
function dbcommand($sql) {
  $q = new dbquery($sql);
  $res = $q->execute();
  return $res;
} // dbcommand
// ......................................................................
/**
* A wrapper which caters for queries which will return
* a record set identifier for returning data.
* @param string $sql An SQL statement in full
* @return resource Returns a resource ID for the recordset
*/
function dbrecordset($sql) {
  $res = new dbrecords($sql);
  return $res;
} // dbrecordset
// ......................................................................
/**
* A wrapper to get the next sequence value from a named sequence..
* @param string $sequencename Name of the sequence
* @param string $column Name of the column sequence is on
* @param string $table Name of the table column is on
* @return integer The value of the next integer in this sequence
*/
function get_next_sequencevalue($sequencename, $table="", $column="") {
  $seq = new dbseq($sequencename);
  return $seq->next_sequencevalue($table, $column);
} // get_next_sequencevalue
// ......................................................................
/**
* A wrapper to get the current sequence value from a named sequence..
* @param string $sequencename Name of the sequence
* @param string $column Name of the column sequence is on
* @param string $table Name of the table column is on
* @return integer The value of the current integer in this sequence
*/
function get_current_sequencevalue($sequencename, $table="", $column="") {
  $seq = new dbseq($sequencename);
  return $seq->current_sequencevalue($table, $column);
} // get_current_sequencevalue
// ......................................................................
// TRANSACTION Functions
/**
* Start a DB transaction. Alias for begin_transaction()
* @see begin_transaction()
*/
function start_transaction() {
  return begin_transaction();
} // start_transaction
// ......................................................................
/**
* Begin a DB transaction
*/
function begin_transaction() {
  global $global_tran;
  $result = $global_tran->begin();
  return $result;
} // begin_transaction
// ......................................................................
/**
* Return DB transaction failure status.
* @return bool True if transaction failed
*/
function transaction_failed() {
  global $global_tran;
  return $global_tran->failed;
} // transaction_failed
// ......................................................................
/**
* Return DB transaction success status.
* @return bool True if transaction succeeded
*/
function transaction_succeeded() {
  return !transaction_failed();
} // transaction_succeeded
// ......................................................................
/**
* Return DB transaction open status.
* @return bool True if transaction already open
*/
function transaction_open() {
  global $global_tran;
  return $global_tran->open;
} // transaction_open
// ......................................................................
/**
* Commit a DB transaction
* @return bool True if transaction committed
*/
function commit() {
  global $global_tran;
  $result = $global_tran->commit();
  return $result;
} // commit
// ......................................................................
/**
* Rollback a DB transaction
* @return bool True if transaction rolled back
*/
function rollback() {
  global $global_tran;
  $result = $global_tran->rollback();
  return $result;
} // rollback

// ----------------------------------------------------------------------
// LOCKING
/**
* Take out a lock on a table or tables, in a given mode. The mode string
* is database-specific and will vary according to the implementation
* of its locking scheme.
* @param string $tablelist List of tables to lock, comma-delimited
* @param string $mode Databes-specific locking-mode or type
*/
function lock($tablelist, $mode) {
  global $RESPONSE;
  return $RESPONSE->datasource->lock($tablelist, $mode);
} // lock

// ----------------------------------------------------------------------
// DATABASE SELECTION
/**
* Global function to set the selected database to a new database. All
* subsequent queries will then act on that database.
* @param string $db_name Name of database to select - nullstring for 'default'.
*/
function select_database($db_name="") {
  global $RESPONSE;
  return $RESPONSE->select_database($db_name);
}
?>