#!/bin/bash
# Upgrade/patch a POSTGRESQL database for an Axyl website.

# NOTE: This is not normally run standalone. The main DB upgrade
# script 'install/upgrade-axyl-databases.sh normally runs this.
#
# THIS SCRIPT REQUIRES THE AXYL COMMON FUNCTIONS TO BE DEFINED
# To do this source ${AXYL_HOME}/install/axyl-common.sh from a
# containing shell script, which then sources this one.
#
# DATABASE UPGRADE PROCESSING
# Upgrades an application database using a patch file containing SQL
# statements. This script is designed to be included by the 'postinst'
# script of the AXYL package, and also by the above-mentioned upgrader
# script.
#
# It relies on there being a special control table in the database
# called 'app_control' which contains a text field 'app_version'. The
# version of this upgrade is then compared and SQL patches sought
# in $SOUNZ_HOME/db/patches. This script will apply multiple patches to
# arrive at the new version, if required.
#
# Files in 'db/upgrade/' must adhere to a strict naming format if they are
# to be automatically picked up by the postinst patch-applier. The format
# is:
#     upgrade_P.Q.R_to_X.Y.Z.sql
#
# where:
#     P.Q.R is the version that the last patch upgraded to 
#     X.Y.Z is the new version this patch will upgrades the schema to
#
# The patch applier will apply mutiple patches to span more than one version
# in the sequence, if required.
#
# In general, there is not a database patch for each version upgrade of the
# package. That's just fine, and you can just create a database patch just for
# the versions which do need one, and the patcher will use it each time.
#
# Eg. if the last patch file was for the upgrade 2.3.0 --> 2.3.1, but there
# is no patch for 2.3.1 --> 2.3.2, or 2.3.2 --> 2.3.3, and then for the
# next upgrade of 2.3.3 --> 2.3.4, there IS a patch file, you would just
# name your patch file as: 'upgrade_2.3.3_to_2.3.4.sql' and the patcher
# will do the right thing.
#
# In other words, just include patch files for the upgrades which need 'em!
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
DBTYPE="postgresql"
SCRIPTNAME="upgrade-db.sh $DBTYPE"

# INCOMING PARAMETERS
#  $1  - DB_NAME
#  $2  - DB_USER
#  $3  - DB_PASSWD ('none' means a blank password)
#  $4  - DB_HOST ('direct' means a local database server)
#  $5  - DB_PORT
#  $6  - AXYL_VERSION (Axyl version, '' or 'n.n.n'
#  $7  - MODE ('upgrade' or 'stamp-only')
#  $8  - DB_DETECT_MODE ('interactive' (default) or 'auto')
DB_NAME=""
DB_USER=""
DB_PASSWD=""
DB_HOST=""
DB_PORT=5432
AXYL_VERSION="unknown"
MODE=upgrade
DB_DETECT_MODE=interactive

# Echo and log messages. We use the common Axyl function 'logit()' to send
# our log messages to this globally defined $LOGFILE. It also logs to
# syslog as well.
LOGFILE=${AXYL_LOGS}/upgrade-db.log
[ ! -f $LOGFILE ] && >$LOGFILE

POS=0
while [ $# -gt 0 ] ; do
	POS=`expr $POS + 1`
	case $POS in
		1)  DB_NAME=$1
			;;
		2)  DB_USER=$1
			;;
		3)  DB_PASSWD=$1
			;;
		4)  DB_HOST=$1
			;;
		5)  DB_PORT=$1
			;;
		6)  AXYL_VERSION=$1
			;;
		7)  MODE=$1
			;;
		8)  DB_DETECT_MODE=$1
			;;
	esac
	shift
done
logit "db=$DBNAME user=$DBUSER host=$DBHOST port=$DBPORT axver=$AXYL_VERSION mode=$MODE detectmode=$DB_DETECT_MODE"

# Cater for standalone running of this script. Normally we are called
# with AXYL_HOME et al already defined..
if [ -z $AXYL_HOME ] ; then
	CONFDIR=/etc/axyl
	CONF=${CONFDIR}/axyl.conf
	if [ ! -f $CONF ] ; then
		echo "Axyl configuration file $CONF not found!"
		exit 2
	else
		. $CONF
		if [ ! -d $AXYL_HOME ] ; then
			echo "FATAL: the Axyl root directory '$AXYL_HOME' does not exist."
			exit 6
		fi
		. ${AXYL_HOME}/install/axyl-common.sh
	fi
fi

# We require Postgres to be locally installed, at least as postgresql-client
# even if no servers are created locally. This is based on the standard
# Debian location, with a few likely fallbacks.

# Detect database, and set up database vars. This set up the following
# variables:
# PG_MULTI_CLUSTER     # Eg. '8.1/main' Postgres version and cluster
# PG_VERSION           # Version of the database eg. '8.1'
# PG_VERSION_SUFFIX    # Version suffix eg. '-8.1'
# PG_BIN               # Path to the Postgres binary files
# PG_CONF              # Path to the Postgre configuration files

. ${AXYL_HOME}/db/postgres/detect-db.sh $DB_DETECT_MODE

# Now set paths to our executables
PSQL=${PG_BIN}/psql
PGDUMP=${PG_BIN}/pg_dump
AXYL_DB_VERSION=

# Upgrade the database contents (tables and data)
if [ -x $PSQL ] ; then
	# Optional host settings for remotely accessed databases..
	HOSTOPTS=""
	[ "$DB_HOST" != "direct" ] && HOSTOPTS="--host $DB_HOST --port $DB_PORT"
	
	# Make sure our database connection user is present..
	USER=`$PSQL --tuples-only --username postgres --dbname template1 $HOSTOPTS --command "SELECT usename FROM pg_user WHERE usename='$DB_USER'" | tr -d ' '`
	if [ "$USER" != "$DB_USER" ] ; then
		logit "creating database user $DB_USER"
		$PSQL --username postgres --dbname template1 $HOSTOPTS --command "CREATE USER $DB_USER WITH PASSWORD '$DB_PASSWD' CREATEDB"
	fi

	# Check ax_control is present, and in latest format
	create_ax_control=0
	AX_CONTROL=`$PSQL --tuples-only --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "SELECT relname FROM pg_class WHERE relname='ax_control'" | tr -d ' '`
	if [ "$AX_CONTROL" = "ax_control" ] ; then
		AX_CHK=`$PSQL --tuples-only --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "SELECT pga.attname FROM pg_class pgc,pg_attribute pga WHERE pgc.relname='ax_control' AND pga.attname='app_version' AND pga.attrelid=pgc.oid" | tr -d ' '`
		if [ "$AX_CHK" != "app_version" ] ; then
			$PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "DROP TABLE ax_control" >>$LOGFILE 2>&1
			logit "dropped legacy format ax_control - will recreate"
			create_ax_control=1
		fi
	else
		create_ax_control=1
	fi			

	# Create ax_control if required
	if [ $create_ax_control -eq 1 ] ; then
		logit "creating ax_control table"
		LAST_DB_PATCH="000000000000-000100000000"
		$PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "CREATE TABLE ax_control (app_version text, last_db_patch text, updated_at timestamp)" >>$LOGFILE 2>&1
		logit "stamping database with version $AXYL_VERSION"
		$PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "INSERT INTO ax_control (app_version,last_db_patch,updated_at) VALUES('$AXYL_VERSION','$LAST_DB_PATCH',CURRENT_TIMESTAMP)" >>$LOGFILE 2>&1
		logit "no patching has been done - please apply any patches manually this time around"
		logit "future upgrades will apply patches automatically"
	fi
			
	# Should have ax_control now, but must check the above went ok
	AX_CONTROL=`$PSQL --tuples-only --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "SELECT relname FROM pg_class WHERE relname='ax_control'" | tr -d ' '`
	if [ "$AX_CONTROL" = "ax_control" ] ; then
		# Acquire application version from ax_control table
		AXYL_DB_VERSION=`$PSQL --tuples-only --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "SELECT app_version FROM ax_control" | tr -d ' '`
		if [ "$MODE" != "stamp-only" -a "$AXYL_DB_VERSION" != "" ] ; then
			if [ "$AXYL_DB_VERSION" = "$AXYL_VERSION" ] ; then
				logit "database ${DB_NAME} is up to date."
			else
				# Ok, we know the registered version, and we know the Axyl installation
				# version, so let's see if we have a patch file series for this upgrade.
				# No patch files, no upgrade.
				logit "current Axyl database version is registered as $AXYL_DB_VERSION"
				gotpatches=0
				skippatches=0
				LAST_DB_PATCH=`$PSQL --tuples-only --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "SELECT last_db_patch FROM ax_control" | tr -d ' '`
				if [ "$LAST_DB_PATCH" = "" ] ; then
					logit "no patch reference, will skip patches this time around"
					skippatches=1
				fi
				UPGRADEROOT=${AXYL_HOME}/db/postgres/upgrade
				cd ${UPGRADEROOT}
				patchSQL=`tempfile --prefix dbupgrade` 
				patchfiles=`${AXYL_HOME}/install/get_patchfiles.pl $UPGRADEROOT $LAST_DB_PATCH`
				logit $patchfiles
				for patchfile in $patchfiles ; do
					chk=`echo $patchfile | cut -d'_' -f1`
					if [ "$chk" = "upgrade" ] ; then
						patchnos=`echo $patchfile | sed "s;.sql;;"`
						if [ "$patchnos" != "" ] ; then
							fromver=`echo $patchnos | cut -d'_' -f2`
							nextver=`echo $patchnos | cut -d'_' -f4`
							if [ $skippatches -eq 1 ] ; then
								logit "skipping patch $fromver --> $nextver"
							else
								logit "adding patch $fromver --> $nextver"
								cat $patchfile >> $patchSQL
							fi
							gotpatches=1
						fi
					else
						LAST_DB_PATCH=$patchfile
					fi
				done
				if [ $gotpatches -eq 1 ] ; then
					if [ $skippatches -eq 0 ] ; then
						tell "dumping database before applying patches"
						dumpdir=${AXYL_DATA}/backup
						if [ ! -d $dumpdir ] ; then
							mkthisdir $dumpdir
							chown ${AXUSER}:${AXUSER} $dumpdir
						fi
						tstamp=`date +"%Y%m%d_%H%M%S"`
						dumpfile=${dumpdir}/${DB_NAME}_${AXYL_DB_VERSION}_${tstamp}.sql.dump
						sudo su $AXUSER -c "$PGDUMP --username $DB_USER --no-owner $DB_NAME >$dumpfile"
						tell "database '$DB_NAME' dumped into $dumpfile"
						logit "database '$DB_NAME' dumped into $dumpfile"
						logit "patching $DB_NAME $AXYL_DB_VERSION --> $AXYL_VERSION"
						$PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --file $patchSQL >>$LOGFILE 2>&1
						if [ $? -ne 0 ] ; then
							logit "errors occurred during the patch process"
						fi
						logit "patches were applied."
					fi
					$PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "UPDATE ax_control SET last_db_patch='$LAST_DB_PATCH'" >>$LOGFILE 2>&1
					logit "database patch record updated"
				else
					tell "no patches to apply"
					logit "no patches to apply"
				fi
			fi

			# Leave the application at the correct version for next time..
			$PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "UPDATE ax_control SET app_version='$AXYL_VERSION',updated_at=CURRENT_TIMESTAMP" >>$LOGFILE 2>&1
			logit "database version record was updated to $AXYL_VERSION"
			# Remove temporary patch file..
			rm -f $patchSQL
		else
			# All we do in this case is assume everything is up to date..
			if [ "$MODE" = "stamp-only" ] ; then
				logit "stamping database $DB_NAME with version $AXYL_VERSION"
			else
				logit "no version data found.. checking record exists"
			fi
			N=`$PSQL --tuples-only --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "SELECT COUNT(*) FROM ax_control" | tr -d ' '`
			if [ $N -eq 0 ] ; then
				logit "no record present. Inserting one with version $AXYL_VERSION"
				LAST_DB_PATCH="000000000000-000100000000"
				$PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "INSERT INTO ax_control (app_version,last_db_patch,updated_at) VALUES('$AXYL_VERSION','$LAST_DB_PATCH',CURRENT_TIMESTAMP)" >>$LOGFILE 2>&1
			else
				logit "record exists, assuming $DB_NAME is up to date and just updating version record to $AXYL_VERSION"
				$PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "UPDATE ax_control SET app_version='$AXYL_VERSION',updated_at=CURRENT_TIMESTAMP" >>$LOGFILE 2>&1
			fi
		fi
	else
		logit "no ax_control table could be found - no upgrade is possible"
	fi

else
	logit "postgres client binaries (psql) not found."
	exit 0
fi

tell "the database upgrade log is in $LOGFILE"
	
# END