<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    catalog-defs.php                                        */
/* Author:      Paul Waite                                              */
/* Description: Handle media catalog items and lists.                   */
/*                                                                      */
/* ******************************************************************** */
/** @package catalog */

/** indexing/searching */
include_once("search-lucene-defs.php");
include_once("search-index-defs.php");
include_once("search-query-defs.php");
/** Filesystem access */
include_once("file-defs.php");

// Definitions
define("STD_THUMB_WIDTH",  "90");
define("STD_THUMB_HEIGHT", "68");

// -----------------------------------------------------------------------
/**
* A class which contains additional versions of the physical media associated
* with an item. If the main item is an image, for example, it may also
* need to have a version of it named 'thumb' which is the thumbnail
* version of the main image. A movie catalogitem might need a short clip
* version of itself called 'movieclip' etc.
* @package catalog
*/
class catalogitem_version {
  var $filepath = "";           // Website URL path to the item (with leading '/')
  var $physicalpath = "";       // Full physical path of image on the filesystem
  var $fileexists = false;      // True if file exists at filepath
  var $width = 0;               // Pixel width of this item
  var $height = 0;              // Pixel height of this item
  
  function catalogitem_version() { }
  function set_paths($physicalpath, $filepath) {
    $this->physicalpath = $physicalpath;
    $this->filepath = $filepath;
  }
  function set_size($width, $height) {
    $this->width = $width;
    $this->height = $height;
  }
} // catalogitem_version class

// -----------------------------------------------------------------------
/**
* A class which encpasulates an item which can be in the catalog.
* @package catalog
*/
class catalogitem extends RenderableObject {
  var $cat_id;                  // Unique catalog record key for the catalog item
  var $cat_name = "";           // Name of this catalog item
  var $cat_desc = "";           // Full description of catalog item
  var $mime_type = "";          // Mime-type of catalog item
  var $mime_category = "";      // Mime category eg: document, movie, audio etc.
  var $keywords = "";           // Any keywords associated with it
  var $category = "";           // Optional user-defined category for item
  var $filesize = "";           // Size of catalog item in bytes
  var $filepath = "";           // Website URL path to the item (with leading '/')
  var $physicalpath = "";       // Full physical path of image on the filesystem
  var $fileexists = false;      // True if file exists at filepath
  var $versions = array();      // Other versions of this catalog item
  var $width = 0;               // Pixel width of this item
  var $height = 0;              // Pixel height of this item
  var $uploaded_ts = 0;         // Unix timestamp of datetime item was added to catalog
  var $uploaded = "";           // Datetime uploaded, NICE_FULLDATETIME format
  var $errmsg = "";             // Contains error message if invalid
  var $valid = false;           // True if item was retreived successfully
  var $newcat = false;          // True if we just created this story

  // Allowed versions for a catalog item
  var $AllowedVersions = array(
    "thumb"     => "image|90|68",
    "medium"    => "image|160|120",
    "flash"     => "image|320|240",
    "movieclip" => "movie|320|240",
    "soundclip" => "audio||"
    );

  // .....................................................................
  /** Constructor
  * @param integer $id Optional unique catalog item ID
  */
  function catalogitem($id=false) {
    // The all-important catalog ID..
    $this->cat_id = $id;
    if (!$id) {
      $this->newcat = true;
      $this->valid = true;
    }
    // Further processing for existing items..
    if (!$this->newcat) {
      // Attempt to get the catlog item
      $this->get();
    }

  } // catalogitem
  // .....................................................................
  /** Get current or nominated catalog item definition from the database.
  * @param integer $id Optional unique catalog item ID to get
  */
  function get($id=false) {
    global $RESPONSE;
    $res = false;
    if ($id) $this->cat_id = $id;
    if ($this->cat_id) {
      $catQ = dbrecordset("SELECT * FROM ax_catalog WHERE cat_id=$this->cat_id");
      if ($catQ->hasdata) {
        $this->cat_name      = $catQ->field("cat_name");
        $this->cat_desc      = $catQ->field("cat_desc");
        $this->mime_type     = $catQ->field("mime_type");
        $this->mime_category = $catQ->field("mime_category");
        $this->keywords      = $catQ->field("keywords");
        $this->category      = $catQ->field("category");
        $this->filesize      = $catQ->field("filesize");
        $this->filepath      = $catQ->field("filepath");
        $this->width         = $catQ->field("width");
        $this->height        = $catQ->field("height");
        $this->uploaded_ts   = datetime_to_timestamp($catQ->field("upload_timestamp"));
        $this->uploaded      = timestamp_to_displaydate(NICE_FULLDATETIME, $this->uploaded_ts);
        $this->physicalpath  = $RESPONSE->site_docroot . $this->filepath;
        $this->fileexists    = file_exists($this->physicalpath);

        // Populate version details, if any versions exist.
        $this->locate_versions();
        $res = true;
        $this->newcat = false;
      }
      else {
        $this->errmsg = "No record of catalog item: $this->cat_id";
      }
    }
    else {
      $this->errmsg = "No catalog ID was given";
    }
    // Did we succeed..?
    $this->valid = $res;
    return $res;
  } // get
  // .....................................................................
  /**
   * Go and look for possible versions out there on disk. Versions are
   * additional versions of the main media, for example 'thumb' for an
   * image thumbnail version. These are just file-named after the main
   * file, but with the version name (eg. 'thumb') added into the name.
   * If found, then we insert the version in our local array.
   */
  function locate_versions() {
    global $RESPONSE;
    $dir  = dirname($this->filepath);
    $name = get_file_stem($this->filepath);
    $extn = get_file_extn($this->filepath);
    $this->versions = array();
    if (count($this->AllowedVersions) > 0) {
      foreach ($this->AllowedVersions as $ver => $info) {
        debugbr("AllowedVersion: $ver => $info");
      }
    }
    else {
      debugbr("AllowedVersions empty!");
    }
    foreach ($this->AllowedVersions as $version => $versioninfo) {
      $verbits = explode("|", $versioninfo);
      $mime_category = $verbits[0];
      if ($this->mime_category == $mime_category) {
        // Derive location details by naming standard
        $verpath = "$dir/$name.$version.$extn";
        $verphys = $RESPONSE->site_docroot . $verpath;
        if (file_exists($verphys)) {
          $ver = new catalogitem_version();
          $ver->set_paths($verphys, $verpath);
          $ver->fileexists = true;
          $ver->width = $verbits[1];
          $ver->height = $verbits[2];
          $this->versions[$version] = $ver;
        }
      }
    } // foreach
  } // get_versions
  // .....................................................................
  /** Save current catalog item definition to the database. Inserts
  * if brand new, else performs an update.
  */
  function save() {
    global $RESPONSE;
    $res = false;
    if ($this->newcat || ($this->cat_id && $this->valid)) {
      if ($this->newcat) {
        $this->cat_id = get_next_sequencevalue("seq_cat_id", "ax_catalog", "cat_id");
        $this->uploaded_ts = time();
        $this->uploaded = timestamp_to_displaydate(NICE_FULLDATETIME, $this->uploaded_ts);
        $cup = new dbinsert("ax_catalog");
        $cup->set("cat_id", $this->cat_id);
        $this->newcat = false;
      }
      else {
        $cup = new dbupdate("ax_catalog");
        $cup->where("cat_id=$this->cat_id");
      }
      $cup->set("cat_name",         $this->cat_name);
      $cup->set("cat_desc",         $this->cat_desc);
      $cup->set("mime_type",        $this->mime_type);
      $cup->set("mime_category",    $this->mime_category);
      $cup->set("keywords",         $this->keywords);
      $cup->set("category",         $this->category);
      $cup->set("filesize",         $this->filesize);
      $cup->set("filepath",         $this->filepath);
      $cup->set("width",            $this->width);
      $cup->set("height",           $this->height);
      $cup->set("upload_timestamp", timestamp_to_datetime($this->uploaded_ts));
      $res = $cup->execute();
      if ($res) {
        // Index to search engine..
        $this->index();
        // Update status of physical file..
        $this->physicalpath = $RESPONSE->site_docroot . $this->filepath;
        $this->fileexists = file_exists($this->physicalpath);
      }
    }
    return $res;
  } // save
  // .....................................................................
  /**
  * Remove the catalog item from the database and disk. This method
  * normally tries to remove the physical file first, and if that succeeds
  * it removes the database record. If $deletefile is false then the file
  * will be left and only the DB record deleted.
  * @param boolean $deletefile If true the physical file will be deleted
  * @return boolean True if the operation succeeded, else false.
  */
  function delete($deletefile=true) {
    global $RESPONSE, $IMAGESDIR;
    $ok = true;
    if ($this->cat_id && $this->valid) {
      // Avoid deleting images which are part of the standard issue..
      if (substr($this->filepath, 0, strlen($IMAGESDIR)) != $IMAGESDIR) {

        // Now deal to the physical file..
        if ($deletefile) {
          if (file_exists($this->physicalpath)) {
            $ok = (is_writeable($this->physicalpath) && unlink($this->physicalpath));
            if ($ok) {
              $this->delete_all_versions();
            }
          }
        }

        // Delete the database record
        if ($ok) {
          $dcat = new dbdelete("ax_catalog");
          $dcat->where("cat_id=$this->cat_id");
          $ok = $dcat->execute();
          if ($ok) {
            // Remove from search engine index..
            $this->unindex();
            $this->valid = false;
          }
        }
      }
    }
    return $ok;
  }  // delete
  // .....................................................................
  /**
  * Remove the catalog item from the database only. This is just a wrapper
  * for a call to delete() with the boolean set false for deleting the
  * physical file.
  * @return boolean True if the operation succeeded, else false.
  */
  function detach() {
    return $this->delete(false);
  }  // detach
  // .....................................................................
  /**
  * Index this catalog item to the search engine.
  * If it exists already, index entry for this item is replaced.
  */
  function index() {
    global $SE_AVAILABLE;
    if ($this->valid && $SE_AVAILABLE) {
      $I = new searchengine_indexer();
      // Make sure these fields are stored..
      $I->define_field("catname", "Text", STORED);
      $I->define_field("category", "Text", STORED);
      $I->define_field("mimecat", "Text", STORED);
      $I->define_field("uploaded", "Date", STORED);
      $I->define_field("path", "text", STORED);
      $I->define_field("width", "text", STORED);
      $I->define_field("height", "text", STORED);
      // Index the fields..
      $I->index_field("Id",       "CAT_" . $this->cat_id);
      $I->index_field("keywords", $this->keywords);
      $I->index_field("catname",  $this->cat_name);
      $I->index_field("category", $this->category);
      $I->index_field("mimecat",  $this->mime_category);
      $I->index_field("uploaded", $this->uploaded_ts);
      $I->index_field("path",     $this->filepath);
      $I->index_field("width",    $this->width);
      $I->index_field("height",   $this->height);
      // Index the content..
      $allcontent[] = $this->cat_name;
      $allcontent[] = $this->cat_desc;
      $allcontent[] = $this->keywords;
      $I->index_content("CAT_" . $this->cat_id, implode(" ", $allcontent));
      $I->execute();
    }
  } // index
  // .....................................................................
  /** Remove this catalog item from the search engine index. */
  function unindex() {
    global $SE_AVAILABLE;
    if ($this->valid && $SE_AVAILABLE) {
      $UI = new searchengine_unindexer();
      $UI->unindex("CAT_" . $this->cat_id);
      $UI->execute();
    }
  } // unindex
  // .....................................................................
  /**
  * Return the keyinfo string for this catalog item. This is encoded
  * as follows, and can be used for such things as key values in selects:
  *   'cat_id|filepath|width|height'
  */
  function keyinfo() {
    $info = array();
    if ($this->valid) {
      $info[] = $this->cat_id;
      $info[] = $this->filepath;
      $info[] = $this->width;
      $info[] = $this->height;
    }
    return implode("|", $info);
  } // keyinfo
  // .....................................................................
  /** Return the catalog item as a clickable icon.
  * @param boolean $autostart Movies etc. if true start automatically
  */
  function AsIcon($showcontrols=false, $autostart=false, $loop=false) {
    $icon = "";
    $tooltip = basename($this->filepath);
    // Set commonsense values for movies..
    if (!$showcontrols) {
      $autostart = true;
      $loop = true;
    }
    switch ($this->mime_category) {
      case "movie":
        $media = new MediaObject($this->filepath, $this->width, $this->height, $autostart, $loop, $showcontrols);
        $media->AsIcon($tooltip);
        $icon = $media->render();
        break;
      case "audio":
        $media = new MediaObject($this->filepath, $this->width, $this->height, $autostart, $loop, $showcontrols);
        $media->AsIcon($tooltip);
        $icon = $media->render();
        break;
      case "flash":
        $media = new FlashObject($this->filepath, $this->width, $this->height, $autostart, $loop);
        $media->AsIcon($tooltip);
        $icon = $media->render();
        break;
      case "document":
        $media = new DocumentObject($this->filepath, $this->width, $this->height);
        $media->AsIcon($tooltip);
        $icon = $media->render();
        break;
      case "image":
        $media = new img($this->filepath, $this->cat_name);
        $media->AsIcon($tooltip);
        $icon = $media->render();
        break;
      default:
        $icon = $this->AsLink();
    } // switch
    return $icon;
  } // AsIcon
  // .....................................................................
  /** Return the catalog item as image, a clickable icon, or otherwise a link.
  * @param boolean $autostart Movies etc. if true start automatically
  */
  function Insitu($showcontrols=false, $autostart=false, $loop=false) {
    $catmedia = "";
    $tooltip = basename($this->filepath);
    // Set commonsense values for movies..
    if (!$showcontrols) {
      $autostart = true;
      $loop = true;
    }
    switch ($this->mime_category) {
      case "movie":
        $media = new MediaObject($this->filepath, $this->width, $this->height, $autostart, $loop, $showcontrols);
        $catmedia = $media->render();
        break;
      case "flash":
        $media = new FlashObject($this->filepath, $this->width, $this->height, $autostart, $loop);
        $catmedia = $media->render();
        break;
      case "image":
        $media = new img(
          $this->filepath,
          str_replace(" ", "", $this->cat_name),
          ($this->cat_desc != "" ? $this->cat_desc : $this->cat_name)
          );
        $catmedia = $media->render();
        break;
      default:
        $catmedia = $this->AsIcon();
    } // switch
    return $catmedia;
  } // Insitu
  // .....................................................................
  /** Return the catalog item as its thumbnail if it is an image, else
   * as an icon or link depending on the type. If the image does not have
   * an existing thumbnail, one will be created to the width/height
   * specified. If the thumbnail exists and its size differs from the
   * passed-in size, and 'recreate' is true, the thumbnail will be rebuilt
   * to the new size.
   * @param integer $width Width of new thumbnail, if it is created
   * @param integer $height Width of a new thumbnail, if created
   * @param boolean $recreate Recreate if thumbnail size differs
   */
  function AsThumbnail($width=STD_THUMB_WIDTH, $height=STD_THUMB_HEIGHT, $recreate=false) {
    $catmedia = "";
    $tooltip = basename($this->filepath);
    // Set commonsense values for movies..
    switch ($this->mime_category) {
      case "image":
        if (!isset($this->versions["thumb"])) {
          $this->make_version("thumb", $width, $height);
        }
        $thumb = $this->versions["thumb"];
        if ($thumb->fileexists) {
          $media = new img(
            $thumb->filepath,
            str_replace(" ", "", $this->cat_name),
            ($this->cat_desc != "" ? $this->cat_desc : $this->cat_name)
            );
          if ( ($media->width != $width || $media->height != $height) && $recreate) {
            $media = $this->make_version("thumb", $width, $height);
          }
        }        
        if (is_object($media)) {
          $catmedia = $media->render();
        }
        break;
      default:
        $catmedia = $this->Insitu();
    } // switch
    return $catmedia;
  } // AsThumbnail
  // .....................................................................
  /**
   * Creates all allowed versions of this current media item. If the
   * version already exists, it will not recreate it unless the flag
   * 'recreate' is set to true.
   * @param boolean $recreate Force re-creation of existing versions
   */
  function make_all_versions($recreate=false) {
    foreach ($this->AllowedVersions as $version => $versioninfo) {
      $info = explode("|", $versioninfo);
      $mimecat = $info[0];
      if ($mimecat == $this->mime_category) {
        if (!isset($this->versions[$version]) || $recreate) {
          $width = $info[1];
          $height = $info[2];
          $this->make_version($version, $width, $height, $recreate);
        }
      }
    }
  } // make_all_vesions
  // .....................................................................
  /**
   * Deletes all existing versions of this current media item.
   */
  function delete_all_versions() {
    foreach ($this->AllowedVersions as $version => $versioninfo) {
      $info = explode("|", $versioninfo);
      $mimecat = $info[0];
      if ($mimecat == $this->mime_category) {
        if (!isset($this->versions[$version])) {
          $ver = $this->versions[$version];
          if (file_exists($ver->physicalpath)) {
            if (unlink($ver->physicalpath)) {
              unset($this->versions[$version]);
            }
          }
        }
      }
    }
  } // delete_all_vesions
  // .....................................................................
  /**
   * Make a version of this catalog item to the width x height given
   * by the incoming parameters. Currently we only support mime-category
   * 'image' for this method. Returns the media object when done. The
   * default mode is to only create missing versions. If you want to
   * recreate then set $recreate to true.
   * @param string Type of version to create: 'thumb', 'medium' or 'flash'
   * @param integer $dest_width Width of new version, if it is created
   * @param integer $dest_height Width of a new version, if created
   * @param boolean $recreate Force re-creation of existing version
   * @return mixed The 'img' object of the new version, or false
   */
  function make_version(
    $version="thumb",
    $dest_width=0,
    $dest_height=0,
    $recreate=false)
  {
    global $RESPONSE;
    
    // Initialise return value to false
    $new_version = false;

    if (isset($this->AllowedVersions[$version])) {
      // Process width settings - set defaults if not specified
      $verinfo = explode("|", $this->AllowedVersions[$version]);
      if ($dest_width == 0) {
        $dest_width = $verinfo[1];
      }
      if ($dest_height == 0) {
        $dest_height = $verinfo[2];
      }
      
      // Derive a destination for this version
      $dir  = dirname($this->filepath);
      $name = get_file_stem($this->filepath);
      $extn = get_file_extn($this->filepath);
      $verpath = "$dir/$name.$version.$extn";
      $verphys = $RESPONSE->site_docroot . $verpath;
      if ($recreate || !file_exists($verphys)) {
        if ($this->fileexists) {
          $org_width  = $this->width;
          $org_height = $this->height;
          
          switch ($this->mime_category) {
            case "image":
              // pick resize which makes the image larger than target size
              $ratio = $org_width / $org_height;
              $test_height = $dest_width / $ratio;
              $test_width  = $dest_height * $ratio;
              $resize_to = "";
              if ($test_height >= $dest_height) {
                  $resize_to = $dest_width . "x";
              }
              elseif ($test_width >= $dest_width) {
                $resize_to = "x" . $dest_height;
              }
              if ($resize_to != "") {
                // create the image in /tmp and then move in place
                $tmp_dest = tempnam("/tmp","make_thumb");
                $convcmd = "convert -resize $resize_to -crop $scale+0+0 $this->physicalpath jpg:$tmp_dest";
                if ( system($convcmd) == 0 ) {
                  system("chmod +r $tmp_dest");
                  if ( system("mv $tmp_dest $verphys") == 0 ) {
                    // get final image to return
                    $new_version = new img(
                                    $verpath,
                                    str_replace(" ", "", $this->cat_name),
                                    ($this->cat_desc != "" ? $this->cat_desc : $this->cat_name)
                                    );
                    if ($new_version->physfileexists) {
                      // Insert into our local versions array
                      $ver = new catalogitem_version();
                      $ver->set_paths($verphys, $verpath);
                      $ver->fileexists = true;
                      $this->versions[$version] = $ver;
                      debugbr("CI VERSION CREATED: at [$verphys]", DBG_DEBUG);
                    }
                    else {
                      debugbr("new version file not found at $verphys", DBG_DEBUG);
                    }
                  }
                  else {
                    unlink($tmp_dest);
                    debugbr("error moving $tmp_dest to $verphys", DBG_DEBUG);
                  }
                }
                else {
                  debugbr("error making image version $verpath ($scale)", DBG_DEBUG);
                }
              }
              break;
          } // switch
        }
      }
    }
    return $new_version;
  } // make_version
  // .....................................................................
  /** Return the catalog item as a clickable hyperlink. */
  function AsLink() {
    $catlink = new anchor($this->filepath, $this->cat_name);
    $catlink->set_linkover_text( basename($this->filepath) );
    return $catlink->render();
  } // AsLink
  // .....................................................................
  /**
  * Create this catalog item from a media file on disk. The media item
  * should be located at a physical path on disk somewhere. It will be grabbed,
  * moved to a new location, and the item record saved to the DB.
  * NOTE: this may be an existing catalogitem, OR a newly created one. This
  * is not determined by this routine, but must be set up before calling this
  * method. The save() method then does whatever is necessary.
  * @param string $srcpath The full physical filesystem path to the source media
  * @param string $destpath The full physical path of where to move the media
  * @param string $filepath The relative path of the media, to save on DB record
  * @param string $mimetype The mime type of the file, eg: "image/jpeg"
  * @param string $name Media file associated name
  * @param string $category Media file associated category
  * @param string $desc Media file full description
  * @param string $keywords Media file associated keywords
  * @return boolean True if all ok, else false
  */
  function create(
      $srcpath, $destpath, $filepath,
      $mimetype="",
      $width=0, $height=0,
      $name="", $category="", $desc="", $keywords=""
  ) {
    global $RESPONSE;

    // Valid flag..
    $this->valid = false;

    // Move file to destination & create our catalog record..
    if (is_readable($srcpath)) {

      // Determine mime-type
      if ($mimetype == "") {
        $mimetype = mimetype_from_filename($srcpath);
      }
      // Determine mime-category..
      $mime_category = mimetype_category($mimetype);

      // Size of this file..
      $filesize = filesize($srcpath);

      // Image metrics..
      if ($mime_category == "image") {
        $imginfo = getimagesize($srcpath);
        $width  = $imginfo[0];
        $height = $imginfo[1];
      }
      // Move to destination, if different to source location..
      if (realpath($destpath) != realpath($srcpath)) {
        if (copy($srcpath, $destpath) === false) {
          $errmsgs[] = "failed to copy new media to its destination: $srcpath --> $destpath";
        }
      }
      // Store file in database..
      if ($name == "") {
        $name = basename($srcpath);
      }
      $this->cat_name = $name;
      $this->cat_desc = $desc;
      $this->keywords = $keywords;
      $this->category = $category;
      $this->mime_category = $mime_category;
      $this->mime_type = $mimetype;
      $this->filesize = $filesize;
      $this->filepath = $filepath;
      $this->physicalpath = $destpath;
      $this->fileexists = file_exists($destpath);
      $this->width = $width;
      $this->height = $height;
      if ($this->save()) {
        $this->valid = true;
      }
      else {
        $errmsgs[] = "failed to save catalog item $this->cat_id to database.";
      }
    }
    // Return the status..
    return $this->valid;

  } // create
  // .....................................................................
  /**
  * Process an uploaded media file, and define this catalog item to be
  * the newly uploaded file. Assuming a valid upload is performed, this
  * catalog item will be added to the database, and the file stashed
  * in the media directory. This method is provided to allow for easy
  * handling of upload form submission particularly for the Axyl media
  * catalog. Ie. use this if you have a form which is just for uploading
  * new images, movies etc. to the Axyl catalog.
  * @param string $name Media file associated name
  * @param string $category Media file associated category
  * @param string $desc Media file full description
  * @param string $keywords Media file associated keywords
  * @return array Error messages, if any occurred
  */
  function upload($name="", $category="", $desc="", $keywords="") {
    global $CATALOGDIR, $RESPONSE;
    global $_overwrite;

    $this->valid = false;
    $errmsgs = array();
    $up = new fileupload();
    if ($up->uploaded_count >= 1) {
      // Process uploaded file..
      $file_mime = $up->mimetype;

      // The relative URL path we will save on the DB record. Note that
      // we get rid of spaces, since these can screw various third-party
      // applications which need to make use of the filepath.
      $filepath = "$CATALOGDIR/" . str_replace(" ", "_", $up->filename);
      // The physical path on the filesystem..
      $destpath = $RESPONSE->site_docroot . $filepath;

      // Check pre-existence/overwrite flag..
      if (!file_exists($destpath) || isset($_overwrite)) {
        // Determine mime_category..
        $mime_category = mimetype_category($file_mime);
        if ($mime_category == "") {
          $file_mime = mimetype_from_filename($up->filename);
          $mime_category = mimetype_category($file_mime);
          if ($mime_category == "") {
            $errmsgs[] = "That file is of a type unknown to the system [$file_mime].";
          }
        }
        // Carry on if legal mime_category..
        if ($mime_category != "") {
          // Store file in catalog..
          $srcpath = tempnam("/tmp", "CATA");
          $tempdir = dirname($srcpath);
          $tempfname = basename($srcpath);
          if ($up->store($tempdir, $tempfname)) {

            // If we are doing this from a microsite, then possibly
            // assign the category, if not already assigned..
            if ($RESPONSE->microsites_mode == MICROSITES_ENABLED) {
              if ($RESPONSE->microsite_detected != "" && $category == "") {
                $category = $RESPONSE->microsite_detected;
              }
            }
            // Move new media file into place, and save data to DB..
            $this->create(
                    $srcpath, $destpath, $filepath,
                    $file_mime,
                    0, 0,
                    $name, $category, $desc, $keywords
                    );
            // Tidy up temporary file..
            if (is_writeable($srcpath)) {
              unlink($srcpath);
            }
          }
          else {
            $errmsgs[] = $up->error_message();
          }
        }
        else {
          $errmsgs[] = "No content type was specified.";
        }
      }
      else {
        $errmsgs[] = "File exists. Check the box to overwrite.";
        $overwrite_chk = true;
      }
    }
    else {
      $errmsgs[] = "Nothing was uploaded.";
    }
    return $errmsgs;
  } // upload

  // -----------------------------------------------------------------------
  /**
   * Resize the image in the catalog. This will limit either the width or
   * height or both to the given values. If either are false, then the scaling
   * limits to the one which is provided, and keeps the same aspect ratio.
   * @parm $maxw mixed False=ignore, else integer pixels for max width
   * @parm $maxh mixed False=ignore, else integer pixels for max height
   */
  function resize($maxw=false, $maxh=false) {
    if ($this->valid) {
      $dest = $this->physicalpath;
      $org_dims = getimagesize($dest);
      $orgw  = $org_dims[0];
      $orgh  = $org_dims[1];
      $ratio = $orgw / $orgh;

      debugbr("resize: orgw=$orgw orgh=$orgh ration=$ratio", DBG_DEBUG);
      debugbr("resize: maxw=$maxw maxh=$maxh", DBG_DEBUG);
    
      // Determine target size..
      $do_resize = false;
      if ($maxw && $maxh) {
        $target_size = $maxw . 'x' . $maxh;
        $do_resize = true;
      }
      elseif ($maxw) {
        if ($orgw > $maxw) {
          $target_size = $maxw . 'x' . intval($maxw / $ratio);
          $do_resize = true;
        }
      }
      elseif ($maxh) {
        if ($orgh > $maxh) {
          $target_size = intval($maxh * $ratio) . 'x' . $maxh;
          $do_resize = true;
        }
      }  
      if ($do_resize) {
        debugbr("resize: resizing to $target_size", DBG_DEBUG);
        // Create the image in /tmp and then move into place
        $tmp_dest = tempnam("/tmp","imgrsiz");
        if (system("convert -resize $target_size $dest jpg:$tmp_dest") != 0) {
          die("error resizing image $dest (catalog id=$catid) to $target_size");
        }
        system("chmod +r $tmp_dest");
        if (system("mv $tmp_dest $dest") != 0) {
          unlink($tmp_dest);
          die("error moving $tmp_dest to $dest");
        }
      
        // Save final image size to catalog, since we've actually
        // resized it on disk, we better had.
        $final_size = getimagesize($dest);
        $this->width  = $final_size[0];
        $this->height = $final_size[1];
        $this->save();
      }
      else {
        debugbr("resize: NOT resizing image!", DBG_DEBUG);
      }
    }
  } // resize

  // .....................................................................
  /**
  * Render the catalog item. We render it as either and icon or a link,
  * both being clickable to view the content.
  * @param string $mode Mode of rendering: 'icon' or 'link'.
  * @return string HTML rendering of this catalog item in given mode
  */
  function html($mode="link") {
    $s = "";
    switch ($mode) {
      case "icon":
        $s = $this->AsIcon();
        break;
      case "insitu":
        $s = $this->Insitu(false);
        break;
      default: // case "link"
        $s = $this->AsLink();
    } // switch
    return $s;
  } // html

} // catalogitem class

// -----------------------------------------------------------------------
// Class to hold all the data items. This is designed such that when
// it is converted to JSON format it is perfect for a DOJO toolkit
// itemDataStore to read and digest.
class dataitems {
  var $identifier;
  var $items = array();
  function dataitems() {
  }
  function additem($item) {
    $this->items[] = $item;
  }
}
class dataitem {
  var $thumb;
  var $large;
  var $title;
  var $link;
  var $catid;
  function dataitem($catid, $thumb, $large, $title, $link) {
    $this->thumb = $thumb;
    $this->large = $large;
    $this->title = $title;
    $this->link = $link;
    $this->catid = $catid;
  }
}
// -----------------------------------------------------------------------
/**
* This class encapsulates a media catalog, which is a collection of
* catalogitem objects.
* @package catalog
*/
class catalog extends RenderableObject {
  /** The array of catalogitem objects in this catalog */
  var $catalogitems = array();
  /** The array of categories (optional) */
  var $categories = array();
  // .....................................................................
  /** Constructor */
  function catalog($catalogitems=false) {
    if (is_array($catalogitems)) {
      $this->catalogitems = $catalogitems;
    }
  } // catalog
  // .....................................................................
  /** Return the count of items in the catalog currently.
  * @return integer Count of items in the catalog.
  */
  function itemcount() {
    return count($this->catalogitems);
  }
  // .....................................................................
  /**
  * Define the list of user categories which is used to categorise
  * the catalog items in this catalog. This is submitted as an
  * associative array with a single word string as key, and a multi-word
  * string as the value (category description).
  * @param array $cats Array of catalog categories: ID => description
  */
  function set_categories($categories) {
    if (is_array($categories)) {
      $this->categories = $categories;
    }
  } // define_categories
  // .....................................................................
  /**
  * Add catalog item to the catalog. The catalog will add the new item
  * by cat_id. If that ID already exists it is overwritten.
  * @param object $catitem The catalog item to add
  */
  function additem($catitem) {
    if (is_object($catitem) && is_a($catitem, "catalogitem")) {
      $this->catalogitems[$catitem->cat_id] = $catitem;
    }
  } // additem
  // .....................................................................
  /**
  * Find a catalogitem by filepath. Just trawls the array of catalog
  * items in this catalog comparing filepaths. Returns false if not found
  * else a copy of the catalogitem object.
  * @param string $filepath Website URL that image is located at.
  * @return mixed False if we didn't match a filepath, else catalog item copy.
  */
  function get_catalogitem_by_filepath($filepath) {
    $found = false;
    foreach ($this->catalogitems as $catitem) {
      if ($catitem->filepath == $filepath) {
        $found = $catitem;
        break;
      }
    }
    return $found;
  } // get_catalogitem_by_filepath
  // .....................................................................
  /**
  * Perform a search for catalog items. This method will use the search
  * engine if it is available and keywords are provided, otherwise it will
  * just go to the database. The end result is that this catalog is
  * populated with the results of the search, having been cleared out
  * beforehand.
  * @param string $keywords Keywords to search for
  * @param array  $mimecats Array of content types (mime categories)
  * @param array  $categories Array of media user-categories
  * @param strong $sortby Sort order for results
  */
  function search($keywords="", $mimecats="", $categories="", $sortby="") {
    global $SE_AVAILABLE;

    // Initialise catalog..
    $this->catalogitems = array();

    // KEYWORD SEARCH..
    if ($keywords != "" && $SE_AVAILABLE) {
      $thesearch = new searchengine_search();

      // CONTENT TYPES
      if (is_array($mimecats) && count($mimecats) > 0) {
        $terms = array();
        foreach ($mimecats as $mimecat) {
          // The nullstring represents 'Any' or 'All'..
          if ($mimecat != "") $terms[] = $mimecat;
        }
        if (count($terms) > 0) {
          $thesearch->must_match( "mimecat:(" . implode(" ", $terms) . ")" );
        }
      }

      // CATEGORIES
      $thesearch->does_not_match( "category:sitecontent" );
      if (is_array($categories) && count($categories) > 0) {
        $terms = array();
        foreach ($categories as $cat) {
          // The nullstring represents 'Any' or 'All'..
          if ($cat != "") $terms[] = $cat;
        }
        if (count($terms) > 0) {
          $thesearch->must_match( "category:(" . implode(" ", $terms) . ")" );
        }
      }

      // KEYWORDS
      // Keywords matching. We search the keywords fields, and
      // also the default text field..
      if ($keywords != "") {
        $words = explode(" ", $keywords);
        $terms = array();
        foreach ($words as $word) {
          if ($word != "") $terms[] = "$word^2";
        }
        $keyword_terms = implode(" ", $terms);

        $terms = array();
        foreach ($words as $word) {
          if ($word != "") $terms[] = $word;
        }
        $text_terms = implode(" ", $terms);
        // Construct the search terms..
        $thesearch->must_match( "(keywords:($keyword_terms) ($text_terms))" );
      }

      // SORT ORDER
      switch ($sortby) {
        case "rank":
          $thesearch->set_sortorder("RANK");
          break;
        case "uploaded":
          $thesearch->set_sortorder("uploaded:Date:Desc");
          break;
        case "catname":
          $thesearch->set_sortorder("catname");
          break;
        case "mimecat":
          $thesearch->set_sortorder("mimecat,uploaded:Date:Desc");
          break;
        case "category":
          $thesearch->set_sortorder("category,uploaded:Date:Desc");
          break;
        default:
          $thesearch->set_sortorder("uploaded:Date:Desc");
          break;
      } // switch

      // RETURN FIELDS
      $thesearch->set_returnfields("catname,category,mimecat,uploaded:Date,path");
      $thesearch->execute();

      // Populate the catalog..
      if ($thesearch->hitcount() > 0) {
        foreach ($thesearch->hit as $hit) {
          $ci = new catalogitem();
          $bits = explode("_", $hit["Id"]);
          $ci->cat_id = $bits[1];
          $ci->get();
          $this->catalogitems[$ci->cat_id] = $ci;
        }
      }

    } // got keywords
    else {
      $cQ  = new dbselect("ax_catalog");
      $wheres = array();

      // CONTENT TYPES
      if (is_array($mimecats) && count($mimecats) > 0) {
        // The nullstring represents 'Any' or 'All'..
        if (!in_array("", $mimecats)) {
          $wheres[] = "mime_category IN ('" . implode("','", $mimecats) . "')";
        }
      }

      // CATEGORIES
      if (is_array($categories) && count($categories) > 0) {
        // The nullstring represents 'Any' or 'All'..
        if (!in_array("", $categories)) {
          $wheres[] = "category IN ('" . implode("','", $categories) . "')";
        }
      }
      if (count($wheres) > 0) {
        $cQ->where( implode(" AND ", $wheres) );
      }

      // SORT ORDER
      switch ($sortby) {
        case "uploaded":
          $cQ->orderby("upload_timestamp DESC");
          break;
        case "catname":
          $cQ->orderby("cat_name");
          break;
        case "mimecat":
          $cQ->orderby("mime_category");
          break;
        case "category":
          $cQ->orderby("category");
          break;
        default:
          $cQ->orderby("upload_timestamp DESC");
          break;
      } // switch

      $cQ->execute();
      if ($cQ->hasdata) {
        do {
          $ci = new catalogitem();
          $ci->cat_id = $cQ->field("cat_id");
          $ci->get();
          $this->catalogitems[$ci->cat_id] = $ci;
        } while ($cQ->get_next());
      }
    }

  } // search
  
  // .....................................................................
  /**
   * This will emit a JSON-formatted stream containing this catalog of
   * items in a format which is compatible with the Dojo toolkit's
   * itemDataStore. Calling it has the side-effect of creating a
   * thumbnail image for each image in the catalog, since the data
   * emitted contains URL's pointing to both the thumbnail and its
   * full image. Thumbnails are stored in the same directory as the
   * image, but with '.thumb' before the extension. Eg. a file called
   * 'foo.jpg would have a thumbnail called 'foo.thumb.jpg'.
   */
  function send_as_json() {
    include_once("json-defs.php");
    $items = new dataitems();
    $items->identifier = "catid";
    foreach ($this->catalogitems as $catid => $ci) {
      if ($ci->fileexists) {
        if (!isset($ci->versions["thumb"])) {
          $ci->make_version("thumb");
        }
        $thumb = $ci->versions["thumb"];
        $item = new dataitem(
                  $ci->cat_id,
                  $thumb->filepath, // thumb url
                  $ci->filepath,    // large url
                  ($ci->cat_desc != "" ? $ci->cat_desc : $ci->cat_name), // title
                  $ci->keyinfo()    // link (ab-used to contain image metadata here)
                  );
        $items->additem($item);
      }
    } // while
    
    // Make JSON and return it
    global $RESPONSE;
    $RESPONSE->discard();
    header("Content-Type: application/json");
    $convert = new JSONconverter();
    $JSON =  $convert->toJSON($items);
    echo $JSON;
  } // send_as_json
  // .....................................................................
  /** Return the HTML for this catalog. Returns the list of catalog items
  * as an HTML table.
  * @return string HTML table containing the whole catalog
  */
  function html() {
    global $RESPONSE, $LIBDIR;
    return "";
  } // html

} // catalog class

// -----------------------------------------------------------------------
?>