<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    search-utility-defs.php                                 */
/* Author:      Paul Waite                                              */
/* Description: Definitions for interfacing to a search engine to       */
/*              perform various "utility" operations.                   */
/*              The parent module which includes this module must also  */
/*              include the underlying defs module for the search       */
/*              engine itself, eg: lucene, solr etc.                    */
/*                                                                      */
/* ******************************************************************** */
/** @package search */

// ----------------------------------------------------------------------
/**
* The lucene purge message class. This class allows you to remove all
* items from the SearchEngine index. Take care!
* @package search
*/
class searchengine_purger extends searchengine_purgemsg {
  // .....................................................................
  /** Constructor
  * Make a new SearchEngine index purger. This message is provided to allow
  * you to delete all items from the SearchEngine index.
  * @param string $application Optional application specifier
  * @param string $host Hostname or IP of SearchEngine server
  * @param string $port Port of SearchEngine server
  */
  function searchengine_purger($application="?", $host="", $port="") {
    $this->searchengine_purgemsg($application, $host, $port);
  } // searchengine_purger
  // .....................................................................
  /**
   * Make SearchEngine purge the index.
   * @param integer $timeoutsecs Override for timeout in seconds
   * @return boolean True if command was successful.
   */ 
  function execute($timeoutsecs="") {
    return $this->send($timeoutsecs);
  } // execute
} // searchengine_purger class

// ----------------------------------------------------------------------
/**
* The SearchEngine utility message class. Used for special SearchEngine
* operations.
* @package search
*/
class searchengine_utilitycmd extends searchengine_utilitymsg {
  /** Constructor
  * @param string $utilitycmd Command for this utility message.
  * @param string $application Optional application specifier
  * @param string $host Hostname or IP of SearchEngine server
  * @param string $port Port of SearchEngine server
  */
  function searchengine_utilitycmd($utilitycmd="", $application="?", $host="", $port="") {
    $this->searchengine_utilitymsg($utilitycmd, $application, $host, $port);
  } // searchengine_utilitycmd
  // .....................................................................
  /**
   * Make SearchEngine execute the given utility command.
   * @param integer $timeoutsecs Override for timeout in seconds
   * @return boolean True if command was successful.
   */ 
  function execute($timeoutsecs="") {
    return $this->send($timeoutsecs);
  } // execute
} // searchengine_utilitycmd class

// ----------------------------------------------------------------------
/**
* Function to optimize the SearchEngine index. This would commonly
* be used after a batch of items have been indexed.
* @param string $application Application name/domain name for searching in
* @param string $host Hostname or IP of SearchEngine server
* @param string $port Port of SearchEngine server
* @return boolean True if the operation was successful.
*/
function searchengine_optimize($application="?", $host="", $port="") {
  $optimizer = new searchengine_utilitycmd("OPTIMIZE", $application, $host, $port);
  $optimizer->execute(SOCK_FOREVER);
  return $optimizer->response->valid;
} // searchengine_optimize

// ----------------------------------------------------------------------
/**
* Function to make a backup of the SearchEngine index. This would commonly
* be used after a batch of items have been successfully optimized (which
* indicates a sound index).
* @param string $application Application name/domain name for searching in
* @param string $host Hostname or IP of SearchEngine server
* @param string $port Port of SearchEngine server
* @return boolean True if the operation was successful.
*/
function searchengine_backup($application="?", $host="", $port="") {
  $backup = new searchengine_utilitycmd("BACKUP", $application, $host, $port);
  $backup->execute(SOCK_FOREVER);
  return $backup->response->valid;
} // searchengine_backup

// ----------------------------------------------------------------------
/**
* Function to purge the SearchEngine index of all indexes to documents. Yes,
* I'll repeat that - it DELETES ALL DOCUMENTS FROM THE INDEX, permanently,
* finito, shazam, ba-boom, as in "Omigod did I *really* mean to do that!?".
* I guess I don't have to warn you to be careful with this, do I?
* @param string $application Application name/domain name for searching in
* @param string $host Hostname or IP of SearchEngine server
* @param string $port Port of SearchEngine server
* @return boolean True if the purging operation was successful.
*/
function searchengine_purge($application="?", $host="", $port="") {
  $purgative = new searchengine_purger($application, $host, $port);
  $purgative->execute(SOCK_FOREVER);
  return $purgative->response->valid;
} // searchengine_purge

// ----------------------------------------------------------------------
/**
* Function to check the SearchEngine index for 'health'. This determines
* that the search engine server is alive, and can access the number of
* documents in its index (which is what it goes and does).
* @param string $application Application name/domain name for searching in
* @param string $host Hostname or IP of SearchEngine server
* @param string $port Port of SearchEngine server
* @return boolean True if the index is healthy.
*/
function searchengine_healthcheck($application="?", $host="", $port="") {
  $health = new searchengine_utilitycmd("HEALTHCHECK", $application, $host, $port);
  $health->execute(SOCK_FOREVER);
  return $health->response->valid;
} // searchengine_healthcheck

// ----------------------------------------------------------------------
/**
* Function to acquire the SearchEngine index codument count. This is
* similar to the healthcheck function, since it uses the same utility
* command, but it just returns the numeric result that the command goes
* and acquires as part of testing whether the server is alive.
* @param string $application Application name/domain name for searching in
* @param string $host Hostname or IP of SearchEngine server
* @param string $port Port of SearchEngine server
* @return mixed The number of documents, else FALSE if server is dead. 
*/
function searchengine_documentcount($application="?", $host="", $port="") {
  $result = false;
  $count = new searchengine_utilitycmd("HEALTHCHECK", $application, $host, $port);
  $count->execute(SOCK_FOREVER);
  if ($count->response->valid) {
    if (isset($count->response->tag_data["NumDocs"])) {
      $result = intval( $count->response->tag_data["NumDocs"]);
    }
  }
  return $result;
} // searchengine_documentcount

// ----------------------------------------------------------------------
?>