/*
 * Created on 2004-2-20
 */
package org.cneclipse.bdcc;

import java.io.*;
import java.net.*;
import java.util.*;
import org.gudy.azureus2.core3.util.*;
import org.gudy.azureus2.plugins.*;
import org.gudy.azureus2.plugins.download.*;
import org.gudy.azureus2.plugins.sharing.*;
import org.gudy.azureus2.plugins.torrent.*;
import org.gudy.azureus2.plugins.tracker.*;
import org.jibble.lz.pircbot.*;

/**
 * @author Leon
 */
public class BDCCAdminCommands {

	private DownloadManager	downloadManager;
	private BDCCIrcClient	bot;
	private PluginConfig	config;
	private TorrentManager	torrentManager;
	private PluginInterface	pluginInterface;
	private int				lowPort		= 0;
	private int				highPort	= 0;

	/**
	 * @param downloadManager
	 */
	public BDCCAdminCommands(BDCCIrcClient bot, PluginInterface pluginInterface) {
		this.torrentManager = pluginInterface.getTorrentManager();
		this.config = pluginInterface.getPluginconfig();
		this.downloadManager = pluginInterface.getDownloadManager();
		this.pluginInterface = pluginInterface;
		this.bot = bot;
		lowPort = config.getPluginIntParameter("BDCC DCC Send Port Low", 0);
		highPort = config.getPluginIntParameter("BDCC DCC Send Port High", 0);
	}

	/**
	 * @return
	 */
	public String[] getAllTorrents() {
		Download[] downloads = downloadManager.getDownloads(true);
		Vector result = new Vector();
		for (int i = 0; i < downloads.length; i++) {
			Download download = downloads[i];
			if (download.getState() == Download.ST_ERROR) {
				result.add("#" + (download.getIndex() + 1) + " ["
						+ DisplayFormatters.formatByteCountToKiBEtc(download.getTorrent().getSize()) + "] "
						+ Colors.RED + " [ERROR] " + Colors.NORMAL + download.getTorrent().getName());
			} else {
				result.add("#" + (download.getIndex() + 1) + " ["
						+ DisplayFormatters.formatByteCountToKiBEtc(download.getTorrent().getSize()) + "] "
						+ download.getTorrent().getName() + " [" + download.getStats().getStatus() + "]");
			}
		}
		String[] ret = new String[result.size()];
		result.toArray(ret);
		return ret;
	}

	/**
	 * @return
	 */
	public String[] getDownloadingTorrents() {
		Download[] downloads = downloadManager.getDownloads(true);
		Vector result = new Vector();
		for (int i = 0; i < downloads.length; i++) {
			Download download = downloads[i];
			if (download.getState() == Download.ST_DOWNLOADING) {
				DownloadStats stats = download.getStats();
				result.add("#" + (download.getIndex() + 1) + " ["
						+ DisplayFormatters.formatByteCountToKiBEtc(download.getTorrent().getSize()) + " ("
						+ DisplayFormatters.formatPercentFromThousands(stats.getCompleted()) + ")] "
						+ download.getTorrent().getName() + "  Down: "
						+ DisplayFormatters.formatByteCountToKiBEtc(stats.getDownloaded()) + "("
						+ DisplayFormatters.formatByteCountToKiBEtcPerSec(stats.getDownloadAverage()) + ") Up: "
						+ DisplayFormatters.formatByteCountToKiBEtc(stats.getUploaded()) + "("
						+ DisplayFormatters.formatByteCountToKiBEtcPerSec(stats.getUploadAverage()) + ")");
			}
		}
		String[] ret = new String[result.size()];
		result.toArray(ret);
		return ret;
	}

	/**
	 * @return
	 */
	public String[] getSeedingTorrents() {
		Download[] downloads = downloadManager.getDownloads(true);
		Vector result = new Vector();
		for (int i = 0; i < downloads.length; i++) {
			Download download = downloads[i];
			if (download.getState() == Download.ST_SEEDING) {
				DownloadStats stats = download.getStats();
				result.add("#" + (download.getIndex() + 1) + " ["
						+ DisplayFormatters.formatByteCountToKiBEtc(download.getTorrent().getSize()) + " ("
						+ DisplayFormatters.formatPercentFromThousands(stats.getCompleted()) + ")] "
						+ download.getTorrent().getName() + "  Up: "
						+ DisplayFormatters.formatByteCountToKiBEtc(stats.getUploaded()) + "("
						+ DisplayFormatters.formatByteCountToKiBEtcPerSec(stats.getUploadAverage()) + ")");
			}
		}
		String[] ret = new String[result.size()];
		result.toArray(ret);
		return ret;
	}

	/**
	 * @return
	 */
	public String[] getQueuedTorrents() {
		Download[] downloads = downloadManager.getDownloads(true);
		Vector result = new Vector();
		for (int i = 0; i < downloads.length; i++) {
			Download download = downloads[i];
			if (download.getState() != Download.ST_DOWNLOADING && download.getState() != Download.ST_SEEDING
					&& download.getState() != Download.ST_ERROR) {
				result.add("#" + (download.getIndex() + 1) + " ["
						+ DisplayFormatters.formatByteCountToKiBEtc(download.getTorrent().getSize()) + "] "
						+ download.getTorrent().getName() + " [Priority: " + download.getPriority() + " ]");
			}
		}
		String[] ret = new String[result.size()];
		result.toArray(ret);
		return ret;
	}

	/**
	 * @param n
	 * @return
	 */
	public String[] getTorrentInfo(int n) {
		Download download = getDownloadById(n);
		if (download == null) return null;
		Torrent torrent = download.getTorrent();
		DownloadStats stats = download.getStats();
		DownloadAnnounceResult announceResult = download.getLastAnnounceResult();
		DownloadScrapeResult scrapeResult = download.getLastScrapeResult();
		Vector result = new Vector();
		result.add("Torrent Info for Torrent #" + n);
		result.add("Name: " + torrent.getName());
		result.add("Hash: " + ByteFormatter.nicePrint(torrent.getHash()));
		result.add("Size: " + DisplayFormatters.formatByteCountToKiBEtc(torrent.getSize()) + " (" + torrent.getSize()
				+ ") Download: " + DisplayFormatters.formatByteCountToKiBEtcPerSec(stats.getDownloadAverage()) + " ("
				+ DisplayFormatters.formatByteCountToKiBEtc(stats.getDownloaded()) + ") Upload: "
				+ DisplayFormatters.formatByteCountToKiBEtcPerSec(stats.getUploadAverage()) + " ("
				+ DisplayFormatters.formatByteCountToKiBEtc(stats.getUploaded()) + ")");
		result.add("Description: " + torrent.getComment() + "  Created By: \"" + torrent.getCreatedBy()
				+ "\" Created On " + new Date(torrent.getCreationDate()));
		result.add("Contains: " + torrent.getPieceCount() + " pieces. Each piece is "
				+ DisplayFormatters.formatByteCountToKiBEtc(torrent.getPieceSize()) + " ");
		result.add("Completed: " + DisplayFormatters.formatPercentFromThousands(stats.getCompleted()) + " ETA: "
				+ stats.getETA() + " Discarded: " + DisplayFormatters.formatByteCountToKiBEtc(stats.getDiscarded())
				+ " Hash Failed: " + stats.getHashFails());
		result.add("Share Rate: " + DisplayFormatters.formatPercentFromThousands(stats.getShareRatio())
				+ " Elapsed Time: " + stats.getElapsedTime());
		result.add("Tracker: " + torrent.getAnnounceURL());
		if (scrapeResult.getResponseType() == DownloadScrapeResult.RT_SUCCESS) result.add("Tracker Status: "
				+ stats.getTrackerStatus() + " Seeds: " + announceResult.getSeedCount() + "("
				+ scrapeResult.getSeedCount() + ") Peers: " + announceResult.getNonSeedCount() + "("
				+ scrapeResult.getNonSeedCount() + ")");
		else result.add("Tracker Status: " + stats.getTrackerStatus() + " Seeds: " + announceResult.getSeedCount()
				+ " Peers: " + announceResult.getNonSeedCount());
		String[] ret = new String[result.size()];
		result.toArray(ret);
		return ret;
	}

	/**
	 * @param n
	 */
	public String queueTorrent(int n) throws DownloadException {
		Download download = getDownloadById(n);
		download.stopAndQueue();
		return download.getName();
	}

	/**
	 * @param n
	 */
	public String startTorrent(int n) throws DownloadException {
		Download download = getDownloadById(n);
		download.restart();
		return download.getName();
	}

	/**
	 * @param n
	 */
	public String stopTorrent(int n) throws DownloadException {
		Download download = getDownloadById(n);
		download.stop();
		return download.getName();
	}

	/**
	 * @param n
	 */
	public String removeTorrent(int n) throws DownloadException, DownloadRemovalVetoException {
		Download download = getDownloadById(n);
		download.remove();
		return download.getName();
	}

	/**
	 * @param nick
	 * @param n
	 */
	public String sendTorrent(String nick, int n) throws IOException, TorrentException {
		Download download = getDownloadById(n);
		if (download == null) return null;
		File torrentTmpFile = File.createTempFile(download.getName(), ".torrent");
		torrentTmpFile.createNewFile();
		torrentTmpFile.deleteOnExit();
		download.getTorrent().writeToFile(torrentTmpFile);
		bot.dccSendFile(torrentTmpFile, nick, 120000, lowPort, highPort);
		return download.getName();
	}

	/**
	 * @param msg
	 * @return
	 */
	public void say(String msg) {
		bot.sendMessage(bot.getChannel(), msg);
	}

	/**
	 * @param msg
	 */
	public void quit(String msg) {
		if (msg == null || msg.length() == 0) bot.quitServer(BDCCIrcClient.VERSION_INFO + " on Azureus. "
				+ pluginInterface.getAzureusVersion() + " running " + bot.getRunningTime());
		else bot.quitServer(msg);
		bot.setQuit(true);
	}

	/**
	 * @param filename
	 * @return
	 */
	public int add(String filename) throws DownloadException, TorrentException {
		File torrentFile = new File(filename);
		if (!torrentFile.exists()) return -1;
		Torrent torrent = torrentManager.createFromBEncodedFile(torrentFile);
		Download download = downloadManager.addDownload(torrent);
		return (download.getIndex() + 1);
	}

	/**
	 * @param url
	 * @return
	 */
	public int addUrl(String url) throws MalformedURLException, TorrentException, DownloadException {
		TorrentDownloader downloader = torrentManager.getURLDownloader(new URL(url));
		Torrent torrent = downloader.download();
		Download download = downloadManager.addDownload(torrent);
		try {
			Thread.sleep(2000);
		} catch (InterruptedException e) {
		}
		return (download.getIndex() + 1);
	}

	/**
	 * @return
	 */
	public String getStatusLines() {
		Download[] downloads = this.downloadManager.getDownloads();
		long uploadSpeed = 0;
		long downloadSpeed = 0;
		for (int i = 0; i < downloads.length; i++) {
			uploadSpeed += downloads[i].getStats().getUploadAverage();
			downloadSpeed += downloads[i].getStats().getDownloadAverage();
		}
		return "** " + downloads.length + " torrents **  Uploading:"
				+ DisplayFormatters.formatByteCountToKiBEtcPerSec(uploadSpeed) + " Downloading:"
				+ DisplayFormatters.formatByteCountToKiBEtcPerSec(downloadSpeed) + " **";
	}

	/**
	 * @param n
	 * @return
	 */
	public Download getDownloadById(int n) {
		n--;
		Download[] downloads = downloadManager.getDownloads();
		for (int i = 0; i < downloads.length; i++) {
			if (downloads[i].getIndex() == n) return downloads[i];
		}
		return null;
	}

	public long getTotalUploadingSpeed() {
		long result = 0;
		Download[] downloads = downloadManager.getDownloads();
		for (int i = 0; i < downloads.length; i++) {
			result += downloads[i].getStats().getUploadAverage();
		}
		return result;
	}

	public long getTotalDownloadingSpeed() {
		long result = 0;
		Download[] downloads = downloadManager.getDownloads();
		for (int i = 0; i < downloads.length; i++) {
			result += downloads[i].getStats().getDownloadAverage();
		}
		return result;
	}

	public long getTotalUploaded() {
		long result = 0;
		Download[] downloads = downloadManager.getDownloads();
		for (int i = 0; i < downloads.length; i++) {
			result += downloads[i].getStats().getUploaded();
		}
		return result;
	}

	public long getTotalDownloaded() {
		long result = 0;
		Download[] downloads = downloadManager.getDownloads();
		for (int i = 0; i < downloads.length; i++) {
			result += downloads[i].getStats().getDownloaded();
		}
		return result;
	}

	/**
	 * @param filename
	 */
	public String addShare(String filename) throws ShareResourceDeletionVetoException, ShareException {
		ShareManager shareManager = pluginInterface.getShareManager();
		ShareResource share = shareManager.addFile(new File(filename));
		return share.getName();
	}

	/**
	 * @param dir
	 */
	public String addShareDir(String dir) throws ShareResourceDeletionVetoException, ShareException {
		ShareManager shareManager = pluginInterface.getShareManager();
		ShareResource share = shareManager.addDir(new File(dir));
		return share.getName();
	}

	/**
	 * @param dir
	 * @param b
	 */
	public String addShareDirContent(String dir, boolean b) throws ShareResourceDeletionVetoException, ShareException {
		ShareManager shareManager = pluginInterface.getShareManager();
		ShareResource share = shareManager.addDirContents(new File(dir), b);
		return share.getName();
	}

	/**
	 * @return
	 */
	public String[] getShares() throws ShareException {
		ShareManager shareManager = pluginInterface.getShareManager();
		ShareResource[] shares = shareManager.getShares();
		Vector result = new Vector();
		for (int i = 0; i < shares.length; i++) {
			ShareResource share = shares[i];
			switch (share.getType()) {
				case ShareResource.ST_FILE:
					result.add("[File] " + share.getName());
					break;
				case ShareResource.ST_DIR:
					result.add("[Directory] " + share.getName());
					break;
				case ShareResource.ST_DIR_CONTENTS:
					result.add("[Directory Content]" + share.getName());
					break;
			}
		}
		String[] ret = new String[result.size()];
		result.toArray(ret);
		return ret;
	}

	/**
	 * @param name
	 */
	public boolean removeShare(String name) throws ShareException {
		ShareManager shareManager = pluginInterface.getShareManager();
		ShareResource[] shares = shareManager.getShares();
		for (int i = 0; i < shares.length; i++) {
			if (shares[i].getName().equalsIgnoreCase(name)) {
				shares[i].delete();
				return true;
			}
		}
		return false;
	}

	/**
	 * @param n
	 * @return
	 */
	public String hostTorrent(int n) throws TrackerException {
		Download download = getDownloadById(n);
		pluginInterface.getTracker().host(download.getTorrent(), true);
		return download.getName();
	}
}