/*
 * File:        sidl_Python.c
 * Revision:    @(#) $Revision: 6148 $
 * Date:        $Date: 2007-10-03 09:20:56 -0700 (Wed, 03 Oct 2007) $
 * Description: Initialize a Python language interpretter
 *
 */

#include "sidl_Python.h"
#include "babel_config.h"
#ifndef included_sidl_DLL_h
#include "sidl_DLL.h"
#endif
#ifndef included_sidl_Loader_h
#include "sidl_Loader.h"
#endif
#ifndef included_sidl_String_h
#include "sidl_String.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include "sidlOps.h"
#include "sidlCas.h"
#ifdef HAVE_PTHREAD
#ifndef CAS32
#include <pthread.h>
static pthread_mutex_t pc_mutex = PTHREAD_MUTEX_INITIALIZER;
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif

static volatile int32_t g_python_object_count = 0;
static void (*g_python_shutdown)(void) = NULL;

static void
sidl_Python_shutdown(void *ignored)
{
  if (g_python_shutdown) {
#if defined(HAVE_PTHREAD) && defined(HAVE_UNISTD_H)
    /* give Java or other threads time or to finalize objects */
    int time_slept = 0;
    while ((g_python_object_count > 0) && (time_slept < 2000)) {
      usleep(100);
      time_slept += 100;
    }
#endif
    if (g_python_object_count <= 0) {
      (*g_python_shutdown)();
    }
    else {
      fprintf(stderr, "Not shutting down Python due to lingering references: %ld\n", 
              (long)g_python_object_count);
    }
  }
}

void sidl_Python_Init(void)
{
  sidl_BaseInterface throwaway_exception; /*TODO: a way to not throw these away? */
  static int python_notinitialized = 1;
#ifdef PYTHON_SHARED_LIBRARY
  static const char libName[] = PYTHON_SHARED_LIBRARY;
#endif
  sidl_DLL dll;
  static const char initName[] = "Py_Initialize";
  static const char finalName[] = "Py_Finalize";
  static const char argvName[] = "PySys_SetArgv";
  void (*pyinit)(void);
  if (python_notinitialized) {
    dll = sidl_Loader_loadLibrary("main:", TRUE, TRUE, &throwaway_exception);
    if (dll) {
      pyinit = (void (*)(void))sidl_DLL_lookupSymbol(dll, initName,&throwaway_exception);
      if (pyinit) {
        void (*setargv)(int, char **);
        static char *fake_args[] = { "" };
        (*pyinit)();
        python_notinitialized = 0;
        g_python_shutdown = (void (*)(void))sidl_DLL_lookupSymbol(dll, finalName,&throwaway_exception);
        if (g_python_shutdown) {
          sidl_atexit(sidl_Python_shutdown, NULL);
        }
        setargv = (void (*)(int, char **))
          sidl_DLL_lookupSymbol(dll, argvName, &throwaway_exception);
        if (setargv) {
          (*setargv)(0, fake_args);
        }
      }
      sidl_DLL_deleteRef(dll,&throwaway_exception);
    }

    if (python_notinitialized) {
#ifdef PYTHON_SHARED_LIBRARY
      char *url = sidl_String_concat2("file:", PYTHON_SHARED_LIBRARY);
      if (url) {
        dll = sidl_Loader_loadLibrary(url, TRUE, TRUE,&throwaway_exception);
        if (dll) {
          pyinit = (void (*)(void))sidl_DLL_lookupSymbol(dll, initName,&throwaway_exception);
          if (pyinit) {
            void (*setargv)(int, char **);
            static char *fake_args[] = { "" };
            python_notinitialized = 0;
            (*pyinit)();
            g_python_shutdown = (void (*)(void))sidl_DLL_lookupSymbol(dll, finalName,&throwaway_exception);
            if (g_python_shutdown) {
              sidl_atexit(sidl_Python_shutdown, NULL);
            }
            setargv = (void (*)(int, char **))
              sidl_DLL_lookupSymbol(dll, argvName, &throwaway_exception);
            if (setargv) {
              (*setargv)(0, fake_args);
            }
          } 
          else {
            fprintf(stderr, "Babel: Error: Unable to find symbol %s in %s",
                    initName, libName);
          }
          sidl_DLL_deleteRef(dll,&throwaway_exception);
        }
        else {
          fprintf(stderr,
                  "Babel: Error: Unable to load library %s\n", libName);
        }
        sidl_String_free(url);
      }
      else {
        fprintf(stderr, "Unable to allocate string or sidl.DDL object\n");
      }
#else
      fprintf(stderr, "Babel: Error: Unable to initialize Python.\n\
The BABEL runtime library was not configured for Python support,\n\
and Python is not already loaded into the global symbol space.\n");
      python_notinitialized = 0;
#endif
    }
  }
}

#ifdef SIDL_C_HAS_INLINE
inline
#endif
static void
changeCount(int32_t delta)
{
#if defined(CAS32) && defined(HAVE_PTHREAD)
  volatile int32_t *ref_ptr = &g_python_object_count;
  int32_t oldRefcount, newRefcount;
  do {
    oldRefcount = *ref_ptr;
    newRefcount = oldRefcount + delta;
  } while (oldRefcount != CAS32(ref_ptr, oldRefcount, newRefcount));
#else
#ifdef HAVE_PTHREAD
  pthread_mutex_lock(&pc_mutex);
  g_python_object_count += delta;
  pthread_mutex_unlock(&pc_mutex);
#else
  g_python_object_count += delta;
#endif  
#endif
}

void
sidl_Python_IncGlobalRef(void)
{
  changeCount(1);
}

void
sidl_Python_DecGlobalRef(void)
{
  changeCount(-1);
}
