<?php

/******************************
 
Class: Properties
Hierarchical property storage designed for fast caching of properties.
Properties is a singleton (there is only ever one instance).

There are basically just three functions: get, set, commit. The Properties
object is not in charge of actually storing this values, it simply caches them.
It is the PageStorage object's job to store the values.

Every time you need a page property, you should ask the Properties object.

For example:

  $title = $properties->get('title',$page);
  
or, for convenience:

  $title = $page->get('title');
  
Which will do the same thing.
  
If the Properties cache is not older than the file storage, it returns the
cached value. Otherwise, it asks the PageStore for the value, caches it, then
returns the value.

note: the properties 'content' and 'name' are handled by the pagestore and not
by the properties object at all.

******************************/

require_once('Page.php');

class Properties {

var $data = array(
	'default' => array(
		'cachedir' => '/tmp/bamboocache',
		'sitename' => 'Untitled Site',
		'siteroot' => '',
		'decorator' => 'default',
		'decoratorurl' => '/bamboo/decorators',
		'liburl' => '/bamboo',
		'access-view' => 'anonymous',
		'access-edit' => 'anonymous',
		'title-prefix' => '',
		'static' => 'true',
		'mtime' => '1',
		'locked' => 'false',
		'title' => '$name',
		'navtitle' => '$title',
		'type' => 'txt',
		'dir' => false,
		'index' => true,
		'visible' => true,
	)
);

# props which are always global
var $globals = array(
	'cachedir','sitename','siteroot',
);

# props which cannot be inherited
var $basic = array(
	'name','title','subject','keywords','description','static','dir','mtime','locked','type','visible'
);

# props which might have translations
var $multilingual = array(
	'title', 'subject', 'description', 'keywords',
);

# 
var $proptypes = array(
	'dir' => 'boolean'
);


function Properties($site_config_file) {	
	if (!is_file($site_config_file)) die("The site configuration file '$site_config_file' does not exist.");
	if (!is_readable($site_config_file)) die("The site configuration file '$site_config_file' could not be read.");

	$this->data['site'] = parse_ini_file($site_config_file);
	if ($this->data['site']['siteroot'] == '/')
		$this->data['site']['siteroot'] = '';
	$this->data['site']['sitedocroot'] = dirname($site_config_file);
	
	// because i am lazy and don't want to write them out 'prop' => 'true'
	// should be removed in the future.
	$this->globals = array_flip($this->globals);
	$this->basic = array_flip($this->basic);
	$this->multilingual = array_flip($this->multilingual);
}

function &get($prop, &$page, $recursive_count=0) {
	$value = null;
	$ps = &$page->ps;
	$siteroot = $this->getGlobal('siteroot');
	
	if ($this->is_global($prop)) {
		$value = $this->getGlobal($prop);
	}
	elseif ($this->is_basic($prop)) {  # ie non-inheritable
	    $id = $page->path;
	    // if memory cache empty, load data
		if ( !isset( $this->data[$id]['basic'] ) ) {
			$this->data[$id]['basic'] = $ps->loadbasic($siteroot,$page->path);
		}
		if ($this->is_multilingual($prop)) {
			if ($value == null) {
				$prop_dot_lang = $prop . '.' . lang();
				if (isset($this->data[$id]['basic'][$prop_dot_lang]))
					$value = &$this->data[$id]['basic'][$prop_dot_lang];
			}
			if ($value == null) {
				$prop_dot_lang = $prop . '.' . default_lang();
				if (isset($this->data[$id]['basic'][$prop_dot_lang]))
					$value = &$this->data[$id]['basic'][$prop_dot_lang];
			}
		}
		if ($value == null) {
			if (isset($this->data[$id]['basic'][$prop]))
				$value = &$this->data[$id]['basic'][$prop];
		}
	}
	else { # inheritable
		$id = '';
		$parentid = '';
		$patharray = split('/',$page->path);
		#d($patharray);
		foreach($patharray as $pathelement) {
			if ($pathelement == '')
				$id = '';
			else
				$id = "$id/$pathelement";
			if ( !isset( $this->data[$id]['inherit'] ) ) {
				if ( !isset($this->data[$parentid]['inherit']) ) {
					$this->data[$id]['inherit'] = $ps->loadinherit($siteroot,$id);
				}
				else {
					// with array_merge, the second arg overwrites the first
					// if the key is the same. hence, inheritence.
					$this->data[$id]['inherit'] = array_merge(
						$this->data[$parentid]['inherit'],
						$ps->loadinherit($siteroot,$id)
					);
				}
			}
			$parentid = $id;
		}
		if (isset($this->data[$id]['inherit'][$prop]))
			$value = &$this->data[$id]['inherit'][$prop];
	}
	
	if ($value == null)
		$value = $this->getGlobal($prop); // last ditch effort.
	
	if ($value != null && $value{0} == '$' && $recursive_count < 4) {
		return $this->get(substr($value,1),$page,$recursive_count+1);
	}
	return $value;
}


function set($propname, $data, &$page) {
	$id = $page->path;
	if ($this->is_multilingual($propname))
		$propnamepluslang = $propname . '.' . lang();
	else
		$propnamepluslang = $propname;
		
	if ($data == 'true')
		$data = true;
	elseif ($data == 'false')
		$data = false;

	if ($this->is_basic($propname)) {
		$this->data[$id]['basic'][$propnamepluslang] = $data;
	}
	else {
		$this->data[$id]['inherit'][$propnamepluslang] = $data;
	}
}

function commit(&$page) {
	$id = $page->path;
	$ar = array();
	$siteroot = $this->getGlobal('siteroot');
	foreach($this->basic as $prop => $dummy) {
		if ($this->is_multilingual($prop))
			$prop = $prop . '.' . lang();
		if (isset($this->data[$id]['basic'][$prop]))
			$ar[$prop] = $this->data[$id]['basic'][$prop];
	}
	
	$err = $page->ps->savebasic($siteroot,$page->path,$ar);
#	if ($err == '')
#		$err = $page->ps->saveinherit($siteroot,$page->path,$ar);
	return $err;
}

/*
 * returns the value of a global property
 * (the same for all pages)
 */
function getGlobal($propname) {
	$value = isset($this->data['default'][$propname]) ? $this->data['default'][$propname] : null;
	$value = isset($this->data['site'][$propname]) ? $this->data['site'][$propname] : $value;
	return $value;
}

function is_global($prop) {
	return array_key_exists($prop,$this->globals);
}

function is_basic($prop) {
	return array_key_exists($prop,$this->basic);
}

function is_multilingual($prop) {
	return array_key_exists($prop,$this->multilingual);
}

function proptype($prop) {
	if (isset($this->proptypes[$prop]))
		return $this->proptypes[$prop];
	else
		return '';
}

} // end class


/* 
 * tries to guess the filename of the site configuration file
 */
function get_site_config() {
	$docroot = preg_replace('#/$#','',$_SERVER['DOCUMENT_ROOT']); // trim trailing slash

	if (isset($_SERVER['REDIRECT_URL']))
		$path = $docroot . $_SERVER['REDIRECT_URL'];
	elseif (isset($_SERVER['SCRIPT_URL']))
		$path = $docroot . $_SERVER['SCRIPT_URL'];
	else
		$path = $_SERVER['PATH_TRANSLATED'];
	$path = preg_replace("|^$docroot(/.*?)/.*$|", '$1', $path);

	if (is_file("$docroot/site.prop"))
		return "$docroot/site.prop";
	else
		return "$docroot$path/site.prop";
}

?>