/*
 * Initial main.c 
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

 #include <pwd.h>
 #include <time.h>
#include <gtk/gtk.h>
#include <glib/gprintf.h>
#include <libgnomeui/libgnomeui.h>
#include <libgnomeui/gnome-ui-init.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <gconf/gconf-client.h>
 
#include "baobab.h"
#include "tv.h"
#include "interface.h"
#include "support.h"
#include "utilities.h"
#include "bbthread.h"
#include "callbacks.h"
#include "props.h"

	static GnomeVFSMonitorHandle * handle_mtab;
	static GnomeVFSMonitorHandle * handle_home;

static 	void prefill_model (gchar* );

static void check_UTF(GString *);
static void  push_iter_in_stack (GtkTreeIter * );
static GtkTreeIter pop_iter_from_stack (void);

static gint currentdepth=0;
static GtkTreeIter currentiter;
static GtkTreeIter firstiter;
static GQueue * iterstack=NULL;
static void prepare_firstcol(GString*, struct BaobabSearchRet* );
static gchar* get_owner(uid_t );
static gchar* get_last_mod(time_t );
static const gchar* get_file_filter (gchar* );
gboolean filter_adv_search(struct BaobabSearchRet *);
static void initialize_search_variables(void);
static void baobab_init(void);
static void baobab_close(void);


/* start scanning on a specific directory
*/
void
start_proc_on_dir (gchar *dir)
{
	
	switch_view(VIEW_TREE);
	if (!check_dir(dir)) { return;}
	g_string_assign(baobab.last_scan_command,dir);
	baobab.STOP_SCANNING=FALSE;
	change_icon("busy.gif");
	check_menu_sens(TRUE);
	gtk_tree_store_clear(baobab.model);
	currentdepth=-1;			//flag
	iterstack = g_queue_new ();
		
	getDir(dir);

	//change the cursor
	GdkCursor *cursor = NULL;
	if (baobab.window->window) {
 		cursor = gdk_cursor_new(GDK_WATCH);
		gdk_window_set_cursor (baobab.window->window, cursor);
			}
	set_statusbar( _("Calculating percentage bars..."));
	gtk_tree_model_foreach((GtkTreeModel*)baobab.model,show_bars,NULL);
	change_icon("done.png");
	check_menu_sens(FALSE);
	set_statusbar(_("Ready"));
	
	/* cursor clean up */
	if (baobab.window->window) {
		gdk_window_set_cursor(baobab.window->window,NULL);
		if (cursor) gdk_cursor_unref(cursor);
		}
		
	gtk_tree_view_columns_autosize((GtkTreeView *)baobab.tree_view); 
	baobab.STOP_SCANNING=TRUE;
	g_queue_free (iterstack);
	baobab.CONTENTS_CHANGED_DELAYED=FALSE;
}


void
start_search (gchar * searchname, gchar * dir)
{
	switch_view(VIEW_SEARCH);
	initialize_search_variables();
	baobab.STOP_SCANNING=FALSE;
	change_icon("busy.gif");
	check_menu_sens(TRUE);
	while (gtk_events_pending()) { gtk_main_iteration(); }
	gtk_list_store_clear(baobab.model_search);

	if (!dir) {
		searchDir("/",searchname);		
	}
	else {
		searchDir(dir,searchname);
	}
	
	change_icon("done.png");
	check_menu_sens(FALSE);
	set_statusbar(_("Ready"));
	set_label_search(baobab.number_found_files,baobab.size_found_files);
	if (get_NB_page()==VIEW_SEARCH) show_label(VIEW_SEARCH);
	gtk_tree_view_columns_autosize((GtkTreeView *)baobab.tree_search); 
	baobab.STOP_SCANNING=TRUE;
	bbSearchOpt.dont_recurse_dir=FALSE;
}
/* fill the search model
*/

void 
fill_search_model(struct BaobabSearchRet *bbret)
{
	
	GtkTreeIter iter;
	GString * testo1;
	GdkPixbuf * picon;
	
	if (!filter_adv_search(bbret)) {
			return;
	}
	
	char* gicon=gnome_icon_lookup_sync(gtk_icon_theme_get_default(),NULL,bbret->fullpath,
								NULL,GNOME_ICON_LOOKUP_FLAGS_NONE,NULL);
	picon= gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),gicon,36,GTK_ICON_LOOKUP_USE_BUILTIN,NULL);
	
	testo1 = g_string_new(bbret->fullpath);
	prepare_firstcol(testo1, bbret);
	
	
	gtk_list_store_append (baobab.model_search, &iter);
	gtk_list_store_set(baobab.model_search, &iter,
					COL0_ICON,picon,
					COL1_STRING,testo1->str,
					//COL2_STRING,testo2->str,
					COL_FULLPATH,bbret->fullpath,
					COL_FILETYPE,get_file_filter(bbret->fullpath),
					COL_SIZE, bbret->size,
					COL_LASTACCESS,bbret->lastacc,
					COL_OWNER, bbret->owner,    
					-1);
	baobab.number_found_files++;
	baobab.size_found_files += bbret->size;
	g_object_unref(picon);
	g_string_free(testo1,TRUE);

}

void 
initialize_search_variables(void)
{
	baobab.number_found_files=0;
	baobab.size_found_files=0;
}


gboolean
filter_adv_search(struct BaobabSearchRet *bbret)
{
	gboolean flag = TRUE;
	GDate * filedate, *today;
	gint days;
	gint maxinterval=100;
	
	
	if ((bbSearchOpt.mod_date==NONE) && bbSearchOpt.size_limit==NONE)
	{return TRUE;}
	
	switch (bbSearchOpt.mod_date) {
		
		case LAST_WEEK:
		case LAST_MONTH:
			filedate=g_date_new();
			today= g_date_new();
			g_date_set_time(filedate,(GTime)bbret->lastacc);
			g_date_set_time(today,time(NULL));
			days = g_date_days_between(filedate,today);
			if (bbSearchOpt.mod_date == LAST_WEEK) {
					maxinterval=7;
					}
			if (bbSearchOpt.mod_date == LAST_MONTH) {
					maxinterval=30;
					}
			if (days > maxinterval) {
				flag = FALSE;
				}
				
			g_date_free(today);
			g_date_free(filedate);
			break;
	}
	
	switch (bbSearchOpt.size_limit) {

		case SIZE_SMALL:
			if (bbret->size >= (guint64)102400) {
				flag = FALSE;
			}
			break;
		
		case SIZE_MEDIUM:
			if (bbret->size >= (guint64)1048576) {
				flag = FALSE;
			}

			break;		
	}
	
	return flag;
}

void
prepare_firstcol(GString* text,struct BaobabSearchRet* bbret)
{
	
	GString* modlabel, * ownlabel, *fullabel, *alloclabel;
	gchar  *basename, *path, *size, *alloc_size;
	
	check_UTF(text);
	basename = g_path_get_basename(text->str);
	path=strdup(text->str);
	modlabel=g_string_new(_("Last Modification:"));
	ownlabel=g_string_new(_("Owner:"));
	fullabel=g_string_new(_("Full path:"));
	alloclabel=g_string_new(_("Allocated bytes:"));
	size = gnome_vfs_format_file_size_for_display(bbret->size);
	alloc_size = gnome_vfs_format_file_size_for_display(bbret->alloc_size);
	g_string_printf(text,"<big><b>%s</b></big>%c<small>%s %s%c%s %s    %s %s%c%s (%s %s) - %s</small>",
			basename,
			'\n',
			fullabel->str,
			path,
			'\n',
			modlabel->str,
			get_last_mod(bbret->lastacc),
			ownlabel->str,
			get_owner(bbret->owner),
			'\n',
			size,
			alloclabel->str,
			alloc_size,
			get_file_filter(bbret->fullpath));
	g_string_free(modlabel,TRUE);
	g_string_free(ownlabel,TRUE);
	g_string_free(fullabel,TRUE);
	g_string_free(alloclabel,TRUE);
	g_free(basename);
	g_free(path);
	g_free(size);
	g_free(alloc_size);

}


gchar*
get_owner(uid_t own_id)
{
	GString * owner;
	struct passwd * pw;
	pw = getpwuid(own_id);
	if (pw==NULL) {
		owner = g_string_new(_("Unknown"));
		}
	else {
		owner = g_string_new(pw->pw_name);
	}

	return g_string_free(owner,FALSE);
}

gchar*
get_last_mod(time_t timeacc)
{
	
	GString * time;
	gchar snum[21];
	struct tm * lt;
	
	lt = localtime(&timeacc);	
	strftime(snum,sizeof(snum),"%Y-%m-%d %R:%S",lt);
	time=g_string_new(snum);
	
	
	return g_string_free(time,FALSE);
}

const gchar*
get_file_filter (gchar* file)
{
	const char* mime_type;
	mime_type= gnome_vfs_get_file_mime_type(file, NULL, FALSE);
	if (mime_type == NULL) {
		return gnome_vfs_mime_get_description (GNOME_VFS_MIME_TYPE_UNKNOWN);
	}
	else {
	return(gnome_vfs_mime_get_description(mime_type));
	}
	
}
/* pre-fills model during scanning
*/
void
prefill_model (gchar* buf)
{
	GString * cdir;
	GtkTreeIter iter, iterparent;
	struct chan_data *data;
	data = (struct chan_data *) buf;

	cdir=g_string_new ("");
	
	if (currentdepth ==-1) {
		gtk_tree_store_append(baobab.model,&iter,NULL);
		firstiter=iter;
	}
	else if(data->depth==1) {
		gtk_tree_store_append(baobab.model,&iter,&firstiter);
		gtk_tree_view_expand_row (
				(GtkTreeView*)baobab.tree_view,gtk_tree_model_get_path(
							GTK_TREE_MODEL(baobab.model),&firstiter),FALSE);
	}
	else if(data->depth > currentdepth) {
		gtk_tree_store_append(baobab.model,&iter,&currentiter);	
	}
	else if (data->depth==currentdepth) {
		gtk_tree_model_iter_parent ((GtkTreeModel*)baobab.model,&iterparent,&currentiter);
		gtk_tree_store_append(baobab.model,&iter,&iterparent);
	}
	else if(data->depth < currentdepth) {
		GtkTreeIter tempiter;
		gint i;
		iter = currentiter;
		for (i=0;i<=(currentdepth-data->depth);i++) {
			gtk_tree_model_iter_parent ((GtkTreeModel*)baobab.model,&tempiter,&iter);
			iter = tempiter;
		}
		gtk_tree_store_append(baobab.model,&iter,&tempiter);
	}
	currentdepth=data->depth;
	push_iter_in_stack(&iter);
	currentiter=iter;

	g_string_assign(cdir, g_filename_display_basename(data->dir));
	
	//check UTF-8 and locale
	check_UTF(cdir);
	g_string_append(cdir," &lt;== ");
	g_string_append(cdir,_("scanning..."));
	gtk_tree_store_set(baobab.model,&iter,COL_DIR_NAME,cdir->str,COL_H_FULLPATH, "",
							COL_H_ELEMENTS,-1,-1);
	while (gtk_events_pending()) { gtk_main_iteration(); }
	g_string_free(cdir,TRUE);
			
}

/* set filesystem first row
*/
void
first_row(void)
{
	char *size;
	gchar textperc[10];
	
	gtk_tree_store_append(baobab.model,&firstiter,NULL);
	size = gnome_vfs_format_file_size_for_display (g_fs.used);
	g_sprintf(textperc," %.1f %%",((gfloat)g_fs.used*100)/(gfloat)g_fs.total);
	
	gtk_tree_store_set(baobab.model,&firstiter,
			COL_DIR_NAME, _("<i>Total filesystem usage:</i>"),	
			COL_H_FULLPATH, "/", 
			COL_BAR, set_bar((g_fs.used*100)/g_fs.total),
			COL_H_PERC, (gfloat)((g_fs.used*100)/g_fs.total),
			COL_DIR_SIZE, size,	
			COL_PERC,textperc,
			COL_H_SIZE, g_fs.used,
			COL_H_ALLOCSIZE, g_fs.used,
			COL_H_ELEMENTS,-1,
			
			-1);
	
	g_free (size);
}


/* fills model during scanning
*  model:
*
*		0, $dir,	                						1, $fullpath,
*		2, set_bar($perc),									3, $perc,
*		4, calc($size),	        							5, $size,
*		6, sprintf("% 4s",$elements)." "._('elements'),		7, $elements,
*		8, text showing hardlink size,						9, $HLsize
*/
void
fill_model (gchar* buf)
{
	GtkTreeIter iter;
	GString *hardlinks;
	GString * basename;
	GString *elementi;
	char *size;
	char *alloc_size;
	
	struct chan_data *data;
	data = (struct chan_data *) buf;
	if (data->elements ==-1) {
			prefill_model(buf);
			return;
			}
	basename= g_string_new ("");
	g_string_assign(basename, g_filename_display_basename(data->dir));
	
	check_UTF(basename);
	g_string_prepend(basename,"<b>");
	g_string_append(basename,"</b>");
				
	iter=pop_iter_from_stack();
	
	hardlinks=g_string_new ("");
	if (data->tempHLsize > 0) {
		size = gnome_vfs_format_file_size_for_display(data->tempHLsize);
		g_string_assign(hardlinks,"<i>(");
		g_string_append(hardlinks,_("contains hardlinks for:"));
		g_string_append(hardlinks," ");
		g_string_append(hardlinks,size);
		g_string_append(hardlinks,")</i>");
		g_free(size);
	}		

	elementi=g_string_new ("");
	g_string_printf(elementi,"% 5d %s",data->elements,
		(data->elements ==1 ?_("object") : _("objects")));
	
	size = gnome_vfs_format_file_size_for_display(data->size);
	alloc_size = gnome_vfs_format_file_size_for_display(data->alloc_size);
	
	gtk_tree_store_set(baobab.model,&iter,
			COL_DIR_NAME, basename->str,			COL_H_FULLPATH, data->dir,
			COL_H_PERC, 0.0,
			COL_DIR_SIZE, baobab.show_allocated ? alloc_size : size,			
			COL_H_SIZE, data->size,
			COL_ELEMENTS, elementi->str,			COL_H_ELEMENTS, data->elements,
			COL_HARDLINK, hardlinks->str,			COL_H_HARDLINK, data->tempHLsize,
			COL_H_ALLOCSIZE,data->alloc_size,
			-1);
	
	while (gtk_events_pending()) { gtk_main_iteration(); }
	g_string_free(hardlinks,TRUE);
	g_string_free(basename,TRUE);
	g_string_free(elementi,TRUE);
	g_free(size);
	g_free(alloc_size);

			
}


///////////////////////////////
void 
push_iter_in_stack (GtkTreeIter *iter)
{

	g_queue_push_head (iterstack,iter->user_data3);
	g_queue_push_head (iterstack,iter->user_data2);
	g_queue_push_head (iterstack,iter->user_data);
	g_queue_push_head (iterstack,GINT_TO_POINTER(iter->stamp));
}

///////////////////////////////
GtkTreeIter
pop_iter_from_stack (void)
{
	GtkTreeIter iter;
	
	iter.stamp      =  GPOINTER_TO_INT(g_queue_pop_head(iterstack));
	iter.user_data  =  g_queue_pop_head(iterstack);
	iter.user_data2 =  g_queue_pop_head(iterstack);
	iter.user_data3 =  g_queue_pop_head(iterstack);
	return iter;
}

///////////////////////////////////
void
check_UTF(GString * name)
{
	g_string_assign(name,g_filename_to_utf8(name->str,-1,NULL,NULL,NULL));
	if (!name->str) {
			g_string_assign(name,g_locale_to_utf8(name->str,-1,NULL,NULL,NULL));
			if (!name->str)		{
				g_string_assign(name,"<i>");
				g_string_append(name, _("Invalid UTF-8 characters"));
				g_string_append(name,"</i>");
			}
		}
	
	g_string_assign(name,g_markup_escape_text (name->str,name->len));
}





void
baobab_init(void)
{
	baobab.label_scan = NULL;
	baobab.label_search = NULL;
	baobab.last_scan_command = g_string_new("/");
	baobab.CONTENTS_CHANGED_DELAYED=FALSE;
	baobab.STOP_SCANNING= TRUE;
	baobab.show_allocated=TRUE;

	//GConf
	GConfClient * gc_client =gconf_client_get_default();
	gconf_client_add_dir(gc_client,BAOBAB_KEY, GCONF_CLIENT_PRELOAD_NONE,NULL);
	gconf_client_notify_add(gc_client,PROPS_SCAN_KEY,props_notify,NULL,NULL,NULL);
	baobab.bbExcludedDirs = gconf_client_get_list(gc_client,PROPS_SCAN_KEY,GCONF_VALUE_STRING,NULL);
    baobab.bbEnableHomeMonitor = gconf_client_get_bool(gc_client,PROPS_ENABLE_HOME_MONITOR,NULL);
	
	//initialize bbSearchOpt variables
	bbSearchOpt.filename = g_string_new("");
	bbSearchOpt.dir = g_string_new(g_get_home_dir());
	bbSearchOpt.mod_date=NONE;
	bbSearchOpt.size_limit=NONE;
	bbSearchOpt.exact=TRUE;
	bbSearchOpt.search_whole=FALSE;
	bbSearchOpt.dont_recurse_dir=FALSE;
	
	//start VFS monitoring
	handle_home=NULL;	

	GnomeVFSVolumeMonitor* volmonitor=gnome_vfs_get_volume_monitor();
	 g_signal_connect (volmonitor, "volume_mounted",
		    G_CALLBACK (volume_changed), NULL);
  	g_signal_connect (volmonitor, "volume_unmounted",
		    G_CALLBACK (volume_changed), NULL);
	
	GnomeVFSResult result =gnome_vfs_monitor_add(&handle_home,g_get_home_dir(),
										GNOME_VFS_MONITOR_DIRECTORY,
										contents_changed_cb,NULL);
	
	if (result != GNOME_VFS_OK) {
		message(_("Cannot initialize GNOME VFS monitoring\n"
				  "Some real-time auto-detect function will not be available!"),NULL);
			g_print("homedir:%s\n",gnome_vfs_result_to_string (result));

	}

}

void
baobab_close(void)
{
		g_free(baobab.label_scan);
		g_free(baobab.label_search);
		g_string_free(baobab.last_scan_command,TRUE);
		g_string_free(bbSearchOpt.filename,TRUE);
		g_string_free(bbSearchOpt.dir,TRUE);
	
	if (handle_mtab) {
		gnome_vfs_monitor_cancel(handle_mtab);
	}
	if (handle_home) {
		gnome_vfs_monitor_cancel(handle_home);
	}
	
	g_slist_foreach(baobab.bbExcludedDirs,(GFunc)g_free,NULL);
	g_slist_free(baobab.bbExcludedDirs);
	gnome_vfs_shutdown();
}
////////////////////////////////main///////////////////////////////
int
main (int argc, char *argv[])
{
 

#ifdef ENABLE_NLS
  setlocale(LC_MESSAGES, "");
  bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
//  bindtextdomain(GETTEXT_PACKAGE,"/usr/share/locale");
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif


  	gtk_set_locale ();
  	//gtk_init (&argc, &argv);
	gnome_program_init(PACKAGE, VERSION, LIBGNOMEUI_MODULE,
	                            argc, argv,GNOME_PARAM_NONE);
	g_type_init();
	
	
	/* remember to initialize GnomeVFS! */
  	if (!gnome_vfs_init ()) {
    	printf ("Could not initialize GnomeVFS\n");
    	return 1;
  	}
	
	add_pixmap_directory(PIX_DIR);
	
	//initialize variables
	baobab_init();
	
	g_fs=getFileSystem();
	set_label_scan(&g_fs);
	
  	baobab.window = create_window ();
	gtk_window_set_position(GTK_WINDOW(baobab.window),GTK_WIN_POS_CENTER);
  	GdkPixbuf * window_icon = create_pixbuf("baobab.png");
  	gtk_window_set_icon(GTK_WINDOW(baobab.window),window_icon);	
  	gtk_window_set_default_size(GTK_WINDOW(baobab.window),-1,500);
	
	//set global pixbufs
	baobab.yellow_bar=create_pixbuf("yellow.png");
	baobab.red_bar=create_pixbuf("red.png");
	baobab.green_bar=create_pixbuf("green.png");
	
	baobab.tree_view = create_treeview();
	baobab.tree_search = create_tree_search();
	set_label_search(0,0);
	
	switch_view(VIEW_SEARCH);
	switch_view(VIEW_TREE);
	show_label(VIEW_TREE);
	change_icon("done.png");
		
	//set allocated space checkbox
	gtk_toggle_button_set_active((GtkToggleButton*)lookup_widget(baobab.window,"ck_allocated"),
								baobab.show_allocated);
								
	gtk_widget_show (baobab.window);

	first_row();
	set_statusbar(_("Ready"));
	
	
	if (install_naut()){
		message(_("A new script has been generated to open Baobab from Nautilus. Just right-click in Nautilus->Script to access Baobab. "),baobab.window);
	}
	
	//commandline
	if (argc > 1){
		start_proc_on_dir(argv[1]);
		}
		
			
 	gtk_main ();
	
	//free variables;
	baobab_close();

return 0;
}
