/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastient.laout@tuxfamily.org                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qlabel.h>
#include <kurl.h>
#include <qlayout.h>
#include <kiconloader.h>
#include <qcursor.h>
#include <klocale.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <kcolorbutton.h>

#include "linklabel.h"
#include "variouswidgets.h"

/** LinkLook */

LinkLook *LinkLook::localLook = 0;
LinkLook *LinkLook::urlLook   = 0;
LinkLook *LinkLook::noUrlLook = 0;

LinkLook::LinkLook(bool underline, bool italic, bool bold, bool hoverUnder,
                   QColor color, QColor hoverColor,
                   bool showIcon, int iconSize, bool onTopIcon)
{
	setLook( underline, italic, bold, hoverUnder, color, hoverColor, showIcon, iconSize, onTopIcon );
}

void LinkLook::setLook(bool underline, bool italic, bool bold, bool hoverUnder,
                       QColor color, QColor hoverColor,
                       bool showIcon, int iconSize, bool onTopIcon)
{
	m_underline  = underline;
	m_italic     = italic;
	m_bold       = bold;
	m_hoverUnder = hoverUnder;
	m_color      = color;
	m_hoverColor = hoverColor;
	m_showIcon   = showIcon;
	m_iconSize   = iconSize;
	m_onTopIcon  = onTopIcon;
}

LinkLook* LinkLook::lookForURL(const KURL &url)
{
	if (url.isEmpty())
		return noUrlLook;
	return url.isLocalFile() ? localLook : urlLook;
}

/** LinkLabel */

LinkLabel::LinkLabel(int hAlign, int vAlign, QWidget *parent, const char *name, WFlags f)
 : QFrame(parent, name, f), m_look(0)
{
	initLabel(hAlign, vAlign);
}

LinkLabel::LinkLabel(const QString &title, const QString &icon, LinkLook *look, int hAlign, int vAlign,
                     QWidget *parent, const char *name, WFlags f)
 : QFrame(parent, name, f), m_look(0)
{
	initLabel(hAlign, vAlign);
	setLink(title, icon, look);
}

void LinkLabel::initLabel(int hAlign, int vAlign)
{
	m_layout  = new QBoxLayout(this, QBoxLayout::LeftToRight);
	m_icon    = new QLabel(this);
	m_title   = new QLabel(this);
	m_spacer1 = new QSpacerItem(0, 0, QSizePolicy::Preferred/*Expanding*/, QSizePolicy::Preferred/*Expanding*/);
	m_spacer2 = new QSpacerItem(0, 0, QSizePolicy::Preferred/*Expanding*/, QSizePolicy::Preferred/*Expanding*/);

	m_hAlign = hAlign;
	m_vAlign = vAlign;
	setCursor(QCursor(Qt::PointingHandCursor));

	// DEGUB:
	//m_icon->setPaletteBackgroundColor("lightblue");
	//m_title->setPaletteBackgroundColor("lightyellow");
}

LinkLabel::~LinkLabel()
{
}

void LinkLabel::setLink(const QString &title, const QString &icon, LinkLook *look)
{
	KIconLoader icol;
	QPixmap     ico;

	if (look)
		m_look = look;   // Needed for icon size

	ico = icol.loadIcon( icon, KIcon::NoGroup, m_look->iconSize(), true );
	m_title->setText(title);
	m_icon->setPixmap(ico);
	if (icon.isEmpty()) {
		m_icon->clear();
		m_icon->hide();
	} else
		m_icon->show();
	if (title.isEmpty())
		m_title->hide(); // Already clear
	else
		m_title->show();

	if (look)
		setLook(look);
}

void LinkLabel::setLook(LinkLook *look)
{
	m_look = look;

	QFont font;
	font.setBold( look->bold() );
	font.setUnderline( look->underline() );
	font.setItalic( look->italic() );
	m_title->setFont( font );
	m_title->setPaletteForegroundColor( look->color() );

	if ( ! look->showIcon() ) {
		m_icon->clear();
		m_icon->hide();
	} else if ( m_icon->pixmap() && ! m_icon->pixmap()->isNull() )
		m_icon->show();

	setAlign(m_hAlign, m_vAlign);
}

void LinkLabel::setAlign(int hAlign, int vAlign)
{
	m_hAlign = hAlign;
	m_vAlign = vAlign;

	if (!m_look)
		return;

	// Define alignment flags :
	//FIXME TODO: Use directly flags !
	int hFlag, vFlag, wBreak;
	switch (hAlign) {
		default:
		case 0: hFlag = Qt::AlignLeft;    break;
		case 1: hFlag = Qt::AlignHCenter; break;
		case 2: hFlag = Qt::AlignRight;   break;
	}
	switch (vAlign) {
		case 0: vFlag = Qt::AlignTop;     break;
		default:
		case 1: vFlag = Qt::AlignVCenter; break;
		case 2: vFlag = Qt::AlignBottom;  break;
	}
	wBreak = Qt::WordBreak * !(!m_look->onTopIcon() && hAlign == 1);

	// Clear the widget :
	m_layout->removeItem(m_spacer1);
	m_layout->remove(m_icon);
	m_layout->remove(m_title);
	m_layout->removeItem(m_spacer2);

	// Otherwise, minimumSize will be incoherent (last size ? )
	m_layout->setResizeMode(QLayout::Minimum);

	// And re-populate the widget with the appropriates things and order
	bool addSpacers = m_look->onTopIcon() || hAlign == 1;
	if (m_look->onTopIcon()) {
		m_layout->setDirection(QBoxLayout::TopToBottom);
		//m_title->setSizePolicy( QSizePolicy(QSizePolicy::Preferred/*Expanding*/, QSizePolicy::Fixed, 0, 0, false) );
		m_icon->setSizePolicy( QSizePolicy(QSizePolicy::Preferred/*Expanding*/, QSizePolicy::Fixed, 0, 0, false) );
		m_spacer1->changeSize( 0, 0, QSizePolicy::Preferred, QSizePolicy::Preferred );
		m_spacer2->changeSize( 0, 0, QSizePolicy::Preferred, QSizePolicy::Preferred );
	} else {
		m_layout->setDirection(QBoxLayout::LeftToRight);
		//m_title->setSizePolicy( QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Maximum/*Expanding*/, 0, 0, false) );
		m_icon->setSizePolicy( QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Preferred/*Expanding*/, 0, 0, false) );
		m_spacer1->changeSize( 0, 0, QSizePolicy::Expanding, QSizePolicy::Preferred/*Expanding*/ );
		m_spacer2->changeSize( 0, 0, QSizePolicy::Expanding, QSizePolicy::Preferred/*Expanding*/ );
	}
	m_icon->setAlignment( hFlag | vFlag );
	m_title->setAlignment( hFlag | vFlag | wBreak );
	if ( addSpacers && !(m_look->onTopIcon() && vAlign == 0) ||
	   (!m_look->onTopIcon() && m_title->text().isEmpty() && hAlign == 2) )
		m_layout->addItem(m_spacer1);
	if ( !m_look->onTopIcon() && hAlign == 2 ) { // If align at right, icon is at right
		m_layout->addWidget(m_title);
		m_layout->addWidget(m_icon);
	} else {
		m_layout->addWidget(m_icon);
		m_layout->addWidget(m_title);
	}
	if ( addSpacers && !(m_look->onTopIcon() && vAlign == 2) ||
	   (!m_look->onTopIcon() && m_title->text().isEmpty() && hAlign == 0) )
		m_layout->addItem(m_spacer2);
}

void LinkLabel::enterEvent(QEvent*)
{
	m_title->setPaletteForegroundColor(m_look->hoverColor());

	if (m_look->hoverUnder()) {
		QFont font = m_title->font();
		font.setUnderline( ! m_look->underline() );
		m_title->setFont( font );
	}
}

void LinkLabel::leaveEvent(QEvent*)
{
	m_title->setPaletteForegroundColor(m_look->color());

	if (m_look->hoverUnder()) {
		QFont font = m_title->font();
		font.setUnderline( m_look->underline() );
		m_title->setFont( font );
	}
}

void LinkLabel::setPaletteBackgroundColor(const QColor &color)
{
	QFrame::setPaletteBackgroundColor(color);
	m_title->setPaletteBackgroundColor(color);
}

/** LinkLookEditWidget **/

LinkLookEditWidget::LinkLookEditWidget(LinkLook *look, QWidget *parent, const char *name, WFlags fl)
 : QWidget(parent, name, fl)
{
	QSpacerItem *spacer;
	QLabel      *label;
	QVBoxLayout *layout = new QVBoxLayout(this, 0, 6);

	m_look = look;

	QHBoxLayout *lay1 = new QHBoxLayout(0, 0, 6);
	label = new QLabel(i18n("Title :"), this);
	m_underline = new QPushButton(i18n("&Underline"), this);
	m_underline->setToggleButton(true);
	m_underline->setOn(look->underline());
	m_italic = new QPushButton(i18n("&Italic"), this);
	m_italic->setToggleButton(true);
	m_italic->setOn(look->italic());
	m_bold = new QPushButton(i18n("&Bold"), this);
	m_bold->setToggleButton(true);
	m_bold->setOn(look->bold());
	m_color = new KColorButton(this);
	m_color->setColor(look->color());
	spacer = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum);
	lay1->addWidget(label);
	lay1->addWidget(m_underline);
	lay1->addWidget(m_italic);
	lay1->addWidget(m_bold);
	lay1->addWidget(m_color);
	lay1->addItem(spacer);
	layout->addLayout(lay1);

	QHBoxLayout *lay2 = new QHBoxLayout(0, 0, 6);
	m_hoverColor = new KColorButton(this);
	label = new QLabel(m_hoverColor, i18n("Mouse hover &color :"), this);
	m_hoverColor->setColor(look->hoverColor());
	spacer = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum);
	lay2->addWidget(label);
	lay2->addWidget(m_hoverColor);
	lay2->addItem(spacer);
	layout->addLayout(lay2);

	m_hoverUnder = new QCheckBox(i18n("(&des-)underline title at mouse hover"), this);
	m_hoverUnder->setChecked(look->hoverUnder());
	layout->addWidget(m_hoverUnder);

	QHBoxLayout *lay3 = new QHBoxLayout(0, 0, 6);
	m_iconProperties = new QWidget(this);
	QHBoxLayout *icoPropLay = new QHBoxLayout(m_iconProperties, 0, 6);
	m_showIcon = new QCheckBox(i18n("&Show icon :"), this);
	m_showIcon->setChecked(look->showIcon());
	m_iconProperties->setEnabled(m_showIcon->isChecked());
	m_onTopIcon = new QComboBox(false, m_iconProperties);
	m_onTopIcon->insertItem(i18n("on side"));
	m_onTopIcon->insertItem(i18n("on top"));
	m_onTopIcon->setCurrentItem(look->onTopIcon());
	m_iconSize = new IconSizeCombo(false, m_iconProperties);
	m_iconSize->setSize(look->iconSize());
	spacer = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum);
	lay3->addWidget(m_showIcon);
	icoPropLay->addWidget(m_onTopIcon);
	icoPropLay->addWidget(m_iconSize);
	lay3->addWidget(m_iconProperties);
	lay3->addItem(spacer);
	layout->addLayout(lay3);

	connect( m_showIcon, SIGNAL(toggled(bool)), this, SLOT(slotShowIcon(bool)) );
}

LinkLookEditWidget::~LinkLookEditWidget()
{
}

void LinkLookEditWidget::saveChanges()
{
	m_look->setLook( m_underline->isOn(), m_italic->isOn(), m_bold->isOn(), m_hoverUnder->isChecked(),
	                 m_color->color(), m_hoverColor->color(),
	                 m_showIcon->isChecked(), m_iconSize->iconSize(), m_onTopIcon->currentItem() );
}

void LinkLookEditWidget::slotShowIcon(bool show)
{
	m_iconProperties->setEnabled(show);
}

#include "linklabel.moc"
